﻿using System.Linq;
using System.Windows.Forms;
using App.Controls;
using App.Data;
using App.Utility;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit
{
    public partial class TextureUsePage : TexturePropertyPage
    {
        public TextureUsePage() :
            base(PropertyPageID.TextureRoot)
        {
            InitializeComponent();

            // 内部でスクロールを管理するので親はスクロールさせない
            HideParentScrollBar = true;
        }

        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_texture_property_window_usage_page;
            }
        }

        public static ObjectPropertyPage CreateInstance(object arg)
        {
            return new TextureUsePage();
        }

        protected override void InitializeFormInternal()
        {
        }

        private class SamplerTriple
        {
            public Model model;
            public Material material;
            public samplerType sampler;
        }

        private class Pattern
        {
            public IHasTexturePatternAnimation animation;
            public int patternIndex;
            public string material;
            public string sampler;
        }

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            string name = ActiveTarget.Name;
            using (var ub = new UpdateBlock(lvwSampler))
            {
                var triples = (from model in DocumentManager.Models.OrderBy(x => x.Name)
                               where model.ReferenceTexturePaths.ContainsKey(name) &&
                                     string.Compare(model.ReferenceTexturePaths[name], ActiveTarget.FilePath, true) == 0
                                from material in model.Materials
                                from sampler in material.ResolvedSamplers
                               where sampler.tex_name == name
                                select new SamplerTriple() { model=model, material=material, sampler=sampler }).ToArray();

                lvwSampler.SetItemCount(triples.Length);

                int index = 0;
                foreach (var triple in triples)
                {
                    var model = triple.model;
                    var material = triple.material;
                    var sampler = triple.sampler;
                    var item = lvwSampler.Items[index];

                    item.Tag = triple;
                    item.SubItems[clhModel.Index].Text = model.Name;
                    item.SubItems[clhMaterial.Index].Text = material.Name;
                    item.SubItems[clhName.Index].Text = sampler.name;
                    item.SubItems[clhHint.Index].Text = sampler.hint;

                    item.SubItems[clhWrapU.Index].Text = UIText.EnumValue(sampler.wrap.u);
                    item.SubItems[clhWrapU.Index].Tag = sampler.wrap.u;
                    item.SubItems[clhWrapV.Index].Text = UIText.EnumValue(sampler.wrap.v);
                    item.SubItems[clhWrapV.Index].Tag = sampler.wrap.v;
                    item.SubItems[clhWrapW.Index].Text = UIText.EnumValue(sampler.wrap.w);
                    item.SubItems[clhWrapW.Index].Tag = sampler.wrap.w;
                    item.SubItems[clhFilterMag.Index].Text = UIText.EnumValue(sampler.filter.mag);
                    item.SubItems[clhFilterMag.Index].Tag = sampler.filter.mag;
                    item.SubItems[clhFilterMin.Index].Text = UIText.EnumValue(sampler.filter.min);
                    item.SubItems[clhFilterMin.Index].Tag = sampler.filter.min;
                    item.SubItems[clhFilterMip.Index].Text = UIText.EnumValue(sampler.filter.mip);
                    item.SubItems[clhFilterMip.Index].Tag = sampler.filter.mip;
                    item.SubItems[clhFilterAniso.Index].Text = UIText.EnumValue(sampler.filter.max_aniso);
                    item.SubItems[clhFilterAniso.Index].Tag = sampler.filter.max_aniso;
                    item.SubItems[clhLodMin.Index].Text = sampler.lod.min.ToString();
                    item.SubItems[clhLodMin.Index].Tag = sampler.lod.min;
                    item.SubItems[clhLodMax.Index].Text = sampler.lod.max.ToString();
                    item.SubItems[clhLodMax.Index].Tag = sampler.lod.max;
                    item.SubItems[clhLodBias.Index].Text = sampler.lod.bias.ToString();
                    item.SubItems[clhLodBias.Index].Tag = sampler.lod.bias;

                    ++index;
                }
            }

            using (var ub = new UpdateBlock(lvwTexturePatternAnim))
            {
                var patterns = (from animation in DocumentManager.Animations.OfType<IHasTexturePatternAnimation>().OrderBy(x => ((AnimationDocument)x).Name)
                                from patternIndex in Enumerable.Range(0, animation.TexPatterns.Count)
                                let pattern = animation.TexPatterns[patternIndex]
                                where pattern.tex_name == name &&
                                    animation.ReferenceTexturePaths.ContainsKey(name) &&
                                    string.Compare(animation.ReferenceTexturePaths[name], ActiveTarget.FilePath, true) == 0
                                let value = (float)patternIndex
                                let curves = (from material in animation.ITexPatternMatAnims
                                              from target in material.PatternAnimTargets
                                              where target.KeyFrames.Any(x => x.Value == value) ||
                                                   (target.GetBaseValue() == value &&
                                                    target.KeyFrames.Any() &&
                                                    (target.KeyFrames[0].Frame > 0 ||
                                                     target.KeyFrames[target.KeyFrames.Count - 1].Frame < ((AnimationDocument)animation).FrameCount))
                                              select new { material, target }).DefaultIfEmpty()
                                from curve in curves
                                select new Pattern {
                                    animation = animation,
                                    patternIndex = patternIndex,
                                    material = (curve != null)? curve.material.mat_name:"",
                                    sampler = curve != null? curve.target.sampler_name: "" }).ToArray();

                lvwTexturePatternAnim.SetItemCount(patterns.Length);

                int index = 0;
                foreach (var pattern in patterns)
                {
                    var item = lvwTexturePatternAnim.Items[index];

                    item.Tag = pattern;
                    var animation = (AnimationDocument)pattern.animation;
                    var fileText = animation.FileName + DocumentManager.GetSameNameIndexText(animation, true);
                    item.SubItems[clhAnimation.Index].Text = fileText;

                    item.SubItems[clhPattern.Index].Text = pattern.patternIndex.ToString();
                    item.SubItems[clhPatternMaterial.Index].Text = pattern.material;
                    item.SubItems[clhPatternSampler.Index].Text = pattern.sampler;

                    index++;
                }
            }
        }

        public override bool ChangeDockStyle
        {
            get
            {
                return false;
            }
        }

        #region コピー＆ペースト
        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public override bool CanCopy()
        {
            return false;
        }

        /// <summary>
        /// ペースト可能か
        /// </summary>
        public override bool CanPaste(object copiedObjectInfo, object copiedObject)
        {
            return false;
        }
        #endregion

        private void lvwList_MouseDoubleClick(object sender, MouseEventArgs e)
        {
            var triple = lvwSampler.SelectedItemData as SamplerTriple;
            if (triple != null)
            {
                using (var wait = new WaitCursor())
                {
                    // プロパティウィンドウを生成
                    var objectGroup = new GuiObjectGroup(triple.material);

                    ObjectPropertyDialog dialog = new ObjectPropertyDialog(null, objectGroup);
                    dialog.TargetFixed = true;

                    // 現在のウィンドウの上に表示する
                    Control parent = Owner;
                    while (parent != null)
                    {
                        if (parent is ObjectPropertyDialog)
                        {
                            dialog.SetLocationNextTo((ObjectPropertyDialog)parent);
                            break;
                        }

                        parent = parent.Parent;
                    }

                    // サンプラを選択
                    var materialPropertyPanel = (MaterialPropertyPanel)dialog.GetPropertyPanel(ObjectPropertyDialog.PanelID.Material);
                    materialPropertyPanel.SelectSampler(triple.sampler);
                    dialog.Show();
                }
            }
        }

        private void lvwTexturePatternAnim_MouseDoubleClick(object sender, MouseEventArgs e)
        {
            var pattern = lvwTexturePatternAnim.SelectedItemData as Pattern;
            if (pattern != null)
            {
                using (var wait = new WaitCursor())
                {
                    // プロパティウィンドウを生成
                    var objectGroup = new GuiObjectGroup((AnimationDocument)pattern.animation);

                    ObjectPropertyDialog dialog = new ObjectPropertyDialog(null, objectGroup);
                    dialog.TargetFixed = true;

                    // 現在のウィンドウの上に表示する
                    Control parent = Owner;
                    while (parent != null)
                    {
                        if (parent is ObjectPropertyDialog)
                        {
                            dialog.SetLocationNextTo((ObjectPropertyDialog)parent);
                            break;
                        }

                        parent = parent.Parent;
                    }
                    IHasTexturePatternCurvePage patternAnimationPanel = null;
                    // サンプラを選択
                    switch (((AnimationDocument)pattern.animation).ObjectID)
                    {
                        case ConfigCommon.GuiObjectID.TexturePatternAnimation:
                            patternAnimationPanel = (IHasTexturePatternCurvePage)dialog.GetPropertyPanel(ObjectPropertyDialog.PanelID.TexturePatternAnimation);
                            break;
                        case ConfigCommon.GuiObjectID.MaterialAnimation:
                            patternAnimationPanel = (IHasTexturePatternCurvePage)dialog.GetPropertyPanel(ObjectPropertyDialog.PanelID.MaterialAnimation);
                            break;
                    }

                    dialog.Show();

                    // 先に選択するとツールストリップのレイアウトが崩れるので Show のあと
                    {
                        var material = DocumentManager.Materials.FirstOrDefault(x => x.Name == pattern.material);

                        if (material != null)
                        {
                            // 選択されているマテリアル。
                            var selectedMaterials = App.AppContext.SelectedTarget.GetObjects(ConfigCommon.GuiObjectID.Material).Select(x => x as Material);

                            // 現時点のカーブエディタでは、ひとつのファイルしか開けないので、
                            // このページのマテリアルのファイルと同じものに限定。
                            var sameFileMaterials = selectedMaterials.Where(x => x.OwnerDocument == material.OwnerDocument);

                            // リスト先頭のマテリアルカーブがツリーノードで選択状態となるので
                            // このページのマテリアルを先頭に配置しておく。
                            var materialNames = sameFileMaterials.Select(x => x.Name).Distinct();
                            materialNames = Enumerable.Repeat(material.Name, 1).Concat(materialNames.Where(x => x != material.Name));

                            patternAnimationPanel.SelectTexturePatternCurvePage(materialNames, pattern.sampler);
                        }
                        else
                        {
                            patternAnimationPanel.SelectTexturePatternCurvePage(Enumerable.Repeat(pattern.material, 1), pattern.sampler);
                        }
                    }

                    // 表示後にフィット
                    patternAnimationPanel.FitCurvePage();
                }
            }
        }
    }
}
