﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using App.Controls;
using App.Data;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit.ShaderParamControls
{
    public partial class EditEnum : ShaderParamControl
    {
        private readonly int optionCount = 0;
        public override ParamType ParamType
        {
            get { return ParamType.option_var; }
        }

        string[] options;
        Dictionary<string, string> aliases;
        public EditEnum(option_varType option)
        {
            InitializeComponent();

            ParamName = option.id;
            cbxValue.Tag = (uint)(1 << 0);
            options = IfShaderOptionUtility.Enum(option.choice);

            aliases = IfShaderOptionUtility.Alias(option.choice);

            optionCount = options.Count();
        }

        ChoiceHelper choiceHelper = new ChoiceHelper();
        public const int minTextWidth = 100;
        public const int textMargin = 30;
        public override bool SetValue(Material material, string value, CustomUI customUI, Definition.ShadingModelTable table, Predicate<string> visibleGroups, HashSet<string> visiblePages, bool showId, bool showOriginalLabel)
        {
            choiceHelper.SetLabel(aliases, customUI, table, visibleGroups, visiblePages, showId, showOriginalLabel);

            var data = options.Concat(Enumerable.Repeat(value, 1)).Distinct().ToArray();
            if (data.Count() != cbxValue.Items.Count ||
                !data.Zip(cbxValue.Items.OfType<UIListControlItem>(), (x, y) => (x == (string)y.Data && (choiceHelper.labels.ContainsKey(x) ? choiceHelper.labels[x] : x) == y.Text)).All(x => x))
            {
                cbxValue.BeginUpdate();
                cbxValue.Items.Clear();
                foreach (var d in data)
                {
                    string text = choiceHelper.labels.ContainsKey(d) ? choiceHelper.labels[d] : d;
                    cbxValue.AddItem(text, d);
                }

                cbxValue.EndUpdate();
            }

            cbxValue.SelectedItemData = value;

            bool updated = false;
            using (var g = CreateGraphics())
            {
                int width = Math.Max(
                    minTextWidth,
                    cbxValue.Items.OfType<UIListControlItem>().Max(x => (int)g.MeasureString(x.Text, cbxValue.Font).Width) + textMargin);

                if (cbxValue.Width != width)
                {
                    cbxValue.Width = width;
                    updated = true;
                }
            }

            return UpdateLinkButtons(updated);
        }

        LinkButton linkButton;
        public bool UpdateLinkButtons(bool updated)
        {
            var hashSet = choiceHelper.LabelHelpers.Where(x => x.Value.linkTarget != null);
            if (hashSet.Any())
            {
                if (linkButton == null)
                {
                    linkButton = LabelHelper.CreateLinkButton();
                    linkButton.Click += (s, e) =>
                        {
                            var value = (string)cbxValue.SelectedItemData;
                            var helper = choiceHelper.LabelHelpers[value];
                            OnLinkClicked(helper.ClickArgs());
                        };
                    Controls.Add(linkButton);
                    updated = true;
                }

                if (updated)
                {
                    linkButton.Location = new Point(cbxValue.Right + 5, cbxValue.Top + 2);
                }

                {
                    var data = (string)cbxValue.SelectedItemData;
                    LabelHelper helper;
                    linkButton.Enabled = choiceHelper.LabelHelpers.TryGetValue(data, out helper) &&
                        helper.linkTarget != null &&
                        !helper.LinkError;
                }
            }
            else if (linkButton != null)
            {
                Controls.Remove(linkButton);
                linkButton.Dispose();
                linkButton = null;
                updated = true;
            }

            UpdateLinkButtonToolTip(false);

            return updated;
        }

        public override void UpdateLinkToolTip(bool force)
        {
            base.UpdateLinkToolTip(force);

        }

        private void UpdateLinkButtonToolTip(bool force)
        {
            var data = (string)cbxValue.SelectedItemData;
            LabelHelper helper;
            choiceHelper.LabelHelpers.TryGetValue(data, out helper);
            var toolTip = helper != null ? helper.GetLinkLabelString() : "";
            var current = parentHint.GetToolTip(linkButton);
            if (toolTip != current || force)
            {
                parentHint.SetToolTip(linkButton, toolTip);
            }
        }
        private void cbxValue_SelectedIndexChanged(object sender, EventArgs e)
        {
            OptionValueChangedEventArgs args = new OptionValueChangedEventArgs();
            {
                args.ParamName	= ParamName;
                args.ParamValue = cbxValue.SelectedItemData;
            }

            InvokeValueChanged(this, args);
        }

        private void cbxValue_CustomDrawItem(object sender, CustomDrawListControlItemEventArgs e)
        {
            if (e.Item != null)
            {
                LabelHelper helper = null;
                if (e.Item.Data is string)
                {
                    choiceHelper.LabelHelpers.TryGetValue((string)e.Item.Data, out helper);
                }
                e.ForeColor = helper != null && helper.color.HasValue ? helper.color.Value:
                    e.Index < optionCount ? SystemColors.ControlText : Color.Red;
                e.BackColor = helper != null && helper.backGroundColor.HasValue ? helper.backGroundColor.Value : SystemColors.Window;
            }
        }
    }
}
