﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Windows.Forms;
using App.Command;
using App.ConfigData;
using App.Controls;
using App.Data;
using App.FileView;
using App.Utility;
using App.res;
using ConfigCommon;
using nw.g3d.nw4f_3dif;
using nw.g3d.iflib.nw3de;
using System.Text.RegularExpressions;
using App.PropertyEdit.ShaderParamControls;

namespace App.PropertyEdit
{
    public partial class MaterialSamplerPage : MaterialPropertyPage
    {
        // static変数
        static CopyData_SingleSampler OneSamplerCopied_ = null; // 行単位でのコピーが行われているか？

        // 静的コンストラクタ
        static MaterialSamplerPage()
        {
        }

        // ３２サイズイメージリスト
        private static readonly ImageList iml32Size_ = new ImageList()
        {
            ImageSize = new Size(32 + 2, 32 + 2),
            ColorDepth = ColorDepth.Depth8Bit,
            TransparentColor = Color.Magenta,
        };

        // ６４サイズイメージリスト
        private static readonly ImageList iml64Size_ = new ImageList()
        {
            ImageSize = new Size(64 + 2, 64 + 2),
            ColorDepth = ColorDepth.Depth8Bit,
            TransparentColor = Color.Magenta,
        };

        public MaterialSamplerPage() :
            base(PropertyPageID.MaterialSampler)
        {
            InitializeComponent();

            // 行の高さを変更
            lvwList.SmallImageList = iml32Size_;
            lvwList.LargeImageList = iml64Size_;

            // リストビューのサイズ調整のために、ヘッダサイズとスクロールバーサイズを取得。
            var lvwHeader = ListViewUtility.GetHeader(lvwList);
            Win32.RECT rcHeader = new Win32.RECT();
            Win32.NativeMethods.GetWindowRect(lvwHeader, ref rcHeader);
            var lvwHeaderWidth = Enumerable.Range(0, lvwList.Columns.Count).Select(x => lvwList.Columns[x].Width).Sum();
            var lvwHeaderHeight = rcHeader.bottom - rcHeader.top;
            var hsbHeight = Win32.NativeMethods.GetSystemMetrics(Win32.NativeMethods.SystemMetric.SM_CXHSCROLL);
            var vsbWidth = Win32.NativeMethods.GetSystemMetrics(Win32.NativeMethods.SystemMetric.SM_CXVSCROLL);

            // リストビューのサイズを調整。
            // 幅: ヘッダがすべて表示される大きさ
            // 高さ: サンプラが 6 個表示される大きさ
            var lvwListClientSize = new Size(
                lvwHeaderWidth + vsbWidth + 4,
                lvwHeaderHeight + iml32Size_.ImageSize.Height * 6 + hsbHeight + 4);
            if (lvwList.ClientSize.Width < lvwListClientSize.Width)
            {
                var add = lvwListClientSize.Width - lvwList.ClientSize.Width;

                // ヘッダがすべて収まるように幅を調整。
                // lvwList の右端はこのクラスの横幅で拡縮されるので、
                // lvwList の幅を直接変更するのではなく、このクラスの幅で変更する。
                Width += add;
            }
            if (lvwList.ClientSize.Height < lvwListClientSize.Height)
            {
                var add = lvwListClientSize.Height - lvwList.ClientSize.Height;

                // リストビューのサイズを調整。
                lvwList.ClientSize = new Size(lvwList.ClientSize.Width, lvwListClientSize.Height);

                // リストビューより下に配置されているコントロールを下にずらす。
                foreach (var ctrl in Controls.OfType<Control>().Where(x => (x != lvwList) && (x.Location.Y > lvwList.Location.Y)))
                {
                    ctrl.Location = new Point(ctrl.Location.X, ctrl.Location.Y + add);
                }

                // 自身の高さを調整。
                Height += add;
            }

            // デザイナが丸めてしまうのでここで値を設定します。
            iepLodBias.Maximum = 31.984375F;

            // サンプラパラメータのマテリアル参照設定用設定
            SetLabelForMaterialReference(mlbTexture, Material.SamplerParamID.tex_name);
            SetLabelForMaterialReference(mlbWrapU, Material.SamplerParamID.wrap_u);
            SetLabelForMaterialReference(mlbWrapV, Material.SamplerParamID.wrap_v);
            SetLabelForMaterialReference(mlbWrapW, Material.SamplerParamID.wrap_w);
            SetLabelForMaterialReference(mlbFilterMag, Material.SamplerParamID.filter_mag);
            SetLabelForMaterialReference(mlbFilterMin, Material.SamplerParamID.filter_min);
            SetLabelForMaterialReference(mlbFilterMipmap, Material.SamplerParamID.filter_mip);
            SetLabelForMaterialReference(mlbFilterAniso, Material.SamplerParamID.filter_max_aniso);
            SetLabelForMaterialReference(mlbLodMin, Material.SamplerParamID.lod_min);
            SetLabelForMaterialReference(mlbLodMax, Material.SamplerParamID.lod_max);
            SetLabelForMaterialReference(mlbLodBias, Material.SamplerParamID.lod_bias);
        }

        private void SetLabelForMaterialReference(UIModifiedMarkLabel label, Material.SamplerParamID id)
        {
            label.ContextMenuStrip = cmsSamplerParam.ProxyContextMenuStrip;
            label.Tag = id;
        }

        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_material_property_window_sampler_page;
            }
        }

        public static ObjectPropertyPage CreateInstance(object arg)
        {
            return new MaterialSamplerPage();
        }

        protected override void InitializeFormInternal()
        {
            using(var ub = new UpdateBlock(cmbWrapU))
            {
                cmbWrapU.AddItem(Strings.Enum_wrap_uvwType_repeat,		wrap_uvwType.repeat);
                cmbWrapU.AddItem(Strings.Enum_wrap_uvwType_mirror,		wrap_uvwType.mirror);
                cmbWrapU.AddItem(Strings.Enum_wrap_uvwType_clamp,		wrap_uvwType.clamp);
                cmbWrapU.AddItem(Strings.Enum_wrap_uvwType_mirror_once,	wrap_uvwType.mirror_once);
            }

            using(var ub = new UpdateBlock(cmbWrapV))
            {
                cmbWrapV.AddItem(Strings.Enum_wrap_uvwType_repeat,		wrap_uvwType.repeat);
                cmbWrapV.AddItem(Strings.Enum_wrap_uvwType_mirror,		wrap_uvwType.mirror);
                cmbWrapV.AddItem(Strings.Enum_wrap_uvwType_clamp,		wrap_uvwType.clamp);
                cmbWrapV.AddItem(Strings.Enum_wrap_uvwType_mirror_once,	wrap_uvwType.mirror_once);
            }

            using(var ub = new UpdateBlock(cmbWrapW))
            {
                cmbWrapW.AddItem(Strings.Enum_wrap_uvwType_repeat,		wrap_uvwType.repeat);
                cmbWrapW.AddItem(Strings.Enum_wrap_uvwType_mirror,		wrap_uvwType.mirror);
                cmbWrapW.AddItem(Strings.Enum_wrap_uvwType_clamp,		wrap_uvwType.clamp);
                cmbWrapW.AddItem(Strings.Enum_wrap_uvwType_mirror_once,	wrap_uvwType.mirror_once);
            }

            using(var ub = new UpdateBlock(cmbFilterMag))
            {
                cmbFilterMag.AddItem(Strings.Enum_filter_mag_minType_point,		filter_mag_minType.point);
                cmbFilterMag.AddItem(Strings.Enum_filter_mag_minType_linear,	filter_mag_minType.linear);
            }

            using(var ub = new UpdateBlock(cmbFilterMin))
            {
                cmbFilterMin.AddItem(Strings.Enum_filter_mag_minType_point,		filter_mag_minType.point);
                cmbFilterMin.AddItem(Strings.Enum_filter_mag_minType_linear,	filter_mag_minType.linear);
            }

            using(var ub = new UpdateBlock(cmbFilterMipmap))
            {
                cmbFilterMipmap.AddItem(Strings.Enum_filter_mipType_none,		filter_mipType.none);
                cmbFilterMipmap.AddItem(Strings.Enum_filter_mipType_point,		filter_mipType.point);
                cmbFilterMipmap.AddItem(Strings.Enum_filter_mipType_linear,		filter_mipType.linear);
            }

            using(var ub = new UpdateBlock(cmbFilterAniso))
            {
                cmbFilterAniso.AddItem(Strings.Enum_filter_max_anisoType_aniso_1,	filter_max_anisoType.aniso_1);
                cmbFilterAniso.AddItem(Strings.Enum_filter_max_anisoType_aniso_2,	filter_max_anisoType.aniso_2);
                cmbFilterAniso.AddItem(Strings.Enum_filter_max_anisoType_aniso_4,	filter_max_anisoType.aniso_4);
                cmbFilterAniso.AddItem(Strings.Enum_filter_max_anisoType_aniso_8,	filter_max_anisoType.aniso_8);
                cmbFilterAniso.AddItem(Strings.Enum_filter_max_anisoType_aniso_16,	filter_max_anisoType.aniso_16);
            }

            using (var ub = new UpdateBlock(cmbChildSamplerStructureRestriction))
            {
                cmbChildSamplerStructureRestriction.AddItem(Strings.Enum_ChildSamplerStructureRestrictionState_None, ChildSamplerStructureRestrictionState.None);
                cmbChildSamplerStructureRestriction.AddItem(Strings.Enum_ChildSamplerStructureRestrictionState_DisallowAdd, ChildSamplerStructureRestrictionState.DisallowAddOwnSampler);
            }
        }

        private bool isFirstUpdate = true;

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            if (ActiveTarget == null)
            {
                return;
            }
            UpdateSamplerParamEditables();
            UpdateList();
            UpdateEdit();
            UpdateModifiedMark();
            UpdateTexture();
            UpdateEnabled();

            // ページを開いた直後は最後を選択状態にする
            if(isFirstUpdate)
            {
                if (lvwList.Items.Count > 0)
                {
                    var lastIndex = lvwList.Items.Count - 1;

                    lvwList.Items[lastIndex].Selected = true;
                    lvwList.Items[lastIndex].Focused = true;
                }

                // ページを開いた直後は、リストビューの「テクスチャ」列の文字列が途中で切れないように調整する。
                var colWidth = Enumerable.Max(lvwList.Items.OfType<ListViewItem>().Select(x => TextRenderer.MeasureText(x.SubItems[clhTexture.Index].Text, x.UseItemStyleForSubItems ? x.Font : x.SubItems[clhName.Index].Font).Width + 10).Concat(new int[] { 0 }));
                if (lvwList.Columns[clhTexture.Index].Width < colWidth)
                {
                    lvwList.Columns[clhTexture.Index].Width = colWidth;
                }

                isFirstUpdate = false;
            }
        }

        private static string[] _samplerParamNames = new[]
        {
            "tex_name",
            "wrap_u",
            "wrap_v",
            "wrap_w",
            "filter_mag",
            "filter_min",
            "filter_mip",
            "filter_max_aniso",
            "lod_min",
            "lod_max",
            "lod_bias",
        };

        private Dictionary<string, HashSet<Material.SamplerParamID>> _samplerParamEditables = new Dictionary<string, HashSet<Material.SamplerParamID>>();

        private Dictionary<string, HashSet<Material.SamplerParamID>> UpdateSamplerParamEditables()
        {
            if (ActiveTarget == null)
            {
                _samplerParamEditables = null;
            }
            else
            {
                _samplerParamEditables.Clear();
                foreach (var sampler in ActiveTarget.Samplers)
                {
                    _samplerParamEditables[sampler.name] = new HashSet<Material.SamplerParamID>(Enum.GetValues(typeof(Material.SamplerParamID)).Cast<Material.SamplerParamID>().Where(x => ActiveTarget.IsResolvedSamplerParamValueEditable(sampler.name, x)));
                }

            }
            return _samplerParamEditables;
        }

        private void UpdateEnabled()
        {
            bool isSelected = lvwList.SelectedIndex != -1;
            var sampler = ResolvedSampler;

            btnUp.Enabled			= isSelected && (lvwList.SelectedIndex >= 1);
            btnDown.Enabled			= isSelected && (lvwList.SelectedIndex >= 0) && (lvwList.SelectedIndex <= lvwList.Items.Count - 2);
            btnRemove.Enabled		= isSelected && GetSelectedSamplers().All(x => !ActiveTarget.IsRequiredChildSampler(x.name));
            var allowAdd = true;
            if (ActiveTarget.DisallowAddSampler())
            {
                var restrictedSamplers = ActiveTarget.GetParentRestrictedSamplers();
                if (restrictedSamplers.Count == ActiveTarget.Samplers.Length)
                {
                    allowAdd = false;
                }
            }

            btnAdd.Enabled          = allowAdd;

            var canChangeName = GetSelectedSamplers().All(x => !ActiveTarget.IsRequiredChildSampler(x.name));

            cbxName.Enabled			= isSelected && canChangeName;
            mlbName.Enabled			= isSelected;
            ltbHint.Enabled         = isSelected;
            mlbHint.Enabled			= isSelected;
            grpImage.Enabled		= isSelected;
            grpWrap.Enabled			= isSelected;
            grpFilter.Enabled		= isSelected;
            grpLod.Enabled			= isSelected;
            cbxName.DropDownStyle = allowAdd ? ComboBoxStyle.DropDown : ComboBoxStyle.DropDownList;

            // ラップ
            if (isSelected)
            {
                UpdateSamplerParamEditables();
                var editables = _samplerParamEditables[sampler.name];

                ltbTexName.Enabled = editables.Contains(Material.SamplerParamID.tex_name);
                cmbFilterMag.Enabled = editables.Contains(Material.SamplerParamID.filter_mag);
                cmbFilterMin.Enabled = editables.Contains(Material.SamplerParamID.filter_min);
                cmbFilterMipmap.Enabled = editables.Contains(Material.SamplerParamID.filter_mip);
                cmbFilterAniso.Enabled = editables.Contains(Material.SamplerParamID.filter_max_aniso);
                iepLodMin.Enabled = editables.Contains(Material.SamplerParamID.lod_min);
                iepLodMax.Enabled = editables.Contains(Material.SamplerParamID.lod_max);
                iepLodBias.Enabled = editables.Contains(Material.SamplerParamID.lod_bias);
                cmbWrapU.Enabled = editables.Contains(Material.SamplerParamID.wrap_u);

                var texture = DocumentManager.Textures.FirstOrDefault(x => x.Name == sampler.tex_name);

                bool enabledV = true;
                bool enabledW = true;

                if (texture != null)
                {
                    switch (texture.Data.texture_info.dimension)
                    {
                        case texture_info_dimensionType.Item1d:
                        case texture_info_dimensionType.Item1d_array:
                            {
                                enabledV = false;
                                enabledW = false;
                                break;
                            }

                        case texture_info_dimensionType.Item2d:
                        case texture_info_dimensionType.Item2d_array:
                        case texture_info_dimensionType.cube:
                        case texture_info_dimensionType.cube_array:
                            {
                                enabledV = true;
                                enabledW = false;
                                break;
                            }

                        case texture_info_dimensionType.Item3d:
                            {
                                enabledV = true;
                                enabledW = true;
                                break;
                            }
                    }
                }

                cmbWrapV.Enabled =  enabledV && editables.Contains(Material.SamplerParamID.wrap_v);
                cmbWrapW.Enabled =  enabledW && editables.Contains(Material.SamplerParamID.wrap_w);

                btnPropertyDialog.Enabled = true;
                btnChange.Enabled = !(ConfigData.ApplicationConfig.Preset.FollowDccSamplerNameRule &&
                    sampler.name.StartsWith("_") &&
                    !string.IsNullOrEmpty(sampler.hint)) && editables.Contains(Material.SamplerParamID.tex_name);
            }
            else
            {
                btnPropertyDialog.Enabled = false;
                btnChange.Enabled = false;
            }
        }


        private IEnumerable<ListViewItem> GetSelectedListViewItems() => lvwList.SelectedItems.OfType<ListViewItem>();
        private IEnumerable<samplerType> GetSelectedSamplers() => GetSelectedListViewItems().Select(x => x.Tag).OfType<samplerType>();

        private samplerType ResolvedSampler
        {
            get
            {
                var sampler = lvwList.SelectedItemData as samplerType;
                return sampler == null ? null : ActiveTarget.GetResolvedSampler(sampler.name);
            }
        }

        private void UpdateList()
        {
            using(var ub = new UpdateBlock(lvwList))
            {
                var samplers = ActiveTarget.Samplers;

                // リストアイテムを予め用意する
                lvwList.SetItemCount(samplers.Count());

                int index = 0;

                foreach (var name in samplers.Select(x => x.name))
                {
                    var sampler = ActiveTarget.GetResolvedSampler(name);
                    var item = lvwList.Items[index];
                    item.Tag = sampler;
                    item.ToolTipText = (sampler != null &&
                                        (ApplicationConfig.Preset.FollowDccSamplerNameRule &&
                                         sampler.name.StartsWith("_") && !string.IsNullOrEmpty(sampler.hint)))
                        ? item.ToolTipText = Strings.MaterialSamplerPage_UpdateList_DccSamplerNameRule
                        : "";

                    item.SubItems[clhIndex.Index].Text = index.ToString();
                    item.SubItems[clhName.Index].Text = sampler.name;
                    item.SubItems[clhHint.Index].Text = sampler.hint;
                    item.SubItems[clhTexture.Index].Text = sampler.tex_name;
                    item.SubItems[clhWrapU.Index].Text = UIText.EnumValue(sampler.wrap.u);
                    item.SubItems[clhWrapV.Index].Text = UIText.EnumValue(sampler.wrap.v);
                    item.SubItems[clhWrapW.Index].Text = UIText.EnumValue(sampler.wrap.w);
                    item.SubItems[clhFilterMag.Index].Text = UIText.EnumValue(sampler.filter.mag);
                    item.SubItems[clhFilterMin.Index].Text = UIText.EnumValue(sampler.filter.min);
                    item.SubItems[clhFilterMip.Index].Text = UIText.EnumValue(sampler.filter.mip);
                    item.SubItems[clhFilterAniso.Index].Text = UIText.EnumValue(sampler.filter.max_aniso);
                    item.SubItems[clhLodMin.Index].Text = sampler.lod.min.ToString();
                    item.SubItems[clhLodMax.Index].Text = sampler.lod.max.ToString();
                    item.SubItems[clhLodBias.Index].Text = sampler.lod.bias.ToString();

                    ++index;
                }
            }
        }

        private void UpdateEdit()
        {
            cmbChildSamplerStructureRestriction.SelectedItemData = ActiveTarget.GetChildSamplerStructureRestriction();
            bool isSelected = lvwList.SelectedIndex != -1;
            var sampler = lvwList.SelectedItemData as samplerType;
            if (isSelected && sampler != null)
            {
                var samplerName = sampler.name;
                Material.ValueResolvedState valueResolvedState;
                var samplerBehaviorItem = ActiveTarget.GetReferenceBehaviorSampler(sampler.name);//, Material.SamplerParamID.tex_name);
                var referenceBehaviorItem = ActiveTarget.GetReferenceBehaviorSamplerParam(sampler.name, Material.SamplerParamID.tex_name);

                cbxName.Items.Clear();
                cbxName.Items.Add(sampler.name);
                cbxName.Value						= sampler.name;
                ltbHint.Text						= sampler.hint;
                cmbWrapU.SelectedItemData			= ActiveTarget.GetResolvedSamplerParamValue(samplerName, Material.SamplerParamID.wrap_u, out valueResolvedState) as wrap_uvwType? ?? sampler.wrap.u;
                mlbWrapU.TailImages = ShaderParamControl.GetReferenceStateIcons(valueResolvedState, ActiveTarget.GetReferenceBehaviorSamplerParam(sampler.name, Material.SamplerParamID.wrap_u));
                cmbWrapV.SelectedItemData			= ActiveTarget.GetResolvedSamplerParamValue(samplerName, Material.SamplerParamID.wrap_v, out valueResolvedState) as wrap_uvwType? ?? sampler.wrap.v;
                mlbWrapV.TailImages = ShaderParamControl.GetReferenceStateIcons(valueResolvedState, ActiveTarget.GetReferenceBehaviorSamplerParam(sampler.name, Material.SamplerParamID.wrap_v));
                cmbWrapW.SelectedItemData			= ActiveTarget.GetResolvedSamplerParamValue(samplerName, Material.SamplerParamID.wrap_w, out valueResolvedState) as wrap_uvwType? ?? sampler.wrap.w;
                mlbWrapW.TailImages = ShaderParamControl.GetReferenceStateIcons(valueResolvedState, ActiveTarget.GetReferenceBehaviorSamplerParam(sampler.name, Material.SamplerParamID.wrap_w));
                cmbFilterMag.SelectedItemData		= ActiveTarget.GetResolvedSamplerParamValue(samplerName, Material.SamplerParamID.filter_mag, out valueResolvedState) as filter_mag_minType? ?? sampler.filter.mag;
                mlbFilterMag.TailImages = ShaderParamControl.GetReferenceStateIcons(valueResolvedState, ActiveTarget.GetReferenceBehaviorSamplerParam(sampler.name, Material.SamplerParamID.filter_mag));
                cmbFilterMin.SelectedItemData		= ActiveTarget.GetResolvedSamplerParamValue(samplerName, Material.SamplerParamID.filter_min, out valueResolvedState) as filter_mag_minType? ?? sampler.filter.min;
                mlbFilterMin.TailImages = ShaderParamControl.GetReferenceStateIcons(valueResolvedState, ActiveTarget.GetReferenceBehaviorSamplerParam(sampler.name, Material.SamplerParamID.filter_min));
                cmbFilterMipmap.SelectedItemData	= ActiveTarget.GetResolvedSamplerParamValue(samplerName, Material.SamplerParamID.filter_mip, out valueResolvedState) as filter_mipType? ?? sampler.filter.mip;
                mlbFilterMipmap.TailImages = ShaderParamControl.GetReferenceStateIcons(valueResolvedState, ActiveTarget.GetReferenceBehaviorSamplerParam(sampler.name, Material.SamplerParamID.filter_mip));
                cmbFilterAniso.SelectedItemData		= ActiveTarget.GetResolvedSamplerParamValue(samplerName, Material.SamplerParamID.filter_max_aniso, out valueResolvedState) as filter_max_anisoType? ?? sampler.filter.max_aniso;
                mlbFilterAniso.TailImages = ShaderParamControl.GetReferenceStateIcons(valueResolvedState, ActiveTarget.GetReferenceBehaviorSamplerParam(sampler.name, Material.SamplerParamID.filter_max_aniso));
                iepLodMin.Value						= ActiveTarget.GetResolvedSamplerParamValue(samplerName, Material.SamplerParamID.lod_min, out valueResolvedState) as float? ?? sampler.lod.min;
                mlbLodMin.TailImages = ShaderParamControl.GetReferenceStateIcons(valueResolvedState, ActiveTarget.GetReferenceBehaviorSamplerParam(sampler.name, Material.SamplerParamID.lod_min));
                iepLodMax.Value						= ActiveTarget.GetResolvedSamplerParamValue(samplerName, Material.SamplerParamID.lod_max, out valueResolvedState) as float? ?? sampler.lod.max;
                mlbLodMax.TailImages = ShaderParamControl.GetReferenceStateIcons(valueResolvedState, ActiveTarget.GetReferenceBehaviorSamplerParam(sampler.name, Material.SamplerParamID.lod_max));
                iepLodBias.Value					= ActiveTarget.GetResolvedSamplerParamValue(samplerName, Material.SamplerParamID.lod_bias, out valueResolvedState) as float? ?? sampler.lod.bias;
                mlbLodBias.TailImages = ShaderParamControl.GetReferenceStateIcons(valueResolvedState, ActiveTarget.GetReferenceBehaviorSamplerParam(sampler.name, Material.SamplerParamID.lod_bias));
            }
            else
            {
                cbxName.Items.Clear();
                cbxName.Items.Add(string.Empty);
                cbxName.Value                       = string.Empty;
                ltbHint.Text						= string.Empty;
                cmbWrapU.SelectedItemData			= null;
                cmbWrapV.SelectedItemData			= null;
                cmbWrapW.SelectedItemData			= null;
                cmbFilterMag.SelectedItemData		= null;
                cmbFilterMin.SelectedItemData		= null;
                cmbFilterMipmap.SelectedItemData	= null;
                cmbFilterAniso.SelectedItemData		= null;
                iepLodMin.Value						= 0;
                iepLodMax.Value						= 0;
                iepLodBias.Value					= 0.0f;
            }
        }

        private string SelectedSamplerName
        {
            get
            {
                return ResolvedSampler?.name;
            }
        }

        private void UpdateModifiedMark()
        {
            // テクスチャは同名であってもパスが違う場合があるので、プレビュー用途で別パスの同名テクスチャに変更されていないか確認する。
            bool isModifiedReferenceTexturePaths = IsModifiedReferenceTexturePaths(ActiveTarget);

            var sampler = (lvwList.SelectedIndex >= 0 && lvwList.SelectedIndex < ActiveTarget.Samplers.Length) ? ActiveTarget.Samplers[lvwList.SelectedIndex] : null;
            var saved = sampler != null ? ActiveTarget.SavedSampler(sampler.name) : null;
            mlbName.IsModified = ActiveTarget.IsValueChanged(sampler, saved, x => x.name) || (sampler != null && saved == null);
            mlbHint.IsModified = ActiveTarget.IsValueChanged(sampler, saved, x => x.hint);
            mlbTexture.IsModified = ActiveTarget.IsValueChanged(sampler, saved, x => x.tex_name) || isModifiedReferenceTexturePaths;
            mlbWrapU.IsModified = ActiveTarget.IsValueChanged(sampler, saved, x => x.wrap.u);
            mlbWrapV.IsModified = ActiveTarget.IsValueChanged(sampler, saved, x => x.wrap.v);
            mlbWrapW.IsModified = ActiveTarget.IsValueChanged(sampler, saved, x => x.wrap.w);
            mlbFilterMag.IsModified = ActiveTarget.IsValueChanged(sampler, saved, x => x.filter.mag);
            mlbFilterMin.IsModified = ActiveTarget.IsValueChanged(sampler, saved, x => x.filter.min);
            mlbFilterMipmap.IsModified = ActiveTarget.IsValueChanged(sampler, saved, x => x.filter.mip);
            mlbFilterAniso.IsModified = ActiveTarget.IsValueChanged(sampler, saved, x => x.filter.max_aniso);
            mlbLodMin.IsModified = ActiveTarget.IsValueChanged(sampler, saved, x => x.lod.min);
            mlbLodMax.IsModified = ActiveTarget.IsValueChanged(sampler, saved, x => x.lod.max);
            mlbLodBias.IsModified = ActiveTarget.IsValueChanged(sampler, saved, x => x.lod.bias);
            mlbChildSamplerStructureRestriction.IsModified = ActiveTarget.IsChildSamplerStructureRestrictionsModified();

            mlbSamplers.IsModified = ActiveTarget.sampler_array.sampler.Length != ActiveTarget.SavedSamplers().Count() || isModifiedReferenceTexturePaths ||
                ActiveTarget.sampler_array.sampler.Zip(ActiveTarget.SavedSamplers(), (y, z) =>
                    ActiveTarget.IsValueChanged(y, z, x => x.name) ||
                    ActiveTarget.IsValueChanged(y, z, x => x.hint) ||
                    ActiveTarget.IsValueChanged(y, z, x => x.tex_name) ||
                    ActiveTarget.IsValueChanged(y, z, x => x.wrap.u) ||
                    ActiveTarget.IsValueChanged(y, z, x => x.wrap.v) ||
                    ActiveTarget.IsValueChanged(y, z, x => x.wrap.w) ||
                    ActiveTarget.IsValueChanged(y, z, x => x.filter.mag) ||
                    ActiveTarget.IsValueChanged(y, z, x => x.filter.min) ||
                    ActiveTarget.IsValueChanged(y, z, x => x.filter.mip) ||
                    ActiveTarget.IsValueChanged(y, z, x => x.filter.max_aniso) ||
                    ActiveTarget.IsValueChanged(y, z, x => x.lod.min) ||
                    ActiveTarget.IsValueChanged(y, z, x => x.lod.max) ||
                    ActiveTarget.IsValueChanged(y, z, x => x.lod.bias)
                    ).Any(x => x);
        }

        public static bool IsModifiedReferenceTexturePaths(Material activeTarget)
        {
            Dictionary<string, string> referenceTexturePaths =
                    MaterialOwnerExtensions.MakeReferenceTexturePaths(activeTarget.Owner);

            foreach (KeyValuePair<string, string> pair in referenceTexturePaths)
            {
                if (activeTarget.Owner.ReferenceTexturePaths.ContainsKey(pair.Key))
                {
                    if (activeTarget.Owner.ReferenceTexturePaths[pair.Key] != pair.Value)
                    {
                        return true;
                    }
                }
            }

            return false;
        }

        public static bool IsModified(Material activeTarget)
        {
            // テクスチャは同名であってもパスが違う場合があるので、プレビュー用途で別パスの同名テクスチャに変更されていないか確認する。
            if (activeTarget != null && IsModifiedReferenceTexturePaths(activeTarget))
            {
                return true;
            }

            return activeTarget != null &&
                (activeTarget.IsChildSamplerStructureRestrictionsModified() ||
                activeTarget.sampler_array.sampler.Length != activeTarget.SavedSamplers().Count() ||
                activeTarget.sampler_array.sampler.Zip(activeTarget.SavedSamplers(), (y, z) =>
                activeTarget.IsValueChanged(y, z, x => x.name) ||
                    activeTarget.IsValueChanged(y, z, x => x.hint) ||
                    activeTarget.IsValueChanged(y, z, x => x.tex_name) ||
                    activeTarget.IsValueChanged(y, z, x => x.wrap.u) ||
                    activeTarget.IsValueChanged(y, z, x => x.wrap.v) ||
                    activeTarget.IsValueChanged(y, z, x => x.wrap.w) ||
                    activeTarget.IsValueChanged(y, z, x => x.filter.mag) ||
                    activeTarget.IsValueChanged(y, z, x => x.filter.min) ||
                    activeTarget.IsValueChanged(y, z, x => x.filter.mip) ||
                    activeTarget.IsValueChanged(y, z, x => x.filter.max_aniso) ||
                    activeTarget.IsValueChanged(y, z, x => x.lod.min) ||
                    activeTarget.IsValueChanged(y, z, x => x.lod.max) ||
                    activeTarget.IsValueChanged(y, z, x => x.lod.bias)
                    ).Any(x => x));
        }

        private static readonly Dictionary<string, string>	mipmapGenFilters = new Dictionary<string, string>()
        {
            {"point",	Strings.MipmapGenFilter_Point},
            {"linear",	Strings.MipmapGenFilter_Linear},
            {"cubic",	Strings.MipmapGenFilter_Cubic},
        };

        private void UpdateTexture()
        {
            bool isSelected = lvwList.SelectedIndex != -1;
            var sampler = lvwList.SelectedItemData as samplerType;
            if (isSelected && sampler != null)
            {
                Material.ValueResolvedState valueResolvedState;
                var texName = ActiveTarget.GetResolvedSamplerParamValue(sampler.name, Material.SamplerParamID.tex_name, out valueResolvedState) as string ?? sampler.tex_name;
                var texture = ActiveTarget.Owner.GetReferenceTexture(texName);
                var referenceBehaviorItem = ActiveTarget.GetReferenceBehaviorSamplerParam(sampler.name, Material.SamplerParamID.tex_name);
                mlbTexture.TailImages = ShaderParamControl.GetReferenceStateIcons(valueResolvedState, referenceBehaviorItem);

                if (texture != null)
                {
                    ltbTexName.ForeColor = SystemColors.WindowText;
                    var name = texture.Name;
                    if (DocumentManager.Textures.Any(x => x.Name == name && x != texture))
                    {
                        string indexText = DocumentManager.GetSameNameIndexText(texture, false);
                        if (indexText.Length > 0)
                        {
                            name += string.Format("  ({0} {1})", texture.FileName, indexText);
                        }
                        else
                        {
                            name += string.Format("  ({0})", texture.FileName);
                        }
                    }

                    ltbTexName.Text = name;
                    ltbTexFullPath.Text = texture.FilePath;

                    tvpImage.Target = texture;

                    var preset = ApplicationConfig.Preset.TexturePresets.FirstOrDefault(x => x.quantize_type == texture.Data.texture_info.quantize_type);

                    ltbDccPreset.Text			= texture.Data.texture_info.dcc_preset;
                    ltbPresetFormat.Text		= (preset == null) ? string.Empty : preset.Name;

                    ltbFormat.Text				= UIText.EnumValue(texture.Data.texture_info.quantize_type);
                    ltbMipmapGenFilter.Text		= mipmapGenFilters.ContainsKey(texture.Data.texture_info.mip_gen_filter) ? mipmapGenFilters[texture.Data.texture_info.mip_gen_filter] : string.Empty;
                    ccsPanel.Value				= texture.Data.texture_info.comp_sel;
                    ltbWeightedCompress.Text	= UIText.FlagEnableDisable(texture.Data.texture_info.weighted_compress);
                    rgbaTable1.Values			= texture.Data.texture_info.linear;

                    ltbDimension.Text			= UIText.EnumValue(texture.Data.texture_info.dimension);
                    ltbWidth.Text				= texture.Data.texture_info.width.ToString();
                    ltbHeight.Text				= texture.Data.texture_info.height.ToString();

                    ltbDepth.Text				= texture.Data.texture_info.depth.ToString();
                    ltbVramSize.Text			= DataSize.XBytesText(texture.Data.texture_info.size);
                    ltbTexMipmapLevel.Text		= texture.Data.texture_info.mip_level.ToString();
                    ltbTexMipmapMinSize.Text	= UIText.MipmapMinSize(texture.Data.texture_info);

                    ltbInitialSwizzle.Text		= texture.Data.texture_info.initial_swizzle.ToString();
                    ltbOriginalImageHash.Text   = texture.Data.texture_info.original_image_hash;
                }
                else
                {
                    ClearTexture();

                    ltbTexName.ForeColor = Const.DisreferenceColor;
                    ltbTexName.Text = texName;
                }
            }
            else
            {
                ClearTexture();
            }

        }

        private void ClearTexture()
        {
            ltbTexName.Text = string.Empty;
            ltbTexFullPath.Text = string.Empty;

            tvpImage.Target = null;

            ltbDccPreset.Text			= string.Empty;
            ltbPresetFormat.Text		= string.Empty;

            ltbFormat.Text				= string.Empty;
            ltbMipmapGenFilter.Text		= string.Empty;
            ccsPanel.Value				= null;
            ltbWeightedCompress.Text	= string.Empty;
            rgbaTable1.Values			= null;

            ltbDimension.Text			= string.Empty;
            ltbWidth.Text				= string.Empty;
            ltbHeight.Text				= string.Empty;

            ltbDepth.Text				= string.Empty;
            ltbVramSize.Text			= string.Empty;
            ltbTexMipmapLevel.Text		= string.Empty;
            ltbTexMipmapMinSize.Text	= string.Empty;

            ltbInitialSwizzle.Text		= string.Empty;
            ltbOriginalImageHash.Text = string.Empty;
        }

        private bool CheckSamplerName(string name)
        {
            // 空文字でないかをチェック
            if (string.IsNullOrEmpty(name))
            {
                UIMessageBox.Error(res.Strings.Sampler_NameEmpty);
                return false;
            }

            // _ アンダーバーから始まる
            if (ApplicationConfig.Preset.FollowDccSamplerNameRule && name.StartsWith("_"))
            {
                UIMessageBox.Error(res.Strings.Sampler_NameStartsWithUnderbar);
                return false;
            }

            // 利用できない文字列が含まれていないかをチェック
            if (!RegexMatch.Check(name, samplerNameRegex))
            {
                UIMessageBox.Error(res.Strings.Sampler_InvalidChar);
                return false;
            }

            // 保持しているGUIオブジェクトを巡回
            foreach(GuiObject obj in Targets.Objects)
            {
                // GUIオブジェクトをマテリアルにキャスト
                var material = (obj as Material);

                // キャストが成功したかをチェック
                if (material != null)
                {
                    // マテリアルが保持している全てのサンプラを巡回
                    foreach (var sampler in material.ResolvedSamplers)
                    {
                        // 名前が重複しているかをチェック
                        if (sampler.name == name)
                        {
                            // 重複しているならエラー出力
                            UIMessageBox.Error(res.Strings.Sampler_NameCollision, name);
                            return false;
                        }
                    }
                }
            }

            return true;
        }

        private void lvwList_SelectionChanged(object sender, EventArgs e)
        {
            UpdateForm(false, false);
        }

        private void lvwList_ItemSelectionChanged(object sender, ListViewItemSelectionChangedEventArgs e)
        {
/*
            var viewItem = e.Item;
            var referenceInfo = e.Item.Tag as Material.SamplerReferenceInfo;
            if (referenceInfo?.Material != ActiveTarget)
            {
                viewItem.Selected = false;
            }
            UpdateForm(false, false);
*/
        }

        private void cmbWrapU_SelectedIndexChanged(object sender, EventArgs e)
        {
            TheApp.CommandManager.Execute(CreateEditCommand_sampler_clamp_x(Targets, SelectedSamplerName, (wrap_uvwType)cmbWrapU.SelectedItemData));
        }

        private void cmbWrapV_SelectedIndexChanged(object sender, EventArgs e)
        {
            TheApp.CommandManager.Execute(CreateEditCommand_sampler_clamp_y(Targets, SelectedSamplerName, (wrap_uvwType)cmbWrapV.SelectedItemData));
        }

        private void cmbWrapW_SelectedIndexChanged(object sender, EventArgs e)
        {
            TheApp.CommandManager.Execute(CreateEditCommand_sampler_clamp_z(Targets, SelectedSamplerName, (wrap_uvwType)cmbWrapW.SelectedItemData));
        }

        private void cmbFilterMag_SelectedIndexChanged(object sender, EventArgs e)
        {
            TheApp.CommandManager.Execute(CreateEditCommand_sampler_filter_mag(Targets, SelectedSamplerName, (filter_mag_minType)cmbFilterMag.SelectedItemData));
        }

        private void cmbFilterMin_SelectedIndexChanged(object sender, EventArgs e)
        {
            TheApp.CommandManager.Execute(CreateEditCommand_sampler_filter_min(Targets, SelectedSamplerName, (filter_mag_minType)cmbFilterMin.SelectedItemData));
        }

        private void cmbFilterMipmap_SelectedIndexChanged(object sender, EventArgs e)
        {
            TheApp.CommandManager.Execute(CreateEditCommand_sampler_filter_mip(Targets, SelectedSamplerName, (filter_mipType)cmbFilterMipmap.SelectedItemData));
        }

        private void cmbFilterAniso_SelectedIndexChanged(object sender, EventArgs e)
        {
            TheApp.CommandManager.Execute(CreateEditCommand_sampler_filter_max_aniso(Targets, SelectedSamplerName, (filter_max_anisoType)cmbFilterAniso.SelectedItemData));
        }
        private void cmbChildSamplerStructureRestriction_SelectedIndexChanged(object sender, EventArgs e)
        {
            TheApp.CommandManager.Add(Material.CreateEditCommandSet_ChildSamplerStructureRestriction(Targets.Objects.OfType<Material>(),
                    (ChildSamplerStructureRestrictionState)cmbChildSamplerStructureRestriction.SelectedItemData));
            App.AppContext.OnMaterialReferenceBehaviorSettingChanged();
        }

        private void iepLodMin_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            if (e.Changing)
            {
                Viewer.SetMaterialSamplerMinLOD.Send(
                    Targets,
                    new Viewer.SetMaterialSamplerMinLOD.PacketData((uint)lvwList.SelectedIndex, iepLodMin.Value),
                    0xFFFFFFFF,
                    0xFFFFFFFF
                );
            }
            else
            {
                TheApp.CommandManager.Execute(CreateEditCommand_sampler_lod_min(Targets, SelectedSamplerName, iepLodMin.Value));
            }
        }

        private void iepLodMax_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            if (e.Changing)
            {
                Viewer.SetMaterialSamplerMaxLOD.Send(
                    Targets,
                    new Viewer.SetMaterialSamplerMaxLOD.PacketData((uint)lvwList.SelectedIndex, iepLodMax.Value),
                    0xFFFFFFFF,
                    0xFFFFFFFF
                );
            }
            else
            {
                TheApp.CommandManager.Execute(CreateEditCommand_sampler_lod_max(Targets, SelectedSamplerName, iepLodMax.Value));
            }
        }

        private void iepLodBias_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            if (e.Changing)
            {
                Viewer.SetMaterialSamplerLODBias.Send(
                    Targets,
                    new Viewer.SetMaterialSamplerLODBias.PacketData((uint)lvwList.SelectedIndex, iepLodBias.Value),
                    0xFFFFFFFF,
                    0xFFFFFFFF
                );
            }
            else
            {
                TheApp.CommandManager.Execute(CreateEditCommand_sampler_lod_bias(Targets, SelectedSamplerName, iepLodBias.Value));
            }
        }

        private void btnAdd_Click(object sender, EventArgs e)
        {
            // カーソル位置の次の位置
            // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1627
            var insertIndex = lvwList.SelectedIndex + 1;

            var samplerArrays = new List<samplerType[]>();
            var materials = Targets.Objects.OfType<Material>().ToArray();
            var uniqueName = StringUtility.UniqueName(
                ApplicationConfig.Preset.FollowDccSamplerNameRule ? "sampler": "_a",
                materials.SelectMany(x => x.ResolvedSamplers.Select(y => y.name)),
                Enumerable.Range(0, int.MaxValue).Select(x => x.ToString()));
            var addedMaterials = new List<Material>();
            foreach (var material in materials)
            {
                var newName = uniqueName;
                // サンプラーが固定されているかどうかによって名前を変える
                // 固定されていて追加不可の場合は何もしない
                if (material.DisallowAddSampler())
                {
                    var parentSamplerNames = ActiveTarget.GetParentRestrictedSamplerNames()?.Where(name => ActiveTarget.Samplers.All(x => x.name != name)).ToArray();
                    if (parentSamplerNames == null || !parentSamplerNames.Any())
                    {
                        continue;
                    }
                    newName = parentSamplerNames.First();
                }
                var newSampler = new samplerType()
                {
                    name = newName,
                    hint = string.Empty,
                    tex_name = string.Empty,
                    wrap = new wrapType()
                    {
                        u = ApplicationConfig.DefaultValue.wrap.u,
                        v = ApplicationConfig.DefaultValue.wrap.v,
                        w = ApplicationConfig.DefaultValue.wrap.w,
                    },
                    filter = new filterType()
                    {
                        mag = ApplicationConfig.DefaultValue.filter.mag,
                        min = ApplicationConfig.DefaultValue.filter.min,
                        mip = ApplicationConfig.DefaultValue.filter.mip,
                        max_aniso = ApplicationConfig.DefaultValue.filter.max_aniso,
                    },
                    lod = new lodType()
                    {
                        min = ApplicationConfig.DefaultValue.lod.min,
                        max = ApplicationConfig.DefaultValue.lod.max,
                        bias = ApplicationConfig.DefaultValue.lod.bias,
                    },
                };

                var newSamplerArray = ObjectUtility.Clone(material.sampler_array.sampler).ToList();
                newSamplerArray.Insert(Math.Min(insertIndex, newSamplerArray.Count()), newSampler);
                samplerArrays.Add(newSamplerArray.ToArray());
                addedMaterials.Add(material);
            }

            TheApp.CommandManager.Add(Material.ExecuteEditCommand_SamplerArray(new GuiObjectGroup(addedMaterials), samplerArrays, false));
        }

        private void btnRemove_Click(object sender, EventArgs e)
        {
            Debug.Assert(lvwList.SelectedIndex != -1);

            var targets = new GuiObjectGroup();
            var samplerArrays = new List<samplerType[]>();
            var samplerName = SelectedSamplerName;

            var commandSet = new EditCommandSet();
            {
                foreach (var material in Targets.Objects.OfType<Material>())
                {
                    if (material.sampler_array.sampler.Any(x => x.name == samplerName))
                    {
                        targets.Add(material);
                        samplerArrays.Add(material.sampler_array.sampler.Where(x => x.name != samplerName).ToArray());
                    }
                }

                var materials = targets.Objects.OfType<Material>().ToArray();
                var materialOwners = materials.Select(x => x.Owner).Distinct().ToArray();
                var models = materialOwners.OfType<Model>().ToArray();
                var materialDocs = materialOwners.OfType<SeparateMaterial>().ToArray();

                commandSet.Add(Material.CreateRemoveReferenceBehaviorSamplerCommand(targets, samplerName).Execute());
                commandSet.Add(Material.ExecuteEditCommand_SamplerArray(targets, samplerArrays, Viewer.Manager.Instance.IsConnected));
                if (models.Any())
                {
                    commandSet.Add(CreateEditCommand_RemoveTexturePaths(new GuiObjectGroup(models), GuiObjectID.Model).Execute());
                }
                if (materialDocs.Any())
                {
                    commandSet.Add(CreateEditCommand_RemoveTexturePaths(new GuiObjectGroup(materialDocs), GuiObjectID.SeparateMaterial).Execute());
                }
            }

            commandSet.Reverse();

            TheApp.CommandManager.Add(commandSet);
        }

        private void btnUp_Click(object sender, EventArgs e)
        {
            Debug.Assert(lvwList.SelectedIndex >= 1);

            var targets = new GuiObjectGroup();
            var samplerArrays = new List<samplerType[]>();
            var name = SelectedSamplerName;
            foreach (var material in Targets.Objects.OfType<Material>())
            {
                var index = Array.FindIndex(material.sampler_array.sampler, x => x.name == name);
                if (1 <= index)
                {
                    var array = material.sampler_array.sampler.ToArray();
                    var tmp = array[index - 1];
                    array[index - 1] = array[index];
                    array[index] = tmp;
                    targets.Add(material);
                    samplerArrays.Add(array);
                }
            }

            // リストのカーソル位置を更新する
            lvwList.Items[lvwList.SelectedIndex - 1].Selected = true;
            TheApp.CommandManager.Add(Material.ExecuteEditCommand_SamplerArray(targets, samplerArrays, Viewer.Manager.Instance.IsConnected));
        }

        private void btnDown_Click(object sender, EventArgs e)
        {
            Debug.Assert(lvwList.SelectedIndex <= lvwList.Items.Count - 2);

            var targets = new GuiObjectGroup();
            var samplerArrays = new List<samplerType[]>();
            var name = SelectedSamplerName;
            foreach (var material in Targets.Objects.OfType<Material>())
            {
                var index = Array.FindIndex(material.sampler_array.sampler, x => x.name == name);
                if (0 <= index && index + 1 < material.sampler_array.sampler.Length)
                {
                    var array = material.sampler_array.sampler.ToArray();
                    var tmp = array[index + 1];
                    array[index + 1] = array[index];
                    array[index] = tmp;
                    targets.Add(material);
                    samplerArrays.Add(array);
                }
            }

            // リストのカーソル位置を更新する
            lvwList.Items[lvwList.SelectedIndex + 1].Selected = true;
            TheApp.CommandManager.Add(Material.ExecuteEditCommand_SamplerArray(targets, samplerArrays, Viewer.Manager.Instance.IsConnected));
        }

        private void btnChange_Click(object sender, EventArgs e)
        {
            var sampler = ResolvedSampler;

            // 予約サンプラのテクスチャは変更不可
            if (IsPreservedSampler(sampler))
            {
                return;
            }

            if (!ActiveTarget.IsResolvedSamplerParamValueEditable(SelectedSamplerName, Material.SamplerParamID.tex_name))
            {
                return;
            }

            var targets = FilterTarget(Targets, SelectedSamplerName);
            var materials = targets.Select(x => x.Item1);
            var materialHash = new HashSet<Material>(materials);
            Func<Material, samplerType, bool> unchanged = (y, s) =>
            {
                if (!materialHash.Contains(y))
                {
                    return true;
                }

                if (s.name != SelectedSamplerName)
                {
                    return true;
                }

                return false;
            };

            var ownerDocs = materials.Select(y => y.Owner).Distinct();
            Predicate<Texture> canSelect = (x => {
                return ownerDocs.All(z => CanAssign_tex_name(z, unchanged, x.Name, x.FilePath));
            });
            var models = ownerDocs.OfType<Model>().ToArray();
            var materialDocs = ownerDocs.OfType<SeparateMaterial>().ToArray();

            Texture current = ActiveTarget.Owner.GetReferenceTexture(sampler.tex_name);

            // ダイアログ表示
            using(TextureSelectDialog dialog = new TextureSelectDialog(current, canSelect))
            {
                if (dialog.ShowDialog(Owner.Owner) != DialogResult.OK) { return; }

                // 同じテクスチャ選択時は何もしない
                // 複数編集時の問題はとりあえず他のコントロールと同じ扱いにしておく
                //Texture texImage = dialog.SelectedTexImage;
                var tex = dialog.SelectedTex;
                if (current == tex) { return; }

                var commandSet = new EditCommandSet();
                if (models.Any())
                {
                    commandSet.Add(CreateEditCommand_ReferenceTexturePath(new GuiObjectGroup(models), GuiObjectID.Model, tex.Name, tex.FilePath));
                    commandSet.Add(CreateEditCommand_tex_name(Targets, SelectedSamplerName, tex.Name));
                    commandSet.Add(new LazyCommand(() => CreateEditCommand_RemoveTexturePaths(new GuiObjectGroup(Targets.Objects.OfType<Model>()), GuiObjectID.Model)));
                    commandSet.Add(new LazyCommand(() => CreateEditCommand_RemoveTexturePaths(new GuiObjectGroup(Targets.Objects.OfType<SeparateMaterial>()), GuiObjectID.SeparateMaterial)));
                    commandSet.OnPostEdit += (s, a) =>
                        {
                            foreach (var model in models)
                            {
                                Viewer.LoadOrReloadModel.Send(model);
                            }
                        };
                }
                if (materialDocs.Any())
                {
                    commandSet.Add(CreateEditCommand_ReferenceTexturePath(new GuiObjectGroup(materialDocs), GuiObjectID.SeparateMaterial, tex.Name, tex.FilePath));
                    commandSet.Add(CreateEditCommand_tex_name(Targets, SelectedSamplerName, tex.Name));
                    commandSet.Add(new LazyCommand(() => CreateEditCommand_RemoveTexturePaths(new GuiObjectGroup(Targets.Objects.OfType<Model>()), GuiObjectID.Model)));
                    commandSet.Add(new LazyCommand(() => CreateEditCommand_RemoveTexturePaths(new GuiObjectGroup(Targets.Objects.OfType<SeparateMaterial>()), GuiObjectID.SeparateMaterial)));
                    commandSet.OnPostEdit += (s, a) =>
                    {
                    };
                }
                TheApp.CommandManager.Execute(commandSet);
            }
        }

        private static bool IsPreservedSampler(samplerType sampler)
        {
            return ConfigData.ApplicationConfig.Preset.FollowDccSamplerNameRule &&
                sampler.name.StartsWith("_") &&
                !string.IsNullOrEmpty(sampler.hint);
        }

        private void lvwList_DoubleClick(object sender, EventArgs e)
        {
            btnChange_Click(sender, e);
        }

        private void cbxName_ValueChanged(object sender, EventArgs e)
        {
            var sampler = lvwList.SelectedItemData as samplerType;
            var newName = cbxName.Value;
            if (string.IsNullOrEmpty(sampler?.name) || sampler.name == newName) return;

            if (CheckSamplerName(newName))
            {
                var materials = Targets.Objects.OfType<Material>().ToList();
                var targets = new GuiObjectGroup();
                var descendantMaterials = materials.SelectMany(x => x.GetAllDescendantSamplerMaterials(sampler.name)).Distinct().ToArray();
                materials.AddRange(descendantMaterials);
                foreach (var material in materials)
                {
                    if (material.DisallowAddSampler())
                    {
                        var restrictedSamplerNames = material.GetParentRestrictedSamplerNames();
                        if (!restrictedSamplerNames.Contains(newName))
                        {
                            //continue;
                        }
                    }
                    targets.Add(material);
                }

                var commandSet = ExecuteEditCommand_name(targets, sampler.name, newName);
                commandSet.Reverse();
                TheApp.CommandManager.Add(commandSet);
            }
            else
            {
                cbxName.Value = sampler.name;
            }
        }

        private void cbxName_DropDown(object sender, EventArgs e)
        {
            // 未確定を確定させる
            cbxName.Value = cbxName.Text;

            UpdateNameList();
        }

        private Regex samplerNameRegex = new Regex(@"[0-9A-Za-z_]+");
        private readonly string[] dccSamplerPrefix = new [] {
            "_a", "_o", "_e", "_n", "_t", "_s", "_r",
        };
        private void UpdateNameList()
        {
            using(var cesb = new UIControlEventSuppressBlock())
            using(var ub = new UpdateBlock(cbxName))
            {
                ActiveTarget.UpdateSamplers();
                var disallowAdd = ActiveTarget.DisallowAddSampler();
                var current = cbxName.Text;

                var parentSamplerNames = ActiveTarget.GetParentSamplerNames().Where(name => ActiveTarget.Samplers.All(x => x.name != name)).ToArray();
                var  presets = new List<string>();
                if (disallowAdd)
                {
                    presets.Add(current);
                    presets.AddRange(ActiveTarget.GetParentRestrictedSamplerNames().Where(name => ActiveTarget.Samplers.All(x => x.name != name)));
                }
                else if (ApplicationConfig.Preset.FollowDccSamplerNameRule || ApplicationConfig.Preset.SamplerNamePresets.Any())
                {
                    presets.AddRange(ApplicationConfig.Preset.SamplerNamePresets.Select(x => x.Name).Distinct()
                        .Where(x => x != null && !x.StartsWith("_") && RegexMatch.Check(x, samplerNameRegex) &&
                                    ActiveTarget.Samplers.All(y => y.name != x)).Union(parentSamplerNames));
                }
                else
                {
                    presets.AddRange(parentSamplerNames);
                    foreach (var prefix in dccSamplerPrefix)
                    {
                        for (var i = 0; ; i++)
                        {
                            var name = string.Format("{0}{1}", prefix, i);
                            if (ActiveTarget.Samplers.All(x => x.name != name) && parentSamplerNames.All(x => x != name))
                            {
                                presets.Add(name);
                                break;
                            }
                        }
                    }
                }

                // 領域を確保する
                if (cbxName.Items.Count != presets.Count)
                {
                    cbxName.Items.Clear();
                    cbxName.Items.AddRange(presets.ToArray());
                }

                {
                    var index = 0;
                    foreach (var preset in presets)
                    {
                        cbxName.Items[index] = preset;
                        index++;
                    }
                }
                cbxName.Text = current;
            }
        }

        private void lvwList_CustomDrawItem(object sender, CustomDrawListViewItemEventArgs e)
        {
            var sampler = e?.Item?.Tag as samplerType;
            if (sampler == null) return;

            var texName = sampler.tex_name ??  "";
            string texturePath;
            ActiveTarget.Owner.ReferenceTexturePaths.TryGetValue(texName, out texturePath);
            var texture = DocumentManager.Textures.FirstOrDefault(x => x.Name == texName && string.Compare(x.FilePath, texturePath, true) == 0);

            if (e.ColumnIndex == clhImage.Index)
            {
                if (texture != null)
                {
                    GraphicsUtility.DrawTextureImage(texture, e.Graphics, e.Bounds);
                }

                e.Handled = true;
            }

            // 文字色
            {
                e.SpecificForeColor =
                    string.IsNullOrEmpty(texName)	? Const.UnsettingColor :
                        (texture == null) ? Const.DisreferenceColor : SystemColors.WindowText;
            }


            // 背景
            if (e.ColumnIndex == clhName.Index)
            {
                var isDccLinkSampler = (ApplicationConfig.Preset.FollowDccSamplerNameRule &&
                                 sampler.name.StartsWith("_") && !string.IsNullOrEmpty(sampler.hint));
                var isRequiredChildSampler = ActiveTarget.IsRequiredChildSampler(sampler.name);
                var isReference = ActiveTarget.IsReferenceSampler(sampler.name);

                if (isDccLinkSampler || isRequiredChildSampler || isReference)
                {
                    var images = new List<Tuple<Bitmap,bool>>();
                    if (isDccLinkSampler)
                    {
                        images.Add(new Tuple<Bitmap, bool>(Properties.Resources.PropertyEdit_MaterialSamplerDccSamplerNameRule, true));
                    }
                    if (isReference)
                    {
                        images.Add(new Tuple<Bitmap, bool>(Properties.Resources.PropertyEdit_MaterialReference_Refer, true));
                    }
                    if (isRequiredChildSampler)
                    {
                        images.Add(new Tuple<Bitmap, bool>(Properties.Resources.PropertyEdit_MaterialSamplerReferenceRequired, true));
                    }

                    e.DrawDefault();

                    var orgClip = e.Graphics.Clip;
                    var xOffset = 0;
                    var xOffsetBottom = 0;
                    foreach (var imageinfo in images)
                    {
                        var image = imageinfo.Item1;
                        var bottom = imageinfo.Item2;
                        var rect = e.Bounds;
                        rect.X += rect.Width - image.Width - (bottom ? xOffsetBottom : xOffset);
                        rect.Y += bottom ? rect.Height - image.Height : 2;
                        if (bottom)
                        {
                            xOffsetBottom += image.Width;
                        }
                        else
                        {
                            xOffset += image.Width;
                        }
                        var clip = new Region(e.Bounds);
                        clip.Intersect(orgClip);
                        e.Graphics.DrawImageSafe(image, rect.X, rect.Y, image.Width, image.Height);
                    }
                    e.Graphics.Clip = orgClip;
                    e.Handled = true;
                }
            }
        }

        /// <summary>
        /// リストからサンプラを選択する
        /// </summary>
        public void SelectSampler(samplerType sampler)
        {
            lvwList.SelectedItemData = sampler;
        }
        #region コマンド

        //
        private static IEnumerable<Tuple<Material, int>> FilterTarget(GuiObjectGroup targets, string name)
        {
            return from material in targets.Objects.OfType<Material>()
                   select new Tuple<Material, int>(material, Array.FindIndex(material.sampler_array.sampler, x => x.name == name));
        }

        private static EditCommandSet ExecuteEditCommand_name(GuiObjectGroup targets, string name, string newName)
        {
            var tuples = FilterTarget(targets, name).Where(x => x.Item2 != -1).ToArray();

            var commandSet = new EditCommandSet();
            using (var propertyBlock = new App.AppContext.PropertyChangedSuppressBlock())
            {
                bool eraseHint = ApplicationConfig.Preset.FollowDccSamplerNameRule && name.StartsWith("_");
                // サンプラの名前変更
                commandSet.Add((new GeneralGroupReferenceEditCommand<Tuple<int, string, string>>(
                    new GuiObjectGroup(tuples.Select(x => x.Item1)),
                    GuiObjectID.Material,
                    tuples.Select(x => new Tuple<int, string, string>(x.Item2, newName, "")),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = (target as Material);
                        var tuple = (Tuple<int, string, string>)data;
                        var sampler = material?.sampler_array?.sampler.ElementAtOrDefault(tuple.Item1);
                        if (sampler == null)
                        {
                            return;
                        }
                        swap = new Tuple<int, string, string>(tuple.Item1, sampler.name, sampler.hint);
                        sampler.name = tuple.Item2;
                        if (eraseHint)
                        {
                            sampler.hint = tuple.Item3;
                        }
                    }
                )).Execute());

                // 名前変更後にサンプラ割り当ての書き換え
                foreach (var material in tuples.Select(x => x.Item1))
                {
                    foreach (var samplerAssign in material.MaterialShaderAssign.SamplerAssigns)
                    {
                        if (samplerAssign.sampler_name == name)
                        {
                            commandSet.Add(ShaderParamControls.ShaderParamControlGroup.CreateEditCommand_material_shader_assign_sampler_name(
                                new GuiObjectGroup(material),
                                samplerAssign.id,
                                newName,
                                false).Execute());
                        }

                    }
                    var behaviorItem = material.GetReferenceBehaviorSampler(name);
                    if (behaviorItem != null)
                    {
                        var newBehaviorItem = ObjectUtility.Clone(behaviorItem);
                        newBehaviorItem.Id = newName;
                        commandSet.Add(material.CreateSetSamplerBehaviorItemCommand(newBehaviorItem).Execute());
                    }

                }

                // バインドを更新
                commandSet.Add(DocumentManager.CreateAnimationUpdateBindCommand(
                    targets.Objects.OfType<Material>().SelectMany(x => x.Referrers).Distinct().SelectMany(x => x.AllAnimations).Distinct()
                ).Execute());

                // 送信
                EventHandler postEdit = (s, e) =>
                {
                    foreach (var model in tuples.SelectMany(x => x.Item1.Referrers).Distinct())
                    {
                        model.UpdateReferenceTexturePaths();
                        Viewer.LoadOrReloadModel.Send(model);
                    }
                };
                commandSet.OnPostEdit += postEdit;
                postEdit(null, null);

                return commandSet;
            }
        }

        public static bool CanAssign_tex_name(IMaterialOwner materialOwner, Func<Material, samplerType, bool> unchanged, string texName, string texPath)
        {
            string path;
            if (materialOwner.ReferenceTexturePaths.TryGetValue(texName, out path))
            {
                if (string.Compare(path, texPath, true) == 0)
                {
                    return true;
                }
            }

            foreach (var material in materialOwner.Materials)
            {
                foreach (var sampler in material.ResolvedSamplers)
                {
                    // 残り続けるなら ng
                    if (sampler.tex_name == texName && unchanged(material, sampler))
                    {
                        return false;
                    }
                }
            }

            return true;
        }

        public static GroupEditCommand CreateEditCommand_ReferenceTexturePath(GuiObjectGroup targets, GuiObjectID id, string texName, string filePath)
        {
            return new GeneralGroupReferenceEditCommand<string>(
                targets,
                id,
                targets.GetObjects(id).Select(x => filePath),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var materialOwner = (IMaterialOwner)target;
                    string old;
                    materialOwner.ReferenceTexturePaths.TryGetValue(texName, out old);
                    swap = old;
                    string path = (string)data;
                    if (path == null)
                    {
                        materialOwner.ReferenceTexturePaths.Remove(texName);
                    }
                    else
                    {
                        materialOwner.ReferenceTexturePaths[texName] = path;
                    }
                });
        }

        public static GroupEditCommand CreateEditCommand_RemoveTexturePaths(GuiObjectGroup targets, GuiObjectID id)
        {
            return new GeneralGroupReferenceEditCommand<Dictionary<string, string>>(
                targets,
                id,
                targets.GetObjects(id).OfType<IMaterialOwner>().Select(x =>
                    {
                        var texNames = (from material in x.Materials
                                        from sampler in material.sampler_array.GetItems()
                                        select sampler.tex_name).Distinct();
                        return texNames.Where(y => x.ReferenceTexturePaths.ContainsKey(y)).ToDictionary(y => y, y => x.ReferenceTexturePaths[y]);
                    }),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var materialOwner = (IMaterialOwner)target;
                    swap = materialOwner.ReferenceTexturePaths;
                    materialOwner.ReferenceTexturePaths = (Dictionary<string, string>)data;
                });
        }



        public static GroupEditCommand CreateEditCommand_tex_name(GuiObjectGroup targets, string name, string texName)
        {
            var tuples = FilterTarget(targets, name).Where(x => x.Item2 != -1).ToArray();

            return
                new GeneralGroupReferenceEditCommand<Tuple<int, string>>(
                    new GuiObjectGroup(tuples.Select(x => x.Item1)),
                    GuiObjectID.Material,
                    tuples.Select(x => new Tuple<int, string>(x.Item2, texName)),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = (target as Material);
                        var tuple = (Tuple<int, string>)data;
                        var sampler = material.sampler_array.sampler[tuple.Item1];
                        swap = new Tuple<int, string>(tuple.Item1, sampler.tex_name);
                        sampler.tex_name = tuple.Item2;
                        foreach (var materialOwner in material.GetAllDescendantSamplerMaterials(name).Select(x => x.Owner).Distinct())
                        {
                            materialOwner.UpdateReferenceTexturePaths();
                        }
                    },
                    postEditDelegate: (editTargets, data) =>
                    {
                        if (!Viewer.Manager.Instance.IsConnected) return;

                        editTargets = GetDescendantSamplerParamMaterials(editTargets, name, Material.SamplerParamID.tex_name);

                        // 送信前に UpdateReferenceTexturePaths を行うと、
                        // プレビュー目的で参照外のテクスチャを割り当てられなくなるので行わない。

                        foreach (var model in editTargets.OfType<Material>().SelectMany(x => x.Referrers).Distinct())
                        {
                            Viewer.LoadOrReloadModel.Send(model);
                        }
                    }

                );
        }

        private static GroupEditCommand CreateEditCommand_sampler_clamp_x(GuiObjectGroup targets, string name, wrap_uvwType wrap_uvw)
        {
            var tuples = FilterTarget(targets, name).Where(x => x.Item2 != -1).ToArray();

            return
                new GeneralGroupReferenceEditCommand<Tuple<int, wrap_uvwType>>(
                    new GuiObjectGroup(tuples.Select(x => x.Item1)),
                    GuiObjectID.Material,
                    tuples.Select(x => new Tuple<int, wrap_uvwType>(x.Item2, wrap_uvw)),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = (target as Material);
                        var tuple = (Tuple<int, wrap_uvwType>)data;
                        var sampler = material.sampler_array.sampler[tuple.Item1];
                        swap = new Tuple<int, wrap_uvwType>(tuple.Item1, sampler.wrap.u);
                        sampler.wrap.u = tuple.Item2;
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        if (!Viewer.Manager.Instance.IsConnected) return;

                        editTargets = GetDescendantSamplerParamMaterials(editTargets, name, Material.SamplerParamID.wrap_u);

                        var sendData = editTargets.OfType<Material>().Zip(data.OfType<Tuple<int, wrap_uvwType>>(),
                            (x, y) =>
                                new Viewer.SetMaterialSamplerClampX.PacketData()
                                {
                                    Index = (uint) y.Item1,
                                    Clamp = y.Item2,
                                }).OfType<object>().ToArray();
                        Viewer.SetMaterialSamplerClampX.Send(
                            editTargets,
                            sendData,
                            0xFFFFFFFF,
                            0xFFFFFFFF
                        );
                    }
                );
        }

        private static GroupEditCommand CreateEditCommand_sampler_clamp_y(GuiObjectGroup targets, string name, wrap_uvwType wrap_uvw)
        {
            var tuples = FilterTarget(targets, name).Where(x => x.Item2 != -1).ToArray();

            return
                new GeneralGroupReferenceEditCommand<Tuple<int, wrap_uvwType>>(
                    new GuiObjectGroup(tuples.Select(x => x.Item1)),
                    GuiObjectID.Material,
                    tuples.Select(x => new Tuple<int, wrap_uvwType>(x.Item2, wrap_uvw)),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = (target as Material);
                        var tuple = (Tuple<int, wrap_uvwType>)data;
                        var sampler = material.sampler_array.sampler[tuple.Item1];
                        swap = new Tuple<int, wrap_uvwType>(tuple.Item1, sampler.wrap.v);
                        sampler.wrap.v = tuple.Item2;
                    },
                    postEditDelegate: (editTargets, data) =>
                    {
                        if (!Viewer.Manager.Instance.IsConnected) return;

                        editTargets = GetDescendantSamplerParamMaterials(editTargets, name, Material.SamplerParamID.wrap_v);

                        var sendData = editTargets.OfType<Material>().Zip(data.OfType<Tuple<int, wrap_uvwType>>(),
                            (x, y) =>
                                new Viewer.SetMaterialSamplerClampY.PacketData()
                                {
                                    Index = (uint) y.Item1,
                                    Clamp = y.Item2,
                                }).OfType<object>().ToArray();
                        Viewer.SetMaterialSamplerClampY.Send(
                            editTargets,
                            sendData,
                            0xFFFFFFFF,
                            0xFFFFFFFF
                        );
                    }
                );
        }

        private static GroupEditCommand CreateEditCommand_sampler_clamp_z(GuiObjectGroup targets, string name, wrap_uvwType wrap_uvw)
        {
            var tuples = FilterTarget(targets, name).Where(x => x.Item2 != -1).ToArray();

            return
                new GeneralGroupReferenceEditCommand<Tuple<int, wrap_uvwType>>(
                    new GuiObjectGroup(tuples.Select(x => x.Item1)),
                    GuiObjectID.Material,
                    tuples.Select(x => new Tuple<int, wrap_uvwType>(x.Item2, wrap_uvw)),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = (target as Material);
                        var tuple = (Tuple<int, wrap_uvwType>)data;
                        var sampler = material.sampler_array.sampler[tuple.Item1];
                        swap = new Tuple<int, wrap_uvwType>(tuple.Item1, sampler.wrap.w);
                        sampler.wrap.w = tuple.Item2;
                    },
                    postEditDelegate: (editTargets, data) =>
                    {
                        if (!Viewer.Manager.Instance.IsConnected) return;

                        editTargets = GetDescendantSamplerParamMaterials(editTargets, name, Material.SamplerParamID.wrap_w);

                        var sendData = editTargets.OfType<Material>().Zip(data.OfType<Tuple<int, wrap_uvwType>>(),
                            (x, y) =>
                                new Viewer.SetMaterialSamplerClampZ.PacketData()
                                {
                                    Index = (uint) y.Item1,
                                    Clamp = y.Item2,
                                }).OfType<object>().ToArray();
                        Viewer.SetMaterialSamplerClampZ.Send(
                            editTargets,
                            sendData,
                            0xFFFFFFFF,
                            0xFFFFFFFF
                        );
                    }
                );
        }

        private static Viewer.SetMaterialSamplerFilter.Data GetMaterialSamplerFilterData(Material material, int index)
        {
            var filter = material.sampler_array.sampler[index].filter;
            var samplerName = material.sampler_array.sampler[index].name;

            return new Viewer.SetMaterialSamplerFilter.Data()
            {
                Index = index,
                minFilter = material.GetResolvedSamplerParamValue(samplerName, Material.SamplerParamID.filter_min) as filter_mag_minType? ?? filter.min,
                magFilter = material.GetResolvedSamplerParamValue(samplerName, Material.SamplerParamID.filter_mag) as filter_mag_minType? ?? filter.mag,
                mipmapFilter = material.GetResolvedSamplerParamValue(samplerName, Material.SamplerParamID.filter_mip) as filter_mipType? ?? filter.mip,
                maxAnisotropy = material.GetResolvedSamplerParamValue(samplerName, Material.SamplerParamID.filter_max_aniso) as filter_max_anisoType? ?? filter.max_aniso,
            };
        }

        private static ArrayList GetDescendantSamplerParamMaterials(ArrayList targets, string samplerName, Material.SamplerParamID id)
        {
            var materials = targets.OfType<Material>().ToList();
            var descendantMaterials = new List<GuiObject>(materials);
            materials.ForEach(x => descendantMaterials.AddRange(x.GetAllDescendantSamplerParamMaterials(samplerName, id)));
            return new ArrayList(descendantMaterials.Distinct().ToArray());
        }

        private static IEnumerable<Material> GetDescendantSamplerParamMaterials(Material material, string samplerName, Material.SamplerParamID id)
        {
            var descendantMaterials = new List<Material> {material};
            descendantMaterials.AddRange(material.GetAllDescendantSamplerParamMaterials(samplerName, id));
            return descendantMaterials.Distinct();
        }

        private static List<Tuple<Material, Viewer.SetMaterialSamplerFilter.Data>> GetSamplerFilterSendData(ArrayList editTargets, object[] datas, string name, Material.SamplerParamID id)
        {
            var newSendData = new List<Tuple<Material, Viewer.SetMaterialSamplerFilter.Data>>();

            for (var i = 0; i < editTargets.Count; i++)
            {
                var material = editTargets[i] as Material;
                var data = datas.ElementAtOrDefault(i);
                if (material == null || data == null)
                {
                    continue;
                }
                var index = (data as Tuple<int, filter_mipType>)?.Item1
                            ?? (data as Tuple<int, filter_mag_minType>)?.Item1
                            ?? (data as Tuple<int, filter_max_anisoType>)?.Item1;
                if (index == null)
                {
                    continue;
                }

                var descendantMaterials = GetDescendantSamplerParamMaterials(material, name, id);
                newSendData.AddRange(descendantMaterials.Select((x) => new Tuple<Material, Viewer.SetMaterialSamplerFilter.Data>(
                    x,
                    GetMaterialSamplerFilterData(x, index.Value)
                )));
            }
            return newSendData;
        }

        private static GroupEditCommand CreateEditCommand_sampler_filter_min(GuiObjectGroup targets, string name, filter_mag_minType filter_min)
        {
            var tuples = FilterTarget(targets, name).Where(x => x.Item2 != -1).ToArray();

            return
                new GeneralGroupReferenceEditCommand<Tuple<int, filter_mag_minType>>(
                    new GuiObjectGroup(tuples.Select(x => x.Item1)),
                    GuiObjectID.Material,
                    tuples.Select(x => new Tuple<int, filter_mag_minType>(x.Item2, filter_min)),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = (target as Material);
                        var tuple = (Tuple<int, filter_mag_minType>)data;
                        var sampler = material.sampler_array.sampler[tuple.Item1];
                        swap = new Tuple<int, filter_mag_minType>(tuple.Item1, sampler.filter.min);
                        sampler.filter.min = tuple.Item2;
                    },
                    postEditDelegate: (editTargets, data) =>
                    {
                        if (!Viewer.Manager.Instance.IsConnected) return;

                        Debug.Assert(editTargets.Count == data.Length);
                        var newSendData = GetSamplerFilterSendData(editTargets, data, name, Material.SamplerParamID.filter_min);
                        Viewer.SetMaterialSamplerFilter.Send(newSendData);
                    }
                );
        }

        private static GroupEditCommand CreateEditCommand_sampler_filter_mag(GuiObjectGroup targets, string name, filter_mag_minType filter_mag)
        {
            var tuples = FilterTarget(targets, name).Where(x => x.Item2 != -1).ToArray();

            return
                new GeneralGroupReferenceEditCommand<Tuple<int, filter_mag_minType>>(
                    new GuiObjectGroup(tuples.Select(x => x.Item1)),
                    GuiObjectID.Material,
                    tuples.Select(x => new Tuple<int, filter_mag_minType>(x.Item2, filter_mag)),
                    delegate (ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = (target as Material);
                        var tuple = (Tuple<int, filter_mag_minType>)data;
                        var sampler = material.sampler_array.sampler[tuple.Item1];
                        swap = new Tuple<int, filter_mag_minType>(tuple.Item1, sampler.filter.mag);
                        sampler.filter.mag = tuple.Item2;
                    },
                    postEditDelegate: (editTargets, data) =>
                    {
                        if (!Viewer.Manager.Instance.IsConnected) return;

                        Debug.Assert(editTargets.Count == data.Length);
                        var newSendData = GetSamplerFilterSendData(editTargets, data, name, Material.SamplerParamID.filter_mag);
                        Viewer.SetMaterialSamplerFilter.Send(newSendData);
                    }
                );
        }

        private static GroupEditCommand CreateEditCommand_sampler_filter_mip(GuiObjectGroup targets, string name, filter_mipType filter_mip)
        {
            var tuples = FilterTarget(targets, name).Where(x => x.Item2 != -1).ToArray();

            return
                new GeneralGroupReferenceEditCommand<Tuple<int, filter_mipType>>(
                    new GuiObjectGroup(tuples.Select(x => x.Item1)),
                    GuiObjectID.Material,
                    tuples.Select(x => new Tuple<int, filter_mipType>(x.Item2, filter_mip)),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = (target as Material);
                        var tuple = (Tuple<int, filter_mipType>)data;
                        var sampler = material.sampler_array.sampler[tuple.Item1];
                        swap = new Tuple<int, filter_mipType>(tuple.Item1, sampler.filter.mip);
                        sampler.filter.mip = tuple.Item2;
                    },
                    postEditDelegate: (editTargets, data) =>
                    {
                        if (!Viewer.Manager.Instance.IsConnected) return;

                        Debug.Assert(editTargets.Count == data.Length);
                        var newSendData = GetSamplerFilterSendData(editTargets, data, name, Material.SamplerParamID.filter_mip);
                        Viewer.SetMaterialSamplerFilter.Send(newSendData);
                    }
                );
        }

        private static GroupEditCommand CreateEditCommand_sampler_filter_max_aniso(GuiObjectGroup targets, string name, filter_max_anisoType filter_max_aniso)
        {
            var tuples = FilterTarget(targets, name).Where(x => x.Item2 != -1).ToArray();

            return
                new GeneralGroupReferenceEditCommand<Tuple<int, filter_max_anisoType>>(
                    new GuiObjectGroup(tuples.Select(x => x.Item1)),
                    GuiObjectID.Material,
                    tuples.Select(x => new Tuple<int, filter_max_anisoType>(x.Item2, filter_max_aniso)),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = (target as Material);
                        var tuple = (Tuple<int, filter_max_anisoType>)data;
                        var sampler = material.sampler_array.sampler[tuple.Item1];
                        swap = new Tuple<int, filter_max_anisoType>(tuple.Item1, sampler.filter.max_aniso);
                        sampler.filter.max_aniso = tuple.Item2;
                    },
                    postEditDelegate: (editTargets, data) =>
                    {
                        if (!Viewer.Manager.Instance.IsConnected) return;

                        Debug.Assert(editTargets.Count == data.Length);
                        var newSendData = GetSamplerFilterSendData(editTargets, data, name, Material.SamplerParamID.filter_max_aniso);
                        Viewer.SetMaterialSamplerFilter.Send(newSendData);
                    }
                );
        }

        private static GroupEditCommand CreateEditCommand_sampler_lod_min(GuiObjectGroup targets, string name, float lod_min)
        {
            var tuples = FilterTarget(targets, name).Where(x => x.Item2 != -1).ToArray();

            return
                new GeneralGroupReferenceEditCommand<Tuple<int, float>>(
                    new GuiObjectGroup(tuples.Select(x => x.Item1)),
                    GuiObjectID.Material,
                    tuples.Select(x => new Tuple<int, float>(x.Item2, lod_min)),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = (target as Material);
                        var tuple = (Tuple<int, float>)data;
                        var sampler = material.sampler_array.sampler[tuple.Item1];
                        swap = new Tuple<int, float>(tuple.Item1, sampler.lod.min);
                        sampler.lod.min = tuple.Item2;
                    },
                    postEditDelegate: (editTargets, data) =>
                    {
                        if (!Viewer.Manager.Instance.IsConnected) return;

                        editTargets = GetDescendantSamplerParamMaterials(editTargets, name, Material.SamplerParamID.lod_min);

                        var sendData = editTargets.OfType<Material>().Zip(data.OfType<Tuple<int, float>>(),
                            (x, y) =>
                            new Viewer.SetMaterialSamplerMinLOD.PacketData()
                            {
                                Index = (uint)y.Item1,
                                Lod = y.Item2,
                            }).OfType<object>().ToArray();
                        Viewer.SetMaterialSamplerMinLOD.Send(
                            editTargets,
                            sendData,
                            0xFFFFFFFF,
                            0xFFFFFFFF
                        );
                    }
                );
        }

        private static GroupEditCommand CreateEditCommand_sampler_lod_max(GuiObjectGroup targets, string name, float lod_max)
        {
            var tuples = FilterTarget(targets, name).Where(x => x.Item2 != -1).ToArray();

            return
                new GeneralGroupReferenceEditCommand<Tuple<int, float>>(
                    new GuiObjectGroup(tuples.Select(x => x.Item1)),
                    GuiObjectID.Material,
                    tuples.Select(x => new Tuple<int, float>(x.Item2, lod_max)),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = (target as Material);
                        var tuple = (Tuple<int, float>)data;
                        var sampler = material.sampler_array.sampler[tuple.Item1];
                        swap = new Tuple<int, float>(tuple.Item1, sampler.lod.max);
                        sampler.lod.max = tuple.Item2;
                    },
                    postEditDelegate: (editTargets, data) =>
                    {
                        if (!Viewer.Manager.Instance.IsConnected) return;

                        editTargets = GetDescendantSamplerParamMaterials(editTargets, name, Material.SamplerParamID.lod_max);

                        var sendData = editTargets.OfType<Material>().Zip(data.OfType<Tuple<int, float>>(),
                            (x, y) =>
                            new Viewer.SetMaterialSamplerMaxLOD.PacketData()
                            {
                                Index = (uint)y.Item1,
                                Lod = y.Item2,
                            }).OfType<object>().ToArray();
                        Viewer.SetMaterialSamplerMaxLOD.Send(
                            editTargets,
                            sendData,
                            0xFFFFFFFF,
                            0xFFFFFFFF
                        );
                    }
                );
        }


        private static GroupEditCommand CreateEditCommand_sampler_lod_bias(GuiObjectGroup targets, string name, float lod_bias)
        {
            var tuples = FilterTarget(targets, name).Where(x => x.Item2 != -1).ToArray();

            return
                new GeneralGroupReferenceEditCommand<Tuple<int, float>>(
                    new GuiObjectGroup(tuples.Select(x => x.Item1)),
                    GuiObjectID.Material,
                    tuples.Select(x => new Tuple<int, float>(x.Item2, lod_bias)),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = (target as Material);
                        var tuple = (Tuple<int, float>)data;
                        var sampler = material.sampler_array.sampler[tuple.Item1];
                        swap = new Tuple<int, float>(tuple.Item1, sampler.lod.bias);
                        sampler.lod.bias = tuple.Item2;
                    },
                    postEditDelegate: (editTargets, data) =>
                    {
                        if (!Viewer.Manager.Instance.IsConnected) return;

                        editTargets = GetDescendantSamplerParamMaterials(editTargets, name, Material.SamplerParamID.lod_bias);

                        var sendData = editTargets.OfType<Material>().Zip(data.OfType<Tuple<int, float>>(),
                            (x, y) =>
                            new Viewer.SetMaterialSamplerLODBias.PacketData()
                            {
                                Index = (uint)y.Item1,
                                Lod = y.Item2,
                            }).OfType<object>().ToArray();
                        Viewer.SetMaterialSamplerLODBias.Send(
                            editTargets,
                            sendData,
                            0xFFFFFFFF,
                            0xFFFFFFFF
                        );
                    }
                );
        }
        #endregion

        #region コピー＆ペースト
        [Serializable]
        public class CopyData
        {
            public sampler_arrayType sampler_array { get; set; }
            public Dictionary<string, string> convertResult { get; set; }
            public Dictionary<string, string> texturePaths { get; set; }
        }

        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public override bool CanCopy()
        {
            return true;
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public override object Copy(ref object copyObjectInfo)
        {
            return Copy(ActiveTarget);
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public static object Copy(Material target, bool notConvert = false)
        {
            var sampler_array = ObjectUtility.Clone(target.sampler_array);
            Dictionary<string, string> convertResult = null;
            if (ApplicationConfig.Preset.FollowDccSamplerNameRule && !notConvert)
            {
                convertResult = convertReservedSamplerNames(sampler_array.sampler);
            }
            var paths = sampler_array.GetItems().Select(x => x.tex_name).Where(x => !string.IsNullOrEmpty(x)).Distinct()
                .ToDictionary(x => x, x => { string y; target.Owner.ReferenceTexturePaths.TryGetValue(x, out y); return y; });
            return
                new CopyData()
                {
                    sampler_array = sampler_array,
                    convertResult = convertResult,
                    texturePaths = paths,
                };
        }

        static Dictionary<string, string> convertReservedSamplerNames(samplerType[] samplers)
        {
            var names = samplers.Select(x => x.name).Where(x => !x.StartsWith("_")).ToDictionary(x => x);
            foreach (var item in samplers)
            {
                var name = item.name;
                if (!name.StartsWith("_"))
                {
                    continue;
                }

                name = name.TrimStart(new[] {'_'});
                var nameBase = name.TrimEnd("0123456789".ToArray());
                int index=0;
                while(names.ContainsKey(name))
                {
                    name = nameBase + index;
                    index++;
                }

                // hint は空にする
                item.hint = "";
                names.Add(item.name, name);
                item.name = name;
            }
            return names;
        }

        public override bool CanPaste(object copiedObjectInfo, object copiedObject)
        {
            if (!CanPaste(Targets, copiedObject))
            {
                return false;
            }

            return base.CanPaste(copiedObjectInfo, copiedObject);
        }

        public static bool CanPaste(GuiObjectGroup Targets, object copiedObject)
        {
            var materials = Targets.GetObjects(GuiObjectID.Material).OfType<Material>().ToArray();
            var materialHash = new HashSet<Material>(materials);
            var materialOwners = materials.Select(x => x.Owner).Distinct();
            var data = (CopyData)copiedObject;
            foreach (var keyValue in data.texturePaths) {
                var texture = DocumentManager.Textures.FirstOrDefault(
                    x => x.Name == keyValue.Key && string.Compare(x.FilePath, keyValue.Value, true) == 0);

                // テクスチャがなければだめ
                if (texture == null)
                {
                    return false;
                }

                if (!materialOwners.Any(x => CanAssign_tex_name(x, (y, z) => !materialHash.Contains(y), keyValue.Key, keyValue.Value)))
                {
                    return false;
                }
            }
            return true;
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public override void Paste(object pasteObject)
        {
            // 通常のペースト
            TheApp.CommandManager.Add(Paste(Targets, pasteObject, true));
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public static ICommand Paste(GuiObjectGroup targets, object pasteObject, bool reload)
        {
            var materialOwners = targets.GetObjects(GuiObjectID.Material).OfType<Material>().Select(x => x.Owner).Distinct().ToArray();
            EditCommandSet commandSet = new EditCommandSet();
            {
                var copyData = (CopyData)pasteObject;

                commandSet.Add(new GeneralGroupReferenceEditCommand<samplerType[]>(
                        targets,
                        GuiObjectID.Material,
                        ObjectUtility.MultipleClone(copyData.sampler_array.sampler, targets.Objects.Count),
                        delegate(ref GuiObject target, ref object data, ref object swap)
                        {
                            var material = (target as Material);

                            swap = ObjectUtility.Clone(material.sampler_array.sampler);
                            material.sampler_array.sampler = (samplerType[])data;
                        },
                        createEventArgsDelegate: (x, y) => new DocumentPropertyChangedSamplerCountArgs(x, y)
                    ));

                var models = materialOwners.OfType<Model>().ToArray();
                if (models.Any())
                {
                    var modelGroup = new GuiObjectGroup(models);
                    foreach (var keyValue in copyData.texturePaths)
                    {
                        commandSet.Add(CreateEditCommand_ReferenceTexturePath(modelGroup, GuiObjectID.Model, keyValue.Key, keyValue.Value));
                    }

                    commandSet.Add(new LazyCommand(() => CreateEditCommand_RemoveTexturePaths(modelGroup, GuiObjectID.Model)));
                }

                var materialDocs = materialOwners.OfType<SeparateMaterial>().ToArray();
                if (materialDocs.Any())
                {
                    var materialDocGroup = new GuiObjectGroup(materialDocs);
                    foreach (var keyValue in copyData.texturePaths)
                    {
                        commandSet.Add(CreateEditCommand_ReferenceTexturePath(materialDocGroup, GuiObjectID.SeparateMaterial, keyValue.Key, keyValue.Value));
                    }

                    commandSet.Add(new LazyCommand(() => CreateEditCommand_RemoveTexturePaths(materialDocGroup, GuiObjectID.SeparateMaterial)));
                }

                // バインドを更新
                commandSet.Add(DocumentManager.CreateAnimationUpdateBindCommand(
                        targets.Objects.OfType<Material>().SelectMany(x => x.Referrers).Distinct().SelectMany(x => x.AllAnimations).Distinct()));

                if (reload)
                {
                    commandSet.OnPostEdit += (s,e)=>{
                        foreach (var model in targets.Objects.OfType<Material>().SelectMany(x => x.Referrers))
                        {
                            Viewer.LoadOrReloadModel.Send(model);
                        }
                    };
                }

                commandSet.Execute();
            }

            return commandSet;
        }
        #endregion


        #region コンテキストメニュー関連
        [Serializable]
        public class CopyData_SingleSampler
        {
            public samplerType sampler { get; set; }
            public string FilePath { get; set; }
        }

        private void cmsSampler_Opening(object sender, CancelEventArgs e)
        {
            // 選択された行があれば、コピー可
            if (lvwList.SelectedItems.Count == 0)
            {
                cmiCopy.Enabled = false;
                cmiIsRequiredForChild.Enabled = false;
            }
            else
            {
                cmiCopy.Enabled = true;
                cmiIsRequiredForChild.Enabled = true;
            }

            var materials = Targets.GetObjects(GuiObjectID.Material).OfType<Material>().ToArray();
            // コピーされたことがあれば、ペーストメニューが有効に
            if (OneSamplerCopied_ != null)
            {
                if (!string.IsNullOrEmpty(OneSamplerCopied_.sampler.tex_name))
                {
                    if (!string.IsNullOrEmpty(OneSamplerCopied_.FilePath))
                    {
                        var texture = DocumentManager.Textures.FirstOrDefault(
                            x => x.Name == OneSamplerCopied_.sampler.tex_name && string.Compare(x.FilePath, OneSamplerCopied_.FilePath, true) == 0);

                        if (texture == null)
                        {
                            cmiPaste.Enabled = false;
                            return;
                        }

                        var materialHash = new HashSet<Material>(materials);
                        var materialOwners = materials.Select(x => x.Owner).Distinct();
                        if (!materialOwners.All(x => CanAssign_tex_name(x,
                            (y,z)=>!materialHash.Contains(y) || IsPreservedSampler(z) || z.name != OneSamplerCopied_.sampler.name,
                            texture.Name,
                            texture.FilePath)))
                        {
                            cmiPaste.Enabled = false;
                            return;
                        }
                    }
                }
                cmiPaste.Enabled = true;
            }
            else
            {
                cmiPaste.Enabled = false;
            }

            if (lvwList.SelectedItems.Count == 0) return;

            var isRequiredForChild = GetSelectedSamplers().Select(x => ActiveTarget.GetReferenceBehaviorSampler(x.name)?.IsRequiredForChild ?? false).Distinct();
            cmiIsRequiredForChild_Yes.Checked = isRequiredForChild.Contains(true);
            cmiIsRequiredForChild_No.Checked = isRequiredForChild.Contains(false);
        }

        private void cmsSamplerParam_Opening(object sender, CancelEventArgs e)
        {
            if (lvwList.SelectedItems.Count == 0) return;

            var samplers = GetSelectedSamplers().ToArray();

            var control = cmsSamplerParam.ProxySourceControl;

            if (control is UIModifiedMarkLabel)
            {
                var label = (UIModifiedMarkLabel)control;
                var paramId = label.Tag as Material.SamplerParamID?;
                if (paramId.HasValue)
                {
                    var id = paramId.Value;
                    var isReferenceEditable = samplers.All(sampler => ActiveTarget.IsResolvedReferenceBehavioSamplerParamEditable(sampler.name, id));

                    var restrictionStates = samplers.Select(sampler => ActiveTarget.GetReferenceBehaviorSamplerParam(sampler.name, id)?.ChildRestriction ?? ShaderItemRestrictionState.None).Distinct().ToArray();
                    cmiMaterialReferenceBehaviorRestriction_None.Checked = restrictionStates.Contains(ShaderItemRestrictionState.None);
                    cmiMaterialReferenceBehaviorRestriction_ForceOverride.Checked = restrictionStates.Contains(ShaderItemRestrictionState.ForceOverride);
                    cmiMaterialReferenceBehaviorRestriction_ForceRefer.Checked = restrictionStates.Contains(ShaderItemRestrictionState.ForceRefer);

                    var valueStates =       samplers.Select(sampler => ActiveTarget.GetReferenceBehaviorSamplerParam(sampler.name, id)?.Value ?? ShaderItemValueState.Refer).Distinct().ToArray();
                    cmiMaterialReferenceBehaviorValue_Refer.Checked = valueStates.Contains(ShaderItemValueState.Refer);
                    cmiMaterialReferenceBehaviorValue_Override.Checked = valueStates.Contains(ShaderItemValueState.Override);
                    cmiMaterialReferenceBehaviorValue_Default.Checked = valueStates.Contains(ShaderItemValueState.Default);

                    cmiMaterialReferenceBehaviorValue.Enabled = isReferenceEditable;
                    cmiMaterialReferenceBehaviorRestriction.Enabled = isReferenceEditable;
                }
            }
        }

        private void cmiCopy_Click(object sender, EventArgs e)
        {
            Debug.Assert(lvwList.SelectedIndex != -1);

            var sampler = ResolvedSampler;
            string path = null;
            if (sampler.tex_name != null)
            {
                var tex = ActiveTarget.Owner.GetReferenceTexture(sampler.tex_name);
                if (tex != null)
                {
                    path = tex.FilePath;
                }
            }
            OneSamplerCopied_ = new CopyData_SingleSampler()
            {
                sampler =  ObjectUtility.Clone(sampler),
                FilePath = path,
            };
        }

        private int SameNameSamplerIndex(List<samplerType> samplers, samplerType sampler)
        {
            int index = 0;
            foreach (var samp in samplers)
            {
                if (samp.name == sampler.name)
                {
                    return index;
                }
                index++;
            }

            return -1;
        }

        private void cmiPaste_Click(object sender, EventArgs e)
        {
            int MultiSelect = Targets.GetObjects(GuiObjectID.Material).Count;
            samplerType copySampler = OneSamplerCopied_.sampler;

            List<samplerType[]> targetSamplers = new List<samplerType[]>();
            GuiObjectGroup group = new GuiObjectGroup();

            // マテリアルごとに、サンプラーのリストを作成する。
            foreach (var obj in Targets.GetObjects(GuiObjectID.Material))
            {
                Material material = obj as Material;
                if (material != null &&
                    material.sampler_array != null &&
                    material.sampler_array.sampler != null)
                {
                    // サンプラーアレイのクローン
                    List<samplerType> samplers = new List<samplerType>();
                    {
                        foreach (var sampler in material.sampler_array.sampler)
                        {
                            samplers.Add(ObjectUtility.Clone(sampler));
                        }
                    }

                    InsertOrOverwriteSampler(samplers, ObjectUtility.Clone(copySampler), lvwList.SelectedIndex, null);

                    group.Add(material);
                    targetSamplers.Add(samplers.ToArray());
                }
            }

            if (targetSamplers.Any())
            {
                var materialOwners = Targets.GetObjects(GuiObjectID.Material).OfType<Material>().Select(x => x.Owner).Distinct().ToArray();
                var commandSet = new EditCommandSet();
                commandSet.Add(Material.ExecuteEditCommand_SamplerArray(group, targetSamplers, false));
                if (!string.IsNullOrEmpty(OneSamplerCopied_.sampler.tex_name))
                {
                    var models = materialOwners.OfType<Model>().ToArray();
                    if (models.Any())
                    {
                        commandSet.Add(CreateEditCommand_ReferenceTexturePath(
                            new GuiObjectGroup(models),
                            GuiObjectID.Model,
                            OneSamplerCopied_.sampler.tex_name,
                            OneSamplerCopied_.FilePath).Execute());
                        commandSet.Add(CreateEditCommand_RemoveTexturePaths(new GuiObjectGroup(models), GuiObjectID.Model).Execute());
                    }

                    var materialDocs = materialOwners.OfType<SeparateMaterial>().ToArray();
                    if (materialDocs.Any())
                    {
                        commandSet.Add(CreateEditCommand_ReferenceTexturePath(
                            new GuiObjectGroup(materialDocs),
                            GuiObjectID.SeparateMaterial,
                            OneSamplerCopied_.sampler.tex_name,
                            OneSamplerCopied_.FilePath).Execute());
                        commandSet.Add(CreateEditCommand_RemoveTexturePaths(new GuiObjectGroup(materialDocs), GuiObjectID.SeparateMaterial).Execute());
                    }
                }

                EventHandler send = (s, a) =>
                {
                    foreach (var model in Targets.GetObjects(GuiObjectID.Material).OfType<Material>().SelectMany(x => x.Referrers))
                    {
                        Viewer.LoadOrReloadModel.Send(model);
                    }
                };

                send(null, EventArgs.Empty);
                commandSet.OnPostEdit += send;
                commandSet.Reverse();

                TheApp.CommandManager.Add(commandSet);
            }
        }

        private void cmiMaterialReferenceBehavior_Click(object sender, EventArgs e)
        {
            var control = cmsSamplerParam.ProxySourceControl;

            if (!(control is UIModifiedMarkLabel)) return;

            var label = (UIModifiedMarkLabel) control;
            var paramId = label.Tag as Material.SamplerParamID?;
            var samplerName = SelectedSamplerName;

            if (!string.IsNullOrEmpty(samplerName) && paramId.HasValue)
            {
                var id = paramId.Value;
                var materials = FilterTarget(Targets, samplerName).Where(x => x.Item2 != -1).Select(x => x.Item1).ToArray();
                var commandSet = new EditCommandSet();
                foreach (var material in materials)
                {
                    var behaviorItem = ObjectUtility.Clone(ActiveTarget.GetReferenceBehaviorSamplerParam(samplerName, id)) ??
                                       new MaterialReferenceBehaviorItem() {Id = Enum.GetName(typeof(Material.SamplerParamID), id)};
                    if (sender == cmiMaterialReferenceBehaviorValue_Refer)
                    {
                        behaviorItem.Value = ShaderItemValueState.Refer;
                    }
                    else if (sender == cmiMaterialReferenceBehaviorValue_Override)
                    {
                        behaviorItem.Value = ShaderItemValueState.Override;
                    }
                    else if (sender == cmiMaterialReferenceBehaviorValue_Default)
                    {
                        behaviorItem.Value = ShaderItemValueState.Default;
                    }
                    else if (sender == cmiMaterialReferenceBehaviorRestriction_None)
                    {
                        behaviorItem.ChildRestriction = ShaderItemRestrictionState.None;
                    }
                    else if (sender == cmiMaterialReferenceBehaviorRestriction_ForceRefer)
                    {
                        behaviorItem.ChildRestriction = ShaderItemRestrictionState.ForceRefer;
                    }
                    else if (sender == cmiMaterialReferenceBehaviorRestriction_ForceOverride)
                    {
                        behaviorItem.ChildRestriction = ShaderItemRestrictionState.ForceOverride;
                    }

                    commandSet.Add(material.CreateSetReferenceBehaviorSamplerCommand(behaviorItem, samplerName).Execute());
                }
                commandSet.Reverse();
                TheApp.CommandManager.Add(commandSet);
                App.AppContext.OnMaterialReferenceBehaviorSettingChanged();
            }
        }

        private void cmiIsRequiredForChild_Click(object sender, EventArgs e)
        {
            var samplerName = SelectedSamplerName;
            if (!string.IsNullOrEmpty(samplerName))
            {
                var materials = FilterTarget(Targets, samplerName).Where(x => x.Item2 != -1).Select(x => x.Item1).ToArray();

                var isRequiredForChild = sender == cmiIsRequiredForChild_Yes;
                var commandSet = new EditCommandSet();
                foreach (var material in materials)
                {
                    var behaviorItem = ObjectUtility.Clone(material.GetReferenceBehaviorSampler(samplerName)) ??
                                       new SamplerBehaviorItem() { Id = samplerName };
                    behaviorItem.IsRequiredForChild = isRequiredForChild;
                    commandSet.Add(material.CreateSetSamplerBehaviorItemCommand(behaviorItem).Execute());
                }
                commandSet.Reverse();
                TheApp.CommandManager.Add(commandSet);
                App.AppContext.OnMaterialReferenceBehaviorSettingChanged();
            }
        }

        public static void InsertOrOverwriteSampler(List<samplerType> samplers, samplerType source, int index, HashSet<string> sourceNames)
        {
            // 上書きする位置
            int pos;
            if (ApplicationConfig.Preset.FollowDccSamplerNameRule)
            {
                pos = samplers.FindIndex(x => x.name == source.name && x.hint == source.hint);
            }
            else
            {
                pos = samplers.FindIndex(x => x.name == source.name);
            }

            // 挿入位置
            if (index < 0)
            {
                index = samplers.Count;
            }
            else
            {
                index = Math.Min(index, samplers.Count);
            }

            var name = source.name;
            var hint = source.hint;
            if (ApplicationConfig.Preset.FollowDccSamplerNameRule)
            {
                // 予約名
                if (name.StartsWith("_"))
                {
                    if (pos != -1 && samplers[pos].hint == hint && hint != "")
                    {
                        // テクスチャ名は引き継ぐ上書き
                        source.tex_name = samplers[pos].tex_name;
                    }
                    else
                    {
                        name = name.TrimStart(new[] { '_' });
                        hint = "";
                        pos = -1;
                    }
                }

                // 同名があれば名前変更
                if (pos == -1)
                {
                    var baseName = name.TrimEnd("0123456789".ToArray());
                    int nameIndex = 0;
                    while (samplers.Any(x => x.name == name) || (nameIndex != 0 && sourceNames != null && sourceNames.Contains(name)))
                    {
                        name = baseName + nameIndex;
                        nameIndex++;
                    }
                }

                // ヒントが重複していれば消去
                if (hint != "" && pos == -1 && samplers.Any(x => x.hint == hint))
                {
                    hint = "";
                }
            }

            source.name = name;
            source.hint = hint;
            if (pos != -1)
            {
                samplers[pos] = source;
            }
            else
            {
                samplers.Insert(index, source);
            }
        }

        #endregion

        private void btnPropertyDialog_Click(object sender, EventArgs e)
        {
            var sampler = lvwList.SelectedItemData as samplerType;
            if (sampler != null)
            {
                var texture = ActiveTarget.Owner.GetReferenceTexture(sampler.tex_name);
                if (texture != null)
                {
                    using (var wait = new WaitCursor())
                    {
                        // プロパティウィンドウを生成
                        var objectGroup = new GuiObjectGroup(texture);

                        ObjectPropertyDialog dialog = new ObjectPropertyDialog(null, objectGroup);
                        dialog.TargetFixed = true;

                        // 現在のウィンドウの上に表示する
                        Control parent = Owner;
                        while (parent != null)
                        {
                            if (parent is ObjectPropertyDialog)
                            {
                                dialog.SetLocationNextTo((ObjectPropertyDialog)parent);
                                break;
                            }

                            parent = parent.Parent;
                        }

                        dialog.Show();
                    }
                }
            }
        }

        private void lvwList_DragDrop(object sender, DragEventArgs e)
        {
            base.OnDragDrop(e);
            if (!e.Data.GetDataPresent(DataFormats.FileDrop)
                && !e.Data.GetDataPresent(typeof(FileTreeView.ViewItem))) return;

            var commandSets = new EditCommandSet();

            var textures = DropedTextures(e, commandSets);

            var pos = lvwList.PointToClient(new Point(e.X, e.Y));
            var targetItem = lvwList.GetItemAt(pos.X, pos.Y);

            // サンプラを追加
            var materials = Targets.Objects.OfType<Material>().ToList();
            // 存在するサンプラー上にドロップされた時はそのサンプラの名前を使う（上書き）
            // 複数ファイルのときは許可しない
            string targetSamplerName = null;
            if (textures.Count() == 1 && targetItem != null && targetItem.Tag is samplerType)
            {
                targetSamplerName = (targetItem.Tag as samplerType).name;
            }

            SetSampler(materials, textures, commandSets, targetSamplerName);

            var models = materials.SelectMany(x => x.Referrers).ToArray();
            EventHandler reload = (s, a) =>
            {
                foreach (var model in models)
                {
                    Viewer.LoadOrReloadModel.Send(model);
                }
            };

            reload(null, EventArgs.Empty);

            commandSets.OnPostEdit += reload;

            commandSets.Reverse();

            TheApp.CommandManager.Add(commandSets);
        }

        public static void SetSampler(List<Material> materials, Texture[] textures, EditCommandSet commandSets, string targetSamplerName = null)
        {
            var newSamplers = materials.ToDictionary(
                material => material,
                material => ObjectUtility.Clone(material.sampler_array.sampler));

            foreach (var texture in textures)
            {
                var texname = texture.Name;
                var samplerArrays = new List<samplerType[]>();
                var samplerName = StringUtility.UniqueName(
                    ApplicationConfig.Preset.FollowDccSamplerNameRule ? "sampler" : "_a",
                    newSamplers.SelectMany(x => x.Value.Select(y => y.name)),
                    Enumerable.Range(0, int.MaxValue).Select(x => x.ToString()));
                foreach (var target in materials)
                {
                    var newSamplerArray = ObjectUtility.Clone(newSamplers[target]).ToList();

                    // ドロップターゲットのサンプラ名が既に存在すれば上書きする。そうでなければ新規作成
                    if (newSamplerArray.Any(x => x.name == targetSamplerName))
                    {
                        var sampler = newSamplerArray.FirstOrDefault(x => x.name == targetSamplerName);

                        if (sampler != null && sampler.tex_name != texname)
                        {
                            sampler.tex_name = texname;
                        }
                    }
                    else
                    {
                        var newSampler = new samplerType()
                                             {
                                                 name = targetSamplerName ?? samplerName,
                                                 hint = string.Empty,
                                                 tex_name = texname,
                                                 wrap =
                                                     new wrapType()
                                                         {
                                                             u = ApplicationConfig.DefaultValue.wrap.u,
                                                             v = ApplicationConfig.DefaultValue.wrap.v,
                                                             w = ApplicationConfig.DefaultValue.wrap.w,
                                                         },
                                                 filter =
                                                     new filterType()
                                                         {
                                                             mag =
                                                                 ApplicationConfig.DefaultValue.filter
                                                                 .mag,
                                                             min =
                                                                 ApplicationConfig.DefaultValue.filter
                                                                 .min,
                                                             mip =
                                                                 ApplicationConfig.DefaultValue.filter
                                                                 .mip,
                                                             max_aniso =
                                                                 ApplicationConfig.DefaultValue.filter
                                                                 .max_aniso,
                                                         },
                                                 lod =
                                                     new lodType()
                                                         {
                                                             min = ApplicationConfig.DefaultValue.lod.min,
                                                             max = ApplicationConfig.DefaultValue.lod.max,
                                                             bias = ApplicationConfig.DefaultValue.lod.bias,
                                                         },
                                             };

                        newSamplerArray.Add(newSampler);
                    }
                    newSamplers[target] = newSamplerArray.ToArray();
                }
            }
            commandSets.Add(Material.ExecuteEditCommand_SamplerArray(
                    new GuiObjectGroup(newSamplers.Select(x => x.Key)),
                    newSamplers.Select(x => x.Value).ToList(),
                    false));

            var materialOwners = materials.Select(x => x.Owner).Distinct().ToArray();
            var models = materialOwners.OfType<Model>().ToArray();
            var materialDocs = materialOwners.OfType<SeparateMaterial>().ToArray();
            foreach (var texture in textures)
            {
                if (models.Any())
                {
                    commandSets.Add(
                        CreateEditCommand_ReferenceTexturePath(
                            new GuiObjectGroup(models),
                            GuiObjectID.Model,
                            texture.Name,
                            texture.FilePath).Execute()
                        );
                }
                if (materialDocs.Any())
                {
                    commandSets.Add(
                        CreateEditCommand_ReferenceTexturePath(
                            new GuiObjectGroup(materialDocs),
                            GuiObjectID.SeparateMaterial,
                            texture.Name,
                            texture.FilePath).Execute()
                        );
                }
            }

            if (models.Any())
            {
                commandSets.Add(
                CreateEditCommand_RemoveTexturePaths(
                    new GuiObjectGroup(models), GuiObjectID.Model).Execute());
            }
            if (materialDocs.Any())
            {
                commandSets.Add(
                CreateEditCommand_RemoveTexturePaths(
                    new GuiObjectGroup(materialDocs), GuiObjectID.SeparateMaterial).Execute());
            }
        }

        public static Texture[] DropedTextures(DragEventArgs e, EditCommandSet commandSets)
        {
            var files = (string[])e.Data.GetData(DataFormats.FileDrop);
            if (files != null) // ファイルをドロップ
            {
                if (
                    files.Select(Path.GetExtension)
                        .Any(
                            x => ObjectIDUtility.ExtToId(x) != GuiObjectID.Texture && ObjectIDUtility.IsTgaExtension(x) == false))
                {
                    return null;
                }

                // 既にファイルが開かれているか確認
                var preOpenFiles =
                    DocumentManager.Textures
                        .Where(
                            tex =>
                            files.Any(file => file.Equals(tex.FilePath, StringComparison.OrdinalIgnoreCase)))
                        .ToArray();

                if (preOpenFiles.Any())
                {
                    using (var dialog = new OkCancelTextBoxDialog())
                    {
                        dialog.lblDescription.Text = Strings.MaterialSamplerPage_lvwList_DragDrop_WarnSameFile;
                        foreach (var texture in preOpenFiles)
                        {
                            dialog.AddLine(texture.FilePath);
                        }

                        var result = dialog.ShowDialog();
                        if (result == DialogResult.Cancel)
                        {
                            return null;
                        }
                    }
                }

                // ファイルを開く
                DocumentManager.LoadFromFile(
                    files.Select(x => new DocumentManager.PathWithName(x, null)),
                    commandSet: commandSets,
                    isUpdateRecentlyUsedFiles: true,
                    isShowErrorDialog: false);

                Texture[] textures;
                textures = files.Select(x => DocumentManager.Textures.FirstOrDefault(y => string.Compare(x, y.FilePath, true) == 0))
                    .Where(x => x != null).ToArray();
                if (textures.Length == 0)
                {
                    return null;
                }

                return textures;
            }
            else // ツリービューからテクスチャをドロップ
            {
                if (e.Data.GetDataPresent(typeof(FileTreeView.ViewItem)))
                {
                    var selectedNodes = App.AppContext.FileTreeView?.SelectedNodes ?? new List<TreeNode>();
                    if (selectedNodes.Count == 0)
                    {
                        return null;
                    }

                    Texture[] textures;
                    textures = new Texture[selectedNodes.Count];
                    for (int idxNode = 0; idxNode < selectedNodes.Count; idxNode++)
                    {
                        textures[idxNode] = selectedNodes[idxNode].Tag as Texture;
                    }

                    return textures;
                }
            }

            return null;
        }

        private void lvwList_DragEnter(object sender, DragEventArgs e)
        {
            base.OnDragEnter(e);
        }

        private void lvwList_DragOver(object sender, DragEventArgs e)
        {
            base.OnDragOver(e);
            e.Effect = DragDropEffects.None;
            var pos = lvwList.PointToClient(new Point(e.X, e.Y));
            var targetItem = lvwList.GetItemAt(pos.X, pos.Y);
            if (targetItem != null)
            {
                //DebugConsole.WriteLine(targetItem.Text);
            }

            var materials = Targets.GetObjects(GuiObjectID.Material).OfType<Material>().ToArray();
            CheckTextureDragOver(materials, e, targetItem);
        }

        public static void CheckTextureDragOver(IEnumerable<Material> materials, DragEventArgs e, ListViewItem targetItem = null)
        {
            if (materials == null)
            {
                return;
            }

            var materialArray = materials as Material[] ?? materials.ToArray();
            if (targetItem == null && materialArray.Any(x => x.DisallowAddSampler()))
            {
                return;
            }

            // ファイルであり、モーダルでないかファイルダイアログが開いている時はドラッグＯＫ
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
            {
                var files = (string[])e.Data.GetData(DataFormats.FileDrop);
                if (files == null)
                {
                    return;
                }

                var materialOwners = materialArray.Select(x => x.Owner).Distinct().ToArray();
                var materialHash = new HashSet<Material>(materialArray);
                var targetSampler = targetItem != null ? ((samplerType)(targetItem.Tag)).name : null;
                var texturePathToName = DocumentManager.Textures.ToDictionary(x => x.FilePath.ToLower(), x => x.Name);
                if (!files.All(x =>
                    {
                        var ext = Path.GetExtension(x);
                        if (ObjectIDUtility.ExtToId(ext) != GuiObjectID.Texture && ObjectIDUtility.IsTgaExtension(ext) == false)
                        {
                            return false;
                        }

                        string name;

                        // ファイル名は大文字小文字を区別せず、オブジェクト名は区別するための処置
                        if (!texturePathToName.TryGetValue(x.ToLower(), out name))
                        {
                            name = Path.GetFileNameWithoutExtension(x);
                        }

                        return materialOwners.All(y => CanAssign_tex_name(y, (z, w) => !materialHash.Contains(z) || w.name != targetSampler, name, x));
                    }))
                {
                    return;
                }


                e.Effect = (targetItem == null || files.Count() > 1)
                               ? DragDropEffects.Copy
                               : (ConfigData.ApplicationConfig.Preset.FollowDccSamplerNameRule
                                  && ((samplerType)targetItem.Tag).name.StartsWith("_")
                                  && !string.IsNullOrEmpty(((samplerType)targetItem.Tag).hint))
                                     ? DragDropEffects.None
                                     : DragDropEffects.Move;
            }
            else if (e.Data.GetDataPresent(typeof(FileTreeView.ViewItem)))
            {
                var source = (FileTreeView.ViewItem)e.Data.GetData(typeof(FileTreeView.ViewItem));
                if (source == null)
                {
                    return;
                }

                var data = source.Tag as Texture;

                if (data != null)
                {
                    var materialOwners = materialArray.Select(x => x.Owner).Distinct().ToArray();
                    var materialHash = new HashSet<Material>(materialArray);
                    var targetSampler = targetItem != null ? ((samplerType)(targetItem.Tag)).name : null;
                    if (!materialOwners.All(y => CanAssign_tex_name(y, (z, w) => !materialHash.Contains(z) || w.name != targetSampler, data.Name, data.FilePath)))
                    {
                        return;
                    }

                    e.Effect = targetItem == null
                                   ? DragDropEffects.Copy
                                   : (ConfigData.ApplicationConfig.Preset.FollowDccSamplerNameRule
                                      && ((samplerType)targetItem.Tag).name.StartsWith("_")
                                      && !string.IsNullOrEmpty(((samplerType)targetItem.Tag).hint))
                                         ? DragDropEffects.None
                                         : DragDropEffects.Move;
                }
            }
        }

    }

    public class DocumentPropertyChangedSamplerCountArgs : DocumentContentsChangedArgs
    {
        public DocumentPropertyChangedSamplerCountArgs(GuiObject target, EditCommand command) : base(target, command) { }
    }
}
