﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using App.Command;
using App.ConfigData;
using App.Controls;
using App.Data;
using App.Utility;

using ConfigCommon;

using nw.g3d.nw4f_3dif;

using Viewer;

namespace App.PropertyEdit
{
    public partial class FogAnimationGeneralPage : FogAnimationPropertyPage
    {
        public FogAnimationGeneralPage() :
            base(PropertyPageID.FogAnimationGeneral)
        {
            InitializeComponent();
            InitComboBox();
            // 減衰
            InitFogAnimFloatEditPanel(fepDistAttnStart, fog_anim_target_targetType.dist_attn_start);
            InitFogAnimFloatEditPanel(fepDistAttnEnd,   fog_anim_target_targetType.dist_attn_end);
            // カラー
            InitFogAnimColorEditPanel(cepColor, new[]{
                                           fog_anim_target_targetType.color_r,
                                           fog_anim_target_targetType.color_g,
                                           fog_anim_target_targetType.color_b});
        }

        private void InitComboBox()
        {
            cmbDistAttnFunc.Items.Clear();

            foreach (var attn in ApplicationConfig.Preset.FogDistAttnPresets)
            {
                cmbDistAttnFunc.AddItem(attn.Label, attn.Function);
            }
        }

        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_animation_property_window_general_page;
            }
        }

        public static ObjectPropertyPage CreateInstance(object arg)
        {
            return new FogAnimationGeneralPage();
        }

        protected override void InitializeFormInternal()
        {
        }

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            var target = ActiveTarget;
            var preset = ApplicationConfig.Preset;

            // 全般
            lblFrameSize.IsModified = target.IsValueChanged(x => x.frame_count);
            iepFrameSize.Value = target.Data.frame_count;

            cbxLoop.IsModified = target.IsValueChanged(x => x.loop);
            cbxLoop.Checked = target.Data.loop;

            lblDistAttnFunc.IsModified = target.IsValueChanged(x => x.dist_attn_func);
            //sibDistAttnFunc.Text = target.Data.dist_attn_func;

            InitComboBox();
            var distfuncidx =
                preset.FogDistAttnPresets.FindIndex(x => x.Function == target.Data.dist_attn_func);

            if (distfuncidx < 0)
            {
                cmbDistAttnFunc.AddItem(target.Data.dist_attn_func, target.Data.dist_attn_func);
                distfuncidx = cmbDistAttnFunc.Items.Count - 1;
            }
            cmbDistAttnFunc.SelectedIndex = distfuncidx;

            // 減衰
            UpdateFogAnimFloatEditPanel(fepDistAttnStart, lblDistAttnStart, fog_anim_target_targetType.dist_attn_start);
            UpdateFogAnimFloatEditPanel(fepDistAttnEnd, lblDistAttnEnd, fog_anim_target_targetType.dist_attn_end);
            // カラー
            UpdateFogAnimColorEditPanel(cepColor, lblColor, new[]{
                                           fog_anim_target_targetType.color_r,
                                           fog_anim_target_targetType.color_g,
                                           fog_anim_target_targetType.color_b});

        }
        private void InitFogAnimFloatEditPanel(FloatEditPanel floatEditPanel, fog_anim_target_targetType targetType)
        {
            // tagに編集前の値を入れて保持する
            floatEditPanel.Tag = null;

            floatEditPanel.SequentialValueChanged += (sender, args) =>
            {
                var animationCurve = new FogAnimationCurveTreeNodeInfo(ActiveTarget, targetType);
                FloatEditValueChanged(floatEditPanel, targetType, args);
            };
        }

        private void FloatEditValueChanged(FloatEditPanel floatEditPanel, fog_anim_target_targetType targetType, SequentialValueChangedEventArgs args)
        {
            var animationCurve = new FogAnimationCurveTreeNodeInfo(ActiveTarget, targetType);
            var value = floatEditPanel.Value;
            if (!(floatEditPanel.Tag is float))
            {
                floatEditPanel.Tag = animationCurve.paramAnim.GetBaseValue();
            }

            if (args.Changing && animationCurve.HasKeyFrame())
            {
                animationCurve.KeyFrames.ForEach(x => x.Value = value);
                CameraAnimationGeneralPage.SendAnimationCurve(ActiveTarget, animationCurve);
            }
            else
            {
                var oldValue = (float)floatEditPanel.Tag;
                floatEditPanel.Tag = null;
                animationCurve.KeyFrames.ForEach(x => x.Value = oldValue);

                var commandSet = CreateEditCommand_ConstantAnimation(new GuiObjectGroup(ActiveTarget), targetType, value);
                TheApp.CommandManager.Add(commandSet.Execute());
            }
        }

        private void UpdateFogAnimFloatEditPanel(FloatEditPanel floatEditPanel, UIModifiedMarkLabel uilabel, fog_anim_target_targetType targetType, bool enable = true)
        {
            var animTarget = ActiveTarget.GetTarget(targetType);
            CameraAnimationGeneralPage.UpdateFloatEditPanel(floatEditPanel, uilabel, animTarget, enable);
        }

        private void InitFogAnimColorEditPanel(ColorEditPanel colorEditPanel, fog_anim_target_targetType[] targetTypes)
        {
            // tagに編集前の値を入れて保持する
            colorEditPanel.Tag = null;
            colorEditPanel.ColorPickerText = res.Strings.ColorPicker;
            colorEditPanel.IsDefaultLinear = ConfigData.ApplicationConfig.Color.GammaCorrection;
            colorEditPanel.SequentialValueChanged += (sender, args) =>
            {
                ColorEditValueChanged(colorEditPanel, targetTypes, args);
            };
        }

        private void ColorEditValueChanged(ColorEditPanel colorEditPanel, fog_anim_target_targetType[] targetTypes, SequentialValueChangedEventArgs args)
        {
            Debug.Assert(targetTypes != null && targetTypes.Count() == 3);
            var animationCurves = targetTypes.Select(x => new FogAnimationCurveTreeNodeInfo(ActiveTarget, x)).ToList();
            var color = colorEditPanel.Color;
            if (!(colorEditPanel.Tag is RgbaColor))
            {
                var colors = new float[3];
                var idx = 0;
                foreach (var animationCurve in animationCurves)
                {
                    colors[idx] = animationCurve.paramAnim.GetBaseValue();
                    idx++;
                }
                colorEditPanel.Tag = new RgbaColor(colors[0], colors[1], colors[2], 1.0f);
            }

            if (args.Changing && animationCurves.All(x => x.HasKeyFrame()))
            {
                var document = (AnimationDocument)ActiveTarget.OwnerDocument;
                animationCurves[0].KeyFrames.ForEach(x => x.Value = color.R);
                animationCurves[1].KeyFrames.ForEach(x => x.Value = color.G);
                animationCurves[2].KeyFrames.ForEach(x => x.Value = color.B);

                foreach (var animationCurve in animationCurves)
                {
                    CameraAnimationGeneralPage.SendAnimationCurve(ActiveTarget, animationCurve);
                }
            }
            else
            {
                var oldColor = (RgbaColor)colorEditPanel.Tag;
                colorEditPanel.Tag = null;
                animationCurves[0].KeyFrames.ForEach(x => x.Value = oldColor.R);
                animationCurves[1].KeyFrames.ForEach(x => x.Value = oldColor.G);
                animationCurves[2].KeyFrames.ForEach(x => x.Value = oldColor.B);
                var commandSet = new EditCommandSet();
                commandSet.Add(CreateEditCommand_ConstantAnimation(Targets, targetTypes[0], color.R));
                commandSet.Add(CreateEditCommand_ConstantAnimation(Targets, targetTypes[1], color.G));
                commandSet.Add(CreateEditCommand_ConstantAnimation(Targets, targetTypes[2], color.B));
                TheApp.CommandManager.Add(commandSet.Execute());
            }
        }

        private void UpdateFogAnimColorEditPanel(ColorEditPanel colorEditPanel, UIModifiedMarkLabel uilabel, fog_anim_target_targetType[] targetTypes)
        {
            Debug.Assert(targetTypes != null && targetTypes.Count() == 3);
            var animTargets = targetTypes.Select(ActiveTarget.GetTarget).ToArray();
            LightAnimationGeneralPage.UpdateColorEditPanel(colorEditPanel, uilabel, animTargets);
        }

        public static bool IsModified(FogAnimation activeTarget)
        {
            if (activeTarget == null)
            {
                return false;
            }
            var targetTypes = Enum.GetValues(typeof(fog_anim_target_targetType)).OfType<fog_anim_target_targetType>();
            var animModified = targetTypes.Select(activeTarget.GetTarget).Any(x => x != null && x.IsModified);

            return animModified ||
                activeTarget.IsValueChanged(x => x.frame_count) ||
                activeTarget.IsValueChanged(x => x.loop) ||
                activeTarget.IsValueChanged(x => x.dist_attn_func);
        }

        #region コマンド
        public static GroupEditCommand CreateEditCommand_frame_count(GuiObjectGroup targets, int FrameCount)
        {
            return
                new GeneralGroupValueEditCommand<int>(
                    targets,
                    GuiObjectID.FogAnimation,
                    FrameCount,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var anim = target as FogAnimation;

                        swap = anim.Data.frame_count;
                        anim.Data.frame_count = (int)data;
                        AnimationDocument.NotifyFrameCountChanged(anim, EventArgs.Empty);

                        //  Viewerへ再転送
                        Viewer.LoadOrReloadAnimation.Send(anim.Owner);
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        ;	// リロードはまとめて送れないのでEditDelegateで送る
                            // 現状複数選択は無いので、こちらでは送らない。
                    }
                );
        }

        public static GroupEditCommand CreateEditCommand_loop(GuiObjectGroup targets, bool value)
        {
            return
                new GeneralGroupValueEditCommand<bool>(
                    targets,
                    GuiObjectID.FogAnimation,
                    value,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var anim = target as FogAnimation;

                        swap = anim.Data.loop;
                        anim.Data.loop = (bool)data;

                        // Viewerへ転送
                        Viewer.LoadOrReloadAnimation.Send(anim.Owner);
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        ;	// リロードはまとめて送れないのでEditDelegateで送る
                            // 現状複数選択は無いので、こちらでは送らない。
                    }
                );
        }

        private static EditCommandSet CreateEditCommand_ConstantAnimation(
            GuiObjectGroup targets,
            fog_anim_target_targetType targetType,
            float value)
        {
            var commandSet = new EditCommandSet();
            var active = targets.Active as FogAnimation;
            if (active == null)
            {
                return commandSet;
            }
            var animCurve = new FogAnimationCurveTreeNodeInfo(active, targetType);

            var newKeys = new List<KeyFrame>();
            if (animCurve.KeyFrames.Any())
            {
                newKeys.AddRange(ObjectUtility.Clone(animCurve.KeyFrames));
                newKeys.ForEach(x => x.Value = value);
            }
            else
            {
                var key = new KeyFrame { Frame = 0, Value = value };
                newKeys.Add(key);
            }
            commandSet.Add(AnimationCurveEditCommand.Create(targets, GuiObjectID.FogAnimation, animCurve, newKeys));

            commandSet.Add(
                new LazyCommand(() => AnimationCurveEditCommand.CreateQuantizeAnalyseAllCommand(targets.Active, false)));
            commandSet.OnPostEdit += (s, e) =>
            {
                if (active.OwnerDocument is AnimationDocument)
                {
                    LoadOrReloadAnimation.Send((AnimationDocument)active.OwnerDocument);
                }
            };
            return commandSet;
        }
        #endregion

        #region コピー＆ペースト
        private class CopyData
        {
            // データ
            public int	frame_count { get; set; }
            public bool	loop { get; set; }
            public string dist_attn_func;
            // アニメーション固定値
            public Dictionary<fog_anim_target_targetType, float> constantAnimTargets = new Dictionary<fog_anim_target_targetType, float>();
        }

        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public override bool CanCopy()
        {
            return true;
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public override object Copy(ref object copyObjectInfo)
        {
            return Copy(ActiveTarget);
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public static object Copy(FogAnimation target)
        {
            var data =
                new CopyData()
                {

                    frame_count	    = target.Data.frame_count,
                    loop			= target.Data.loop,
                    dist_attn_func  = target.Data.dist_attn_func,
                };
            // アニメーションの固定値だけをコピー
            foreach (var animTarget in target.FogAnimTargets.Where(animTarget => animTarget != null))
            {
                float baseValue;
                if (animTarget.KeyFrames.Count == 0 || animTarget.KeyFrames.IsConstantCurve(out baseValue))
                {
                    baseValue = animTarget.GetBaseValue();
                    data.constantAnimTargets.Add(animTarget.targetType, baseValue);
                }
            }
            return data;
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public override void Paste(object pasteObject)
        {
            TheApp.CommandManager.Add(Paste(Targets, pasteObject));
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public static ICommand Paste(GuiObjectGroup targets, object pasteObject)
        {
            EditCommandSet commandSet = new EditCommandSet();
            {
                var copyData = (CopyData)pasteObject;

                commandSet.Add(CreateEditCommand_frame_count(targets, copyData.frame_count));
                commandSet.Add(CreateEditCommand_loop(targets, copyData.loop));
                commandSet.Add(CreateEditCommand_dist_attn_func(targets, copyData.dist_attn_func));

                // アニメーションの固定値をペースト
                var fogAnim = targets.Active as FogAnimation;
                if (fogAnim != null)
                {
                    foreach (var constAnim in copyData.constantAnimTargets)
                    {
                        var animTarget = fogAnim.GetTarget(constAnim.Key);
                        if (animTarget != null)
                        {
                            float basevalue;
                            // ペースト先にアニメーションが設定されているときはペーストしない
                            if (animTarget.KeyFrames.Count == 0 || animTarget.KeyFrames.IsConstantCurve(out basevalue))
                            {
                                commandSet.Add(CreateEditCommand_ConstantAnimation(new GuiObjectGroup(fogAnim), constAnim.Key, constAnim.Value));
                            }
                        }
                    }
                }
            }

            return commandSet.Execute();
        }
        #endregion

        private static ICommand CreateEditCommand_dist_attn_func(GuiObjectGroup targets, string dist_attn_func)
        {
            return new GeneralGroupReferenceEditCommand<string>(
                targets,
                GuiObjectID.FogAnimation,
                Enumerable.Repeat(dist_attn_func, targets.Objects.Count),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var anim = target as FogAnimation;
                    swap = anim.Data.dist_attn_func;
                    anim.Data.dist_attn_func = (string)data;

                    // Viewerへ転送
                    Viewer.LoadOrReloadAnimation.Send(anim.Owner);
                });
        }

        #region イベント
        private void cmbDistAttnFunc_SelectedIndexChanged(object sender, EventArgs e)
        {
            var func = cmbDistAttnFunc.SelectedItemData.ToString();
            if (func == ActiveTarget.Data.dist_attn_func)
            {
                return;
            }
            TheApp.CommandManager.Execute(CreateEditCommand_dist_attn_func(Targets, func));
        }

        private void cmbDistAttnFunc_CustomDrawItem(object sender, CustomDrawListControlItemEventArgs e)
        {
            if (e.Item != null)
            {
                e.ForeColor =
                    ApplicationConfig.Preset.FogDistAttnPresets
                    .Any(x => x.Function == (string)e.Item.Data) ? SystemColors.ControlText : Color.Red;
            }
        }

        private void iepFrameSize_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            if (e.Changing)
            {
                ;
            }
            else
            {
                TheApp.CommandManager.Execute(CreateEditCommand_frame_count(Targets, iepFrameSize.Value));
            }

        }

        private void cbxLoop_CheckedChanged(object sender, EventArgs e)
        {
            TheApp.CommandManager.Execute(CreateEditCommand_loop(Targets, cbxLoop.Checked));
        }
        #endregion //イベント
    }
}
