﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using App.ConfigData;
using App.Controls;
using App.Data;
using App.res;
using Viewer;

namespace App
{
//	[ToolboxItem(true)]
    public partial class PlayControl : UIUserControl
    {
        public NintendoWare.G3d.Edit.PlayPolicy	PlayPolicy
        {
            get
            {
                return playPolicy_;
            }

            private set
            {
                playPolicy_ = value;

                if (TheApp.IsInitializedMainFrame)
                {
                    Viewer.SetPlayPolicy.Send(PlayPolicy, (float)CurrentFrame);
                }

                UpdateState();
            }
        }

        public bool			IsPlaying
        {
            get
            {
                return isPlaying_;
            }

            private set
            {
                //DebugConsole.WriteLine("IsPlaying set");
                isPlaying_ = value;
                tmrPainter.Enabled = value;

                if (isPlaying_)
                {
                    playOldTickCount_ = System.Environment.TickCount;
                }

                SendPlayFrameCtrl();

                UpdateState();
            }
        }

        public double		StartFrame
        {
            get
            {
                return startFrame_;
            }

            set
            {
                startFrame_ = value;
                ftbCurrentFrame.Minimum = (float)startFrame_;
                pnlTimeSlider.Invalidate();
            }
        }

        public double		EndFrame
        {
            get
            {
                return endFrame_;
            }

            set
            {
                endFrame_ = value;
                ftbCurrentFrame.Maximum = (float)endFrame_;

                if (CurrentFrame > EndFrame)
                {
                    CurrentFrame = EndFrame;
                }
                pnlTimeSlider.Invalidate();
            }
        }

        /// <summary>
        /// ランタイムに送る FrameCount (0 のときに EndFrame と食い違う可能性あり)
        /// </summary>
        public float FrameCount { get; set; }

        sealed public class NoSendPlayFrameCtrlBlock : IDisposable
        {
            public static bool IsEnabled
            {
                get
                {
                    return count > 0;
                }
            }

            public static int count = 0;

            public NoSendPlayFrameCtrlBlock()
            {
                count++;
            }

            public void Dispose()
            {
                count--;
            }
        }

        public double		CurrentFrame
        {
            get
            {
                return currentFrame_;
            }

            set
            {
                if (inCurrentFrameSet_ == false)
                {
                    inCurrentFrameSet_ = true;
                    {
                        double nextCurrentFrame = Math.Min(Math.Max(value, StartFrame), EndFrame);

                        if (CurrentFrame != nextCurrentFrame)
                        {
                            currentFrame_ = nextCurrentFrame;
                            Viewer.AnimationMessage.CurrentFrame = (float)currentFrame_;
                            // 720: アニメーション再生中のタイムスライダの左ドラッグ動作について
                            // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=720
                            if (NoSendPlayFrameCtrlBlock.IsEnabled == false)
                            {
                                if (IsPlaying && IsTimeSliderDragging)
                                {
                                    SendPlayFrameCtrl(false);
                                }
                                else
                                {
                                    SendPlayFrameCtrl();
                                }
                            }

                            UpdateState();
                            pnlTimeSlider.Invalidate();

                            App.AppContext.NotifyPreviewCurrentFrameChanged();
                        }
                    }
                    inCurrentFrameSet_ = false;
                }
            }
        }

        private bool inCurrentFrameSet_ = false;

        public double		FrameRange
        {
            get
            {
                return EndFrame - StartFrame + 1.0f;
            }
        }

        public double FrameScale { get; private set; }

        public void SetFrameScaleExternally(float frameScale)
        {
            FrameScale = frameScale;
            using (var suppressBlock = new UIControlEventSuppressBlock())
            {
                fepFrameScale.Value = frameScale;
            }
        }

        public bool			IsAnyAnimationLoop
        {
            get
            {
                // アニメーションが一つでも有るかどうか
                bool hasAnimations = false;

                // シーンアニメーション側のアニメーションで、ループ再生があるものを探す。
                if (DocumentManager.PreviewSceneAnimSet != null)
                {
                    hasAnimations = DocumentManager.PreviewSceneAnimSet.Animations.Any();

                    foreach (var animName in DocumentManager.PreviewSceneAnimSet.Animations)
                    {
                        var anim = DocumentManager.Animations.FirstOrDefault(x => new AnimationSetItem(x.FileName, x.FileLocation) == animName);
                        if (anim != null)
                        {
                            Debug.Assert(anim is AnimationDocument);

                            if ((anim as AnimationDocument).Loop)
                            {
                                return true;
                            }
                        }
                    }
                }

                foreach(var model in DocumentManager.Models.Where(x => x.IsVisible))
                {
                    //Debug.Assert(model.PreviewAnimSet != null);

                    if (model.PreviewAnimSet != null)
                    {
                        hasAnimations = hasAnimations || model.PreviewAnimSet.Animations.Any();

                        foreach (var animName in model.PreviewAnimSet.Animations)
                        {
                            var anim = DocumentManager.Animations.FirstOrDefault(x => new AnimationSetItem(x.FileName, x.FileLocation) == animName);
                            if (anim != null)
                            {
                                Debug.Assert(anim is AnimationDocument);

                                if ((anim as AnimationDocument).Loop)
                                {
                                    return true;
                                }
                            }
                        }
                    }
                }

                // アニメーションがなければループ扱い
                return !hasAnimations;
            }
        }

        private NintendoWare.G3d.Edit.PlayPolicy	playPolicy_;
        private bool								isPlaying_;
        private double								currentFrame_;
        private double								startFrame_;
        private double								endFrame_;
        private int									playOldTickCount_;

        public PlayControl()
        {
            InitializeComponent();

            tsbLoopMode.ToolTipText	= Strings.PlayControl_LoopMode;
            tsbStop.Text			= Strings.PlayControl_StopText;
            tsbStop.ToolTipText		= Strings.PlayControl_StopToolTipText;
            tsbBack.Text			= Strings.PlayControl_BackText;
            tsbBack.ToolTipText		= Strings.PlayControl_BackToolTipText;
            tsbPlay.Text			= Strings.PlayControl_PlayText;
            tsbPlay.ToolTipText		= Strings.PlayControl_PlayToolTipText;
            tsbPause.Text			= Strings.PlayControl_PauseText;
            tsbPause.ToolTipText	= Strings.PlayControl_PauseToolTipText;
            tsbForward.Text			= Strings.PlayControl_ForwardText;
            tsbForward.ToolTipText	= Strings.PlayControl_ForwardToolTipText;

            tsmLoopModelAuto.Tag = NintendoWare.G3d.Edit.PlayPolicy.Auto;
            tsmLoopModelLoop.Tag = NintendoWare.G3d.Edit.PlayPolicy.Loop;
            tsmLoopModelOnce.Tag = NintendoWare.G3d.Edit.PlayPolicy.Once;

            fepFrameScale.SetSliderToolTipHintMessage(res.Strings.PlayControl_FrameScaleSliderTipHint);

            PlayPolicy		= NintendoWare.G3d.Edit.PlayPolicy.Auto;
            IsPlaying		= false;
            CurrentFrame	= 0.0;
            StartFrame		= 0.0;
            EndFrame		= 59.94;

            FrameScale = 1.0;

            App.AppContext.DocumentAddedOrRemoved	+= (s, a, r, swaped, reload) =>	UpdateAll();
            App.AppContext.PropertyChanged		+= (s, a) =>			UpdateAll();
            AnimationDocument.FrameCountChanged	+= (s, a) =>			UpdateAll();

            UpdateAll();
        }

        public void SendCurrentFrame()
        {
            if (IsPlaying)
            {
                Viewer.PlayFrameCtrl.Send((float)CurrentFrame);
            }
            else
            {
                Viewer.StopFrameCtrl.Send((float)CurrentFrame);
            }
        }

        private void UpdateAll()
        {
            UpdateEndFrame();
            UpdateState();
        }

        public void Play()
        {
            IsPlaying = true;
        }

        public void Stop()
        {
            using(var block = new NoSendPlayFrameCtrlBlock())
            {
                CurrentFrame = 0.0f;
            }
            IsPlaying = false;
        }

        public void Back()
        {
            using(var block = new NoSendPlayFrameCtrlBlock())
            {
                CurrentFrame = CorrectFrame(Math.Round(CurrentFrame - 1.0), true, true);
            }
            IsPlaying = false;
        }

        public void Forward()
        {
            using(var block = new NoSendPlayFrameCtrlBlock())
            {
                CurrentFrame = CorrectFrame(Math.Round(CurrentFrame + 1.0), true, true);
            }
            IsPlaying = false;
        }

        public void Pause()
        {
            using(var block = new NoSendPlayFrameCtrlBlock())
            {
                UpdateCurrentFrame();
                CurrentFrame = AlignCurrentFrame(CurrentFrame, FrameScale);
            }
            IsPlaying = false;
            Viewer.Connecter.PulseMessageThread();
        }

        /// <summary>
        /// 再生可能状態かどうか？
        /// </summary>
        public bool FrameControlEnabled { get; private set; }

        private void UpdateEndFrame()
        {
            var frameCount = 0;
            var isUpdateFrameCount = false;
            {
                // シーンアニメーション側のアニメーションフレーム数をカウントする。
                if (DocumentManager.PreviewSceneAnimSet != null)
                {
                    // アニメーションがなければループ扱い
                    if (DocumentManager.PreviewSceneAnimSet.Animations.Any() == false)
                    {
                        isUpdateFrameCount = true;
                    }

                    foreach (var animName in DocumentManager.PreviewSceneAnimSet.Animations)
                    {
                        var anim = DocumentManager.Animations.FirstOrDefault(x => new AnimationSetItem(x.FileName, x.FileLocation) == animName);
                        if (anim != null)
                        {
                            Debug.Assert(anim is AnimationDocument);

                            int animFrameCount = (anim as AnimationDocument).FrameCount;

                            if (animFrameCount > frameCount)
                            {
                                frameCount = animFrameCount;
                                isUpdateFrameCount = true;
                            }
                        }
                    }
                }

                foreach(var model in DocumentManager.Models.Where(x => x.IsVisible))
                {
                    if (model.PreviewAnimSet == null)
                    {
                        continue;
                    }

                    // アニメーションがなければループ扱い
                    if (model.PreviewAnimSet.Animations.Any() == false)
                    {
                        isUpdateFrameCount = true;
                    }

                    Debug.Assert(model.PreviewAnimSet != null);
                    foreach(var animName in model.PreviewAnimSet.Animations)
                    {
                        var anim = DocumentManager.Animations.FirstOrDefault(x => new AnimationSetItem(x.FileName, x.FileLocation) == animName);
                        if (anim != null)
                        {
                            Debug.Assert(anim is AnimationDocument);

                            int animFrameCount = (anim as AnimationDocument).FrameCount;

                            if (animFrameCount > frameCount)
                            {
                                frameCount = animFrameCount;
                                isUpdateFrameCount = true;
                            }
                        }
                    }
                }
            }

            if (FrameCount != frameCount)
            {
                Viewer.SetFrameCount.Send(frameCount);
            }
            FrameCount = frameCount;


            FrameControlEnabled = DocumentManager.Animations.Any() && (frameCount > 0);
            pnlTimeSlider.Enabled = FrameControlEnabled;
            if (!FrameControlEnabled)
            {
                if (IsPlaying)
                {
                    AutoPaused = true;
                    Pause();
                }
            }
            else if (AutoPaused)
            {
                AutoPaused = false;
                Play();
            }

            if (isUpdateFrameCount)
            {
                EndFrame = (frameCount > 0) ? Math.Max(1, frameCount) : EndFrame;
            }
            else if (IsPlaying)
            {
                IsPlaying = false;
            }
        }

        private bool AutoPaused;

        private void UpdateState()
        {
            using (var block = new UIControlEventSuppressBlock())
            {
                ftbCurrentFrame.Enabled	= FrameControlEnabled && !IsPlaying;
                ftbCurrentFrame.Value	= FrameControlEnabled ? (float)CurrentFrame : 0.0f;
                tsbLoopMode.Enabled		= FrameControlEnabled && !IsPlaying;
                tsbLoopMode.Text		= PlayPolicy.ToString();
                tsbPlay.Visible			= IsPlaying == false;
                tsbPause.Visible		= IsPlaying;

                tsbBack.Enabled = FrameControlEnabled;
                tsbPause.Enabled = FrameControlEnabled;
                tsbPlay.Enabled = FrameControlEnabled;
                tsbStop.Enabled = FrameControlEnabled;
                tsbForward.Enabled = FrameControlEnabled;
            }
        }

        private void tsmLoopModel_Click(object sender, EventArgs e)
        {
            Debug.Assert(sender is UIToolStripMenuItem);
            var menu = sender as UIToolStripMenuItem;

            PlayPolicy = (NintendoWare.G3d.Edit.PlayPolicy)menu.Tag;
        }

        public void PlayClick()
        {
            bool isOnRestart = false;

            if (CurrentFrame >= EndFrame)
            {
                if (( PlayPolicy == NintendoWare.G3d.Edit.PlayPolicy.Once) ||
                    ((PlayPolicy == NintendoWare.G3d.Edit.PlayPolicy.Auto) && IsAnyAnimationLoop == false))
                {
                    using(var block = new NoSendPlayFrameCtrlBlock())
                    {
                        CurrentFrame = 0.0;
                    }
                    IsPlaying = true;
                    isOnRestart = true;
                }
            }

            if (isOnRestart == false)
            {
                IsPlaying = true;
            }
        }

        private void tsbPlay_Click(object sender, EventArgs e)
        {
            PlayClick();
        }

        // フレームをアラインする。frameをscale倍の値にする
        private static double AlignCurrentFrame(double frame, double scale)
        {
            return Math.Ceiling(frame / scale) * scale;
        }

        private void tsbPause_Click(object sender, EventArgs e)
        {
            Pause();
        }

        private void tsbStop_Click(object sender, EventArgs e)
        {
            Stop();
        }

        private void tsbBack_Click(object sender, EventArgs e)
        {
            Back();
        }

        private void tsbForward_Click(object sender, EventArgs e)
        {
            Forward();
        }

        private void ftbCurrentFrame_ValueChanged(object sender, EventArgs e)
        {
            CurrentFrame = ftbCurrentFrame.Value;
        }

        private void tmrPainter_Tick(object sender, EventArgs e)
        {
            UpdateCurrentFrame();
        }

        public void UpdateCurrentFrame()
        {
            if (tmrPainter.Enabled)
            {
                int nowTickCount = System.Environment.TickCount;
                int time = nowTickCount - playOldTickCount_;
                double tickFrame = time / (1000.0 / Math.Max(1.0, ApplicationConfig.Preview.Fps));

                if (IsTimeSliderDragging == false)
                {
                    double nextFrame = CurrentFrame + tickFrame * FrameScale;

                    using (var block = new NoSendPlayFrameCtrlBlock())
                    {
                        CurrentFrame = CorrectFrame(nextFrame, false, false);
                    }
                }

                playOldTickCount_ = nowTickCount;
            }
        }

        private double CorrectFrame(double srcFrame, bool isForceLoop, bool isStep)
        {
            var playPolicy = isForceLoop ? NintendoWare.G3d.Edit.PlayPolicy.Loop : PlayPolicy;

            if (isStep ? (srcFrame > EndFrame) : (srcFrame >= EndFrame))
            {
                switch(playPolicy)
                {
                    case NintendoWare.G3d.Edit.PlayPolicy.Auto:
                    {
                        if (IsAnyAnimationLoop)
                        {
                            srcFrame -= EndFrame - StartFrame;
                        }
                        else
                        {
                            srcFrame = EndFrame;
                            using(var block = new NoSendPlayFrameCtrlBlock())
                            {
                                CurrentFrame = srcFrame;
                                IsPlaying = false;
                            }
                            Viewer.UpdatePlaying.Send(false);
                        }

                        break;
                    }

                    case NintendoWare.G3d.Edit.PlayPolicy.Loop:
                    {
                        srcFrame -= EndFrame - StartFrame;
                        break;
                    }

                    case NintendoWare.G3d.Edit.PlayPolicy.Once:
                    {
                        srcFrame = EndFrame;
                        using(var block = new NoSendPlayFrameCtrlBlock())
                        {
                            CurrentFrame = srcFrame;
                            IsPlaying = false;
                        }
                        Viewer.UpdatePlaying.Send(false);
                        break;
                    }
                }
            }
            else
            if (srcFrame < StartFrame)
            {
                switch(playPolicy)
                {
                    case NintendoWare.G3d.Edit.PlayPolicy.Auto:
                    {
                        if (IsAnyAnimationLoop)
                        {
                            srcFrame += EndFrame - StartFrame;
                        }
                        else
                        {
                            srcFrame = StartFrame;
                            using(var block = new NoSendPlayFrameCtrlBlock())
                            {
                                CurrentFrame = srcFrame;
                                IsPlaying = false;
                            }
                            Viewer.UpdatePlaying.Send(false);
                        }

                        break;
                    }

                    case NintendoWare.G3d.Edit.PlayPolicy.Loop:
                    {
                        srcFrame += EndFrame - StartFrame;
                        break;
                    }

                    case NintendoWare.G3d.Edit.PlayPolicy.Once:
                    {
                        srcFrame = StartFrame;
                        using(var block = new NoSendPlayFrameCtrlBlock())
                        {
                            CurrentFrame = srcFrame;
                            IsPlaying = false;
                        }
                        Viewer.UpdatePlaying.Send(false);
                        break;
                    }
                }
            }

            return srcFrame;
        }

        private void pnlTimeSlider_Resize(object sender, EventArgs e)
        {
            pnlTimeSlider.Invalidate();
        }

#region タイムスライダ
        // 左右の余白
        private int TimeSliderPaddingLeft
        {
            get
            {
                return 14;
            }
        }

        private int TimeSliderPaddingRight
        {
            get
            {
                // 桁数を求める
                var n = ((int)EndFrame).ToString().Length;
                return 14 + n * 7;
            }
        }

        private double TimeSliderFrameWidth
        {
            get
            {
                return TimeSliderClientWidth / FrameRange;
            }
        }

        private double TimeSliderCurrentFrameX
        {
            get
            {
                return MakePosXFromFrame(CurrentFrame);
            }
        }

        private double TimeSliderClientWidth
        {
            get
            {
                return pnlTimeSlider.Width - (TimeSliderPaddingLeft + TimeSliderPaddingRight);
            }
        }

        private void pnlTimeSlider_Paint(object sender, PaintEventArgs e)
        {
            if(pnlTimeSlider.Enabled)
            {
                // 現在フレームを描画する
                e.Graphics.FillRectangle(Brushes.DimGray, (int)TimeSliderCurrentFrameX, 0, (int)Math.Max(TimeSliderFrameWidth, 1), Height);

                // スケールを描画する
                DrawTimeSliderScale(e);

                // 現在フレーム数を描画する
                var currentString = CurrentFrame.ToString("0.##");
                var currentX = MakeStringClampX(currentString, TheApp.GuiFontBold, (float)TimeSliderCurrentFrameX, e.Graphics) + 3.0f;
                e.Graphics.DrawString(currentString, TheApp.GuiFontBold, Brushes.White, currentX, 14.0f);
            }
        }

        private float MakeStringClampX(string srcString, Font font, float srcX, Graphics graphics)
        {
            const int padding = 2;

            var panelWidth = pnlTimeSlider.Width - padding;

            var size = graphics.MeasureString(srcString, font);

            if (srcX + size.Width > panelWidth)
            {
                srcX -= (srcX + size.Width) - panelWidth;
            }

            return srcX;
        }

        private void DrawStringWithEdge(string src, Font font, Brush inBrush, Brush edgeBrush, float x, float y, Graphics graphics)
        {
#if true
            // 縁
            graphics.DrawString(src, font, edgeBrush, x - 2.0f, y       );
            graphics.DrawString(src, font, edgeBrush, x + 2.0f, y       );
            graphics.DrawString(src, font, edgeBrush, x       , y - 1.0f);
            graphics.DrawString(src, font, edgeBrush, x       , y + 1.0f);
#else
            var size = graphics.MeasureString(src, font);

            graphics.FillRectangle(edgeBrush, new RectangleF() { X = x, Y = y, Width = size.Width, Height = size.Height });
#endif

            // 中
            graphics.DrawString(src, font, inBrush, x, y);
        }

        private void DrawTimeSliderScale(PaintEventArgs e)
        {
            // 間のフレームを描画する
            {
                int startFrame = (int)StartFrame;
                int endFrame   = (int)EndFrame;
                int step       = MakeTimeSliderScaleStep();

                int lineY0 = pnlTimeSlider.Height / 2;
                int lineY1 = pnlTimeSlider.Height;

                using(var largePen = new Pen(Color.Black, 2))
                {
                    for (int frame = startFrame, i = 0; frame <= endFrame + 1; frame += step, ++ i)
                    {
                        var x = (int)MakePosXFromFrame(frame);
                        var largeStep = (i % 5) == 0;
                        var lineY     = largeStep ? 0 : lineY0;

                        e.Graphics.DrawLine(largeStep ? largePen : Pens.Black, x, lineY, x, lineY1);

                        if (frame <= endFrame)
                        {
                            if ((frame != (int)StartFrame) &&
                                (frame != (int)EndFrame  ))
                            {
                                e.Graphics.DrawString(frame.ToString("0.##"), largeStep ? TheApp.GuiFont : TheApp.GuiFontSmall, Brushes.Black, (float)(x + 1), largeStep ? 0.0f : 3.0f);
                            }
                        }
                    }
                }
            }

            // 開始・終了フレーム数を描画する
            {
                var startFrameString = StartFrame.ToString("0.##");
                var endFrameString   = EndFrame.ToString("0.##");

                var startX = (float)MakePosXFromFrame(StartFrame) + 3.0f;
                var endX   = (float)MakePosXFromFrame(EndFrame)   + 3.0f;

                startX = MakeStringClampX(startFrameString, TheApp.GuiFontBold, startX, e.Graphics);
                endX   = MakeStringClampX(endFrameString,   TheApp.GuiFontBold, endX,   e.Graphics);

                DrawStringWithEdge(startFrameString, TheApp.GuiFontBold, Brushes.Black, SystemBrushes.ButtonShadow, startX, 0.0f, e.Graphics);
                DrawStringWithEdge(endFrameString,   TheApp.GuiFontBold, Brushes.Black, SystemBrushes.ButtonShadow, endX,   0.0f, e.Graphics);
            }
        }

        // タイムスライダスケールのステップ数を作る
        private static readonly int[] baseStep_ = new []{1, 2, 5};
        private int MakeTimeSliderScaleStep()
        {
            const double minInterval = 30;

            var orgX = MakePosXFromFrame(0);

            for(int scale = 1;;scale *= 10)
            {
                foreach(int s in baseStep_)
                {
                    var step     = s * scale;
                    var unitX    = MakePosXFromFrame(step);
                    var interval = unitX - orgX;

                    if (interval >= minInterval)
                    {
                        return step;
                    }
                }
            }
        }

        private void SendPlayFrameCtrl(bool? isPlaying = null)
        {
            if (TheApp.IsInitializedMainFrame)
            {
                if (NoSendPlayFrameCtrlBlock.IsEnabled == false)
                {
                    if (isPlaying == null)
                    {
                        if (IsPlaying)
                        {
                            Viewer.PlayFrameCtrl.Send((float)CurrentFrame);
                        }
                        else
                        {
                            Viewer.StopFrameCtrl.Send((float)CurrentFrame);
                        }
                    }
                    else
                    {
                        if ((bool)isPlaying)
                        {
                            Viewer.PlayFrameCtrl.Send((float)CurrentFrame);
                        }
                        else
                        {
                            Viewer.StopFrameCtrl.Send((float)CurrentFrame);
                        }
                    }
                }
            }
        }

        private bool isTimeSliderDragging_ = false;
        public bool IsTimeSliderDragging
        {
            get{ return isTimeSliderDragging_; }
            set
            {
                isTimeSliderDragging_ = value;

                if (isTimeSliderDragging_ == false)
                {
                    // 720: アニメーション再生中のタイムスライダの左ドラッグ動作について
                    // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=720
                    if (IsPlaying)
                    {
                        SendPlayFrameCtrl(true);
                    }
                }
            }
        }

        private void pnlTimeSlider_MouseDown(object sender, MouseEventArgs e)
        {
            if (IsTimeSliderDragging == false)
            {
                IsTimeSliderDragging = true;

                CurrentFrame = (int)MakeFrameFromMouseX(e.X);
            }
        }

        private void pnlTimeSlider_MouseMove(object sender, MouseEventArgs e)
        {
            if (IsTimeSliderDragging)
            {
                CurrentFrame = (int)MakeFrameFromMouseX(e.X);
            }
        }

        private void pnlTimeSlider_MouseUp(object sender, MouseEventArgs e)
        {
            if (IsTimeSliderDragging)
            {
                CurrentFrame = (int)MakeFrameFromMouseX(e.X);

                IsTimeSliderDragging = false;
            }
        }

        private double MakeFrameFromMouseX(int x)
        {
            return FrameRange * (x - TimeSliderPaddingLeft) / TimeSliderClientWidth;
        }

        private double MakePosXFromFrame(double frame)
        {
            return (frame * TimeSliderClientWidth / FrameRange) + TimeSliderPaddingLeft;
        }
#endregion

        private void fepFrameScale_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            if (TheApp.IsInitializedMainFrame)
            {
                FrameScale = fepFrameScale.Value;
                Viewer.SetFrameStep.Send((float)FrameScale);
            }
        }
    }
}
