﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Drawing;
using System.Windows.Forms;
using App.Controls;
using App.Utility;

namespace App.ObjectView.Schematic
{
    /// <summary>
    /// オブジェクトスケマティックビュー操作バークラス。
    /// </summary>
    public sealed partial class ObjectSchematicViewOpBar : UIUserControl
    {
        // ビュー
        private readonly ObjectSchematicView _view;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ObjectSchematicViewOpBar(ObjectSchematicView view)
        {
            InitializeComponent();

            // ビュー設定
            _view = view;
            _view.MagnifyChanged += view_MagnifyChanged;
            UpdateForm();

            // イベント登録
            btnAlign.Click += btnAlign_Click;
            trbMagnify.SequentialValueChanged += trbMagnify_SequentialValueChanged;
            btnMagnifyMinus.Click += btnMagnifyMinus_Click;
            btnMagnifyPlus.Click += btnMagnifyPlus_Click;
            Application.Idle += Application_Idle;

            base.SetStyle(ControlStyles.ResizeRedraw, true);
            base.TabStop = false;
            DoubleBuffered = true;

            // 整列ボタンの状態を更新
            UpdateAlignButton();
        }

        /// <summary>
        /// 拡大。
        /// </summary>
        public void MagnifyPlus()
        {
            _view.Magnify += 0.1f;//_view.Magnify * 1.2f;
            UpdateMagnifyLabel();
            UpdateMagnifyTrackBar();
        }

        /// <summary>
        /// 縮小。
        /// </summary>
        public void MagnifyMinus()
        {
            _view.Magnify -= 0.1f; //_view.Magnify * 0.8f;
            UpdateMagnifyLabel();
            UpdateMagnifyTrackBar();
        }

        #region フォーム更新
        /// <summary>
        /// フォーム更新。
        /// </summary>
        private void UpdateForm()
        {
            UpdateMagnifyLabel();
            UpdateMagnifyTrackBar();
        }

        /// <summary>
        /// 倍率ラベル更新。
        /// </summary>
        private void UpdateMagnifyLabel()
        {
            lblMagnify.Text = string.Format("{0:f2}%", _view.Magnify * 100.0f);
        }

        /// <summary>
        /// 倍率トラックバー更新。
        /// </summary>
        private void UpdateMagnifyTrackBar()
        {
            trbMagnify.Value = MagnifyToTrackBarValue(_view.Magnify);
        }

        /// <summary>
        /// トラックバー値を倍率に変換。
        /// </summary>
        private float TrackBarValueToMagnify(int value)
        {
            Range  valRange = new Range(trbMagnify.Minimum, trbMagnify.Maximum);
            RangeF magRange = _view.MagnifyRange;

            float magnify = MathUtility.Interpolate(
                valRange.Minimum,
                valRange.Maximum,
                value,
                magRange.Minimum,
                magRange.Maximum
            );

            return magRange.Truncate(magnify);
        }

        /// <summary>
        /// 倍率をトラックバー値に変換。
        /// </summary>
        private int MagnifyToTrackBarValue(float magnify)
        {
            Range  valRange = new Range(trbMagnify.Minimum, trbMagnify.Maximum);
            RangeF magRange = _view.MagnifyRange;

            int value = (int)Math.Round(MathUtility.Interpolate(
                magRange.Minimum,
                magRange.Maximum,
                magnify,
                valRange.Minimum,
                valRange.Maximum
            ));

            return valRange.Truncate(value);
        }
        #endregion

        #region イベントハンドラ
        //---------------------------------------------------------------------
        // 整列
        private void btnAlign_Click(object sender, EventArgs e)
        {
            _view.Align(ObjectSchematicView.AlignMode.Schematic);
            UpdateAlignButton();
        }

        private void btnAlignVertical_Click(object sender, EventArgs e)
        {
            _view.Align(ObjectSchematicView.AlignMode.Vertical);
            UpdateAlignButton();
        }

        private void btnAlignHorizontal_Click(object sender, EventArgs e)
        {
            _view.Align(ObjectSchematicView.AlignMode.Horizontal);
            UpdateAlignButton();
        }

        //---------------------------------------------------------------------
        // 倍率
        private void trbMagnify_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            _view.Magnify = TrackBarValueToMagnify(trbMagnify.Value);
            UpdateMagnifyLabel();
        }

        private void btnMagnifyMinus_Click(object sender, EventArgs e)
        {
            MagnifyMinus();
        }

        private void btnMagnifyPlus_Click(object sender, EventArgs e)
        {
            MagnifyPlus();
        }

        //---------------------------------------------------------------------
        // ビュー
        private void view_MagnifyChanged(object sender, EventArgs e)
        {
            using (var block = new UIControlEventSuppressBlock())
            {
                UpdateForm();
            }
        }

        //---------------------------------------------------------------------
        // アイドル時
        private void Application_Idle(object sender, EventArgs e)
        {
            // コンポーネント状態更新
            if (Visible)
            {
                // 倍率
                trbMagnify.Enabled      = _view.IsEnableMagnifyMinus || _view.IsEnableMagnifyPlus;
                btnMagnifyMinus.Enabled = _view.IsEnableMagnifyMinus;
                btnMagnifyPlus.Enabled  = _view.IsEnableMagnifyPlus;
            }
        }
        #endregion

        private void ObjectSchematicViewOpBar_Paint(object sender, PaintEventArgs e)
        {
            e.Graphics.DrawLine(SystemPens.ControlDark, ClientRectangle.Left, ClientRectangle.Top, ClientRectangle.Left, ClientRectangle.Bottom - 1);
            e.Graphics.DrawLine(SystemPens.ControlDark, ClientRectangle.Right - 1, ClientRectangle.Top, ClientRectangle.Right - 1, ClientRectangle.Bottom - 1);
            e.Graphics.DrawLine(SystemPens.ControlDark, ClientRectangle.Left, ClientRectangle.Bottom - 1, ClientRectangle.Right - 1, ClientRectangle.Bottom - 1);
        }

        public void UpdateAlignButton()
        {
            // 現在選択中の整列モードのカラーを変更
            switch ((ObjectSchematicView.AlignMode)ConfigData.ApplicationConfig.Setting.MainFrame.ViewAlignMode)
            {
                case ObjectSchematicView.AlignMode.Schematic:
                    btnAlign.BackColor = Color.Gray;
                    btnAlignVertical.BackColor = SystemColors.Control;
                    btnAlignHorizontal.BackColor = SystemColors.Control;
                    break;
                case ObjectSchematicView.AlignMode.Vertical:
                    btnAlign.BackColor = SystemColors.Control;
                    btnAlignVertical.BackColor = Color.Gray;
                    btnAlignHorizontal.BackColor = SystemColors.Control;
                    break;
                case ObjectSchematicView.AlignMode.Horizontal:
                    btnAlign.BackColor = SystemColors.Control;
                    btnAlignVertical.BackColor = SystemColors.Control;
                    btnAlignHorizontal.BackColor = Color.Gray;
                    break;
                default:
                    break;
            }
        }
    }
}
