﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Text;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using App.Command;
using App.Controls;
using App.Data;
using App.ObjectView.List;
using App.Properties;
using App.Utility;
using App.res;
using ConfigCommon;

namespace App.ObjectView.Schematic
{
    public class ObjectSchematicView : MouseOperatableControl, IObjectView
    {
        // オブジェクトとノードの対応
        private readonly Dictionary<GuiObject, List<Node>> NodeTable = new Dictionary<GuiObject, List<Node>>();

        // 選択の更新がスケマティックビューから行われたかどうか
        private readonly SuppressBlockCounter UpdatingSelectionCounter;

        // ルートノード
        private List<Node> RootNodes;

        private List<Node> RootNodesAll;

        private Node[] Nodes;

        // シフト選択開始オブジェクト
        private Node _shiftSelectTarget = null;
        // 操作モード
        private OperationMode _operationMode = OperationMode.None;

        /// <summary>
        /// 操作モード。
        /// </summary>
        private enum OperationMode
        {
            None,			// 何もなし
            ObjectMoveWait,	// オブジェクト移動待機
            ObjectMove,		// オブジェクト移動
            ObjectTracking,	// オブジェクトトラッキング
            ViewScroll,		// ビュースクロール
            ViewScale		// ビュースケール
        };

        // 整列モード
        public enum AlignMode
        {
            Schematic,  // スケマティック
            Vertical,   // 垂直
            Horizontal, // 水平
        };

        public AlignMode CurrentAlignMode
        {
            // 再起動時にも適用されるようにコンフィグに記録
            get
            {
                return (AlignMode)ConfigData.ApplicationConfig.Setting.MainFrame.ViewAlignMode;
            }
            set
            {
                ConfigData.ApplicationConfig.Setting.MainFrame.ViewAlignMode = (int)value;
            }
        }

        // クライアントサイズ保存値
        private Size _clientSize = Size.Empty;
        // 原点
        private PointF _origin = PointF.Empty;
        // 表示倍率
        private float _magnify = 1.0f;
        // 表示倍率範囲
        private readonly RangeF _magnifyRange = new RangeF(0.1f, 1.0f);

        // ラベル編集用テキストボックス
        private readonly UITextBox tbxComment;

        // ツールチップ
        private readonly HintToolTip nameToolTip = new HintToolTip();

        // カラー編集
        private readonly GuiObjectColorPickerTrigger colorPickerTrigger;
        private readonly ColorPickerAdapter colorPickerAdapter;

        private bool isDefaultLinear;
        public bool IsDefaultLinear
        {
            get
            {
                return isDefaultLinear;
            }
            set
            {
                isDefaultLinear = value;
                colorPickerTrigger.IsDefaultLinear = value;
            }
        }

        public ObjectSchematicView()
        {
            // 初期設定
            base.FocusRedraw = true;
            DoubleBuffered = true;

            RootNodes = new List<Node>();
            RootNodesAll = new List<Node>();
            Nodes = new Node[0];

            UpdatingSelectionCounter = new SuppressBlockCounter();
            UpdatingSelectionCounter.ReleasedEvent += Invalidate;

            // テキストボックス
            tbxComment = new UITextBox()
            {
                BorderStyle = BorderStyle.FixedSingle,
                Visible = false,
            };
            tbxComment.TextChanged += (o, e) => UpdateViewContent();
            tbxComment.LostFocus += (o, e) => UpdateComment();
            tbxComment.KeyDown += TextBoxKeyDown;
            Controls.Add(tbxComment);

            // トリム用のStringFormat
            sf = new StringFormat();
            sf.Trimming = StringTrimming.None;
            sf.Alignment = StringAlignment.Near;
            sf.LineAlignment = StringAlignment.Center;
            sf.FormatFlags |= StringFormatFlags.NoClip;
            sf.FormatFlags |= StringFormatFlags.NoWrap;

            // カラーピッカーの設定
            colorPickerTrigger = new GuiObjectColorPickerTrigger();
            colorPickerAdapter = new ColorPickerAdapter(colorPickerTrigger);
            colorPickerAdapter.ColorEdit += ColorEdit;
            colorPickerAdapter.Nullify += NullifySymbolColor;

            IsDefaultLinear = false;

            App.AppContext.DocumentAddedOrRemoved += (s, a, r, swaped, reloaded) =>
                {
                    // reloaded 分については再構築
                    foreach (var model in reloaded.OfType<Model>())
                    {
                        int index = RootNodes.FindIndex(x => x.Content == model);

                        if (index >= 0)
                        {
                            var newNode = new ModelNode(model);
                            newNode.PosAligned = CopyPosition(newNode, RootNodes[index]);
                            RootNodes[index] = newNode;
                        }
                    }

                    for (int i = 0; i < RootNodes.Count; i++)
                    {
                        GuiObject obj;
                        if (swaped.TryGetValue(RootNodes[i].Content, out obj))
                        {
                            var newNode = new ModelNode((Model)obj);
                            newNode.PosAligned = CopyPosition(newNode, RootNodes[i]);
                            RootNodes[i] = newNode;
                        }
                    }

                    // モデルファイルが追加、削除、リロードされた時のみ再描画する
                    bool doRefresh = false;

                    if (a != null && a.OfType<Model>().Any())
                    {
                        doRefresh = true;
                    }
                    else if (r != null && r.OfType<Model>().Any())
                    {
                        doRefresh = true;
                    }
                    else if (reloaded != null && reloaded.OfType<Model>().Any())
                    {
                        doRefresh = true;
                    }

                    if (doRefresh)
                    {
                        RefreshNodes();
                    }
                };
            App.AppContext.PropertyChanged += (s, e) =>
            {
                if (e.Any(x => x is App.FileView.DocumentPropertyChangedShowInObjViewArgs))
                {
                    RefreshNodes();
                }
                else
                {
                    ResetShowingSecondaryLabel();
                }

                // カラーピッカーを更新
                colorPickerAdapter.NotifyUpdate();
                Invalidate();
            };

            // イベント登録
            App.AppContext.SelectedTargetChanged += (x, y) => UpdateSelection();

            // ノード更新
            RefreshNodes();
        }

        void ResetShowingSecondaryLabel()
        {
            foreach (var node in Nodes)
            {
                if (ShowSecondaryLabel(node))
                {
                    if (!node.ShowingSecondaryLabel)
                    {
                        node.ShowingSecondaryLabel = true;
                        if (node.PosAligned)
                        {
                            foreach (var child in node.Children)
                            {
                                child.MovePos(0, 16, true);
                            }
                        }
                    }
                }
                else
                {
                    if (node.ShowingSecondaryLabel)
                    {
                        node.ShowingSecondaryLabel = false;
                        if (node.PosAligned)
                        {
                            foreach (var child in node.Children)
                            {
                                child.MovePos(0, -16, true);
                            }
                        }
                    }
                }
            }
        }

        Control IObjectView.Control
        {
            get { return this; }
        }

        void IObjectView.FocusClient()
        {
            Focus();
        }

        AppConfig.ObjectViewMode IObjectView.ViewMode
        {
            get { return AppConfig.ObjectViewMode.SchematicView; }
        }

        /// <summary>
        /// ノード更新
        /// </summary>
        private void RefreshNodes()
        {
            // オペレーションを解除
            _operationMode = OperationMode.None;

            // ルートを再構築
            var newRootNodes = new List<Node>();
            newRootNodes.Clear();

            bool updateAlign = false;
            foreach (var model in DocumentManager.OrderedModels)
            {
                Node node = RootNodes.Find(x => x.Content == model);
                if (node == null)
                {
                    node = new ModelNode(model);
                    updateAlign = true;
                }
                newRootNodes.Add(node);
            }

            updateAlign |= RootNodes.Count != newRootNodes.Count;
            RootNodes = newRootNodes;
            var oldNodesCount = Nodes.Length;
            Nodes = RootNodes.Where(x => (x.Content is Model) && (x.Content as Model).IsShowInObjView).SelectMany(x => x.Descendants).ToArray();
            updateAlign |= oldNodesCount != Nodes.Length;

            NodeTable.Clear();
            foreach (var node in Nodes)
            {
                if (!NodeTable.ContainsKey(node.Content))
                {
                    NodeTable.Add(node.Content, new List<Node>(Enumerable.Repeat(node, 1)));
                }
                else
                {
                    NodeTable[node.Content].Add(node);
                }
            }

            bool empty = true;
            RectangleF rect = RectangleF.Empty;

            foreach (var node in Nodes)
            {
                if (node.UpdateNodeControl)
                {
                    UpdateNodeControls(node);
                    node.UpdateNodeControl = false;
                }
            }

            // コメントの表示の更新
            ResetShowingSecondaryLabel();

            // 配置の決定
            foreach (var root in RootNodes.Where(x => x.PosAligned))
            {
                if (empty)
                {
                    rect = root.TreeBound;
                    empty = false;
                }
                else
                {
                    rect = RectangleF.Union(rect, root.TreeBound);
                }
            }

            foreach (var root in RootNodes)
            {
                if (root.PosAligned)
                {
                    continue;
                }

                AlignPos(root, new PointF(0, 0), CurrentAlignMode);
                RectangleF treeBound = root.TreeBound;
                if (!empty)
                {
                    var isHorizontalAlign = false;

                    if (CurrentAlignMode == AlignMode.Schematic)
                    {
                        // サイズによって下に並ぶか横に並ぶか決定される
                        if (treeBound.Width + rect.Width < treeBound.Height + rect.Height)
                        {
                            isHorizontalAlign = true;
                        }
                    }
                    else if (CurrentAlignMode == AlignMode.Horizontal)
                    {
                        isHorizontalAlign = true;
                    }

                    if (isHorizontalAlign)
                    {
                        // 横に配置
                        root.MovePos(
                            rect.Right - treeBound.Left + 24,
                            rect.Top - treeBound.Top,
                            true);
                    }
                    else
                    {
                        // 下に配置
                        root.MovePos(
                            ((rect.Right + rect.Left) - (treeBound.Right + treeBound.Left)) / 2,
                            rect.Bottom - treeBound.Top + 24,
                            true);
                    }
                    rect = RectangleF.Union(rect, root.TreeBound);
                }
                else
                {
                    rect = treeBound;
                    empty = false;
                }
                root.PosAligned = true;
            }

            // 表示設定
            if (updateAlign)
            {
                if (RootNodes.Any())
                {
                    // 中央表示指定
                    CenteringWhole(true);
                }
                else
                {
                    _origin = PointF.Empty;
                    _magnify = 1.0f;

                    OnMagnifyChanged(EventArgs.Empty);
                    UpdateViewContent();
                    Invalidate();
                }
            }
            else
            {
                Invalidate();
            }
        }

        private bool CopyPosition(Node newNode, Node oldNode)
        {
            if (newNode.Content.Name == oldNode.Content.Name &&
                newNode.Content.ObjectID == oldNode.Content.ObjectID &&
                newNode.Children.Count() == oldNode.Children.Count())
            {
                newNode.Location = oldNode.Location;
                newNode.ShowingSecondaryLabel = oldNode.ShowingSecondaryLabel;

                foreach (var pair in newNode.Children.Zip(oldNode.Children, (x, y)=>new {x, y}))
                {
                    pair.x.PosAligned = CopyPosition(pair.x, pair.y);
                    if (!pair.x.PosAligned)
                    {
                        return false;
                    }
                }

                return true;
            }

            return false;
        }

        #region イベント
        //---------------------------------------------------------------------
        private static readonly object EVENT_MAGNIFYCHANGED = new object();

        /// <summary>
        /// 表示倍率変更イベント。
        /// </summary>
        public event EventHandler MagnifyChanged
        {
            add { base.Events.AddHandler(EVENT_MAGNIFYCHANGED, value); }
            remove { base.Events.RemoveHandler(EVENT_MAGNIFYCHANGED, value); }
        }

        /// <summary>
        /// 表示倍率変更ハンドラ。
        /// </summary>
        private void OnMagnifyChanged(EventArgs e)
        {
            if (UIControlEventSuppressBlock.Enabled) return;
            var handler = (EventHandler)base.Events[EVENT_MAGNIFYCHANGED];
            if (handler != null) { handler(this, e); }
        }
        #endregion

        #region プロパティ
        /// <summary>
        /// 表示倍率。
        /// </summary>
        public float Magnify
        {
            get { return _magnify; }
            set
            {
                // 外部からの呼び出し専用なのでイベント発行を抑制する
                using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
                {
                    ScaleView(value - _magnify);
                }
            }
        }

        /// <summary>
        /// 表示倍率範囲。
        /// </summary>
        public RangeF MagnifyRange
        {
            get { return _magnifyRange; }
        }

        /// <summary>
        /// 拡大可能かどうか。
        /// </summary>
        public bool IsEnableMagnifyPlus
        {
            get { return Nodes.Any() && _magnify < _magnifyRange.Maximum; }
        }

        /// <summary>
        /// 縮小可能かどうか。
        /// </summary>
        public bool IsEnableMagnifyMinus
        {
            get { return Nodes.Any() && _magnify > _magnifyRange.Minimum; }
        }

        /// <summary>
        /// 全体を中央表示可能かどうか。
        /// </summary>
        public bool IsEnableCenteringWhole
        {
            get { return Nodes.Any(); }
        }

        /// <summary>
        /// 選択オブジェクトを中央表示可能かどうか。
        /// </summary>
        public bool IsEnableCenteringSelected
        {
            get { return SelectedNodes.Any(); }
        }

        /// <summary>
        /// 全体を整列可能かどうか。
        /// </summary>
        public bool IsEnableAlign
        {
            get { return Nodes.Any(); }
        }

        /// <summary>
        /// 選択オブジェクトを整列可能かどうか。
        /// </summary>
        public bool IsEnableAlignSelected
        {
            get { return SelectedNodes.Any(); }
        }
        #endregion

        #region 表示操作
        /// <summary>
        /// 全体を中央表示。
        /// </summary>
        public void CenteringWhole(bool defaultSize)
        {
            RectangleF objBounds = GetObjectBounds(false);
            if (!objBounds.IsEmpty)
            {
                // デフォルトではあまり小さくさせない
                RangeF range = new RangeF(_magnifyRange);
                if (defaultSize)
                {
                    range.Minimum = 0.6f;
                }

                OptimizeView(objBounds, range);
                UpdateViewContent();
                Invalidate();
            }
        }

        /// <summary>
        /// 選択オブジェクトを中央表示。
        /// </summary>
        public void CenteringSelected()
        {
            RectangleF objBounds = GetObjectBounds(true);
            if (!objBounds.IsEmpty)
            {
                OptimizeView(objBounds, _magnifyRange);
                UpdateViewContent();
                Invalidate();
            }
        }

        public void AlignSelected()
        {
            bool aligned = false;
            foreach (Node node in SelectedNodes)
            {
                if (node.Children.Any())
                {
                    AlignPos(node, node.Location, CurrentAlignMode);
                    aligned = true;
                }
            }
            if (aligned)
            {
                Invalidate();
            }
        }

        public void Align(AlignMode alignMode)
        {
            // 最後に行った整列モードを保存しておく
            CurrentAlignMode = alignMode;

            // 配置の決定
            bool empty = true;
            RectangleF rect = RectangleF.Empty;
            foreach (var root in RootNodes.Where(x => x.PosAligned))
            {
                if (empty)
                {
                    rect = root.TreeBound;
                    empty = false;
                }
                else
                {
                    rect = RectangleF.Union(rect, root.TreeBound);
                }
            }

            foreach (var root in RootNodes)
            {
                AlignPos(root, new PointF(0, 0), CurrentAlignMode);
                RectangleF treeBound = root.TreeBound;
                var isHorizontalAlign = false;

                if (CurrentAlignMode == AlignMode.Schematic)
                {
                    // サイズによって下に並ぶか横に並ぶか決定される
                    if (treeBound.Width + rect.Width < treeBound.Height + rect.Height)
                    {
                        isHorizontalAlign = true;
                    }
                }
                else if (CurrentAlignMode == AlignMode.Horizontal)
                {
                    isHorizontalAlign = true;
                }

                if (isHorizontalAlign)
                {
                    // 横に配置
                    root.MovePos(
                        rect.Right - treeBound.Left + 24,
                        rect.Top - treeBound.Top,
                        true);
                }
                else
                {
                    // 下に配置
                    root.MovePos(
                        ((rect.Right + rect.Left) - (treeBound.Right + treeBound.Left)) / 2,
                        rect.Bottom - treeBound.Top + 24,
                        true);
                }

                rect = RectangleF.Union(rect, root.TreeBound);
                root.PosAligned = true;
            }

            // 整列時に形状が大きく変化するので、ツリーを見失う場合があるためデフォルトサイズでセンタリングを行う
            CenteringWhole(true);
        }

        /// <summary>
        /// ビューを最適化。
        /// </summary>
        private void OptimizeView(RectangleF target, RangeF range)
        {
            //target.Inflate(10, 10);

            // ターゲットの中心位置
            PointF ptTarget = RectangleUtility.GetCenter(target);


            // クライアント領域
            var rectClient = new RectangleF(ClientRectangle.X, ClientRectangle.Y, ClientRectangle.Width, ClientRectangle.Height);
            // クライアントにパディングをつける
            rectClient.Inflate(-10, -10);

            // まず等倍で中央表示
            _origin.X = ptTarget.X - rectClient.Width / 2.0f;
            _origin.Y = ptTarget.Y - rectClient.Height / 2.0f;
            _magnify = 1.0f;

            // クライアント領域とのサイズ比較
            bool overW = target.Width > rectClient.Width;
            bool overH = target.Height > rectClient.Height;

            // クライアント領域とのサイズ比率
            float wRatio = rectClient.Width / target.Width;
            float hRatio = rectClient.Height / target.Height;

            {
                // 横方向のみ収まらない場合
                if (overW && !overH)
                {
                    ScaleView(wRatio - _magnify, range);
                }
                // 縦方向のみ収まらない場合
                else if (!overW && overH)
                {
                    ScaleView(hRatio - _magnify, range);
                }
                // 縦も横も収まらない場合
                else if (overW && overH)
                {
                    // 横に合わせる
                    if (wRatio < hRatio)
                    {
                        ScaleView(wRatio - _magnify, range);
                    }
                    // 縦に合わせる
                    else
                    {
                        ScaleView(hRatio - _magnify, range);
                    }
                }
            }

            // 一旦等倍にしたため必ず呼ぶ
            OnMagnifyChanged(EventArgs.Empty);
        }

        /// <summary>
        /// 指定点を中心にビューをスケーリング。
        /// </summary>
        private void ScaleView(PointF point, float delta)
        {
            ScaleView(point, delta, _magnifyRange);
        }

        /// <summary>
        /// 指定点を中心にビューをスケーリング。
        /// </summary>
        private void ScaleView(PointF point, float delta, RangeF range)
        {
            float prevMag = _magnify;
            float nextMag = _magnify;
            bool changed = false;

            // 拡大
            if (delta > 0.0f && nextMag < range.Maximum)
            {
                nextMag += delta;
                if (nextMag >= range.Maximum)
                {
                    nextMag = range.Maximum;
                }
                changed = true;
            }
            // 縮小
            if (delta < 0.0f && nextMag > range.Minimum)
            {
                nextMag += delta;
                if (nextMag <= range.Minimum)
                {
                    nextMag = range.Minimum;
                }
                changed = true;
            }

            // 倍率変更時のみ調整する
            if (changed)
            {
                SizeF sizeDelta = new SizeF(
                    point.X * (1.0f / nextMag - 1.0f / prevMag),
                    point.Y * (1.0f / nextMag - 1.0f / prevMag)
                );

                _origin.X -= sizeDelta.Width;
                _origin.Y -= sizeDelta.Height;
                _magnify = nextMag;

                OnMagnifyChanged(EventArgs.Empty);
                UpdateViewContent();
                Invalidate();
            }
        }

        /// <summary>
        /// 中央を中心にビューをスケーリング。
        /// </summary>
        private void ScaleView(float delta)
        {
            ScaleView(delta, _magnifyRange);
        }

        /// <summary>
        /// 中央を中心にビューをスケーリング。
        /// </summary>
        private void ScaleView(float delta, RangeF range)
        {
            // クライアントサイズ
            Size sizeClient = ClientSize;

            // 中心点を求める
            PointF center = new PointF(
                sizeClient.Width / 2.0f,
                sizeClient.Height / 2.0f
            );

            // スケーリング
            ScaleView(center, delta, range);
        }
        #endregion

        #region オブジェクト選択

        // 選択状態のノード
        private readonly HashSet<Node> SelectedNodes = new HashSet<Node>();

        // アクティブな選択状態のノード
        private Node ActiveSelectedNode = null;

        /// <summary>
        /// 選択更新
        /// </summary>
        private void UpdateSelection()
        {
            if (!UpdatingSelectionCounter.CheckBlock)
            {
                SelectedNodes.Clear();
                foreach (var obj in App.AppContext.SelectedTarget.Objects)
                {
                    if (NodeTable.ContainsKey(obj))
                    {
                        foreach (var node in NodeTable[obj])
                        {
                            SelectedNodes.Add(node);
                        }
                    }
                }

                if ((App.AppContext.SelectedTarget.Active != null) && NodeTable.ContainsKey(App.AppContext.SelectedTarget.Active))
                {
                    ActiveSelectedNode = NodeTable[App.AppContext.SelectedTarget.Active][0];
                }
                else
                {
                    ActiveSelectedNode = null;
                }
                Invalidate();

                // 空なら閉じる
                if (App.AppContext.SelectedTarget.Objects.Count() == 0)
                {
                    ColorPickerDialog.EndConnection(colorPickerAdapter);
                }
            }
        }

        /// <summary>
        /// 選択設定
        /// </summary>
        private void SetSelected(Node obj)
        {
            using (var block = new SuppressBlock(UpdatingSelectionCounter))
            {
                SelectedNodes.Clear();
                SelectedNodes.Add(obj);
                ActiveSelectedNode = obj;
                App.AppContext.SelectedTarget.Set(obj.Content, true);
            }
        }

        /// <summary>
        /// 選択設定
        /// </summary>
        private void SetSelected(IEnumerable<Node> nodes, Node active)
        {
            using (var block = new SuppressBlock(UpdatingSelectionCounter))
            {
                SelectedNodes.Clear();
                foreach (var node in nodes)
                {
                    SelectedNodes.Add(node);
                }

                if (active == null)
                {
                    active = SelectedNodes.FirstOrDefault();
                }

                ActiveSelectedNode = active;
                App.AppContext.SelectedTarget.Set(nodes.Select(x => x.Content).Distinct(), active != null ? active.Content : null, true);
            }
        }

        public bool EnableSelectAll
        {
            get
            {
                return Nodes.Any();
            }
        }

        /// <summary>
        /// すべて選択
        /// </summary>
        public void SelectAll()
        {
            SetSelected(Nodes, ActiveSelectedNode);
        }

#if true
        public bool EnableToggleSelection
        {
            get
            {
                return Nodes.Any();
            }
        }

        /// <summary>
        /// 選択状態反転
        /// </summary>
        public void ToggleSelection()
        {
            SetSelected(Nodes.Where(x => !SelectedNodes.Contains(x)).ToArray(), null);
        }

#endif
        /// <summary>
        /// 選択解除
        /// </summary>
        private void Deselect(Node obj)
        {
            using (var block = new SuppressBlock(UpdatingSelectionCounter))
            {
                Debug.Assert(SelectedNodes.Contains(obj));
                SelectedNodes.Remove(obj);

                if (ActiveSelectedNode == obj)
                {
                    ActiveSelectedNode = null;
                }

                Debug.Assert(App.AppContext.SelectedTarget.Contains(obj.Content));
                if (!NodeTable[obj.Content].Any(x => SelectedNodes.Contains(x)))
                {
                    App.AppContext.SelectedTarget.Remove(obj.Content);
                    if (ActiveSelectedNode == null)
                    {
                        if (App.AppContext.SelectedTarget.Active != null && NodeTable.ContainsKey(App.AppContext.SelectedTarget.Active))
                        {
                            ActiveSelectedNode = NodeTable[App.AppContext.SelectedTarget.Active][0];
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 選択追加
        /// </summary>
        private void AddSelected(Node obj)
        {
            using (var block = new SuppressBlock(UpdatingSelectionCounter))
            {
                SelectedNodes.Add(obj);

                ActiveSelectedNode = obj;
                if (!App.AppContext.SelectedTarget.Contains(obj.Content))
                {
                    App.AppContext.SelectedTarget.Add(obj.Content);
                }
            }
        }

        /// <summary>
        /// オブジェクトを選択。
        /// </summary>
        private SelectObjectResult SelectObject(Point point, MouseButtons button)
        {
            // オブジェクト検索
            Node obj = SearchNode(point);

            return SelectNode(obj, button);
        }



        /// <summary>
        /// オブジェクトを選択。
        /// </summary>
        private SelectObjectResult SelectNode(Node obj, MouseButtons button)
        {
            if (obj == null)
            {
                return SelectObjectResult.Noting;
            }

            // 修飾状態
            var buttonL = button == MouseButtons.Left;
            var withCtrl = buttonL && Control.ModifierKeys == Keys.Control;
            var withShift = buttonL && Control.ModifierKeys == Keys.Shift;
            var withCtrlShift = buttonL && Control.ModifierKeys == (Keys.Control | Keys.Shift);
            var selected = SelectedNodes.Contains(obj);

            // 選択状態
            if (withCtrl)
            {
                if (selected)
                {
                    Deselect(obj);
                }
                return SelectObjectResult.Deselected;
            }
            else if (withShift)
            {
                if (selected)
                {
                    Deselect(obj);
                    return SelectObjectResult.Deselected;
                }
                else
                {
                    AddSelected(obj);
                }
            }
            else if (withCtrlShift)
            {
                if (selected)
                {
                    ActiveSelectedNode = obj;
                    App.AppContext.SelectedTarget.Active = obj.Content;
                }
                else
                {
                    AddSelected(obj);
                }
            }
            else
            {
                if (selected)
                {
                    ActiveSelectedNode = obj;
                    App.AppContext.SelectedTarget.Active = obj.Content;
                }
                else
                {
                    SetSelected(obj);
                }
            }
            return SelectObjectResult.Selected;

        }

        /// <summary>
        /// オブジェクト選択結果。
        /// </summary>
        private enum SelectObjectResult
        {
            Noting,			// 何もしていない
            Selected,		// 選択した
            Deselected		// 選択解除した
        };

        /// <summary>
        /// オブジェクトを選択。
        /// </summary>
        private void SelectObject(ArrowDirection direction)
        {
            // オブジェクト検索
            Node obj = SearchObject(direction, ActiveSelectedNode);
            if (obj == null)
            {
                return;
            }

            // ビューの範囲外なら位置補正
            RectangleF rectObj = TransformObjectToView(obj.Bound);
            RectangleF rectClient = ClientRectangle;
            PointF ptObjLT = RectangleUtility.GetTopLeft(rectObj);
            PointF ptObjRB = RectangleUtility.GetBottomRight(rectObj);
            if (!(rectClient.Contains(ptObjLT) && rectClient.Contains(ptObjRB)))
            {
                SizeF offset = SizeF.Empty;
                const float margin = 10.0f;
                if (rectObj.Left < rectClient.Left) { offset.Width = rectObj.Left - rectClient.Left - margin; }
                else if (rectObj.Right > rectClient.Right) { offset.Width = rectObj.Right - rectClient.Right + margin; }
                if (rectObj.Top < rectClient.Top) { offset.Height = rectObj.Top - rectClient.Top - margin; }
                else if (rectObj.Bottom > rectClient.Bottom) { offset.Height = rectObj.Bottom - rectClient.Bottom + margin; }

                offset = TransformViewToObject(offset);
                _origin.X += offset.Width;
                _origin.Y += offset.Height;
            }

            // シフト選択
            if (Control.ModifierKeys == Keys.Shift) { ShiftSelectObject(obj); }
            // 単一選択
            else { SetSelected(obj); }
        }

        /// <summary>
        /// オブジェクトのシフト選択。
        /// </summary>
        private void ShiftSelectObject(Node obj)
        {
            // 原点オブジェクト設定
            if (_shiftSelectTarget == null)
            {
                Node active = ActiveSelectedNode;
                _shiftSelectTarget = active ?? obj;
            }

            // 同じオブジェクトの時は単一選択
            if (_shiftSelectTarget == obj)
            {
                SetSelected(obj);
            }
            // 原点オブジェクトからの範囲選択
            else
            {
                RectangleF rectCheck = RectangleF.Union(
                    _shiftSelectTarget.Bound,
                    obj.Bound
                );

                List<Node> selectedNodes = new List<Node>();
                foreach (var node in Nodes)
                {
                    if (rectCheck.IntersectsWith(node.Bound))
                    {
                        selectedNodes.Add(node);
                    }
                }

                SetSelected(selectedNodes, obj);
            }
        }

        /// <summary>
        /// トラッキング終了処理。
        /// </summary>
        private void EndTracking()
        {
            var oldSelected = SelectedNodes.ToList();
            var selectedNext = new List<Node>();
            Node activeNext = null;

            // 選択オブジェクト登録
            var dragBounds = TransformViewToObject(MouseDragBounds);
            var hitNodes = Nodes.Where(obj => dragBounds.IntersectsWith(obj.Bound)).ToList();

            switch (Control.ModifierKeys)
            {
                case Keys.Shift:
                    selectedNext = oldSelected.Union(hitNodes).Except(oldSelected.Intersect(hitNodes)).ToList();
                    break;
                case Keys.Control:
                    selectedNext = oldSelected.Except(hitNodes).ToList();
                    break;
                case Keys.Control | Keys.Shift:
                    selectedNext = oldSelected.Union(hitNodes).ToList();
                    break;
                default:
                    selectedNext = hitNodes;
                    break;
            }

            foreach (var node in selectedNext)
            {
                if (node == ActiveSelectedNode || activeNext == null)
                {
                    activeNext = node;
                }
            }

            // 選択オブジェクト変更
            if (ActiveSelectedNode != activeNext ||
                selectedNext.Count != SelectedNodes.Count ||
                selectedNext.Any(x => !SelectedNodes.Contains(x))
                )
            {
                ActiveSelectedNode = activeNext;
                SetSelected(selectedNext, activeNext);
            }
            else
            {
                SetSelected(selectedNext, activeNext);
                Invalidate();
            }
        }
        #endregion

        #region オブジェクト検索
        /// <summary>
        /// オブジェクトを検索。
        /// </summary>
        private Node SearchNode(Point point)
        {
            // オブジェクト座標にする
            PointF ptEvent = TransformViewToObject(point);

            // 描画順と逆順に検索
            foreach (var node in Nodes.Reverse())
            {
                if (node.Bound.Contains(ptEvent))
                {
                    return node;
                }
            }

            return null;
        }

        /// <summary>
        /// オブジェクトを検索。
        /// </summary>
        private Node SearchObject(ArrowDirection direction, Node target)
        {
            Node result = null;

            // ターゲット指定なし
            if (target == null)
            {
                PointF ptTgt = PointF.Empty;
                bool first = true;
                foreach (Node obj in Nodes)
                {
                    PointF ptObj = obj.Center;

                    // １個目は無条件に取得
                    if (first)
                    {
                        result = obj;
                        ptTgt = ptObj;
                        first = false;
                    }
                    // ２個目以降から位置を比較
                    else
                    {
                        bool change = false;
                        switch (direction)
                        {
                            case ArrowDirection.Up: change = ptObj.Y > ptTgt.Y; break;
                            case ArrowDirection.Down: change = ptObj.Y < ptTgt.Y; break;
                            case ArrowDirection.Left: change = ptObj.X > ptTgt.X; break;
                            case ArrowDirection.Right: change = ptObj.X < ptTgt.X; break;
                        }

                        if (change)
                        {
                            result = obj;
                            ptTgt = ptObj;
                        }
                    }
                }
            }
            // ターゲット指定あり
            else
            {
                RectangleF rectTgt = target.Bound;
                RectangleF rectObj = RectangleF.Empty;

                PointF ptTgt = target.Center;
                PointF ptObj = PointF.Empty;

                PointF ptTgtChk = PointF.Empty;
                PointF ptObjChk = PointF.Empty;

                float minDist = 100000.0f;	// でかい値にしとく
                foreach (var obj in Nodes)
                {
                    // 自分と比べちゃ駄目
                    if (target == obj)
                    {
                        continue;
                    }

                    rectObj = obj.Bound;
                    ptObj = obj.Center;

                    // 特定点間の距離が近いのを探す
                    float distance = 0.0f;
                    switch (direction)
                    {
                        case ArrowDirection.Up:
                            ptTgtChk.X = ptTgt.X;
                            ptTgtChk.Y = rectTgt.Top;
                            ptObjChk.X = ptObj.X;
                            ptObjChk.Y = rectObj.Bottom;
                            distance = PointUtility.GetDistance(ptTgtChk, ptObjChk);
                            if (ptTgtChk.Y > ptObjChk.Y && distance < minDist)
                            {
                                result = obj;
                                minDist = distance;
                            }
                            break;
                        case ArrowDirection.Down:
                            ptTgtChk.X = ptTgt.X;
                            ptTgtChk.Y = rectTgt.Bottom;
                            ptObjChk.X = ptObj.X;
                            ptObjChk.Y = rectObj.Top;
                            distance = PointUtility.GetDistance(ptTgtChk, ptObjChk);
                            if (ptTgtChk.Y < ptObjChk.Y && distance < minDist)
                            {
                                result = obj;
                                minDist = distance;
                            }
                            break;
                        case ArrowDirection.Left:
                            ptTgtChk.X = rectTgt.Left;
                            ptTgtChk.Y = ptTgt.Y;
                            ptObjChk.X = rectObj.Right;
                            ptObjChk.Y = ptObj.Y;
                            distance = PointUtility.GetDistance(ptTgtChk, ptObjChk);
                            if (ptTgtChk.X > ptObjChk.X && distance < minDist)
                            {
                                result = obj;
                                minDist = distance;
                            }
                            break;
                        case ArrowDirection.Right:
                            ptTgtChk.X = rectTgt.Right;
                            ptTgtChk.Y = ptTgt.Y;
                            ptObjChk.X = rectObj.Left;
                            ptObjChk.Y = ptObj.Y;
                            distance = PointUtility.GetDistance(ptTgtChk, ptObjChk);
                            if (ptTgtChk.X < ptObjChk.X && distance < minDist)
                            {
                                result = obj;
                                minDist = distance;
                            }
                            break;
                    }
                }
            }

            return result;
        }
        #endregion

        /// <summary>
        /// オブジェクト領域を取得。
        /// </summary>
        private RectangleF GetObjectBounds(bool selectedOnly)
        {
            bool empty = true;
            RectangleF result = RectangleF.Empty;
            IEnumerable<Node> nodes = selectedOnly ? (IEnumerable<Node>)SelectedNodes : (IEnumerable<Node>)Nodes;
            foreach (Node obj in nodes)
            {
                if (empty)
                {
                    result = obj.Bound;
                    empty = false;
                }
                else
                {
                    result = RectangleF.Union(result, obj.Bound);
                }
            }

            return result;
        }

        void AlignPos(Node node, PointF pos, AlignMode alignMode)
        {
            node.ResetChildWidthAndHeight(alignMode);
            node.ResetChildPos(pos, alignMode);
        }

        public sealed class DrawContext : IDisposable
        {
            public DrawContext(Color color)
            {
                BrushBase = new SolidBrush(color);
                PenBase = new Pen(color);
                PenLight = new Pen(ControlPaint.LightLight(color));
                PenDark = new Pen(ControlPaint.Dark(color));
            }

            public void Dispose()
            {
                BrushBase.Dispose();
                PenBase.Dispose();
                PenLight.Dispose();
                PenDark.Dispose();
            }

            public Brush BrushBase { get; private set; }
            public Pen PenBase { get; private set; }
            public Pen PenLight { get; private set; }
            public Pen PenDark { get; private set; }
        }

        private static readonly DrawContext ActiveDC = new DrawContext(Color.FromArgb(255, 255, 96));
        private static readonly DrawContext SelectedDC = new DrawContext(Color.FromArgb(255, 255, 255));


        #region ノード

        public const float NodeControlMargin = 3;
        public const float NodeControlUnit = 16;
        public PointF NodeControlPos(int x, int y, float marginX, float marginY)
        {
            return new PointF(marginX + (NodeControlUnit + 2) * x, marginY + (NodeControlUnit + 2) * y);
        }

        private void UpdateNodeControls(Node node)
        {
            if (node is ShapeNode)
            {
                UpdateShapeNodeControls(node);
                return;
            }

            Func<string> toolTipMessage = () =>
            {
                // ラベルがあれば「ラベル(名前)」、なければ「名前」。
                var header = !string.IsNullOrEmpty(node.Content.Label) ? string.Format("{0}({1})", node.Content.Label, node.Content.Name) : node.Content.Name;

                // コメント。
                var body = node.Content.Comment;

                // header と body を改行で結合。
                var message = !string.IsNullOrEmpty(body) ? (header + System.Environment.NewLine + body) : header;

                return message;
            };

            // アイコン
            node.Icon.ClientRectangle = ()=>new RectangleF(NodeControlPos(0, 0, NodeControlMargin, NodeControlMargin), new SizeF(NodeControlUnit, NodeControlUnit));
            node.Icon.Draw = (g, r) => DrawColorEdit(g, node, node.Icon, r);
            node.Icon.CatchEnter = true;
            node.Icon.OperateOnButtonUp = true;
            node.Icon.Operate = () => StartColorPicker(node);

            // ラベルがあれば「ラベル」、なければ「名前」。
            node.MainLabel.ClientRectangle = ()=>new RectangleF(NodeControlPos(1, 0, NodeControlMargin, NodeControlMargin), new SizeF(node.Width - NodeControlMargin * 2 - NodeControlUnit, NodeControlUnit));
            node.MainLabel.Draw = (g, r) => DrawMainLabel(g, node, r);
            node.MainLabel.ToolTipMessage = toolTipMessage;

            // ラベルがあれば「(名前)」、なければ空文字。
            node.SecondaryLabel.ClientRectangle = ()=>new RectangleF(NodeControlPos(0, 1, NodeControlMargin, NodeControlMargin), new SizeF(node.Width - NodeControlMargin * 2, NodeControlUnit));
            node.SecondaryLabel.Draw = (g, r) => DrawSecondaryLabel(g, node, r);
            node.SecondaryLabel.ToolTipMessage = toolTipMessage;
//			node.ModifiedCheck.ClientRectangle = ()=> new RectangleF(node.Width - 6, (float)Math.Floor((node.Height - 2) / 2.0), 5, 5);//new RectangleF(NodeControlPos(5, 0, 0, 0), new SizeF(NodeControlUnit, NodeControlUnit));
            node.ModifiedCheck.ClientRectangle = ()=> new RectangleF(node.Width - 9, (float)Math.Floor((node.Height - 6) / 2.0), 8, 8);
            node.ModifiedCheck.Draw = DrawModifiedCheck;
        }

        private void UpdateShapeNodeControls(Node node)
        {
            node.MainLabel.ClientRectangle = () => new RectangleF(NodeControlPos(0, 0, NodeControlMargin, NodeControlMargin), new SizeF(node.Width - NodeControlMargin * 2, NodeControlUnit));
            node.MainLabel.Draw = (g, r) => DrawMainLabel(g, node, r);
            node.MainLabel.ToolTipMessage = () =>
            {
                var text = !string.IsNullOrEmpty(node.Content.Label) ? node.Content.Label : node.Content.Name;
                return node.MainLabelTrimed || _magnify < 0.601F ? text : "";
            };
//			node.ModifiedCheck.ClientRectangle = () => new RectangleF(node.Width - 6, (float)Math.Floor((node.Height - 2) / 2.0), 5, 5);//new RectangleF(NodeControlPos(5, 0, 0, 0), new SizeF(NodeControlUnit, NodeControlUnit));
            node.ModifiedCheck.ClientRectangle = ()=> new RectangleF(node.Width - 9, (float)Math.Floor((node.Height - 6) / 2.0), 8, 8);
            node.ModifiedCheck.Draw = DrawModifiedCheck;
        }

        private readonly Image EditColor = Resources.EditColor;
        private readonly Image EditColorNull2 = Resources.EditColorNull2;
        private readonly Image EditColor_MouseDown = Resources.EditColor_MouseDown;
        private readonly Image EditColor_MouseOver = Resources.EditColor_MouseOver;
        private readonly Image Control_Star = Resources.Control_Star;

        private void DrawColorEdit(Graphics g, Node node, NodeControl control, RectangleF r)
        {
            if (pointingNodeControl != control || (_operationMode != OperationMode.None && _operationMode != OperationMode.ObjectMoveWait))
            {
                if (node.Content.EditColor.HasValue)
                {
                    RectangleF colorRectangle = r;
                    colorRectangle.Inflate(-1, -1);
                    using (var brush = new SolidBrush(ColorUtility.Pow(node.Content.EditColor.Value.ToColor(), IsDefaultLinear ? 1/2.2:1)))
                    {
                        g.FillRectangle(brush, colorRectangle);
                    }
                    g.DrawImage(EditColor, r);
                }
                else
                {
                    g.DrawImage(EditColorNull2, r);
                }
            }
            else
            {
                RectangleF colorRectangle = r;
                colorRectangle.Inflate(-1, -1);
                using (var brush = new SolidBrush(node.Content.EditColor.HasValue ? ColorUtility.Pow(node.Content.EditColor.Value.ToColor(), IsDefaultLinear ? 1/2.2:1) : Color.White))
                {
                    g.FillRectangle(brush, colorRectangle);
                }

                g.DrawImage(pushedNodeControl == control ? EditColor_MouseDown : EditColor_MouseOver, r);
            }
        }

        private void DrawModifiedCheck(Graphics g, RectangleF r)
        {
            g.DrawImage(Control_Star, r);
        }

        /// <summary>
        /// ラベルがあれば「(名前)」、なければ空文字。
        /// </summary>
        private void DrawSecondaryLabel(Graphics g, Node node, RectangleF r)
        {
            DrawString(g, (!string.IsNullOrEmpty(node.Content.Label) || (node == CommentEditingNode)) ? string.Format("({0})", node.Content.Name) : string.Empty, r, TheApp.GuiFont);
        }

        /// <summary>
        /// ラベルがあれば「ラベル」、なければ「名前」。
        /// </summary>
        private void DrawMainLabel(Graphics g, Node node, RectangleF rect)
        {
            var text = !string.IsNullOrEmpty(node.Content.Label) ? node.Content.Label : node.Content.Name;
            if (node.Content is Material)
            {
                var mtl = node.Content as Material;
                node.TextureValidity = mtl.TextureValidity();
                node.ParentMaterialsValidity = mtl.ParentMaterialsValidity();
                node.CombinerValidity = mtl.CombinerValidity();
                if (!node.TextureValidity || !node.ParentMaterialsValidity || !node.CombinerValidity)
                {
                    node.MainLabelTrimed = DrawString(g, text, rect, TheApp.GuiFontBold, Color.Red);
                    return;
                }
            }
            node.MainLabelTrimed = DrawString(g, text, rect, TheApp.GuiFontBold);
            node.TextureValidity = true;
            node.ParentMaterialsValidity = true;
            node.CombinerValidity = true;
        }

        private bool DrawString(Graphics g, string text, RectangleF rect, Font font, Color? color = null)
        {
            float scale = Math.Min(1 / _magnify, 10F / 8F);
            bool trimed;
            Matrix mat = g.Transform;
            {
                g.ScaleTransform(scale, scale);
                RectangleF rectScaled = new RectangleF(rect.X / scale, rect.Y / scale, rect.Width / scale, rect.Height / scale);
                trimed = DrawTrimString(g, font, text, rectScaled, color);
            }
            g.Transform = mat;
            return trimed;
        }

        static private StringFormat sf;
        static private readonly StringBuilder builder = new StringBuilder();
        private bool DrawTrimString(Graphics g, Font font, string content, RectangleF rect, Color? color = null)
        {
            int trim = 0;
            string trimedContent = content;
            while (content.Length - trim > 4)
            {
                var size = g.MeasureString(trimedContent, font, rect.Size, sf);
                if (size.Width < rect.Width)
                {
                    break;
                }
                trim++;
                builder.Append(content, 0, content.Length - trim - 3);
                builder.Append("..");
                builder.Append(content, content.Length - 3, 3);
                trimedContent = builder.ToString();
                builder.Clear();
            }

            var b = new SolidBrush(color ?? Color.Black);
            g.DrawString(trimedContent, font, b, rect, sf);

            return trim > 0;
        }

        bool ShowSecondaryLabel(Node node)
        {
            return CommentEditingNode == node || !string.IsNullOrEmpty(node.Content.Label);
        }

        private Node CommentEditingNode;

        private abstract class Node
        {
            public GuiObject Content { get; private set; }

            protected Node(GuiObject content)
            {
                Content = content;

                // ラベルがあればラベルをメインラベルに表示し、二次ラベルに名前を表示する。
                // ここで値を設定しておかなければ、ファイル読み込み直後の非表示オブジェクトでノード境界が正しく算出できない。
                // 詳細は ObjectSchematicView.RefreshNodes() を参照。
                ShowingSecondaryLabel = !string.IsNullOrEmpty(Content.Label);

                PosAligned = false;
                MainLabel = new NodeControl();
                Icon = new PushableNodeControl();
                UpdateNodeControl = true;
                SecondaryLabel = new NodeControl();
                ModifiedCheck = new NodeControl();
                TextureValidity = true;
                ParentMaterialsValidity = true;
                CombinerValidity = true;
            }

            public bool ShowingSecondaryLabel;
            public PointF Location;
            public bool PosAligned { get; set; }
            const float width = 100;
            const float height = 21;
            const float heightWithSecondaryLabel = 39;
            public virtual float Width { get { return 100; } }
            public virtual float Height { get { return ShowingSecondaryLabel? heightWithSecondaryLabel : height; } }
            private float MarginX { get { return 16; } }
            private float MarginY { get { return 16; } }
            public PointF TopMiddle { get { return new PointF(Location.X + Width / 2, Location.Y); } }
            public PointF BottomMiddle { get { return new PointF(Location.X + Width / 2, Location.Y + Height); } }
            public PointF Center { get { return new PointF(Location.X + Width / 2, Location.Y + Height / 2); } }
            public bool UpdateNodeControl { get; set; }
            public readonly NodeControl MainLabel;   // メインラベル (コメントラベルがあれば「ラベル」、なければ「名前」)
            public readonly PushableNodeControl Icon;
            public readonly NodeControl SecondaryLabel; // 二次ラベル (コメントラベルがあれば「(名前)」を表示、なければ非表示)
            public readonly NodeControl ModifiedCheck;
            public abstract IEnumerable<Node> Children { get; }
            public virtual GraphicsPath RoundishRectangle
            {
                get
                {
                    return ShowingSecondaryLabel? _roundishRectangleWithComment: _roundishRectangle;
                }
            }
            static private readonly GraphicsPath _roundishRectangle = CreateRoundishRectangle(0, 0, width, height, 6);

            public virtual GraphicsPath OffsetedRoundishRectangle
            {
                get
                {
                    return ShowingSecondaryLabel? _offsetRoundishRectangleWithComment: _offsetRoundishRectangle;
                }
            }
            static private readonly GraphicsPath _offsetRoundishRectangle = CreateRoundishRectangle(-1.5F, -1.5F, width + 3, height + 3, 6 + 1.5F);

            static private readonly GraphicsPath _roundishRectangleWithComment = CreateRoundishRectangle(0, 0, width, heightWithSecondaryLabel, 6);
            static private readonly GraphicsPath _offsetRoundishRectangleWithComment = CreateRoundishRectangle(-1.5F, -1.5F, width + 3, heightWithSecondaryLabel + 3, 6 + 1.5F);
            public bool MainLabelTrimed;
            public bool TextureValidity;
            public bool ParentMaterialsValidity;
            public bool CombinerValidity;
            static private GraphicsPath CreateRoundishRectangle(float x, float y, float width, float height, float radius)
            {
                GraphicsPath path = new GraphicsPath();
                path.AddArc(x, y, 2 * radius, 2 * radius, 180, 90);
                path.AddArc(x + width - 2 * radius, y, 2 * radius, 2 * radius, 270, 90);
                path.AddArc(x + width - 2 * radius, y + height - 2 * radius, 2 * radius, 2 * radius, 0, 90);
                path.AddArc(x, y + height - 2 * radius, 2 * radius, 2 * radius, 90, 90);
                path.CloseFigure();
                return path;
            }

            // 自分自身も含めた子孫
            public IEnumerable<Node> Descendants
            {
                get
                {
                    return Enumerable.Repeat(this, 1).Concat(Children.SelectMany(x => x.Descendants));
                }
            }

            // 親ノード(複数可)
            public abstract IEnumerable<Node> Parents { get; }

            public RectangleF Bound
            {
                get
                {
                    return new RectangleF(Location.X, Location.Y, Width, Height);
                }
            }

            public RectangleF TreeBound
            {
                get
                {
                    RectangleF rect = Bound;
                    foreach (var node in Descendants.Where(x => x != this))
                    {
                        rect = RectangleF.Union(rect, node.Bound);
                    }
                    return rect;
                }
            }

            private float ChildWidth { get; set; }
            private float ChildHeight { get; set; }
            public void ResetChildWidthAndHeight(AlignMode alignMode)
            {
                float width = 0;
                float height = 0;
                foreach (var child in Children)
                {
                    child.ResetChildWidthAndHeight(alignMode);

                    if (alignMode == AlignMode.Schematic)
                    {
                        width += child.ChildWidth;
                        height = Math.Max(child.ChildHeight, height);
                    }
                    else
                    {
                        width = Math.Max(child.ChildWidth, width);
                        height += child.ChildHeight;
                    }
                }

                if (alignMode == AlignMode.Schematic)
                {
                    ChildWidth = Math.Max(width, Width + MarginX);
                    ChildHeight = height + Height + MarginY;
                }
                else
                {
                    ChildWidth = width + Width + MarginX;
                    ChildHeight = Math.Max(height, Height + MarginY);
                }
            }

            public void ResetChildPos(PointF point, AlignMode alignMode)
            {
                Location = point;

                if (alignMode == AlignMode.Schematic)
                {
                    point.X += -ChildWidth / 2;
                    point.Y += Height + MarginY;
                    foreach (var child in Children)
                    {
                        point.X += child.ChildWidth / 2;
                        child.ResetChildPos(point, alignMode);
                        point.X += child.ChildWidth / 2;
                    }
                }
                else
                {
                    point.X += Width + MarginX;
                    foreach (var child in Children)
                    {
                        child.ResetChildPos(point, alignMode);
                        point.Y += child.ChildHeight;
                    }
                }
            }

            public void MovePos(float x, float y, bool withChild)
            {
                if (withChild)
                {
                    foreach (var node in Descendants)
                    {
                        node.Location.X += x;
                        node.Location.Y += y;
                    }
                }
                else
                {
                    Location.X += x;
                    Location.Y += y;
                }
            }



            public abstract DrawContext DrawContext { get; }
            public virtual IEnumerable<NodeControl> NodeControls
            {
                get
                {
                    yield return Icon;
                    yield return MainLabel;
                    if (ShowingSecondaryLabel)
                    {
                        yield return SecondaryLabel;
                    }
                    if (Content.IsModifiedObject)
                    {
                        yield return ModifiedCheck;
                    }
                }
            }
        }

        private class ModelNode : Node
        {
            public ModelNode(Model model) : base(model)
            {
                childNodes = new List<Node>();

                // ボーンの生成
                Dictionary<string, BoneNode> boneToNode = new Dictionary<string, BoneNode>();
                foreach (var bone in model.Bones)
                {
                    var node = new BoneNode(bone);
                    boneToNode.Add(bone.Name, node);
                }

                // シェイプの生成
                List<ShapeNode> shapeNodes = new List<ShapeNode>();
                foreach (var shape in model.Shapes)
                {
                    var node = new ShapeNode(shape);
                    BoneNode parent = boneToNode[shape.Data.shape_info.bone_name];
                    parent.childNodes.Add(node);
                    node.Parent = parent;
                    shapeNodes.Add(node);
                }

                // ノードの生成
                foreach (var bone in model.Bones)
                {
                    if (bone.Parent == null)
                    {
                        boneToNode[bone.Name].Parent = this;
                        childNodes.Add(boneToNode[bone.Name]);
                    }
                    else
                    {
                        BoneNode node = boneToNode[bone.Name];
                        BoneNode parent = boneToNode[bone.Parent.Name];
                        node.Parent = parent;
                        parent.childNodes.Add(node);
                    }
                }

                // マテリアルの生成
                List<MaterialNode> materialNodes = new List<MaterialNode>();
                Dictionary<string, MaterialNode> nameMaterialNodeTable = new Dictionary<string, MaterialNode>();
                foreach (var material in model.Materials)
                {
                    var node = new MaterialNode(material);
                    materialNodes.Add(node);
                    nameMaterialNodeTable.Add(material.Name, node);
                }

                foreach (var shapeNode in shapeNodes)
                {
                    var shape = (Shape)shapeNode.Content;
                    var materialNode = nameMaterialNodeTable[shape.Data.shape_info.mat_name];
                    if (!materialNode.ParentList.Any())
                    {
                        shapeNode.childNodes.Add(materialNode);
                    }
                    materialNode.ParentList.Add(shapeNode);
                }
            }

            private readonly List<Node> childNodes;
            public override IEnumerable<Node> Children
            {
                get
                {
                    return childNodes;
                }
            }

            public override IEnumerable<Node> Parents
            {
                get { return Enumerable.Empty<Node>(); }
            }

            public override DrawContext DrawContext
            {
                get { return drawContext; }
            }
            private static readonly DrawContext drawContext = new DrawContext(Color.FromArgb(240, 208, 176));
        }

        private class BoneNode : Node
        {
            public BoneNode(Bone bone)
                : base(bone)
            {
                childNodes = new List<Node>();
            }
            public override DrawContext DrawContext
            {
                get { return drawContext; }
            }
            private static readonly DrawContext drawContext = new DrawContext(Color.FromArgb(192, 224, 240));
            public override IEnumerable<Node> Children
            {
                get { return childNodes; }
            }
            public readonly List<Node> childNodes;
            public Node Parent { private get; set; }
            public override IEnumerable<Node> Parents
            {
                get { yield return Parent; }
            }
        }

        private class ShapeNode : Node
        {
            public ShapeNode(Shape shape)
                : base(shape)
            {
                childNodes = new List<Node>();
            }
            public override DrawContext DrawContext
            {
                get { return drawContext; }
            }
            private static readonly DrawContext drawContext = new DrawContext(Color.FromArgb(176, 240, 176));
            public override IEnumerable<Node> Children
            {
                get { return childNodes; }
            }
            public readonly List<Node> childNodes;
            public Node Parent { private get; set; }
            public override IEnumerable<Node> Parents
            {
                get { yield return Parent; }
            }

            public override IEnumerable<NodeControl> NodeControls
            {
                get
                {
                    //yield return Icon;
                    yield return MainLabel;
                    if (Content.IsModifiedObject)
                    {
                        yield return ModifiedCheck;
                    }
                }
            }
        }

        private class MaterialNode : Node
        {
            public MaterialNode(Material material) : base(material)
            {
                ParentList = new List<Node>();
            }

            public override DrawContext DrawContext
            {
                get { return drawContext; }
            }
            private static readonly DrawContext drawContext = new DrawContext(Color.FromArgb(240, 192, 208));

            public override IEnumerable<Node> Children
            {
                get { return Enumerable.Empty<Node>(); }
            }

            public List<Node> ParentList { get; private set; }
            public override IEnumerable<Node> Parents
            {
                get { return ParentList; }
            }
        }
        #endregion

        #region NodeControl
        public class NodeControl
        {
            public Func<RectangleF> ClientRectangle;
            public Action<Graphics, RectangleF> Draw;
            public bool CatchEnter;
            public Func<string> ToolTipMessage;
            public Func<Color> SpecificNameColor;
        }

        public class PushableNodeControl : NodeControl
        {
            public Action Operate;
            public bool OperateOnButtonUp = true;
        }
        #endregion

        #region 描画
        protected override void OnPaint(System.Windows.Forms.PaintEventArgs e)
        {
            Graphics g = e.Graphics;

            // 行列設定
            Matrix preTransform = g.Transform;
            g.ScaleTransform(_magnify, _magnify);
            g.TranslateTransform((float)Math.Round(-_origin.X), (float)Math.Round(-_origin.Y));

            // スムージング設定
            SmoothingMode preSmoothingMode = g.SmoothingMode;
            InterpolationMode preInterpolationMode = g.InterpolationMode;

            // テキスト描画法設定
            TextRenderingHint preTextRenderingHint = g.TextRenderingHint;
            if (_magnify == 1)
            {
                g.TextRenderingHint = System.Drawing.Text.TextRenderingHint.ClearTypeGridFit;//SystemDefault;
            }
            else
            {
                g.SmoothingMode = SmoothingMode.AntiAlias;
                g.TextRenderingHint = System.Drawing.Text.TextRenderingHint.ClearTypeGridFit;//AntiAlias;
            }

            // 背景描画
            g.Clear(Color.DimGray);

            foreach (var node in Nodes)
            {
                foreach (var parent in node.Parents)
                {
                    DrawConnectionLine(node, parent, g);
                }
            }

            foreach (var node in Nodes)
            {
                DrawNode(node, g);
            }

            // 設定を元に戻す
            g.InterpolationMode = preInterpolationMode;
            g.TextRenderingHint = preTextRenderingHint;
            g.SmoothingMode = preSmoothingMode;
            g.Transform = preTransform;

            // トラッキング枠
            if (_operationMode == OperationMode.ObjectTracking)
            {
                ControlPaint.DrawFocusRectangle(g, MouseDragBounds, Color.White, Color.Transparent);
            }

            // フォーカス枠
            if (Focused)
            {
                ControlPaint.DrawFocusRectangle(g, ClientRectangle);
            }

            base.OnPaint(e);
        }

        private void DrawConnectionLine(Node node, Node parent, Graphics g)
        {
            PointF pt1 = node.TopMiddle;
            PointF pt2 = parent.BottomMiddle;

            // 2点からなる領域
            RectangleF rect = RectangleUtility.FromPoints(pt1, pt2);

            if (rect.IntersectsWith(g.ClipBounds))
            {
                DrawContext context = node.DrawContext;
                bool isSelected = SelectedNodes.Contains(node) || SelectedNodes.Contains(parent);
                SmoothingMode tmp = g.SmoothingMode;
                g.SmoothingMode = SmoothingMode.AntiAlias;

                float tmp2 = context.PenBase.Width;
                context.PenBase.Width = isSelected ? 4: 3;
                g.DrawLine(context.PenBase, pt1, pt2);
                context.PenBase.Width = tmp2;

                if (isSelected)
                {
                    Pen selectPen = (ActiveSelectedNode == node || ActiveSelectedNode == parent) ? ActiveDC.PenBase : SelectedDC.PenBase;
                    float tmp3 = selectPen.Width;
                    float nx = pt2.Y - pt1.Y;
                    float ny = pt1.X - pt2.X;
                    float d = (float)Math.Sqrt(nx * nx + ny * ny);
                    if (d > 0.5)
                    {
                        nx *= 2F / d;
                        ny *= 2F / d;
                        g.DrawLine(selectPen, pt1.X - nx, pt1.Y - ny, pt2.X - nx, pt2.Y - ny);
                        g.DrawLine(selectPen, pt1.X + nx, pt1.Y + ny, pt2.X + nx, pt2.Y + ny);
                    }
                    selectPen.Width = tmp3;
                }
                g.SmoothingMode = tmp;
            }
        }

        private void DrawNode(Node node, Graphics g)
        {
            Matrix transform = g.Transform;
            g.TranslateTransform((float)Math.Round(node.Location.X), (float)Math.Round(node.Location.Y));

            // 描画
            RectangleF rectObj = new RectangleF(0, 0, node.Width, node.Height);

            // 選択枠分拡張
            rectObj.Inflate(2, 2);


            if (rectObj.IntersectsWith(g.ClipBounds))
            {
                DrawContext context = node.DrawContext;

                SmoothingMode smoothingMode = g.SmoothingMode;
                g.SmoothingMode = SmoothingMode.AntiAlias;
                g.FillPath(context.BrushBase, node.RoundishRectangle);
                g.DrawPath(context.PenDark, node.RoundishRectangle);

                if (SelectedNodes.Contains(node))
                {
                    Pen pen = ActiveSelectedNode == node ? ActiveDC.PenBase : SelectedDC.PenBase;
                    float tmp = pen.Width;
                    pen.Width = 2;
                    g.DrawPath(pen, node.OffsetedRoundishRectangle);
                    pen.Width = tmp;
                }

                g.SmoothingMode = smoothingMode;

                //　NodeControl の描画
                foreach (var control in node.NodeControls)
                {
                    control.Draw(g, control.ClientRectangle());
                }
            }

            g.Transform = transform;
        }

        #endregion

        #region 描画座標変換
        /// <summary>
        /// オブジェクト座標→表示座標変換。
        /// </summary>
        private PointF TransformObjectToView(PointF point)
        {
            point.X -= _origin.X;
            point.Y -= _origin.Y;

            point.X *= _magnify;
            point.Y *= _magnify;

            // 丸め込む
            return Point.Round(point);
        }

        /// <summary>
        /// オブジェクト座標→表示座標変換。
        /// </summary>
        private RectangleF TransformObjectToView(RectangleF rect)
        {
            rect.X -= _origin.X;
            rect.Y -= _origin.Y;

            rect.X *= _magnify;
            rect.Y *= _magnify;
            rect.Width *= _magnify;
            rect.Height *= _magnify;

            // 丸め込む
            return Rectangle.Round(rect);
        }

        /// <summary>
        /// 表示座標→オブジェクト座標変換。
        /// </summary>
        private PointF TransformViewToObject(PointF point)
        {
            point.X /= _magnify;
            point.Y /= _magnify;

            point.X += _origin.X;
            point.Y += _origin.Y;

            return point;
        }

        /// <summary>
        /// 表示座標→オブジェクト座標変換。
        /// </summary>
        private RectangleF TransformViewToObject(RectangleF rect)
        {
            rect.X /= _magnify;
            rect.Y /= _magnify;
            rect.Width /= _magnify;
            rect.Height /= _magnify;

            rect.X += _origin.X;
            rect.Y += _origin.Y;

            return rect;
        }

        /// <summary>
        /// 表示座標→オブジェクト座標変換。
        /// </summary>
        private SizeF TransformViewToObject(SizeF size)
        {
            size.Width /= _magnify;
            size.Height /= _magnify;

            return size;
        }
        #endregion

        #region マウス操作
        Node pushedObject;
        PushableNodeControl pushedNodeControl;

        Node pointingObject;
        NodeControl pointingNodeControl;

        /// <summary>
        /// マウスアップ
        /// </summary>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            // オペレーション実行
            if (pushedNodeControl != null && pushedNodeControl.Operate != null && pushedNodeControl.OperateOnButtonUp)
            {
                PointF location = TransformViewToObject(e.Location);
                location.X -= pushedObject.Location.X;
                location.Y -= pushedObject.Location.Y;
                if (pushedNodeControl.ClientRectangle().Contains(location))
                {
                    pushedNodeControl.Operate();
                }
            }

            UpdatePushingObject(null, null);
            base.OnMouseUp(e);
        }

        /// <summary>
        /// マウス移動
        /// </summary>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            UpdatePointingObject(e.Location);
            base.OnMouseMove(e);
        }

        private void UpdatePointingObject(Point viewCooridinate)
        {
            Node obj = SearchNode(viewCooridinate);
            NodeControl control = null;
            if (obj != null)
            {
                PointF location = TransformViewToObject(viewCooridinate);
                SearchNodeControl(obj, location, out control);
            }

            bool invalidate = false;
            RectangleF region = Rectangle.Empty;
            if (obj != pointingObject || control != pointingNodeControl)
            {
                if (control != null && control.CatchEnter)
                {
                    RectangleF rectF = control.ClientRectangle();
                    rectF.Offset(obj.Location);
                    rectF.Inflate(1, 1);
                    rectF = TransformObjectToView(rectF);
                    region = !invalidate ? rectF : RectangleF.Union(region, rectF);
                    invalidate = true;
                }

                if (pointingNodeControl != null && pointingNodeControl.CatchEnter)
                {
                    RectangleF rectF = pointingNodeControl.ClientRectangle();
                    rectF.Offset(pointingObject.Location);
                    rectF.Inflate(1, 1);
                    rectF = TransformObjectToView(rectF);
                    region = !invalidate ? rectF : RectangleF.Union(region, rectF);
                    invalidate = true;
                }
            }

            pointingObject = obj;
            pointingNodeControl = control;

            UpdateToolTip();

            if (invalidate)
            {
                Invalidate(new Rectangle((int)region.X, (int)region.Y, (int)region.Width, (int)region.Height));
            }
        }

        private void UpdateToolTip()
        {
            if (pointingObject != null && pointingNodeControl != null && pointingNodeControl.ToolTipMessage != null)
            {
                string message = pointingNodeControl.ToolTipMessage();

                if (nameToolTip.GetToolTip(this) != message)
                {
                    if (message != "")
                    {
                        nameToolTip.SetToolTip(this, message);
                    }
                    else
                    {
                        nameToolTip.Active = false;
                    }
                }
                else
                {
                    nameToolTip.Active = true;
                }
            }
            else
            {
                nameToolTip.Active = false;
            }
        }

        private void InvalidatePointingObject()
        {
            if (pointingNodeControl != null && pointingNodeControl.CatchEnter)
            {
                RectangleF rectF = pointingNodeControl.ClientRectangle();
                rectF.Offset(pointingObject.Location);
                rectF.Inflate(1, 1);
                rectF = TransformObjectToView(rectF);
                Invalidate(new Rectangle((int)rectF.X, (int)rectF.Y, (int)rectF.Width, (int)rectF.Height));
            }
        }

        private void UpdatePushingObject(Node obj, PushableNodeControl control)
        {
            bool invalidate = false;
            RectangleF region = Rectangle.Empty;
            if (control != null && control.CatchEnter)
            {
                RectangleF rectF = control.ClientRectangle();
                rectF.Offset(obj.Location);
                rectF.Inflate(1, 1);
                rectF = TransformObjectToView(rectF);
                region = !invalidate ? rectF : RectangleF.Union(region, rectF);
                invalidate = true;
            }

            if (pushedNodeControl != null && pushedNodeControl.CatchEnter)
            {
                RectangleF rectF = pushedNodeControl.ClientRectangle();
                rectF.Offset(pushedObject.Location);
                rectF.Inflate(1, 1);
                rectF = TransformObjectToView(rectF);
                region = !invalidate ? rectF : RectangleF.Union(region, rectF);
                invalidate = true;
            }

            pushedObject = obj;
            pushedNodeControl = control;
            if (invalidate)
            {
                Invalidate(new Rectangle((int)region.X, (int)region.Y, (int)region.Width, (int)region.Height));
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnBeginMouseDrag(MouseBeginDragEventArgs e)
        {
            //DebugConsole.WriteLine("OnBeginMouseDrag");
            switch (e.Button)
            {
                // 左ボタン
                case MouseButtons.Left:
                    Node obj = SearchNode(e.Location);
                    if (obj != null)
                    {
                        PointF location = TransformViewToObject(e.Location);
                        NodeControl control;
                        if (SearchNodeControl(obj, location, out control))
                        {
                            PushableNodeControl pushableControl = control as PushableNodeControl;
                            UpdatePushingObject(obj, pushableControl);
                        }
                    }

                    // オブジェクト選択
                    switch (SelectNode(obj, e.Button))
                    {
                        // トラッキング開始
                        case SelectObjectResult.Noting:
                            ChangeOperationMode(OperationMode.ObjectTracking);
                            break;

                        // オブジェクト移動開始
                        case SelectObjectResult.Selected:
                            ChangeOperationMode(OperationMode.ObjectMoveWait);
                            break;

                        default:
                            break;
                    }
                    break;

                // 中ボタン
                case MouseButtons.Middle:
                    // ビュースケール開始
                    ChangeOperationMode(Control.ModifierKeys == Keys.Control
                                            ? OperationMode.ViewScale
                                            : OperationMode.ViewScroll);
                    break;

                // 右ボタン
                case MouseButtons.Right:
                    if (Control.ModifierKeys == Keys.Alt)
                    {
                        ChangeOperationMode(OperationMode.ViewScale);
                    }
                    else
                    {
                        // オブジェクト選択
                        switch (SelectObject(e.Location, e.Button))
                        {
                            // トラッキング開始
                            case SelectObjectResult.Noting:
                                ChangeOperationMode(OperationMode.ObjectTracking);
                                break;
                            default:
                                break;
                        }
                    }
                    break;
            }
            base.OnBeginMouseDrag(e);
        }

        private bool SearchNodeControl(Node obj, PointF location, out NodeControl control)
        {
            location.X -= obj.Location.X;
            location.Y -= obj.Location.Y;
            foreach (var candidate in obj.NodeControls)
            {
                if (candidate.ClientRectangle().Contains(location))
                {
                    control = candidate;
                    return true;
                }
            }
            control = null;
            return false;
        }

        // オブジェクト移動待機中から移動を開始するまでのマウス移動量(ピクセル)
        private const int ObjectMoveThreshold = 4;

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseDrag(MouseEventArgs e)
        {
            // オブジェクト移動待機状態なら
            if (_operationMode == OperationMode.ObjectMoveWait)
            {
                // 開始点からの移動量が一定範囲を超えれば解除
                if (TestMouseDragSizeOver(ObjectMoveThreshold))
                {
                    _operationMode = OperationMode.ObjectMove;
                    UpdatePushingObject(null, null);
                }
            }

            // ドラッグ操作
            switch (_operationMode)
            {
                // オブジェクト移動
                case OperationMode.ObjectMove:
                    {
                        // 移動量
                        SizeF delta = TransformViewToObject(MouseDragDeltaSize);

                        // 複数選択を個別に移動
                        if (SelectedNodes.Count >= 2)
                        {
                            foreach (Node obj in SelectedNodes)
                            {
                                obj.MovePos(delta.Width, delta.Height, false);
                            }
                        }
                        // 単一オブジェクトを移動
                        else
                        {
                            bool single = Control.ModifierKeys == Keys.Shift;
                            ActiveSelectedNode.MovePos(delta.Width, delta.Height, !single);
                        }

                        UpdateViewContent();
                        Invalidate();
                    }
                    break;

                // オブジェクトトラッキング
                case OperationMode.ObjectTracking:
                    Invalidate();
                    break;

                // 原点移動
                case OperationMode.ViewScroll:
                    {
                        SizeF delta = TransformViewToObject(MouseDragDeltaSize);
                        _origin.X -= delta.Width;
                        _origin.Y -= delta.Height;

                        UpdateViewContent();
                        Invalidate();
                    }
                    break;

                // 表示倍率変更
                case OperationMode.ViewScale:
                    {
                        Point ptCenter = new Point(ClientSize.Width / 2, ClientSize.Height / 2);
                        Point ptRelay = MouseDragRelayPoint;

                        // 変更比率は非線形
                        float delta = 0.050f * _magnify;
                        if (e.X < ptRelay.X) { ScaleView(ptCenter, -delta); }
                        else if (e.X > ptRelay.X) { ScaleView(ptCenter, delta); }
                    }
                    break;

                default:
                    break;
            }
            base.OnMouseDrag(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnEndMouseDrag(MouseEventArgs e)
        {
            //DebugConsole.WriteLine("OnEndMouseDrag");

            // トラッキング終了
            if (_operationMode == OperationMode.ObjectTracking)
            {
                EndTracking();
            }
            // ドラッグ操作終了
            if (_operationMode != OperationMode.None)
            {
                ChangeOperationMode(OperationMode.None);
            }

            InvalidatePointingObject();
            base.OnEndMouseDrag(e);
        }

        /// <summary>
        /// 操作モードを変更。
        /// </summary>
        private void ChangeOperationMode(OperationMode mode)
        {
            _operationMode = mode;

            // カーソル設定
            Cursor cursor = null;
            switch (mode)
            {
                case OperationMode.None:
                    cursor = Cursors.Default;
                    break;
                case OperationMode.ViewScroll:
                    cursor = ResCursors.ViewScroll;
                    break;
                case OperationMode.ViewScale:
                    cursor = ResCursors.ViewScale;
                    break;

                default: break;
            }
            if (cursor != null)
            {
                if (Cursor != cursor)
                {
                    Cursor = cursor;
                }
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseDoubleClick(MouseEventArgs e)
        {
            DebugConsole.WriteLine("OnMouseDoubleClick");
            if (e.Button == MouseButtons.Left && _operationMode != OperationMode.ObjectMove)
            {
                    // プロパティウィンドウ表示
                    if (SelectedNodes.Any())
                    {
                        App.PropertyEdit.ObjectPropertyDialog.ShowPropertyDialog();
                    }
            }
            base.OnMouseDoubleClick(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseWheel(MouseEventArgs e)
        {
            // ドラッグ中は無視
            if (IsMouseDragging()) { return; }

            // クライアント領域内のみ
            if (ClientRectangle.Contains(e.Location))
            {
                float unit = (float)e.Delta / (float)Win32.Constants.WHEEL_DELTA;
                const float delta = 0.1f; //0.20f * _magnify;
                // スケーリング
                ScaleView(e.Location, delta * unit);
            }
            base.OnMouseWheel(e);
        }

        #endregion

        #region キー入力
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool IsInputKey(Keys keyData)
        {
            switch (keyData)
            {
                // 方向キーを通常キー扱いにする
                case Keys.Up:
                case Keys.Down:
                case Keys.Left:
                case Keys.Right:
                case (Keys.Shift | Keys.Up):
                case (Keys.Shift | Keys.Down):
                case (Keys.Shift | Keys.Left):
                case (Keys.Shift | Keys.Right):
                    break;

                default:
                    return base.IsInputKey(keyData);
            }
            return true;
        }

        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            // ドラッグ中とコメント入力中は無視
            if (IsMouseDragging() || tbxComment.Visible) { return false; }

            switch (keyData & Keys.KeyCode)
            {
                // 選択オブジェクト変更
                case Keys.Up:
                    SelectObject(ArrowDirection.Up);
                    break;
                case Keys.Down:
                    SelectObject(ArrowDirection.Down);
                    break;
                case Keys.Left:
                    SelectObject(ArrowDirection.Left);
                    break;
                case Keys.Right:
                    SelectObject(ArrowDirection.Right);
                    break;

                // 表示倍率拡大
                case Keys.Add:
                case Keys.Oemplus:
                    ScaleView(0.1f);
                    break;

                // 表示倍率縮小
                case Keys.Subtract:
                case Keys.OemMinus:
                    ScaleView(-0.1f);
                    break;

                case Keys.A:
                    if ((keyData & Keys.Control) == Keys.Control)
                    {
                        // すべて選択
                        SelectAll();
                    }
                    else
                    {
                        // 全体を中央表示
                        CenteringWhole(false);
                    }
                    break;
                // 選択オブジェクトを中央表示
                case Keys.F:
                    CenteringSelected();
                    break;
                // コメント編集
                case Keys.F3:
                    if (ActiveSelectedNode != null && !(ActiveSelectedNode is ShapeNode))
                    {
                        ShowEditCommentTextBox(ActiveSelectedNode);
                    }
                    break;
                default:
                    return base.ProcessCmdKey(ref msg, keyData);
            }

            return true;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnKeyUp(KeyEventArgs e)
        {
            switch (e.KeyCode)
            {
                // シフト選択開始オブジェクトクリア
                case Keys.ShiftKey:
                    if (_shiftSelectTarget != null)
                    {
                        _shiftSelectTarget = null;
                    }
                    break;

                default:
                    break;
            }
            base.OnKeyUp(e);
        }
        #endregion

        #region その他
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnResize(EventArgs e)
        {
            Size size = ClientSize;

            // 原点調整
            if (_clientSize != size)
            {
                SizeF delta = new SizeF(
                    (size.Width - _clientSize.Width) / 2.0f,
                    (size.Height - _clientSize.Height) / 2.0f
                );
                delta = TransformViewToObject(delta);

                _origin.X -= delta.Width;
                _origin.Y -= delta.Height;
            }

            // サイズを保持
            _clientSize = size;
            base.OnResize(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnContextMenuPopup(ContextMenuPopupEventArgs e)
        {
            // コンテキストメニュー
            TheApp.MainFrame.ShowObjectMenu(PointToScreen(e.Location), this);
            base.OnContextMenuPopup(e);
        }
        #endregion
        #region コメント

        // モデル向き?
        private void ShowEditCommentTextBox(Node node)
        {
            CommentEditingNode = node;
            tbxComment.Text = node.Content.Label;
            if (node.ShowingSecondaryLabel == false)
            {
                node.ShowingSecondaryLabel = true;
                foreach (var child in node.Children)
                {
                    child.MovePos(0, 16, true);
                }
            }
            UpdateViewContent();
            tbxComment.Visible = true;
            tbxComment.Focus();
            tbxComment.Select(tbxComment.Text.Length, 0);
        }

        // 表示内容を更新する
        private void UpdateViewContent()
        {
            //DebugConsole.WriteLine("UpdateViewContent");
            SetTextBoxLocation();
            UpdatePointingObject(PointToClient(MousePosition));
        }

        /// <summary>
        /// コメント入力用テキストボックスの位置と大きさを調整する。
        /// </summary>
        private void SetTextBoxLocation()
        {
            if (CommentEditingNode == null)
            {
                return;
            }

            tbxComment.SuspendLayout();

            // 文字幅以上にする
            float fontSize = (_magnify * TheApp.GuiFont.SizeInPoints);
            if (tbxComment.Font.SizeInPoints != fontSize)
            {
                tbxComment.Font = new Font(TheApp.GuiFont.Name, fontSize, FontStyle.Regular, GraphicsUnit.Point);
            }
            Size sz = TextRenderer.MeasureText(tbxComment.Text + " ", tbxComment.Font);
            tbxComment.Width = (int)(sz.Width);
            tbxComment.Height = sz.Height;

            // 位置を調整する
            PointF point = CommentEditingNode.MainLabel.ClientRectangle().Location;

            // 幅以上にする
            int width = (int)(_magnify * (CommentEditingNode.Width - 3 - point.X));
            tbxComment.Width = Math.Max(width, tbxComment.Width);

            point.X += (float)Math.Round(CommentEditingNode.Location.X);
            point.Y += (float)Math.Round(CommentEditingNode.Location.Y);

            point = TransformObjectToView(point);
            tbxComment.Location = new Point((int)point.X, (int)point.Y);
            if (tbxComment.Width + tbxComment.Location.X > ClientSize.Width)
            {
                tbxComment.Width = ClientSize.Width - tbxComment.Location.X;
            }

            tbxComment.ResumeLayout();

            // カレット位置を調整
            int start = tbxComment.SelectionStart;
            tbxComment.SelectionStart = 0;
            tbxComment.SelectionStart = start;
        }

        /// <summary>
        /// テキストボックスコマンドハンドラ
        /// </summary>
        private void TextBoxKeyDown(object sender, System.Windows.Forms.KeyEventArgs e)
        {
            switch (e.KeyData)
            {
                // エンターで確定
                case Keys.Return:
                    UpdateComment();
                    // フォーカスをテキストボックスから取り返す。
                    Focus();
                    break;
                case Keys.Escape:
                    UpdateComment(true);
                    Focus();
                    break;
            }
        }

        // コメントをアップデートする
        private void UpdateComment(bool cancel = false)
        {
            if (CommentEditingNode == null)
            {
                return;
            }

            string label = tbxComment.Text;
            Node node = CommentEditingNode;
            CommentEditingNode = null;
            tbxComment.Visible = false;
            if (!cancel && node.Content.Label != label)
            {
                TheApp.CommandManager.Execute(
                    new GeneralGroupReferenceEditCommand<string>(
                    new GuiObjectGroup(App.AppContext.SelectedTarget),
                    null,
                    Enumerable.Repeat(label, App.AppContext.SelectedTarget.Objects.Count),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        swap = ObjectUtility.Clone(target.Label);
                        target.Label = (string)data;
                    }
                ));
            }
            else
            {
                if (string.IsNullOrEmpty(node.Content.Label))
                {
                    node.ShowingSecondaryLabel = false;
                    foreach (var child in node.Children)
                    {
                        child.MovePos(0, -16, true);
                    }
                }
                Invalidate();
            }
        }
        #endregion

        #region カラーピッカー
        /// <summary>
        /// カラーピッカートリガー
        /// </summary>
        class GuiObjectColorPickerTrigger : IColorPickerTrigger
        {
            public Node obj;
            public RgbaColor Color
            {
                get
                {
                    return obj.Content.EditColor.HasValue ? obj.Content.EditColor.Value : RgbaColor.FromColor(System.Drawing.Color.White);
                }
            }

            /// <summary>
            /// 編集用カラーかどうか。
            /// </summary>
            public bool IsMarkColor
            {
                get { return true; }
            }

            public bool EnableAlpha
            {
                get
                {
                    return false;
                }
            }

            public bool IsDefaultLinear
            {
                get;
                set;
            }

            public bool ReadOnly
            {
                get
                {
                    return false;
                }
            }

            /// <summary>
            /// HDR の上限。
            /// </summary>
            public float HDRUpperBound
            {
                get
                {
                    return 1.0f;
                }
            }
        }

        /// <summary>
        /// カラー編集コマンドハンドラー
        /// </summary>
        private void ColorEdit(object sender, ColorEditEventArgs e)
        {
            if (e.EditFixed)
            {
                TheApp.CommandManager.Execute(CreateEditColorEditCommand(RgbaColor.Round(e.Color)));
                //TheApp.CommandManager.Execute(new SymbolColorEditCommand(GuiObjectManager.Selected, e.Color.ToColor()));
            }
        }

        private void NullifySymbolColor()
        {
            foreach (GuiObject item in App.AppContext.SelectedTarget.Objects)
            {
                if (item.EditColor.HasValue)
                {
                    TheApp.CommandManager.Execute(CreateEditColorEditCommand(null));
                    break;
                }
            }
        }
        private GroupEditCommand CreateEditColorEditCommand(RgbaColor? color)
        {
            return
                new GeneralGroupReferenceEditCommand</*Color?*/object>(
                    new GuiObjectGroup(App.AppContext.SelectedTarget),
                    null,
                    ObjectUtility.MultipleClone((object)color, App.AppContext.SelectedTarget.Objects.Count),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        swap = ObjectUtility.Clone(target.EditColor);
                        target.EditColor = (RgbaColor?)data;
                    }
                );
        }
        /// <summary>
        /// カラーピッカ接続
        /// </summary>
        private void StartColorPicker(Node obj)
        {
            if (!obj.Content.Editable)
            {
                return;
            }

            colorPickerTrigger.obj = obj;
            colorPickerAdapter.ColorPickerText = obj.Content.Name;
            ColorPickerDialog.StartConnection(colorPickerAdapter);
        }
        #endregion
    }
}
