﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.ComponentModel;
using System.Linq;
using App.Controls;
using App.Data;
using App.Utility;

namespace App.FileView
{
    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class FileData
    {
        public FileData()
        {
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class ProjectFileData : FileData
    {
        private readonly ProjectDocument target_ = null;

        public ProjectFileData(ProjectDocument target)
        {
            target_ = target;
        }

        [PropertyOrder(0)]
        [LocalizedCategory("Fileview_Information_Category_General")]
        [LocalizedPropertyName("Fileview_Information_Property_General_FullPath")]
        public string FullPath
        {
            get
            {
                return
                    (target_						!= null) ?
                        target_.FilePath :
                        string.Empty;
            }
        }

        [PropertyOrder(10)]
        [LocalizedCategory("Fileview_Information_Category_General")]
        [LocalizedPropertyName("Fileview_Information_Property_General_Creator")]
        public string Creator
        {
            get
            {
                return
                    (target_						!= null) &&
                    (target_.FileInfo				!= null) &&
                    (target_.FileInfo.create		!= null) &&
                    (target_.FileInfo.create.user	!= null) ?
                        target_.FileInfo.create.user :
                        string.Empty;
            }
        }

        [PropertyOrder(11)]
        [LocalizedCategory("Fileview_Information_Category_General")]
        [LocalizedPropertyName("Fileview_Information_Property_General_Modifier")]
        public string Modifier
        {
            get
            {
                return
                    (target_						!= null) &&
                    (target_.FileInfo				!= null) &&
                    (target_.FileInfo.modify		!= null) &&
                    (target_.FileInfo.modify.user	!= null) ?
                        target_.FileInfo.modify.user :
                        string.Empty;
            }
        }

        [PropertyOrder(12)]
        [LocalizedCategory("Fileview_Information_Category_General")]
        [LocalizedPropertyName("Fileview_Information_Property_General_CreateHost")]
        public string CreateHost
        {
            get
            {
                return
                    (target_						!= null) &&
                    (target_.FileInfo				!= null) &&
                    (target_.FileInfo.create		!= null) &&
                    (target_.FileInfo.create.host	!= null) ?
                        target_.FileInfo.create.host :
                        string.Empty;
            }
        }

        [PropertyOrder(13)]
        [LocalizedCategory("Fileview_Information_Category_General")]
        [LocalizedPropertyName("Fileview_Information_Property_General_ModifyHost")]
        public string ModifyHost
        {
            get
            {
                return
                    (target_						!= null) &&
                    (target_.FileInfo				!= null) &&
                    (target_.FileInfo.modify		!= null) &&
                    (target_.FileInfo.modify.host	!= null) ?
                        target_.FileInfo.modify.host :
                        string.Empty;
            }
        }

        [PropertyOrder(14)]
        [LocalizedCategory("Fileview_Information_Category_General")]
        [LocalizedPropertyName("Fileview_Information_Property_General_CreateDate")]
        public string CreateDate
        {
            get
            {
                return
                    (target_							!= null) &&
                    (target_.FileInfo					!= null) &&
                    (target_.FileInfo.create			!= null) &&
                    (target_.FileInfo.create.date_time	!= null) ?
                        target_.FileInfo.create.date_time.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(15)]
        [LocalizedCategory("Fileview_Information_Category_General")]
        [LocalizedPropertyName("Fileview_Information_Property_General_ModifyDate")]
        public string ModifyDate
        {
            get
            {
                return
                    (target_							!= null) &&
                    (target_.FileInfo					!= null) &&
                    (target_.FileInfo.modify			!= null) &&
                    (target_.FileInfo.modify.date_time	!= null) ?
                        target_.FileInfo.modify.date_time.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(16)]
        [LocalizedCategory("Fileview_Information_Category_General")]
        [LocalizedPropertyName("Fileview_Information_Property_General_CreateTool")]
        public string CreateTool
        {
            get
            {
                if( (target_							!= null) &&
                    (target_.FileInfo					!= null) &&
                    (target_.FileInfo.create			!= null) &&
                    (target_.FileInfo.create.tool_name	!= null))
                {
                    return
                        (target_.FileInfo.create.tool_version != null) ?
                            target_.FileInfo.create.tool_name + "    " + target_.FileInfo.create.tool_version :
                            target_.FileInfo.create.tool_name;
                }
                else
                {
                    return string.Empty;
                }
            }
        }

        [PropertyOrder(17)]
        [LocalizedCategory("Fileview_Information_Category_General")]
        [LocalizedPropertyName("Fileview_Information_Property_General_ModifyTool")]
        public string ModifyTool
        {
            get
            {
                if( (target_							!= null) &&
                    (target_.FileInfo					!= null) &&
                    (target_.FileInfo.modify			!= null) &&
                    (target_.FileInfo.modify.tool_name	!= null))
                {
                    return
                        (target_.FileInfo.modify.tool_version != null) ?
                            target_.FileInfo.modify.tool_name + "    " + target_.FileInfo.modify.tool_version :
                            target_.FileInfo.modify.tool_name;
                }
                else
                {
                    return string.Empty;
                }
            }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class IntermediateFileData : FileData
    {
        private readonly IntermediateFileDocument target_ = null;

        public IntermediateFileData(IntermediateFileDocument target)
        {
            target_ = target;
        }

        [PropertyOrder(0)]
        [LocalizedCategory("Fileview_Information_Category_General")]
        [LocalizedPropertyName("Fileview_Information_Property_General_FullPath")]
        public string FullPath
        {
            get
            {
                return
                    (target_						!= null) ?
                        target_.FilePath :
                        string.Empty;
            }
        }

        [PropertyOrder(1)]
        [LocalizedCategory("Fileview_Information_Category_General")]
        [LocalizedPropertyName("Fileview_Information_Property_General_ProjectPath")]
        public string ProjectPath
        {
            get
            {
                return
                    (target_						!= null) ?
                        target_.ProjectPath :
                        string.Empty;
            }
        }

        [PropertyOrder(2)]
        [LocalizedCategory("Fileview_Information_Category_General")]
        [LocalizedPropertyName("Fileview_Information_Property_General_SearchPath")]
        public string SearchPath
        {
            get
            {
                return
                    (target_						!= null) &&
                    (target_.BaseName				!= null) ?
                        target_.BaseName :
                        string.Empty;
            }
        }

        [PropertyOrder(16)]
        [LocalizedCategory("Fileview_Information_Category_General")]
        [LocalizedPropertyName("Fileview_Information_Property_General_CreateTool")]
        public string CreateTool
        {
            get
            {
                if( (target_							!= null) &&
                    (target_.file_info					!= null) &&
                    (target_.file_info.create			!= null) &&
                    (target_.file_info.create.tool_name	!= null))
                {
                    return
                        (target_.file_info.create.tool_version != null) ?
                            target_.file_info.create.tool_name + "    " + target_.file_info.create.tool_version :
                            target_.file_info.create.tool_name;
                }
                else
                {
                    return string.Empty;
                }
            }
        }

        [PropertyOrder(17)]
        [LocalizedCategory("Fileview_Information_Category_General")]
        [LocalizedPropertyName("Fileview_Information_Property_General_ModifyTool")]
        public string ModifyTool
        {
            get
            {
                if( (target_							!= null) &&
                    (target_.file_info					!= null) &&
                    (target_.file_info.modify			!= null) &&
                    (target_.file_info.modify.tool_name	!= null))
                {
                    return
                        (target_.file_info.modify.tool_version != null) ?
                            target_.file_info.modify.tool_name + "    " + target_.file_info.modify.tool_version :
                            target_.file_info.modify.tool_name;
                }
                else
                {
                    return string.Empty;
                }
            }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class ModelFileData : IntermediateFileData
    {
        private readonly Model target_ = null;

        public ModelFileData(Model target) :
            base(target)
        {
            target_ = target;
        }

        [PropertyOrder(100)]
        [LocalizedCategory("Fileview_Information_Category_Model")]
        [LocalizedPropertyName("Fileview_Information_Category_Model_MaterialCount")]
        public string MaterialCount
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.material_array			!= null) &&
                    (target_.Data.material_array.material	!= null) ?
                        target_.Data.material_array.material.Length.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(101)]
        [LocalizedCategory("Fileview_Information_Category_Model")]
        [LocalizedPropertyName("Fileview_Information_Category_Model_BoneCount")]
        public string BoneCount
        {
            get
            {
                if (target_ != null)
                {
                    bool hasSkelton = target_.Data.skeleton != null;

                    return hasSkelton ? target_.Data.skeleton.bone_array.bone.Length.ToString() : "0";
                }
                else
                {
                    return string.Empty;
                }
            }
        }

        [PropertyOrder(102)]
        [LocalizedCategory("Fileview_Information_Category_Model")]
        [LocalizedPropertyName("Fileview_Information_Category_Model_ShapeCount")]
        public string ShapeCount
        {
            get
            {
                return
                    (target_						!= null) &&
                    (target_.Data					!= null) &&
                    (target_.Data.shape_array		!= null) &&
                    (target_.Data.shape_array.shape	!= null) ?
                        target_.Data.shape_array.shape.Length.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(103)]
        [LocalizedCategory("Fileview_Information_Category_Model")]
        [LocalizedPropertyName("Fileview_Information_Category_Model_ProcessVertexCount")]
        public string ProcessVertexCount
        {
            get
            {
                return
                    (target_ != null) ?
                    (target_.HasProcessVertexCount ? target_.ProcessVertexCount.ToString() : res.Strings.ProcessVertexCount_UnsupportedPlatform) :
                    string.Empty;
            }
        }

        [PropertyOrder(104)]
        [LocalizedCategory("Fileview_Information_Category_Model")]
        [LocalizedPropertyName("Fileview_Information_Category_Model_ProcessVertexPerTriangleCount")]
        public string ProcessVertexPerTriangleCount
        {
            get
            {
                return
                    (target_ != null) ?
                    (target_.HasProcessVertexCount ? target_.ProcessVertexPerTriangleCount.ToString() : res.Strings.ProcessVertexCount_UnsupportedPlatform) :
                    string.Empty;
            }
        }

        [PropertyOrder(105)]
        [LocalizedCategory("Fileview_Information_Category_Model")]
        [LocalizedPropertyName("Fileview_Information_Category_Model_VertexCount")]
        public string VertexCount
        {
            get
            {
                return
                    (target_						!= null) ?
                        target_.VertexCount.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(106)]
        [LocalizedCategory("Fileview_Information_Category_Model")]
        [LocalizedPropertyName("Fileview_Information_Category_Model_PolygonCount")]
        public string PolygonCount
        {
            get
            {
                return
                    (target_						!= null) ?
                        target_.PolygonCount.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(107)]
        [LocalizedCategory("Fileview_Information_Category_Model")]
        [LocalizedPropertyName("Fileview_Information_Category_Model_UnitePosQuantize")]
        public string UnitePosQuantize
        {
            get
            {
                return
                    (target_						!= null) &&
                    (target_.Data					!= null) &&
                    (target_.Data.model_info		!= null) ?
                        UIText.FlagYesNo(target_.Data.model_info.unite_pos_quantize) :
                        string.Empty;
            }
        }

        [PropertyOrder(110)]
        [LocalizedCategory("Fileview_Information_Category_Model")]
        [LocalizedPropertyName("Fileview_Information_Category_Model_SmoothSkinning")]
        public string SmoothSkinning
        {
            get
            {
                return
                    (target_						!= null) ?
                        target_.SmoothSkinningCount.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(111)]
        [LocalizedCategory("Fileview_Information_Category_Model")]
        [LocalizedPropertyName("Fileview_Information_Category_Model_RigidSkinningCount")]
        public string RigidSkinningCount
        {
            get
            {
                return
                    (target_						!= null) ?
                        target_.RigidSkinningCount.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(112)]
        [LocalizedCategory("Fileview_Information_Category_Model")]
        [LocalizedPropertyName("Fileview_Information_Category_Model_RigidBodyCount")]
        public string RigidBodyCount
        {
            get
            {
                return
                    (target_						!= null) ?
                        target_.RigidBodyCount.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(113)]
        [LocalizedCategory("Fileview_Information_Category_Model")]
        [LocalizedPropertyName("Fileview_Information_Category_Model_UseScale")]
        public string UseScale
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.skeleton					!= null) &&
                    (target_.Data.skeleton.skeleton_info	!= null) ?
                        (target_.HasSkelton ? UIText.FlagYesNo(target_.Data.skeleton.skeleton_info.scale_enable) : string.Empty) :
                        string.Empty;
            }
        }

        [PropertyOrder(114)]
        [LocalizedCategory("Fileview_Information_Category_Model")]
        [LocalizedPropertyName("Fileview_Information_Category_Model_ScaleMode")]
        public string ScaleMode
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.skeleton					!= null) &&
                    (target_.Data.skeleton.skeleton_info	!= null) ?
                        (target_.HasSkelton ? UIText.EnumValue(target_.Data.skeleton.skeleton_info.scale_mode) : string.Empty) :
                        string.Empty;
            }
        }

        [PropertyOrder(115)]
        [LocalizedCategory("Fileview_Information_Category_Model")]
        [LocalizedPropertyName("Fileview_Information_Category_Model_RotateMode")]
        public string RotateMode
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.skeleton					!= null) &&
                    (target_.Data.skeleton.skeleton_info	!= null) ?
                        (target_.HasSkelton ? UIText.EnumValue(target_.Data.skeleton.skeleton_info.rotate_mode) : string.Empty) :
                        string.Empty;
            }
        }

        [PropertyOrder(116)]
        [LocalizedCategory("Fileview_Information_Category_Model")]
        [LocalizedPropertyName("Fileview_Information_Category_Model_UseMotionMirroring")]
        public string UseMotionMirroring
        {
            get
            {
                return
                    (target_ != null) &&
                    (target_.Data != null) &&
                    (target_.Data.skeleton != null) &&
                    (target_.Data.skeleton.skeleton_info != null) ?
                        (target_.HasSkelton ? UIText.FlagYesNo(target_.Data.skeleton.skeleton_info.motion_mirroring_enable) : string.Empty) :
                        string.Empty;
            }
        }

        [PropertyOrder(120)]
        [LocalizedCategory("Fileview_Information_Category_Model")]
        [LocalizedPropertyName("Fileview_Information_Category_Model_DccPreset")]
        public string DccPreset
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.model_info				!= null) ?
                        target_.Data.model_info.dcc_preset :
                        string.Empty;
            }
        }

        [PropertyOrder(121)]
        [LocalizedCategory("Fileview_Information_Category_Model")]
        [LocalizedPropertyName("Fileview_Information_Category_Model_DccMagnify")]
        public string DccMagnify
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.model_info				!= null) ?
                        target_.Data.model_info.dcc_magnify.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(122)]
        [LocalizedCategory("Fileview_Information_Category_Model")]
        [LocalizedPropertyName("Fileview_Information_Category_Model_DccStartFrame")]
        public string DccStartFrame
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.model_info				!= null) ?
                        target_.Data.model_info.dcc_start_frame.ToString() :
                        string.Empty;
            }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class TextureFileData : IntermediateFileData
    {
        private readonly Texture target_ = null;

        public TextureFileData(Texture target) :
            base(target)
        {
            target_ = target;
        }

        [PropertyOrder(100)]
        [LocalizedCategory("Fileview_Information_Category_Texture")]
        [LocalizedPropertyName("Fileview_Information_Category_Texture_Dimension")]
        public string Dimension
        {
            get
            {
                return
                    (target_					!= null) &&
                    (target_.Data				!= null) &&
                    (target_.Data.texture_info	!= null) ?
                        UIText.EnumValue(target_.Data.texture_info.dimension) :
                        string.Empty;
            }
        }

        [PropertyOrder(101)]
        [LocalizedCategory("Fileview_Information_Category_Texture")]
        [LocalizedPropertyName("Fileview_Information_Category_Texture_Format")]
        public string Format
        {
            get
            {
                return
                    (target_					!= null) &&
                    (target_.Data				!= null) &&
                    (target_.Data.texture_info	!= null) ?
                        UIText.EnumValue(target_.Data.texture_info.quantize_type) :
                        string.Empty;
            }
        }

        [PropertyOrder(102)]
        [LocalizedCategory("Fileview_Information_Category_Texture")]
        [LocalizedPropertyName("Fileview_Information_Category_Texture_ComponentSelector")]
        public string ComponentSelector
        {
            get
            {
                return
                    (target_					!= null) &&
                    (target_.Data				!= null) &&
                    (target_.Data.texture_info	!= null) ?
                        UIText.ComponentSelector(target_.Data.texture_info.comp_sel) :
                        string.Empty;
            }
        }

        [PropertyOrder(103)]
        [LocalizedCategory("Fileview_Information_Category_Texture")]
        [LocalizedPropertyName("Fileview_Information_Category_Texture_WeightedCompress")]
        public string WeightedCompress
        {
            get
            {
                return
                    (target_					!= null) &&
                    (target_.Data				!= null) &&
                    (target_.Data.texture_info	!= null) ?
                        UIText.FlagYesNo(target_.Data.texture_info.weighted_compress) :
                        string.Empty;
            }
        }

        [PropertyOrder(104)]
        [LocalizedCategory("Fileview_Information_Category_Texture")]
        [LocalizedPropertyName("Fileview_Information_Category_Texture_Size")]
        public string Size
        {
            get
            {
                if ((target_ != null) &&
                    (target_.Data				!= null) &&
                    (target_.Data.texture_info	!= null))
                {
                    return
                        string.Format(
                            "{0} x {1}",
                            target_.Data.texture_info.width,
                            target_.Data.texture_info.height
                        );
                }
                else
                {
                    return string.Empty;
                }
            }
        }

        [PropertyOrder(105)]
        [LocalizedCategory("Fileview_Information_Category_Texture")]
        [LocalizedPropertyName("Fileview_Information_Category_Texture_Depth")]
        public string Depth
        {
            get
            {
                return
                    (target_					!= null) &&
                    (target_.Data				!= null) &&
                    (target_.Data.texture_info	!= null) ?
                        target_.Data.texture_info.depth.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(106)]
        [LocalizedCategory("Fileview_Information_Category_Texture")]
        [LocalizedPropertyName("Fileview_Information_Category_Texture_DataSize")]
        public string DataSize
        {
            get
            {
                return
                    (target_					!= null) &&
                    (target_.Data				!= null) &&
                    (target_.Data.texture_info	!= null) ?
                        Utility.DataSize.XBytesText(target_.Data.texture_info.size) :
                        string.Empty;
            }
        }

        [PropertyOrder(107)]
        [LocalizedCategory("Fileview_Information_Category_Texture")]
        [LocalizedPropertyName("Fileview_Information_Category_Texture_Mipmap")]
        public string Mipmap
        {
            get
            {
                return
                    (target_					!= null) &&
                    (target_.Data				!= null) &&
                    (target_.Data.texture_info	!= null) ?
                        UIText.MipLevel(target_.Data.texture_info.mip_level) :
                        string.Empty;
            }
        }

        [PropertyOrder(108)]
        [LocalizedCategory("Fileview_Information_Category_Texture")]
        [LocalizedPropertyName("Fileview_Information_Category_Texture_MipmapMinSize")]
        public string MipmapMinSize
        {
            get
            {
                return
                    (target_					!= null) &&
                    (target_.Data				!= null) &&
                    (target_.Data.texture_info	!= null) ?
                        UIText.MipmapMinSize(target_.Data.texture_info) :
                        string.Empty;
            }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class SkeletalAnimationFileData : IntermediateFileData
    {
        private readonly SkeletalAnimation target_ = null;

        public SkeletalAnimationFileData(SkeletalAnimation target) :
            base(target)
        {
            target_ = target;
        }

        [PropertyOrder(100)]
        [LocalizedCategory("Fileview_Information_Category_SkeletalAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_SkeletalAnimation_FrameCount")]
        public string FrameCount
        {
            get
            {
                return
                    (target_							!= null) &&
                    (target_.Data						!= null) &&
                    (target_.Data.skeletal_anim_info	!= null) ?
                        target_.Data.skeletal_anim_info.frame_count.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(101)]
        [LocalizedCategory("Fileview_Information_Category_SkeletalAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_SkeletalAnimation_FrameResolution")]
        public string FrameResolution
        {
            get
            {
                return
                    (target_							!= null) &&
                    (target_.Data						!= null) &&
                    (target_.Data.skeletal_anim_info	!= null) ?
                        target_.Data.skeletal_anim_info.frame_resolution.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(102)]
        [LocalizedCategory("Fileview_Information_Category_SkeletalAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_SkeletalAnimation_BakeAll")]
        public string BakeAll
        {
            get
            {
                return
                    (target_							!= null) &&
                    (target_.Data						!= null) &&
                    (target_.Data.skeletal_anim_info	!= null) ?
                        UIText.FlagYesNo(target_.Data.skeletal_anim_info.bake_all) :
                        string.Empty;
            }
        }

        [PropertyOrder(103)]
        [LocalizedCategory("Fileview_Information_Category_SkeletalAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_SkeletalAnimation_Loop")]
        public string Loop
        {
            get
            {
                return
                    (target_							!= null) &&
                    (target_.Data						!= null) &&
                    (target_.Data.skeletal_anim_info	!= null) ?
                        UIText.FlagYesNo(target_.Data.skeletal_anim_info.loop) :
                        string.Empty;
            }
        }

        [PropertyOrder(104)]
        [LocalizedCategory("Fileview_Information_Category_SkeletalAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_SkeletalAnimation_ScaleMode")]
        public string ScaleMode
        {
            get
            {
                return
                    (target_							!= null) &&
                    (target_.Data						!= null) &&
                    (target_.Data.skeletal_anim_info	!= null) ?
                        UIText.EnumValueSimple(target_.Data.skeletal_anim_info.scale_mode) :
                        string.Empty;
            }
        }

        [PropertyOrder(105)]
        [LocalizedCategory("Fileview_Information_Category_SkeletalAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_SkeletalAnimation_RotateMode")]
        public string RotateMode
        {
            get
            {
                return
                    (target_							!= null) &&
                    (target_.Data						!= null) &&
                    (target_.Data.skeletal_anim_info	!= null) ?
                        UIText.EnumValueSimple(target_.Data.skeletal_anim_info.rotate_mode) :
                        string.Empty;
            }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class ShapeAnimationFileData : IntermediateFileData
    {
        public ShapeAnimationFileData(ShapeAnimation target) :
            base(target)
        {
        }
    }

    // TODO: 要確認
    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MaterialAnimationFileData : IntermediateFileData
    {
        private readonly MaterialAnimation target_ = null;

        public MaterialAnimationFileData(MaterialAnimation target) :
            base(target)
        {
            target_ = target;
        }

        [PropertyOrder(100)]
        [LocalizedCategory("Fileview_Information_Category_MaterialAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_MaterialAnimation_FrameCount")]
        public string FrameCount
        {
            get
            {
                return
                    (target_ != null) &&
                    (target_.Data != null) &&
                    (target_.Data.material_anim_info != null) ?
                        target_.Data.material_anim_info.frame_count.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(101)]
        [LocalizedCategory("Fileview_Information_Category_MaterialAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_MaterialAnimation_FrameResolution")]
        public string FrameResolution
        {
            get
            {
                return
                    (target_ != null) &&
                    (target_.Data != null) &&
                    (target_.Data.material_anim_info != null) ?
                        target_.Data.material_anim_info.frame_resolution.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(102)]
        [LocalizedCategory("Fileview_Information_Category_MaterialAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_MaterialAnimation_BakeAll")]
        public string BakeAll
        {
            get
            {
                return
                    (target_ != null) &&
                    (target_.Data != null) &&
                    (target_.Data.material_anim_info != null) ?
                        UIText.FlagYesNo(target_.Data.material_anim_info.bake_all) :
                        string.Empty;
            }
        }

        [PropertyOrder(104)]
        [LocalizedCategory("Fileview_Information_Category_MaterialAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_MaterialAnimation_Loop")]
        public string Loop
        {
            get
            {
                return
                    (target_ != null) &&
                    (target_.Data != null) &&
                    (target_.Data.material_anim_info != null) ?
                        UIText.FlagYesNo(target_.Data.material_anim_info.loop) :
                        string.Empty;
            }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class ShaderParameterAnimationFileData : IntermediateFileData
    {
        private readonly ShaderParameterAnimation target_ = null;

        public ShaderParameterAnimationFileData(ShaderParameterAnimation target) :
            base(target)
        {
            target_ = target;
        }

        [PropertyOrder(100)]
        [LocalizedCategory("Fileview_Information_Category_ShaderParameterAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_ShaderParameterAnimation_FrameCount")]
        public string FrameCount
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.shader_param_anim_info	!= null) ?
                        target_.Data.shader_param_anim_info.frame_count.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(101)]
        [LocalizedCategory("Fileview_Information_Category_ShaderParameterAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_ShaderParameterAnimation_FrameResolution")]
        public string FrameResolution
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.shader_param_anim_info	!= null) ?
                        target_.Data.shader_param_anim_info.frame_resolution.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(102)]
        [LocalizedCategory("Fileview_Information_Category_ShaderParameterAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_ShaderParameterAnimation_BakeAll")]
        public string BakeAll
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.shader_param_anim_info	!= null) ?
                        UIText.FlagYesNo(target_.Data.shader_param_anim_info.bake_all) :
                        string.Empty;
            }
        }

        [PropertyOrder(104)]
        [LocalizedCategory("Fileview_Information_Category_ShaderParameterAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_ShaderParameterAnimation_Loop")]
        public string Loop
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.shader_param_anim_info	!= null) ?
                        UIText.FlagYesNo(target_.Data.shader_param_anim_info.loop) :
                        string.Empty;
            }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class ColorAnimationFileData : IntermediateFileData
    {
        private readonly ColorAnimation target_ = null;

        public ColorAnimationFileData(ColorAnimation target) :
            base(target)
        {
            target_ = target;
        }

        [PropertyOrder(100)]
        [LocalizedCategory("Fileview_Information_Category_ColorAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_ShaderParameterAnimation_FrameCount")]
        public string FrameCount
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.shader_param_anim_info	!= null) ?
                        target_.Data.shader_param_anim_info.frame_count.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(101)]
        [LocalizedCategory("Fileview_Information_Category_ColorAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_ShaderParameterAnimation_FrameResolution")]
        public string FrameResolution
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.shader_param_anim_info	!= null) ?
                        target_.Data.shader_param_anim_info.frame_resolution.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(102)]
        [LocalizedCategory("Fileview_Information_Category_ColorAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_ShaderParameterAnimation_BakeAll")]
        public string BakeAll
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.shader_param_anim_info	!= null) ?
                        UIText.FlagYesNo(target_.Data.shader_param_anim_info.bake_all) :
                        string.Empty;
            }
        }

        [PropertyOrder(104)]
        [LocalizedCategory("Fileview_Information_Category_ColorAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_ShaderParameterAnimation_Loop")]
        public string Loop
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.shader_param_anim_info	!= null) ?
                        UIText.FlagYesNo(target_.Data.shader_param_anim_info.loop) :
                        string.Empty;
            }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class TextureSrtAnimationFileData : IntermediateFileData
    {
        private readonly TextureSrtAnimation target_ = null;

        public TextureSrtAnimationFileData(TextureSrtAnimation target) :
            base(target)
        {
            target_ = target;
        }

        [PropertyOrder(100)]
        [LocalizedCategory("Fileview_Information_Category_TextureSrtAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_ShaderParameterAnimation_FrameCount")]
        public string FrameCount
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.shader_param_anim_info	!= null) ?
                        target_.Data.shader_param_anim_info.frame_count.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(101)]
        [LocalizedCategory("Fileview_Information_Category_TextureSrtAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_ShaderParameterAnimation_FrameResolution")]
        public string FrameResolution
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.shader_param_anim_info	!= null) ?
                        target_.Data.shader_param_anim_info.frame_resolution.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(102)]
        [LocalizedCategory("Fileview_Information_Category_TextureSrtAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_ShaderParameterAnimation_BakeAll")]
        public string BakeAll
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.shader_param_anim_info	!= null) ?
                        UIText.FlagYesNo(target_.Data.shader_param_anim_info.bake_all) :
                        string.Empty;
            }
        }

        [PropertyOrder(104)]
        [LocalizedCategory("Fileview_Information_Category_TextureSrtAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_ShaderParameterAnimation_Loop")]
        public string Loop
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.shader_param_anim_info	!= null) ?
                        UIText.FlagYesNo(target_.Data.shader_param_anim_info.loop) :
                        string.Empty;
            }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class MaterialVisibilityAnimationFileData : IntermediateFileData
    {
        private readonly MaterialVisibilityAnimation target_ = null;

        public MaterialVisibilityAnimationFileData(MaterialVisibilityAnimation target) :
            base(target)
        {
            target_ = target;
        }

        [PropertyOrder(100)]
        [LocalizedCategory("Fileview_Information_Category_MaterialVisibilityAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_MaterialVisibilityAnimation_FrameCount")]
        public string FrameCount
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.mat_visibility_anim_info	!= null) ?
                        target_.Data.mat_visibility_anim_info.frame_count.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(101)]
        [LocalizedCategory("Fileview_Information_Category_MaterialVisibilityAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_MaterialVisibilityAnimation_FrameResolution")]
        public string FrameResolution
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.mat_visibility_anim_info	!= null) ?
                        target_.Data.mat_visibility_anim_info.frame_resolution.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(102)]
        [LocalizedCategory("Fileview_Information_Category_MaterialVisibilityAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_MaterialVisibilityAnimation_BakeAll")]
        public string BakeAll
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.mat_visibility_anim_info	!= null) ?
                        UIText.FlagYesNo(target_.Data.mat_visibility_anim_info.bake_all) :
                        string.Empty;
            }
        }

        [PropertyOrder(103)]
        [LocalizedCategory("Fileview_Information_Category_MaterialVisibilityAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_MaterialVisibilityAnimation_Loop")]
        public string Loop
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.mat_visibility_anim_info	!= null) ?
                        UIText.FlagYesNo(target_.Data.mat_visibility_anim_info.loop) :
                        string.Empty;
            }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class BoneVisibilityAnimationFileData : IntermediateFileData
    {
        private readonly BoneVisibilityAnimation target_ = null;

        public BoneVisibilityAnimationFileData(BoneVisibilityAnimation target) :
            base(target)
        {
            target_ = target;
        }

        [PropertyOrder(100)]
        [LocalizedCategory("Fileview_Information_Category_BoneVisibilityAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_BoneVisibilityAnimation_FrameCount")]
        public string FrameCount
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.bone_visibility_anim_info	!= null) ?
                        target_.Data.bone_visibility_anim_info.frame_count.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(101)]
        [LocalizedCategory("Fileview_Information_Category_BoneVisibilityAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_BoneVisibilityAnimation_FrameResolution")]
        public string FrameResolution
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.bone_visibility_anim_info	!= null) ?
                        target_.Data.bone_visibility_anim_info.frame_resolution.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(102)]
        [LocalizedCategory("Fileview_Information_Category_BoneVisibilityAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_BoneVisibilityAnimation_BakeAll")]
        public string BakeAll
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.bone_visibility_anim_info	!= null) ?
                        UIText.FlagYesNo(target_.Data.bone_visibility_anim_info.bake_all) :
                        string.Empty;
            }
        }

        [PropertyOrder(103)]
        [LocalizedCategory("Fileview_Information_Category_BoneVisibilityAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_BoneVisibilityAnimation_Loop")]
        public string Loop
        {
            get
            {
                return
                    (target_								!= null) &&
                    (target_.Data							!= null) &&
                    (target_.Data.bone_visibility_anim_info	!= null) ?
                        UIText.FlagYesNo(target_.Data.bone_visibility_anim_info.loop) :
                        string.Empty;
            }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class TexturePatternAnimationFileData : IntermediateFileData
    {
        private readonly TexturePatternAnimation target_ = null;

        public TexturePatternAnimationFileData(TexturePatternAnimation target) :
            base(target)
        {
            target_ = target;
        }

        [PropertyOrder(100)]
        [LocalizedCategory("Fileview_Information_Category_TexturePatternAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_TexturePatternAnimation_FrameCount")]
        public string FrameCount
        {
            get
            {
                return
                    (target_							!= null) &&
                    (target_.Data						!= null) &&
                    (target_.Data.tex_pattern_anim_info	!= null) ?
                        target_.Data.tex_pattern_anim_info.frame_count.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(101)]
        [LocalizedCategory("Fileview_Information_Category_TexturePatternAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_TexturePatternAnimation_FrameResolution")]
        public string FrameResolution
        {
            get
            {
                return
                    (target_							!= null) &&
                    (target_.Data						!= null) &&
                    (target_.Data.tex_pattern_anim_info	!= null) ?
                        target_.Data.tex_pattern_anim_info.frame_resolution.ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(102)]
        [LocalizedCategory("Fileview_Information_Category_TexturePatternAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_TexturePatternAnimation_BakeAll")]
        public string BakeAll
        {
            get
            {
                return
                    (target_							!= null) &&
                    (target_.Data						!= null) &&
                    (target_.Data.tex_pattern_anim_info	!= null) ?
                        UIText.FlagYesNo(target_.Data.tex_pattern_anim_info.bake_all) :
                        string.Empty;
            }
        }

        [PropertyOrder(103)]
        [LocalizedCategory("Fileview_Information_Category_TexturePatternAnimation")]
        [LocalizedPropertyName("Fileview_Information_Category_TexturePatternAnimation_Loop")]
        public string Loop
        {
            get
            {
                return
                    (target_							!= null) &&
                    (target_.Data						!= null) &&
                    (target_.Data.tex_pattern_anim_info	!= null) ?
                        UIText.FlagYesNo(target_.Data.tex_pattern_anim_info.loop) :
                        string.Empty;
            }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class ShaderDefinitionFileData : IntermediateFileData
    {
        private readonly ShaderDefinition target_ = null;

        public ShaderDefinitionFileData(ShaderDefinition target) :
            base(target)
        {
            target_ = target;
        }

        [PropertyOrder(100)]
        [LocalizedCategory("Fileview_Information_Category_ShaderDefinition")]
        [LocalizedPropertyName("Fileview_Information_Category_ShaderDefinition_DefinitionCount")]
        public string DefinitionCount
        {
            get
            {
                return
                    (target_						!= null) &&
                    (target_.Definitions			!= null) ?
                        target_.Definitions.Count.ToString() :
                        string.Empty;
            }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class SeparateMaterialFileData : IntermediateFileData
    {
        private readonly SeparateMaterial target_ = null;

        public SeparateMaterialFileData(SeparateMaterial target) :
            base(target)
        {
            target_ = target;
        }

        [PropertyOrder(100)]
        [LocalizedCategory("Fileview_Information_Category_SeparateMaterial")]
        [LocalizedPropertyName("Fileview_Information_Category_SeparateMaterial_MaterialName")]
        public string MaterialName
        {
            get
            {
                return
                    (target_ != null) &&
                    (target_.Materials.FirstOrDefault() != null) ?
                        target_.Materials.First().Name :
                        string.Empty;
            }
        }

        [PropertyOrder(101)]
        [LocalizedCategory("Fileview_Information_Category_SeparateMaterial")]
        [LocalizedPropertyName("Fileview_Information_Category_SeparateMaterial_ShaderDefinitionFileName")]
        public string ShaderDefinitionFileName
        {
            get
            {
                return
                    (target_ != null) &&
                    (target_.Materials.FirstOrDefault() != null) ?
                        target_.Materials.First().MaterialShaderAssign.ShaderDefinitionFileName :
                        string.Empty;
            }
        }

        [PropertyOrder(102)]
        [LocalizedCategory("Fileview_Information_Category_SeparateMaterial")]
        [LocalizedPropertyName("Fileview_Information_Category_SeparateMaterial_ShaderName")]
        public string ShaderName
        {
            get
            {
                return
                    (target_ != null) &&
                    (target_.Materials.FirstOrDefault() != null) ?
                        target_.Materials.First().MaterialShaderAssign.ShaderName :
                        string.Empty;
            }
        }
    }

    [TypeConverter(typeof(LocalizedPropertyGridConverter))]
    public class DefinitionFileData : FileData
    {
        private readonly Definition target_ = null;

        public DefinitionFileData(Definition target)
        {
            target_ = target;
        }

        [PropertyOrder(100)]
        [LocalizedCategory("Fileview_Information_Category_Definition")]
        [LocalizedPropertyName("Fileview_Information_Category_Definition_MacroCount")]
        public string MacroCount
        {
            get
            {
                return
                    (target_						!= null) &&
                    (target_.Data					!= null) &&
                    (target_.Data.macro_array		!= null) &&
                    (target_.Data.macro_array.macro	!= null) ?
                        target_.Data.macro_array.macro.Length.ToString() :
                        "0";
            }
        }

        [PropertyOrder(101)]
        [LocalizedCategory("Fileview_Information_Category_Definition")]
        [LocalizedPropertyName("Fileview_Information_Category_Definition_VertexShader")]
        public string VertexShader
        {
            get
            {
                return
                    (target_						!= null) ?
                        target_.VertexShaderPath :
                        string.Empty;
            }
        }

        [PropertyOrder(102)]
        [LocalizedCategory("Fileview_Information_Category_Definition")]
        [LocalizedPropertyName("Fileview_Information_Category_Definition_FragmentShader")]
        public string FragmentShader
        {
            get
            {
                return
                    (target_						!= null) ?
                        target_.FragmentShaderPath :
                        string.Empty;
            }
        }

        [PropertyOrder(103)]
        [LocalizedCategory("Fileview_Information_Category_Definition")]
        [LocalizedPropertyName("Fileview_Information_Category_Definition_GeometryShader")]
        public string GeometryShader
        {
            get
            {
                return
                    (target_						!= null) ?
                        target_.GeometryShaderPath :
                        string.Empty;
            }
        }

        [PropertyOrder(104)]
        [LocalizedCategory("Fileview_Information_Category_Definition")]
        [LocalizedPropertyName("Fileview_Information_Category_Definition_UniformCount")]
        public string UniformCount
        {
            get
            {
                return
                    (target_						!= null) &&
                    (target_.Data					!= null) ?
                        target_.Data.Uniforms().Count().ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(105)]
        [LocalizedCategory("Fileview_Information_Category_Definition")]
        [LocalizedPropertyName("Fileview_Information_Category_Definition_SamplerCount")]
        public string SamplerCount
        {
            get
            {
                return
                    (target_						!= null) &&
                    (target_.Data					!= null) ?
                        target_.Data.Samplers().Count().ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(106)]
        [LocalizedCategory("Fileview_Information_Category_Definition")]
        [LocalizedPropertyName("Fileview_Information_Category_Definition_AttribCount")]
        public string AttribCount
        {
            get
            {
                return
                    (target_						!= null) &&
                    (target_.Data					!= null) ?
                        target_.Data.Attributes().Count().ToString() :
                        string.Empty;
            }
        }

        [PropertyOrder(107)]
        [LocalizedCategory("Fileview_Information_Category_Definition")]
        [LocalizedPropertyName("Fileview_Information_Category_Definition_OptionCount")]
        public string OptionCount
        {
            get
            {
                return
                    (target_						!= null) &&
                    (target_.Data					!= null) ?
                        target_.Data.Options().Count().ToString() :
                        string.Empty;
            }
        }
    }
}
