﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading;
using App.Controls;
using App.Utility;

namespace App
{
    // シェーダーコンバーターマネージャー
    static public class ShdrcvtrManager
    {
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate int InitDelegate();

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool ShutdownDelegate();

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate void InitGLContextDelegate();

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate void ShutdownGLContextDelegate();

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool ClearDelegate();

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate UInt32 GetCvtrVersionDelegate();

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool SetOptionsDelegate(
            [MarshalAs(UnmanagedType.LPArray, ArraySubType = UnmanagedType.LPWStr)] string[] options
        );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool AddFileDelegate(
            IntPtr pData,
            int dataSize,
            [MarshalAs(UnmanagedType.LPArray, ArraySubType = UnmanagedType.LPWStr)] string[] paths,
            [MarshalAs(UnmanagedType.LPArray, ArraySubType = UnmanagedType.LPWStr)] string[] options
        );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate int CalcArchiveSizeDelegate(
        );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool ConvertDelegate(
            IntPtr ppData,
            int pDataSize
        );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool EndianSwapDelegate(
            IntPtr ppData,
            int pDataSize
        );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool MakeDelegate();

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate int CalcDefinitionSizeDelegate(
            [MarshalAs(UnmanagedType.LPWStr)] string fullpath
        );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool WriteDelegate(
            IntPtr ppData,
            int pDataSize
        );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate IntPtr OpenLogFileDelegate(
            [MarshalAs(UnmanagedType.LPArray, ArraySubType = UnmanagedType.LPWStr)] string[] outArgv,
            [MarshalAs(UnmanagedType.LPArray, ArraySubType = UnmanagedType.LPWStr)] string[] errArgv
        );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate void CloseLogFileDelegate(
        );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate void ForceNwArchiveDelegate(
        );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate void ForceNnArchiveDelegate(
        );

        public static InitDelegate					Init;
        public static ShutdownDelegate				Shutdown;
        public static InitGLContextDelegate			InitGLContext;
        public static ShutdownGLContextDelegate		ShutdownGLContext;
        public static ClearDelegate					Clear;
        public static GetCvtrVersionDelegate		GetCvtrVersion;
        public static SetOptionsDelegate			SetOptions;
        public static AddFileDelegate				AddFile;
        public static CalcArchiveSizeDelegate		CalcArchiveSize;
        public static ConvertDelegate				Convert;
        public static EndianSwapDelegate			EndianSwap;
        public static MakeDelegate					Make;
        public static CalcDefinitionSizeDelegate	CalcDefinitionSize;
        public static WriteDelegate					Write;
        public static OpenLogFileDelegate			OpenLogFile;
        public static CloseLogFileDelegate			CloseLogFile;
        public static ForceNwArchiveDelegate ForceNwArchive;
        public static ForceNnArchiveDelegate ForceNnArchive;

        public static object lockObject = new object();

        private static string DllFilepath
        {
            get
            {
                var path = string.Format(
                        @"{0}\..\G3dTool\{1}\NW4F_g3dshdrcvtr.dll",
                        Environment.GetEnvironmentVariable("NW4F_3DEDITOR_ROOT"),
                        Environment.Is64BitProcess ?
                            "win64" :
                            "win32"
                    );

                // 新構成対応
                if (!File.Exists(path))
                {
                    path = string.Format(
                        @"{0}\..\3dTools\{1}3dShaderConverter.dll",
                        Environment.GetEnvironmentVariable("NW4F_3DEDITOR_ROOT"),
                        Environment.Is64BitProcess ?
                            "" :
                            @"x86\"
                    );
                }

                return path;
            }
        }

        private static string ExeFilePath
        {
            get
            {
                var path = Path.GetFullPath(string.Format(
                        @"{0}\..\G3dTool\{1}\NW4F_g3dshdrcvtr.exe",
                        Environment.GetEnvironmentVariable("NW4F_3DEDITOR_ROOT"),
                        Environment.Is64BitProcess ?
                            "win64" :
                            "win32"
                    ));

                // 新構成対応
                if (!File.Exists(path))
                {
                    path = Path.GetFullPath(string.Format(
                        @"{0}\..\3dTools\{1}",
                        Environment.GetEnvironmentVariable("NW4F_3DEDITOR_ROOT"),
                        Environment.Is64BitProcess ?
                            "3dShaderConverter.exe" :
                            "3dShaderConverter32.exe"
                    ));
                }

                return path;
            }
        }

        private static string Exe32FilePath
        {
            get
            {
                var path = Path.GetFullPath(string.Format(
                        @"{0}\..\G3dTool\{1}\NW4F_g3dshdrcvtr.exe",
                        Environment.GetEnvironmentVariable("NW4F_3DEDITOR_ROOT"),
                        "win32"));

                // 新構成対応
                if (!File.Exists(path))
                {
                    path = Path.GetFullPath(string.Format(
                        @"{0}\..\3dTools\{1}",
                        Environment.GetEnvironmentVariable("NW4F_3DEDITOR_ROOT"),
                            "3dShaderConverter32.exe"
                    ));
                }

                return path;
            }
        }

        private static IntPtr dllHandle_ = IntPtr.Zero;
        public static void Initialize()
        {
            string currentDir = System.Environment.CurrentDirectory;
            try
            {
                System.IO.Directory.SetCurrentDirectory(Path.GetDirectoryName(DllFilepath));
                dllHandle_ = App.Win32.NativeMethods.LoadLibrary(DllFilepath);

                SetExportFunction();

                Init();
            }
            catch(Exception e)
            {
                UIMessageBox.Error(e.ToString());
            }
            finally
            {
                System.Environment.CurrentDirectory = currentDir;
            }
        }

        private delegate IntPtr wglCreateContextAttribsARBDelegate(IntPtr hdc, IntPtr hglrc, [In, MarshalAs(UnmanagedType.LPArray)] int[] attribList);

        public static bool glInitialized = false;
        public static void InitializeGL()
        {
            ForceNwArchive();
            try
            {
                // チーム設定で指定された GLSL バージョンを取得する
                TeamConfig.PlatformPreset platformPreset = App.ConfigData.ApplicationConfig.PlatformPresets.FirstOrDefault(x => x.Name == "Win");
                if (platformPreset != null)
                {
                    string[] args = platformPreset.PcOption.ShaderConverterAdditionalArgs.Split(' ');
                    if (args != null)
                    {
                        int index = Array.IndexOf(args, "--glsl-version");
                        if (index >= 0 && index + 1 < args.Length)
                        {
                            if (int.TryParse(args[index + 1], out int value))
                            {
                                GLSLVersion = value;
                            }
                        }
                    }
                }

                if (GLSLVersion != 0)
                {
                    // 既にバージョンが決定済み
                    DebugConsole.WriteLine(string.Format("GLSLVersion {0} was set from team settings", GLSLVersion));
                }
                else if (App.AppContext.SelectedPlatformPreset.UseNw)
                {
                    // TODO: GL コンテキストの初期化は廃止予定
                    // GL関連の初期化は、こちらで行う。
                    InitGLContext();
                    TheApp.MainFrame.HandleDestroyed += (s, e) => ShutdownGLContext();
                    glInitialized = true;

                    unsafe
                    {
                        // glGetString は InitGLContext で初期化が行われた後に行う必要がある。
                        var s = new string(App.Win32.NativeMethods.glGetString(App.Win32.NativeMethods.GL_SHADING_LANGUAGE_VERSION));

                        // たとえば、"4.50 NVIDIA" を "450" に変換する
                        // https://www.opengl.org/wiki/Get_Context_Info
                        // The string returned starts with "<major version>.<minor version>"
                        // the minor version always has two digits.
                        var splited = s.Split(new[] { '.' }, 2);
                        GLSLVersion = int.Parse(splited[0]) * 100 + int.Parse(splited[1].Substring(0, 2));
                        DebugConsole.WriteLine(string.Format("GLSLVersion {0} was set from gl context", GLSLVersion));
                    }
                }
                else
                {
                    // 3DEditor で GL コンテキストを作成して GLSL バージョンを取得する。
                    using (var ctrl = new System.Windows.Forms.Control())
                    using (var gfx = ctrl.CreateGraphics())
                    {
                        var hdc = gfx.GetHdc();
                        IntPtr defaultCtx = IntPtr.Zero;
                        IntPtr latestCtx = IntPtr.Zero;
                        string glslVersionStr = null;
                        do
                        {
                            // レンダリングに使うわけではないので
                            // 最低限の設定で PIXELFORMATDESCRIPTOR を作成する。
                            var pfd = new App.Win32.PIXELFORMATDESCRIPTOR
                            {
                                nSize = (short)Marshal.SizeOf(typeof(App.Win32.PIXELFORMATDESCRIPTOR)),
                                nVersion = 1,
                                dwFlags = App.Win32.NativeMethods.PFD_DRAW_TO_WINDOW | App.Win32.NativeMethods.PFD_SUPPORT_OPENGL,
                                iPixelType = App.Win32.NativeMethods.PFD_TYPE_RGBA,
                                cColorBits = 24,
                                cRedBits = 0,
                                cRedShift = 0,
                                cGreenBits = 0,
                                cGreenShift = 0,
                                cBlueBits = 0,
                                cBlueShift = 0,
                                cAlphaBits = 0,
                                cAlphaShift = 0,
                                cAccumBits = 0,
                                cAccumRedBits = 0,
                                cAccumGreenBits = 0,
                                cAccumBlueBits = 0,
                                cAccumAlphaBits = 0,
                                cDepthBits = 0,
                                cStencilBits = 0,
                                cAuxBuffers = 0,
                                iLayerType = App.Win32.NativeMethods.PFD_MAIN_PLANE,
                                bReserved = 0,
                                dwLayerMask = 0,
                                dwVisibleMask = 0,
                                dwDamageMask = 0
                            };
                            var pfmt = App.Win32.NativeMethods.ChoosePixelFormat(hdc, ref pfd);

                            // コンテキストの作成にはピクセルフォーマットの設定が必須。
                            if (!App.Win32.NativeMethods.SetPixelFormat(hdc, pfmt, ref pfd))
                            {
                                break;
                            }

                            // 標準コンテキストを作成する。
                            // ARB 拡張を使うためには、このコンテキストが必須。
                            defaultCtx = App.Win32.NativeMethods.wglCreateContext(hdc);
                            if (defaultCtx == null)
                            {
                                break;
                            }

                            // 標準コンテキストをカレントに設定して、その GLSL バージョンを取得。
                            if (!App.Win32.NativeMethods.wglMakeCurrent(hdc, defaultCtx))
                            {
                                break;
                            }
                            unsafe
                            {
                                // まずは標準コンテキストの GLSL バージョンを取得しておく。
                                var glslVerStr = new string(App.Win32.NativeMethods.glGetString(App.Win32.NativeMethods.GL_SHADING_LANGUAGE_VERSION));
                                if ((App.Win32.NativeMethods.glGetError() == App.Win32.NativeMethods.GL_NO_ERROR) && (glslVerStr != null))
                                {
                                    glslVersionStr = glslVerStr;
                                }
                            }

                            // wglCreateContextAttribsARB() のために、標準コンテキストの OpenGL バージョンを取得しておく。
                            // OpenGL のバージョン取得は拡張命令なので、標準コンテキストのバージョンによっては取得に失敗する可能性もある。
                            // 取得に成功したかどうかを hasDefaultVersion で判断し、wglCreateContextAttribsARB() の attribList 引数を調整する。
                            var major = new int[1];
                            var minor = new int[1];
                            bool hasDefaultVersion = true;
                            App.Win32.NativeMethods.glGetIntegerv(App.Win32.NativeMethods.GL_MAJOR_VERSION, major);
                            hasDefaultVersion &= App.Win32.NativeMethods.glGetError() == App.Win32.NativeMethods.GL_NO_ERROR;
                            App.Win32.NativeMethods.glGetIntegerv(App.Win32.NativeMethods.GL_MINOR_VERSION, minor);
                            hasDefaultVersion &= App.Win32.NativeMethods.glGetError() == App.Win32.NativeMethods.GL_NO_ERROR;

                            // 標準コンテキストから wglCreateContextAttribsARB() を取得。
                            var wglCreateContextAttribsARB = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.wglGetProcAddress("wglCreateContextAttribsARB"), typeof(wglCreateContextAttribsARBDelegate)) as wglCreateContextAttribsARBDelegate;
                            if (wglCreateContextAttribsARB == null)
                            {
                                break;
                            }

                            // 最新のコンテキストを作成する。
                            // 標準コンテキストが最新コンテキストの場合もある。
                            if (hasDefaultVersion)
                            {
                                // デフォルトバージョン以上かつ 3.2 以上のコンテキストを作成する。
                                int[] attribList =
                                {
                                    App.Win32.NativeMethods.WGL_CONTEXT_MAJOR_VERSION_ARB, major[0],
                                    App.Win32.NativeMethods.WGL_CONTEXT_MINOR_VERSION_ARB, minor[0],
                                    App.Win32.NativeMethods.WGL_CONTEXT_PROFILE_MASK_ARB, App.Win32.NativeMethods.WGL_CONTEXT_CORE_PROFILE_BIT_ARB,
                                    0
                                };
                                latestCtx = wglCreateContextAttribsARB(hdc, IntPtr.Zero, attribList);
                            }
                            else
                            {
                                // バージョン 3.2 以上のコンテキストを作成する。
                                // WGL_CONTEXT_MAJOR/MINOR_VERSION_ARB に 3.2 を指定したいところだが、
                                // 3.2 非対応環境での動作が不明なので、敢えて何も指定しない。
                                // WGL_CONTEXT_CORE_PROFILE_BIT_ARB は 3.2 以上なので、バージョン指定は不要なはず。
                                int[] attribList =
                                {
                                   App.Win32.NativeMethods. WGL_CONTEXT_PROFILE_MASK_ARB, App.Win32.NativeMethods.WGL_CONTEXT_CORE_PROFILE_BIT_ARB,
                                    0
                                };
                                latestCtx = wglCreateContextAttribsARB(hdc, IntPtr.Zero, attribList);
                            }
                            if (latestCtx == null)
                            {
                                break;
                            }

                            // 最新コンテキストをカレントに設定して、その GLSL バージョンを取得。
                            if (!App.Win32.NativeMethods.wglMakeCurrent(hdc, latestCtx))
                            {
                                break;
                            }
                            unsafe
                            {
                                var glslVerStr = new string(App.Win32.NativeMethods.glGetString(App.Win32.NativeMethods.GL_SHADING_LANGUAGE_VERSION));
                                if ((App.Win32.NativeMethods.glGetError() == App.Win32.NativeMethods.GL_NO_ERROR) && (glslVerStr != null))
                                {
                                    glslVersionStr = glslVerStr;
                                }
                            }
                        }
                        while (false);

                        // バージョン取得が完了したので、コンテキストを破棄。
                        {
                            App.Win32.NativeMethods.wglMakeCurrent(hdc, IntPtr.Zero);
                            if (latestCtx != null)
                            {
                                App.Win32.NativeMethods.wglDeleteContext(latestCtx);
                                latestCtx = IntPtr.Zero;
                            }
                            if (defaultCtx != null)
                            {
                                App.Win32.NativeMethods.wglDeleteContext(defaultCtx);
                                defaultCtx = IntPtr.Zero;
                            }
                            if (hdc != null)
                            {
                                gfx.ReleaseHdc(hdc);
                                hdc = IntPtr.Zero;
                            }
                        }

                        try
                        {
                            // たとえば、"4.50 NVIDIA" を "450" に変換する
                            // https://www.opengl.org/wiki/Get_Context_Info
                            // The string returned starts with "<major version>.<minor version>"
                            // the minor version always has two digits.
                            var splited = glslVersionStr.Split(new[] { '.' }, 2);
                            GLSLVersion = int.Parse(splited[0]) * 100 + int.Parse(splited[1].Substring(0, 2));
                            DebugConsole.WriteLine(string.Format("GLSLVersion {0} was set from gl context", GLSLVersion));
                        }
                        catch (Exception e)
                        {
                            // とりあえず 330 にしておく
                            GLSLVersion = 330;
                            DebugConsole.WriteLine(e.ToString());
                        }
                    }
                    glInitialized = true;
                }
            }
            catch (Exception e)
            {
                // とりあえず 330 にしておく
                GLSLVersion = 330;
                DebugConsole.WriteLine(e.ToString());
            }
        }

        public static void Initialize2()
        {
            try
            {
                if (Clear != null)
                {
                    Clear();
                }
            }
            catch (Exception e)
            {
                UIMessageBox.Error(e.ToString());
            }
        }

        public static void Destroy()
        {
            Clear = null;

            if (dllHandle_ != IntPtr.Zero)
            {
                Shutdown();

                App.Win32.NativeMethods.FreeLibrary(dllHandle_);
                dllHandle_ = IntPtr.Zero;
            }
        }

        private static void SetExportFunction()
        {
            Init = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolShaderConverterInitialize"), typeof(InitDelegate)) as InitDelegate;
            Shutdown = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolShaderConverterShutdown"), typeof(ShutdownDelegate)) as ShutdownDelegate;
            InitGLContext = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolShaderConverterInitializeGlContext"), typeof(InitGLContextDelegate)) as InitGLContextDelegate;
            ShutdownGLContext = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolShaderConverterShutdownGlContext"), typeof(ShutdownGLContextDelegate)) as ShutdownGLContextDelegate;
            Clear = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolShaderConverterClear"), typeof(ClearDelegate)) as ClearDelegate;
            GetCvtrVersion = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolShaderConverterGetConverterVersion"), typeof(GetCvtrVersionDelegate)) as GetCvtrVersionDelegate;
            SetOptions = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolShaderConverterSetOptions"), typeof(SetOptionsDelegate)) as SetOptionsDelegate;
            AddFile = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolShaderConverterAddFile"), typeof(AddFileDelegate)) as AddFileDelegate;
            CalcArchiveSize = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolShaderConverterCalculateArchiveSize"), typeof(CalcArchiveSizeDelegate)) as CalcArchiveSizeDelegate;
            Convert = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolShaderConverterConvert"), typeof(ConvertDelegate)) as ConvertDelegate;
            EndianSwap = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolShaderConverterSwapEndian"), typeof(EndianSwapDelegate)) as EndianSwapDelegate;
            Make = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolShaderConverterMake"), typeof(MakeDelegate)) as MakeDelegate;
            CalcDefinitionSize = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolShaderConverterCalculateDefinitionSize"), typeof(CalcDefinitionSizeDelegate)) as CalcDefinitionSizeDelegate;
            Write = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolShaderConverterWrite"), typeof(WriteDelegate)) as WriteDelegate;
            OpenLogFile = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolShaderConverterOpenLogFile"), typeof(OpenLogFileDelegate)) as OpenLogFileDelegate;
            CloseLogFile = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolShaderConverterCloseLogFile"), typeof(CloseLogFileDelegate)) as CloseLogFileDelegate;
            ForceNwArchive = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolBinaryConverterForceNwArchive"), typeof(ForceNwArchiveDelegate)) as ForceNwArchiveDelegate;
            ForceNnArchive = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolBinaryConverterForceNnArchive"), typeof(ForceNnArchiveDelegate)) as ForceNnArchiveDelegate;
        }

        private static string GetLogFilePath()
        {
            return TemporaryFileUtility.MakeTemporaryFileName(".txt");
        }

        public static bool ConvertShaderToBinaryForViewer(string fsdFilePath, string bfshaFilePath, bool gl_source, bool gl_binary, TargetType targetType, TeamConfig.PlatformPreset platform, bool useConverter32, string additionalOptions)
        {
            return ConvertShader(Enumerable.Repeat(fsdFilePath, 1), false, null, bfshaFilePath, true, gl_source, gl_binary, targetType, platform, useConverter32, additionalOptions);
        }

        public static int GLSLVersion { get; set; }

        public enum TargetType
        {
            Cafe,
            PC,
        }

        public static bool ConvertShader(IEnumerable<string> sourceFiles, bool test, string fsdbOutputPath, string bfshaFilePath, bool showLog, bool gl_source, bool gl_binary, TargetType targetType, TeamConfig.PlatformPreset platform, bool useConverter32, string additionalOptions = null)
        {
            var builder = new StringBuilder();

            // MarkerCreateOldBinary があるとエラーになり、ないと別のエラーになるので指定しないように修正
            /*
            if (platform.UseNw)
            {
                // SIGLO-76235 対応。(SIGLO-71286 で削除されたオプション)
                // builder.Append("--force-nw-archive ");
            }
            */


            bool hasFsv = sourceFiles.Any(x =>
            {
                var ext = Path.GetExtension(x).ToLower();
                return ext == ".fsva" || ext == ".fsvb";
            });

            // 重複するオプションは後が優先して解釈される。
            // 追加オプションは先に指定する。
            if (hasFsv && !string.IsNullOrEmpty(ConfigData.ApplicationConfig.FileIo.ShaderConverterAdditionalArgsForOptimizedShader))
            {
                builder.Append(Environment.ExpandEnvironmentVariables(ConfigData.ApplicationConfig.FileIo.ShaderConverterAdditionalArgsForOptimizedShader) + " ");
            }
            else if (!string.IsNullOrEmpty(ConfigData.ApplicationConfig.FileIo.ShaderConverterAdditionalArgs))
            {
                builder.Append(Environment.ExpandEnvironmentVariables(ConfigData.ApplicationConfig.FileIo.ShaderConverterAdditionalArgs) + " ");
            }

            if (hasFsv && !string.IsNullOrEmpty(platform.ActiveDeviceOption?.ShaderConverterAdditionalArgsForOptimizedShader))
            {
                builder.Append(Environment.ExpandEnvironmentVariables(platform.ActiveDeviceOption.ShaderConverterAdditionalArgsForOptimizedShader) + " ");
            }
            else if (!string.IsNullOrEmpty(platform.ActiveDeviceOption?.ShaderConverterAdditionalArgs))
            {
                builder.Append(Environment.ExpandEnvironmentVariables(platform.ActiveDeviceOption.ShaderConverterAdditionalArgs) + " ");
            }

            builder.Append("--unified-annotation ");
            if (!string.IsNullOrEmpty(fsdbOutputPath))
            {
                builder.AppendFormat("--output=\"{0}\" ", fsdbOutputPath);
            }

            if (!string.IsNullOrEmpty(bfshaFilePath))
            {
                Debug.Assert(bfshaFilePath.EndsWith(".bfsha", StringComparison.OrdinalIgnoreCase));
                builder.AppendFormat("--output=\"{0}\" ", bfshaFilePath);
            }

            string exePath;
            if (platform.UseNw)
            {
                exePath = gl_binary ? Exe32FilePath : ExeFilePath;
            }
            else
            {
                exePath = useConverter32 ? Exe32FilePath : ExeFilePath;
            }

            var glslVersion = platform.ActiveDeviceOption != null ? platform.ActiveDeviceOption.GLSLVersion : 0;
            builder.AppendFormat("--glsl-version={0} ", glslVersion != 0 ? glslVersion : GLSLVersion);

            if (gl_source && platform.UseNw)
            {
                // SIGLO-76235 対応。(SIGLO-71286 で削除されたオプション)
                // builder.AppendFormat("--gl-source ");
            }

            if (gl_binary)
            {
                if (platform.UseNw)
                {
                    // SIGLO-76235 対応。(SIGLO-71286 で削除されたオプション)
                    // builder.Append("--gl-binary ");
                }
            }

            if (!string.IsNullOrEmpty(additionalOptions))
            {
                builder.Append(additionalOptions);
                builder.Append(" ");
            }

            foreach (var path in sourceFiles)
            {
                builder.AppendFormat("\"{0}\" ", path);
            }

            DebugConsole.WriteLine(string.Format("ShaderConvert {0} {1}", exePath, builder.ToString()));
            var args = exePath + " " + builder.ToString();
            var info = new ProcessStartInfo(exePath, builder.ToString())
                           {
                               CreateNoWindow = true,
                               UseShellExecute = false,
                               RedirectStandardError = true,
                               StandardErrorEncoding = Encoding.Default,
                               RedirectStandardOutput = true,
                               StandardOutputEncoding = Encoding.Default
                           };
            Process process = null;
            var outputBuilder = new StringBuilder();
            var errorBuilder = new StringBuilder();

            try
            {
                var logStopWatch = new MessageLogStopWatch(res.Strings.Log_ShaderConvert, App.ConfigData.ApplicationConfig.FileIo.LogShaderConvertTime);
                {
                    process = Process.Start(info);

                    process.OutputDataReceived += (s, a) =>
                    {
                        if (!string.IsNullOrEmpty(a.Data))
                        {
                            DebugConsole.WriteLine(string.Format("ShaderConverterLog: {0}", a.Data));
                            outputBuilder.AppendLine(a.Data);
                        }
                    };
                    process.BeginOutputReadLine();
                    process.ErrorDataReceived += (s, a) =>
                    {
                        if (!string.IsNullOrEmpty(a.Data))
                        {
                            DebugConsole.WriteLine(string.Format("ShaderConverterError: {0}", a.Data));
                            errorBuilder.AppendLine(a.Data);
                        }
                    };
                    process.BeginErrorReadLine();

                    //process.WaitForExit(); だと Abort されない。詳しい理由は不明
                    while (!process.WaitForExit(10000)) { }
                }

                // エラーや例外が起きたときはログ出力しない
                if (process.ExitCode == 0)
                {
                    logStopWatch.WriteLog();
                }
            }
            catch
            {
                if (process != null)
                {
                    try
                    {
                        // プロセスが走っているかもしれないので止める
                        process.Kill();
                    }
                    catch
                    {
                        // 何もしない
                        DebugConsole.WriteLine("Failed to Kill process");
                    }
                }

                if (showLog && !Thread.CurrentThread.ThreadState.HasFlag(System.Threading.ThreadState.AbortRequested))
                {
                    var log = res.Strings.ShaderConvertError_FailedToStart;

                    // ログ出力
                    ErrorLog.WriteLog(log, ErrorLog.LogKind.ShaderConvert);

                    // エラーダイアログの表示
                    ShdrcvtrManager.ShowShaderConvertErrorMsg(new List<string> { args, log }, false);
                }

                return false;
            }

            // 正常終了しなかったときのみエラー表示
            if (showLog && process.ExitCode != 0)
            {
                var error = errorBuilder.ToString();
                var output = outputBuilder.ToString();

                var messages = new StringBuilder();
                if (process.ExitCode != 0)
                {
                    messages.AppendLine(res.Strings.ShaderConvertError);
                }
                messages.AppendLine();
                messages.AppendLine("From ShaderConverter -------------------------------");
                messages.AppendLine(output);
                messages.AppendLine(error);

                var log = messages.ToString();

                // ログ出力
                ErrorLog.WriteLog(log, ErrorLog.LogKind.ShaderConvert);

                // エラーダイアログの表示
                ShdrcvtrManager.ShowShaderConvertErrorMsg(new List<string> { args, log }, process.ExitCode == 0);
            }

#if DEBUG
            DebugConsole.WriteLine(errorBuilder.ToString());
            DebugConsole.WriteLine(outputBuilder.ToString());
#endif
            return process.ExitCode == 0;
        }

        public static bool ConvertShaderToBinaryForViewer(List<Tuple<byte[], string>> fileImages, string bfshaFilePath, TargetType targetType, TeamConfig.PlatformPreset platform, bool useConverter32, string additionalOptions)
        {
            try
            {
                // ファイルに出力
                foreach (var tuple in fileImages)
                {
                    DebugConsole.WriteLine("ConvertShader write input file" + tuple.Item2);
                    File.WriteAllBytes(tuple.Item2, tuple.Item1);
                }
            }
            catch (Exception e)
            {
                if (!Thread.CurrentThread.ThreadState.HasFlag(System.Threading.ThreadState.AbortRequested))
                {
                    // ログ出力
                    ErrorLog.WriteLog(e.Message, ErrorLog.LogKind.ShaderConvert);

                    // エラーダイアログ表示
                    ShdrcvtrManager.ShowShaderConvertErrorMsg(new List<string> { e.Message }, false);
                }
                return false;
            }

            return ConvertShader(fileImages.Select(x => x.Item2), false, null, bfshaFilePath, true, targetType == TargetType.PC, gl_binary: false, targetType: targetType, platform: platform, useConverter32:useConverter32, additionalOptions: additionalOptions);
        }

        public static bool ConvertFscToFsd(
            byte[] fscData,
            string fscFilePath,
            string fsdFilePath, // null のときは出力しない
            bool log,
            bool output,
            out byte[] outputFsd,
            bool compile_test,
            bool specificShadingModel,
            TeamConfig.PlatformPreset platform)
        {
            outputFsd = null;
            lock (lockObject)
            using (var logWriter = new ShaderConvertLog(log))
            {
                // ログファイル名設定
                logWriter.SetShaderConverterLogFileStream(GetLogFilePath(), GetLogFilePath());
                using (var watch = new DebugStopWatch("ConvertFscToFsd " + fsdFilePath ?? fscFilePath))
                {
                    try
                    {
                        DebugConsole.WriteLine("ConvertFscToFsd ");
                        if (Clear == null)
                        {
                            Initialize();
                            if (glInitialized == false)
                            {
                                InitializeGL();
                            }
                        }
                        if (!Clear())
                        {
                            DebugConsole.WriteLine("Clear failed");
                            logWriter.AddErrorMsg(res.Strings.ShaderConvertError_UnknownErr);
                            return false;
                        }

                        if (platform.UseNw)
                        {
                            ForceNwArchive();
                        }
                        else
                        {
                            ForceNnArchive();
                        }

                        var options = new List<string>();
                        if (!specificShadingModel)
                        {
                            options.Add("--unified-annotation");
                        }

                        // 追加引数 ShaderConverterAdditionalArgs は指定しない

                        if (options.Any())
                        {
                            // nullInserted は、次のような形で要素を埋める
                            // { オプション, 値またはnull, オプション, 値またはnull, ... , オプション, 値またはnull, null}
                            var nullInserted = new List<string>();
                            for (int i=0; i< options.Count;i++)
                            {
                                var item = options[i];
                                if (item.StartsWith("---"))
                                {
                                    // 位置引数を指定することはできない
                                    if (i + 1 < options.Count)
                                    {
                                        DebugConsole.WriteLine("ConvertFscToFsd Error PosArg");
                                        logWriter.AddErrorMsg(string.Format(res.Strings.ShaderConverterError_PosArg, options[i+1]));
                                        return false;
                                    }
                                    break;
                                }

                                bool isOption = item.StartsWith("-");

                                // オプションは偶数番から開始させる
                                if (isOption && nullInserted.Count % 2 == 1)
                                {
                                    nullInserted.Add(null);
                                }

                                // 偶数番に値は埋めない
                                if (!isOption && nullInserted.Count % 2 == 0)
                                {
                                    DebugConsole.WriteLine("ConvertFscToFsd Error PosArg");
                                    logWriter.AddErrorMsg(string.Format(res.Strings.ShaderConverterError_PosArg, options[i + 1]));
                                    return false;
                                }

                                if (isOption)
                                {
                                    string[] splitted;
                                    if (item.StartsWith("--"))
                                    {
                                        // ロングネーム
                                        splitted = item.Split(new[] { '=' }, 2);
                                    }
                                    else
                                    {
                                        // ショートネーム
                                        if (item.Length <= 2)
                                        {
                                            splitted = new[] { item };
                                        }
                                        else
                                        {
                                            splitted = new[] { item.Substring(0, 2), item.Substring(2) };
                                        }
                                    }

                                    Debug.Assert(splitted.Length == 1 || splitted.Length == 2);
                                    foreach (var s in splitted)
                                    {
                                        nullInserted.Add(s);
                                    }
                                }
                                else
                                {
                                    // 値はそのまま追加
                                    nullInserted.Add(item);
                                }
                            }

                            // 奇数番を埋める
                            if (nullInserted.Count % 2 == 1)
                            {
                                nullInserted.Add(null);
                            }

                            // 末尾を埋める
                            nullInserted.Add(null);

                            logWriter.AddNotErrorMsg(string.Concat(nullInserted.Select(x => (x ?? string.Empty) + " ")));
                            if (!SetOptions(nullInserted.ToArray()))
                            {
                                DebugConsole.WriteLine("ConvertFscToFsd SetOptions failed");
                                logWriter.AddErrorMsg(res.Strings.ShaderConvertError_SetOptions);
                                return false;
                            }
                        }

                        unsafe
                        {
                            fixed (byte* pData = fscData)
                            {
                                logWriter.AddNotErrorMsg(fscFilePath);
                                if (!AddFile(
                                    (IntPtr)pData,
                                    Marshal.SizeOf(fscData[0]) * fscData.Length,
                                    new string[] { fscFilePath, Path.GetFileNameWithoutExtension(fscFilePath), Path.GetExtension(fscFilePath) },
                                    new string[] { null }
                                    ))
                                {
                                    DebugConsole.WriteLine("ConvertFscToFsd AddFile failed");
                                    logWriter.AddErrorMsg(res.Strings.ShaderConvertError_AddFile);
                                    return false;
                                }

                                if (!Make())
                                {
                                    DebugConsole.WriteLine("ConvertFscToFsd Make failed");
                                    logWriter.AddErrorMsg(res.Strings.ShaderConvertError_Make);
                                    return false;
                                }

                                int size = CalcDefinitionSize(fsdFilePath);
                                if (size == 0)
                                {
                                    DebugConsole.WriteLine("ConvertFscToFsd CalcDefinitionSize() failed");
                                    logWriter.AddErrorMsg(res.Strings.ShaderConvertError_CalcDefinitionSize);
                                    return false;
                                }

                                outputFsd = new byte[size / Marshal.SizeOf(typeof(byte))];
                                fixed (byte* pOutput = outputFsd)
                                {
                                    if (!Write((IntPtr)pOutput, size))
                                    {
                                        DebugConsole.WriteLine("ConvertFscToFsd Write() failed");
                                        logWriter.AddErrorMsg(res.Strings.ShaderConvertError_SaveFile);
                                        return false;
                                    }

                                    if (output)
                                    {
                                        // バイナリ形式でファイルに書き出し。
                                        using (BinaryWriter w = new BinaryWriter(File.OpenWrite(fsdFilePath)))
                                        {
                                            w.Write(outputFsd, 0, size);
                                        }
                                    }
                                }
                            }
                        }
                    }
                    catch (ThreadAbortException)
                    {
                        // Abortに対しては特に何も行わない
                    }
                    catch (Exception e)
                    {
                        DebugConsole.WriteLine("ConvertFscToFsd exception: " + e.ToString());
                        logWriter.AddErrorMsg(res.Strings.ShaderConvertError_UnknownErr);
                        return false;
                    }
                }

                logWriter.warning = true;

                // コンパイルテスト中は警告を表示しない
                logWriter.hideError = compile_test;
                return true;
            }
        }

        // 書き込みデリゲート
        private delegate void ShowShaderConvertErrorMsgDelegate(List<string> errMsgs, bool warning);
        // Invoke用書き込みメソッド
        private static void _ShowShaderConvertErrorMsg(List<string> errMsgs, bool warning) {
            // ダイアログ表示
            using (var dialog = new OkListBoxDialog())
            {
                if (warning)
                {
                    dialog.Text = res.Strings.ShaderConvertWarning_DlgCaption;
                    dialog.lblDescription.Text = res.Strings.ShaderConvertWarning_DlgDescription;
                }
                else
                {
                    dialog.Text = res.Strings.ShaderConvertError_DlgCaption;
                    dialog.lblDescription.Text = res.Strings.ShaderConvertError_DlgDescription;
                }
                foreach (var file in errMsgs)
                {
                    dialog.AddLine(file);
                }
                dialog.ShowDialog();
            }
        }
        public static void ShowShaderConvertErrorMsg(List<string> errMsgs, bool warning)
        {
            // 別スレッドからの呼び出し用
            // また同一スレッドでもエラーメッセージで後の処理を止めないBeginInvokeする。
            {
                TheApp.MainFrame.BeginInvoke(new ShowShaderConvertErrorMsgDelegate(_ShowShaderConvertErrorMsg), new object[] { errMsgs, warning });
                return;
            }


        }
    }

    public sealed class ShaderConvertLog : IDisposable
    {
        private string stdOutLogFilePath_;
        private string stdErrLogFilePath_;
        private readonly List<string> errMsgs_ = new List<string>();
        public bool warning = false;
        public bool hideError = false;
        public bool showLog = false;

        // デバッグ用
        private static bool logging = false;

        private readonly bool write_;
        public ShaderConvertLog(bool write = true)
        {
            write_ = write;
        }

        public void SetShaderConverterLogFileStream(string stdOutLogFilePath, string stdErrLogFilePath)
        {
            if (!write_)
            {
                return;
            }

            // ログ出力中に別のログファイルを設定してはいけない。
            // 本当は singleton にしたほうがよさそう。
            Debug.Assert(logging == false);
            logging = true;

            stdOutLogFilePath_ = stdOutLogFilePath;
            stdErrLogFilePath_ = stdErrLogFilePath;

            if (ShdrcvtrManager.OpenLogFile == null)
            {
                ShdrcvtrManager.Initialize();
                if (ShdrcvtrManager.glInitialized == false)
                {
                    ShdrcvtrManager.InitializeGL();
                }
            }

            try
            {
                const string mode = "w+";
                ShdrcvtrManager.OpenLogFile(
                    new string[] { stdOutLogFilePath, mode },
                    new string[] { stdErrLogFilePath, mode }
                    );
            }
            catch (Exception e)
            {
                DebugConsole.WriteLine("SetBinaryConverterLogFileStream Exception {0}", e.Message);
            }
        }

        public void AddErrorMsg(string msg)
        {
            showLog = true;
            errMsgs_.Add(msg);
        }

        public void AddNotErrorMsg(string msg)
        {
            errMsgs_.Add(msg);
        }

        private static void ResetShaderConverterLogFileStream()
        {
            try
            {
                // バイナリコンバーターのLogFileStreamをもとに戻す。
                ShdrcvtrManager.CloseLogFile();
            }
            catch (Exception e)
            {
                DebugConsole.WriteLine("ResetBinaryConverterLogFileStream Exception {0}", e.Message);
            }

        }

        private static string GetLogFileString(string path)
        {
            string logFileStr = string.Empty;

            try
            {
                using (StreamReader reader = new StreamReader(path, System.Text.Encoding.Default))
                {
                    logFileStr = reader.ReadToEnd();
                }

                // 一時ファイルも削除しちゃう。
                System.IO.File.Delete(path);
            }
            catch (Exception e)
            {
                DebugConsole.WriteLine("GetLogFileString Exception {0}", e.Message);
            }

            return logFileStr;
        }

        private void SetErrMsgFromLogfile(string stdErrLogFilePath, string stdOutLogFilePath)
        {
            string ErrorlogStr = GetLogFileString(stdErrLogFilePath);
            string OutlogStr = GetLogFileString(stdOutLogFilePath);
            if (!string.IsNullOrEmpty(ErrorlogStr))
            {
                const string fromConverter = "From ShaderConverter -------------------------------";
                AddErrorMsg(string.Empty);
                AddErrorMsg(fromConverter);
                AddErrorMsg(OutlogStr);
                AddErrorMsg(ErrorlogStr);
            }
        }

        public void Dispose()
        {
            if (!write_)
            {
                // リソースを解放
                ShdrcvtrManager.Clear();
                return;
            }

            // バイナリコンバーターDll側から出力されたログを読み込む
            ResetShaderConverterLogFileStream();

            logging = false;

            SetErrMsgFromLogfile(stdErrLogFilePath_, stdOutLogFilePath_);

            if (!showLog)
            {
                return;
            }

            if (!hideError)
            {
                string msgSum = string.Empty;
                foreach (var msg in errMsgs_)
                {
                    msgSum += (msg + "\r\n");
                }
                // ログ出力
                ErrorLog.WriteLog(msgSum, ErrorLog.LogKind.ShaderConvert);

                // abort によって中断されたときはダイアログを表示しない
                if (System.Threading.Thread.CurrentThread.ThreadState != System.Threading.ThreadState.AbortRequested)
                {
                    // エラーダイアログの表示
                    ShdrcvtrManager.ShowShaderConvertErrorMsg(errMsgs_, warning);
                }
            }

#if DEBUG
            foreach (var msg in errMsgs_)
            {
                DebugConsole.WriteLine(msg);
            }
#endif

            // リソースを解放
            ShdrcvtrManager.Clear();
        }
    }

}
