﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Xml;
using System.Xml.Serialization;
using App.Command;
using App.ConfigData;
using App.Controls;
using App.Utility;
using App.res;
using ConfigCommon;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using App.PropertyEdit;

namespace App.Data
{
    public class DocumentSaver
    {
        public DocumentSaver()
        {
            _disableUpdateFileInfo = false;
        }

        public DocumentSaver(bool disableUpdateFileInfo)
        {
            _disableUpdateFileInfo = disableUpdateFileInfo;
        }

        private readonly bool _disableUpdateFileInfo;

        #region イベント
        //---------------------------------------------------------------------
        /// <summary>
        /// ドキュメント保存イベント。
        /// </summary>
        public event EventHandler DocumentSaved = null;

        /// <summary>
        /// ドキュメント保存ハンドラ。
        /// </summary>
        protected virtual void OnDocumentSaved(EventArgs e)
        {
            if (DocumentSaved != null)
            {
                DocumentSaved(this, e);
            }
        }
        #endregion

        /// <summary>
        /// ファイルを保存します。
        /// </summary>
        /// <param name="document">保存対象</param>
        /// <param name="filePath">保存パス</param>
        /// <param name="commandSet">実行済みのパス変更コマンド</param>
        /// <param name="confirmReadOnly"></param>
        /// <returns>保存されたかどうか</returns>
        private bool SetSavePath(Document document, string filePath, EditCommandSet commandSet, bool confirmReadOnly = false)
        {
            /*
            if (document.ObjectID == GuiObjectID.Project)
            {
                // 未確定パスの確定
                if (!SaveNotExistsFiles(commandSet, savedDocuments))
                {
                    return false;
                }
            }*/

            if (!document.ConfirmBeforeSave())
            {
                return false;
            }

            {
                var defaultExt = document.FileDotExt.Substring(1);
                var filename = document.FileName;
                var filter = ObjectIDUtility.FileFilter(document.ObjectID);
                var fileLocation = "";

                if (document.FileLocation != string.Empty)
                {
                    fileLocation = document.FileLocation;
                }
                else
                {
                    fileLocation = ApplicationConfig.Setting.IO.SaveAsPath;

                    // 保存場所をモデルのフォルダにします。
                    if (document.ObjectID == GuiObjectID.Project)
                    {
                        var model = DocumentManager.Models.
                            OrderBy(x => x.Name == document.Name ? 0 : 1).
                            FirstOrDefault(x => !string.IsNullOrEmpty(x.FileLocation) && !x.OpenedFromStartUp);
                        if (model != null)
                        {
                            fileLocation = model.FileLocation;
                        }
                    }
                }

                while (true)
                {
                    // 名前指定
                    if (string.IsNullOrEmpty(filePath))
                    {
                        filename = App.AppContext.NotificationHandler.SaveFileDialog(
                            filter,
                            defaultExt,
                            Strings.DocumentSaver_SetSavePath_SaveAs,
                            fileName: filename,
                            fileLocation: fileLocation);

                        if (string.IsNullOrEmpty(filename))
                        {
                            return false;
                        }

                        var dialogDotExt = Path.GetExtension(filename);
                        if (dialogDotExt != document.FileDotExt)
                        {
                            var dialogDotExtFormat	= dialogDotExt.Substring(0, dialogDotExt.Length - 1);
                            var dialogDotExtAB		= dialogDotExt.Substring(dialogDotExt.Length - 1);
                            var docFileDotExtFormat	= document.FileDotExt.Substring(0, document.FileDotExt.Length - 1);
                            var docFileDotExtAB		= document.FileDotExt.Substring(document.FileDotExt.Length - 1);

                            if ((dialogDotExtFormat == docFileDotExtFormat) &&
                                ((dialogDotExtAB  == G3dPath.Text) || (dialogDotExtAB  == G3dPath.Binary)) &&
                                ((docFileDotExtAB == G3dPath.Text) || (docFileDotExtAB == G3dPath.Binary)))
                            {
                                // アスキー、バイナリの違いは問題なし
                                ;
                            }
                            else
                            {
                                // TODO:.txt とか .doc とかだと補完してくれない。なぜ？
                                //      なので自前で補完する
                                filename += document.FileDotExt;
                            }
                        }

                        string name = Path.GetFileNameWithoutExtension(filename);

                        // 名前のチェック
                        if (!RegexMatch.Check(name, "[0-9A-Za-z\\-\\._]+"))
                        {
                            App.AppContext.NotificationHandler.MessageBoxWarning(string.Format(Strings.IO_InvalidFileName, name));
                            fileLocation = Path.GetDirectoryName(filename);
                            filename = Path.GetFileName(filename);
                            continue;
                        }

                        string ext = Path.GetExtension(filename);


                        // マテリアルアニメーションの接尾辞のチェック
                        if (ApplicationConfig.Preset.SeparateMaterialAnimCreationMenu && document is MaterialAnimation)
                        {
                            var subType = MaterialAnimation.NameToType(name);
                            var materialAnimation = (MaterialAnimation)document;
                            if (subType != materialAnimation.MaterialAnimationSubType)
                            {
                                var postFix = MaterialAnimation.PostFix(materialAnimation.MaterialAnimationSubType);
                                if (!string.IsNullOrEmpty(postFix))
                                {
                                    App.AppContext.NotificationHandler.MessageBoxWarning(
                                        string.Format(
                                        Strings.Name_PostFix,
                                        MaterialAnimation.SubTypeString(materialAnimation.MaterialAnimationSubType),
                                        postFix));
                                }
                                else
                                {
                                    App.AppContext.NotificationHandler.MessageBoxWarning(
                                        string.Format(
                                        Strings.Name_NotPostFix,
                                        MaterialAnimation.SubTypeString(materialAnimation.MaterialAnimationSubType),
                                        MaterialAnimation.PostFix(subType)));
                                }

                                fileLocation = Path.GetDirectoryName(document.FileName);
                                filename = Path.GetFileName(filename);
                                continue;
                            }
                        }

                        // 名前の重複チェック
                        if (ObjectIDUtility.CanOpenSameNameFile(ext))
                        {
                            // ファイルパスの重複チェック
                            if (DocumentManager.Objects(document.ObjectID).OfType<Document>().Any(
                                x => x != document && x.FilePath == filename))
                            {
                                App.AppContext.NotificationHandler.MessageBoxWarning(string.Format(Strings.Save_SamePath, filename));
                                fileLocation = Path.GetDirectoryName(document.FileName);
                                filename = Path.GetFileName(filename);
                                continue;
                            }

                            // アニメーションセット内の重複
                            var duplication = (from model in DocumentManager.Models
                                               let fileName = DocumentSaver.DuplicatedName(model, document, name)
                                               where fileName != null
                                               select new { modelName = model.Name, fileName }).FirstOrDefault();

                            if (duplication != null)
                            {
                                App.AppContext.NotificationHandler.MessageBoxWarning(string.Format(Strings.Rename_NameDuplicateUnderModel, duplication.modelName, duplication.fileName));
                                fileLocation = Path.GetDirectoryName(document.FileName);
                                filename = Path.GetFileName(filename);
                                continue;
                            }
                        }
                        else if (DocumentManager.Objects(document.ObjectID).Any(x => (x != document) && (x.Name == name)))
                        {
                            // ファイル名の重複
                            App.AppContext.NotificationHandler.MessageBoxWarning(string.Format(Strings.Rename_NameDuplicate, name));
                            fileLocation = Path.GetDirectoryName(document.FileName);
                            filename = Path.GetFileName(filename);
                            continue;
                        }

                        filePath = filename;
                    }

                    // 読み取り専用チェック
                    if (confirmReadOnly && File.Exists(filePath) && (File.GetAttributes(filePath) & FileAttributes.ReadOnly) != 0)
                    {
                        if (App.AppContext.NotificationHandler.MessageBoxOkCancel(MessageContext.IO_SaveFileReadOnly, string.Format(Strings.IO_SaveFileReadOnly, filePath)))
                        {
                            filePath = string.Empty;
                            continue;
                        }
                        else
                        {
                            return false;
                        }
                    }

                    if (!document.ConfirmPathBeforeSave(filePath))
                    {
                        return false;
                    }

                    // パス変更
                    SetFilePath(document, filePath, commandSet);

                    return true;
                }
            }
        }

        public static string DuplicatedName(Model owner, Document document, string name)
        {
            // Rename 後に重複するのは NG
            string sameName = null;
            bool contains = false;
            foreach (var item in owner.AnimationSetsWithDefault.SelectMany(x => x.Animations))
            {
                if (item.Name == document.FileName && item.Directory.Equals(document.FileLocation, StringComparison.OrdinalIgnoreCase))
                {
                    contains = true;
                }
                else
                {
                    var itemName = Path.GetFileNameWithoutExtension(item.Name);
                    if (name == itemName)
                    {
                        var itemExt = Path.GetExtension(item.Name);
                        var id = ObjectIDUtility.ExtToId(itemExt);
                        if (id == document.ObjectID)
                        {
                            sameName = item.Name;
                        }
                    }
                }
            }

            return contains ? sameName: null;
        }

        private void SetFilePath(Document document, string filePath, EditCommandSet commandSet)
        {
            if (filePath != document.FilePath)
            {
                using (var viewerBlock = new Viewer.ViewerDrawSuppressBlock())
                using (var block = new App.AppContext.PropertyChangedSuppressBlock())
                {
                    string fileName = Path.GetFileName(filePath);
                    string directoryName = Path.GetDirectoryName(filePath);
                    Document[] referenceDocuments = new Document[0];
                    if (document is Texture)
                    {
                        referenceDocuments = DocumentManager.DocumentsWithoutProject.Where(x => x.ReferenceDependDocuments.Contains(document)).ToArray();
                    }
                    else if (document is AnimationDocument)
                    {
                        foreach (var hasAnimation in DocumentManager.HasModelAnimations)
                        {
                            if (hasAnimation.Animations.Any(
                                x => x.Name == document.FileName && x.Directory.Equals(document.FileLocation, StringComparison.OrdinalIgnoreCase)))
                            {
                                commandSet.Add(
                                    DocumentManager.CreateAnimationsEditCommand(hasAnimation, hasAnimation.Animations.Select(
                                    x => x.Name == document.FileName && x.Directory.Equals(document.FileLocation, StringComparison.OrdinalIgnoreCase) ?
                                        new AnimationSetItem(fileName, directoryName) : x).ToArray(), false).Execute());
                            }
                        }

                        foreach (var hasAnimation in DocumentManager.SceneAnimationSets.Concat(Enumerable.Repeat(DocumentManager.DefaultSceneAnimationSet, 1)))
                        {
                            if (hasAnimation.Animations.Any(
                                x => x.Name == document.FileName && x.Directory.Equals(document.FileLocation, StringComparison.OrdinalIgnoreCase)))
                            {
                                commandSet.Add(
                                    DocumentManager.CreateAnimationsEditCommand(hasAnimation, hasAnimation.Animations.Select(
                                    x => x.Name == document.FileName && x.Directory.Equals(document.FileLocation, StringComparison.OrdinalIgnoreCase) ?
                                        new AnimationSetItem(fileName, directoryName) : x).ToArray(), false).Execute());
                            }
                        }
                    }

                    commandSet.Add(DocumentManager.CreatePathChangeCommand(document, filePath).Execute());

                    // 参照情報の更新
                    {
                        var models = referenceDocuments.OfType<Model>();
                        if (models.Any())
                        {
                            commandSet.Add(MaterialSamplerPage.CreateEditCommand_RemoveTexturePaths(new GuiObjectGroup(models), GuiObjectID.Model).Execute());
                        }

                        var texturePatterns = referenceDocuments.OfType<TexturePatternAnimation>();
                        if (texturePatterns.Any())
                        {
                            commandSet.Add(TexturePatternAnimationPatternPage.CreateEditCommand_RemoveTexturePaths(new GuiObjectGroup(texturePatterns)).Execute());
                        }
                    }


                    // Undo 専用
                    if (document is IntermediateFileDocument)
                    {
                        var doc = (IntermediateFileDocument)document;
                        commandSet.Add(DocumentManager.CreateFileInfoEditCommand(doc, doc.file_info).Execute());
                    }
                    else if (document is ProjectDocument)
                    {
                        var doc = (ProjectDocument)document;
                        commandSet.Add(DocumentManager.CreateFileInfoEditCommand(doc, doc.FileInfo).Execute());
                    }
                }

                TheApp.MainFrame.FileAdded(document);
            }
        }

        /// <summary>
        /// ファイル出力
        /// </summary>
        /// <remarks>例外を飛ばす可能性があります。</remarks>
        public bool WriteDocument(Document document, string filePath, bool updateFileInfo, bool outputRawFileInfo = false)
        {
            Debug.Assert(document != null);

            // 出力できるかどうか？
            if (document.IsFileOutputable == false)
            {
                // できなければなにもしないで抜ける。
                throw new FileNotOutputException(document.FileNotOutputErrorMessage);
            }

            using (var watch = new DebugStopWatch("Save " + document.FilePath))
            {
                if (document is IntermediateFileDocument)
                {
                    IntermediateFileDocument intermediateFile = (IntermediateFileDocument)document;
                    nw4f_3difType nw4f_3dif = intermediateFile.Create_nw4f_3difType();

                    if (updateFileInfo)
                    {
                        nw4f_3dif.file_info = intermediateFile.file_info;

                        if (nw4f_3dif.file_info != null)
                        {
                            if (nw4f_3dif.file_info.create == null)
                            {
                                IfFileLogUtility.SetCreate(nw4f_3dif, string.Empty);
                                intermediateFile.file_info = nw4f_3dif.file_info;
                            }
                            else
                            {
                                IfFileLogUtility.SetModify(nw4f_3dif);
                            }
                        }
                    }
                    else if (outputRawFileInfo)
                    {
                        // 変更なしで出力されるようにする
                        if (intermediateFile.file_info != null && intermediateFile.file_info.create != null)
                        {
                            nw4f_3dif.file_info = intermediateFile.file_info;
                        }
                    }

// デバッグ用
#if false
                    {
                        // 加工しやすいようにアトリビュート単位で改行する
                        XmlWriterSettings writerSettings = new XmlWriterSettings();
                        writerSettings.Indent = true;
                        //writerSettings.IndentChars = "";
                        writerSettings.NewLineOnAttributes = true;

                        // 文字列へシリアライズする
                        using (XmlWriter xmlWriter = XmlWriter.Create(filePath + ".txt", writerSettings))
                        {
                            XmlSerializer serializer = new XmlSerializer(typeof(nw4f_3difType));
                            serializer.Serialize(xmlWriter, nw4f_3dif);
                        }
                    }
#endif

                    //if (G3dPath.IsTextPath(filePath))
                    //{
                    //    if (!G3dStreamUtility.HasStreamArray(nw4f_3dif))
                    //    {
                    //        nw4f_3dif.RootElement.stream_array =
                    //            G3dStreamUtility.ToStreamArray(intermediateFile.BinaryStreams);
                    //    }
                    //    IfWriteUtility.Write(nw4f_3dif, filePath);
                    //}
                    //else
                    {
                        List<G3dStream> streams = intermediateFile.BinaryStreams;
                        if (G3dStreamUtility.HasStreamArray(nw4f_3dif))
                        {
                            nw4f_3dif.RootElement.stream_array = null;
                        }

                        // 書き込み中は監視を無効化する。
                        var watching = string.Equals(filePath, document.FilePath, StringComparison.OrdinalIgnoreCase) && document.IsWatched;
                        if (watching)
                        {
                            document.SuspendWatcher();
                        }

                        IfWriteUtility.Write(nw4f_3dif, streams, filePath);

                        // 書き込みが終了したので監視を再開する。
                        if (watching)
                        {
                            document.ResumeWatcher();
                        }
                    }
                }
                else if (document is ProjectDocument)
                {
                    // プロジェクトファイルの保存
                    var data = ((ProjectDocument)document).GetProject(filePath);
                    if (data == null)
                    {
                        return false;
                    }

                    using (FileStream fileStream = new FileStream(filePath, FileMode.Create))
                    {
                        XmlWriterSettings xmlWriterSettings = new XmlWriterSettings()
                        {
                            Encoding = new UTF8Encoding(true),			// BOM付き
                            Indent = true,
                            IndentChars = "\t",
                            CloseOutput = false,
                        };

                        XmlWriter xmlWriter = XmlTextWriter.Create(fileStream, xmlWriterSettings);
                        XmlSerializerNamespaces ns = new XmlSerializerNamespaces();
                        ns.Add(string.Empty, string.Empty);
                        (new XmlSerializer(typeof(Project), "")).Serialize(xmlWriter, data, ns);
#if false
                            fileStream.Seek(0, SeekOrigin.Begin);
                            StreamReader reader = new StreamReader(fileStream);
                            DebugConsole.WriteLine(reader.ReadToEnd());
#endif
                    }

                    // 保存した状態を更新する
                    ((ProjectDocument)document).savedProject = data;
                }
                else
                {
                    Debug.Assert(false);
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// 依存ファイルも含めて保存
        /// </summary>
        public bool SaveWithChildren(Document document)
        {
            return SaveWithChildren(Enumerable.Repeat(document, 1));
        }

        /// <summary>
        /// 依存ファイルも含めて保存
        /// </summary>
        public bool SaveWithChildren(IEnumerable<Document> documents)
        {
            var saveDocuments = new List<Document>(documents);
            {
                for (int i = 0; i < saveDocuments.Count; i++)
                {
                    saveDocuments.AddRange(saveDocuments[i].ReferenceDocuments.Where(x => x.Editable && !saveDocuments.Contains(x)));
                }
            }

            var selectedDocuments = App.AppContext.NotificationHandler.SaveDirectoryDialog(saveDocuments);

            if (selectedDocuments == null)
            {
                return false;
            }
            return SaveDocuments(selectedDocuments);
        }

        public bool SaveDocument(Document document)
        {
            return SaveDocuments(Enumerable.Repeat(document, 1));
        }

        public bool SaveDocumentAs(Document document)
        {
            return SaveDocuments(new [] {new Tuple<Document, string>(document, string.Empty)});
        }

        public bool SaveDocuments(IEnumerable<Document> documents, EditCommandSet commandSet = null)
        {
            return SaveDocuments(documents.Select(x => new Tuple<Document, string>(x, x.FilePath)).ToArray(), commandSet);
        }
        /// <summary>
        /// 複数のファイルの保存
        /// </summary>
        public bool SaveDocuments(IEnumerable<Tuple<Document, string>> documents, EditCommandSet commandSet = null)
        {
            // クリア処理があるのでこの条件が必要
            Debug.Assert(commandSet == null || commandSet.CommandCount == 0);

            var externalCommandSet = commandSet != null;
            if (!externalCommandSet)
            {
                commandSet = new EditCommandSet();
            }

            if (documents.Any(x => x.Item1.ObjectID == GuiObjectID.Project))
            {
                var notSavedFiles =
                    DocumentManager.Documents.Where(x => string.IsNullOrEmpty(x.FileLocation))
                        .Except(documents.Select(x => x.Item1))
                        .ToArray();
                if (notSavedFiles.Any())
                {
                    if (App.AppContext.NotificationHandler.OkCancelListBoxDialog(
                        MessageContext.NoFileLocation,
                        TheApp.AssemblyName,
                        Strings.IO_Save_following_Items,
                        App.AppContext.NotificationHandler.StringFormat.UnsavedFile,
                        notSavedFiles))
                    {
                        documents =
                            documents.Concat(notSavedFiles.Select(x => new Tuple<Document, string>(x, string.Empty)))
                                .ToArray();
                    }
                }
            }

            var originalPaths = new Dictionary<Document, string>();
            foreach (var document in documents)
            {
                originalPaths.Add(document.Item1, document.Item1.FilePath);
            }

            // 保存のチェックにファイル間の依存関係があるので種類ごとにパスを決める
            var textures = new List<Tuple<Document, string>>();
            var models = new List<Tuple<Document, string>>();
            var materialDocs = new List<Tuple<Document, string>>();
            var others = new List<Tuple<Document, string>>();
            Tuple<Document, string> project = null;
            int totalCount = 0;
            foreach (var document in documents)
            {
                switch (document.Item1.ObjectID)
                {
                    case GuiObjectID.Project:
                        project = document;
                        break;
                    case GuiObjectID.Model:
                        models.Add(document);
                        break;
                    case GuiObjectID.SeparateMaterial:
                        materialDocs.Add(document);
                        break;
                    case GuiObjectID.Texture:
                        textures.Add(document);
                        break;
                    default:
                        others.Add(document);
                        break;
                }
                totalCount++;
            }

            var savingDocuments = new List<Document>();

            using (var block = new App.AppContext.PropertyChangedSuppressBlock())
            {
                // テクスチャのパス設定
                savingDocuments.AddRange(
                    SetFilePaths(textures, commandSet, 0, totalCount));

                // モデルのパス設定
                savingDocuments.AddRange(
                    SetFilePaths(models, commandSet, textures.Count, totalCount));

                // マテリアルファイルのパス設定
                savingDocuments.AddRange(
                    SetFilePaths(materialDocs, commandSet, textures.Count + models.Count, totalCount));

                // その他のファイルのパス設定
                savingDocuments.AddRange(
                    SetFilePaths(others, commandSet, textures.Count + models.Count + materialDocs.Count, totalCount));

                // プロジェクトのパス設定
                if (project != null)
                {
                    if (!DocumentManager.DocumentsWithoutProject.Any(x => string.IsNullOrEmpty(x.FileLocation)))
                    {
                        if (SetSavePath(project.Item1, project.Item2, commandSet))
                        {
                            savingDocuments.Add(project.Item1);
                        }
                    }
                }
            }

            if (savingDocuments.Count == 0)
            {
                return false;
            }

            {
                var newPaths = new List<string>();
                foreach (var document in savingDocuments)
                {
                    var original = originalPaths[document];
                    if (string.IsNullOrEmpty(original) ||
                        string.Compare(Path.GetFullPath(original), Path.GetFullPath(document.FilePath), true) != 0)
                    {
                        newPaths.Add(document.FilePath);
                    }
                }

                // セーブデータを格納
                foreach (var document in savingDocuments)
                {
                    DocumentManager.AddSaveDocument(document);
                }

                // ファイルがないものに対しては PreOpenCommand は実行しない
                var existingPaths = newPaths.Where(x => File.Exists(x)).ToArray();
                if (existingPaths.Any())
                {
                    if (!PrePostIO.ExecutePreOpen(existingPaths))
                    {
                        // 元に戻す
                        commandSet.Reverse();
                        commandSet.Execute();
                        return false;
                    }
                }
            }

            var saved = WriteDocuments(savingDocuments.ToArray(), commandSet);

            // モデルとテクスチャが同時にセーブされた場合はテクスチャ参照を修正する。
            foreach (var doc in models)
            {
                var model = doc.Item1 as Model;
                Debug.Assert(model != null, "model != null");
                var savedTextures =
                    originalPaths.Where(pair => pair.Key is Texture && model.ReferenceTexturePaths.ContainsValue(pair.Value)).ToList();
                var samplers = model.Materials.SelectMany(x => x.sampler_array.sampler);
                foreach (var sampler in samplers)
                {
                    var tex = savedTextures.FirstOrDefault(pair => ((Texture)pair.Key).Name == sampler.tex_name);
                    if (tex.Key != null)
                    {
                        commandSet.Add(
                            MaterialSamplerPage.CreateEditCommand_ReferenceTexturePath(new GuiObjectGroup(model),
                                GuiObjectID.Model, sampler.tex_name, tex.Key.FilePath).Execute());
                    }
                }
            }

            // マテリアルファイルとテクスチャが同時にセーブされた場合はテクスチャ参照を修正する。
            foreach (var doc in materialDocs)
            {
                var materialDoc = doc.Item1 as SeparateMaterial;
                Debug.Assert(materialDoc != null, "materialDoc != null");
                var savedTextures =
                    originalPaths.Where(pair => pair.Key is Texture && materialDoc.ReferenceTexturePaths.ContainsValue(pair.Value)).ToList();
                var samplers = materialDoc.Materials.SelectMany(x => x.sampler_array.sampler);
                foreach (var sampler in samplers)
                {
                    var tex = savedTextures.FirstOrDefault(pair => ((Texture)pair.Key).Name == sampler.tex_name);
                    if (tex.Key != null)
                    {
                        commandSet.Add(
                            MaterialSamplerPage.CreateEditCommand_ReferenceTexturePath(new GuiObjectGroup(materialDoc),
                                GuiObjectID.SeparateMaterial, sampler.tex_name, tex.Key.FilePath).Execute());
                    }
                }
            }

            // テクスチャパターンアニメーションまたはマテリアルアニメーションとテクスチャが同時にセーブされた場合はテクスチャ参照を修正する。
            foreach (var doc in others.Where(tuple => tuple.Item1.ObjectID == GuiObjectID.MaterialAnimation || tuple.Item1.ObjectID == GuiObjectID.TexturePatternAnimation))
            {
                var texPatAnim = doc.Item1 as TexturePatternAnimation;
                var mtlAnim = doc.Item1 as MaterialAnimation;
                Debug.Assert(texPatAnim != null || mtlAnim != null, "texPatAnim != null || mtlAnim != null");
                var referenceTexturePaths = (texPatAnim != null)
                                            ? texPatAnim.ReferenceTexturePaths
                                            : mtlAnim.ReferenceTexturePaths;
                var texPatterns = (texPatAnim != null)
                                            ? texPatAnim.TexPatterns
                                            : mtlAnim.TexPatterns;
                var savedTextures =
                    originalPaths.Where(pair => pair.Key is Texture && referenceTexturePaths.ContainsValue(pair.Value)).ToList();
                foreach (var pattern in texPatterns)
                {
                    var tex = savedTextures.FirstOrDefault(pair => ((Texture)pair.Key).Name == pattern.tex_name);
                    if (tex.Key != null)
                    {
                        if (texPatAnim != null)
                        {
                            commandSet.Add(
                                TexturePatternAnimationPatternPage.CreateEditCommand_ReferenceTexturePath(
                                    new GuiObjectGroup(texPatAnim), pattern.tex_name, tex.Key.FilePath).Execute());
                        }
                        else
                        {
                            commandSet.Add(
                                MaterialAnimationPatternPage.CreateEditCommand_ReferenceTexturePath(
                                    new GuiObjectGroup(mtlAnim), pattern.tex_name, tex.Key.FilePath).Execute());
                        }
                    }
                }
            }

            {
                var newPaths = new List<Tuple<Document, string>>();
                var oldPaths = new List<Tuple<Document, string>>();
                foreach (var document in savingDocuments)
                {
                    var original = originalPaths[document];
                    if (string.IsNullOrEmpty(original))
                    {
                        newPaths.Add(Tuple.Create(document, document.FilePath));
                    }
                    else if (Path.GetFullPath(original).ToLower() != Path.GetFullPath(document.FilePath).ToLower())
                    {
                        newPaths.Add(Tuple.Create(document, document.FilePath));
                        oldPaths.Add(Tuple.Create(document, original));
                    }
                }

                if (newPaths.Any())
                {
                    commandSet.canRedo += () => PrePostIO.ExecutePreOpenUndoRedo(newPaths, false);
                }

                if (oldPaths.Any())
                {
                    commandSet.canUndo += () => PrePostIO.ExecutePreOpenUndoRedo(oldPaths, true);
                }
            }

            if (commandSet.CommandCount > 0)
            {
                commandSet.Reverse();
                if (!externalCommandSet)
                {
                    TheApp.CommandManager.Add(commandSet);
                }
            }

            if (saved && ApplicationConfig.UserSetting.EditHistory.ClearAfterFileSaved && !externalCommandSet)
            {
                TheApp.CommandManager.Clear();
            }

            return saved;
        }

        private IEnumerable<Document> SetFilePaths(
            IEnumerable<Tuple<Document, string>> others,
            EditCommandSet commandSet,
            int initialCount,
            int totalCount)
        {
            if (others.Any(x => string.IsNullOrEmpty(x.Item2) ||
                (File.Exists(x.Item2) && (File.GetAttributes(x.Item2) & FileAttributes.ReadOnly) != 0) ||
                x.Item1.NeedToAskBeforeSave(x.Item2)))
            {
                using (var indicator = new DocumentSaverStatusIndicator(this, totalCount, initialCount))
                {
                    foreach (var document in others)
                    {
                        if (SetSavePath(document.Item1, document.Item2, commandSet))
                        {
                            yield return document.Item1;
                        }
                    }
                }
            }
            else
            {
                foreach (var document in others)
                {
                    SetFilePath(document.Item1, document.Item2, commandSet);
                    yield return document.Item1;
                }
            }
        }

        static public bool PreSaveCommand(string[] paths)
        {
            DebugConsole.WriteLine("PreSaveCommand");
            return PrePostIO.ExecutePreSave(paths);
        }

        static public bool PostSaveCommand(string[] paths)
        {
            DebugConsole.WriteLine("PostSaveCommand");
            return PrePostIO.ExecutePostSave(paths);
        }

        private bool WriteDocuments(Document[] documents, EditCommandSet commandSet)
        {
            // 保存前コマンド
            if (!PreSaveCommand(documents.Select(x => x.FilePath).ToArray()))
            {
                return false;
            }

            List<Document> savedDocuments = new List<Document>();
            if (documents.Any(x => File.Exists(x.FilePath) && (File.GetAttributes(x.FilePath) & FileAttributes.ReadOnly) != 0))
            {
                foreach (var document in documents)
                {
                    var path = document.FilePath;
                    if (SetSavePath(document, document.FilePath, commandSet, true))
                    {
                        if (path != document.FilePath)
                        {
                            // オープン前コマンド
                            if (File.Exists(document.FilePath) && !PrePostIO.ExecutePreOpen(new[] { document.FilePath }))
                            {
                                continue;
                            }

                            // 保存前コマンド
                            if (!PreSaveCommand(new[] { document.FilePath }))
                            {
                                continue;
                            }
                        }

                        try
                        {
                            // 書き込み権限がない
                            if (FileUtility.HasWritePermission(document.FilePath) == false)
                            {
                                App.AppContext.NotificationHandler.MessageBoxWarning(document.FilePath + "\n" + Strings.IO_BinaryResourceFilenameConnotWriteAccessPermission);
                                continue;
                            }

                            // 保存
                            using (WaitCursor wait = new WaitCursor())
                            {
                                if (WriteDocument(document, document.FilePath, !_disableUpdateFileInfo, _disableUpdateFileInfo))
                                {
                                    AfterDocumentSaved(document);
                                    OnDocumentSaved(EventArgs.Empty);
                                    savedDocuments.Add(document);
                                }
                            }
                        }
                        catch (FileNotOutputException e)
                        {
                            App.AppContext.NotificationHandler.MessageBoxError(res.Strings.IO_SaveFailed + "\n" + e.Message, document.FilePath);
                        }
                        catch (Exception e)
                        {
#if DEBUG
                            App.AppContext.NotificationHandler.MessageBoxError(res.Strings.IO_SaveFailed + "\n" + e.Message + "\n" + e.StackTrace, document.FilePath);
#else
                        App.AppContext.NotificationHandler.MessageBoxError(res.Strings.IO_SaveFailed + "\n" + e.Message, document.FilePath);
#endif
                        }
                    }
                }
            }
            else
            {
                var written = new bool[documents.Length];
                var exceptions = new Exception[documents.Length];

                // パラレル保存
                using(WaitCursor wait = new WaitCursor())
                using (var parallelSave = new DebugStopWatch("parallelSave"))
                {
                    G3dParallel.For(0, documents.Length,
                        i =>
                        {
                            try
                            {
                                // 書き込み権限がない
                                if (FileUtility.HasWritePermission(documents[i].FilePath) == false)
                                {
                                    throw new Exception(Strings.IO_BinaryResourceFilenameConnotWriteAccessPermission);
                                }

                                written[i] = WriteDocument(documents[i], documents[i].FilePath, !_disableUpdateFileInfo, _disableUpdateFileInfo);
                            }
                            catch (Exception e)
                            {
                                exceptions[i] = e;
                            }
                        });
                }

                for (int i = 0; i < documents.Length; i++)
                {
                    if (exceptions[i] != null)
                    {
                        var e = exceptions[i];
                        var filePath = documents[i].FilePath;

                        if (e is FileNotOutputException)
                        {
                            App.AppContext.NotificationHandler.MessageBoxError(res.Strings.IO_SaveFailed + "\n" + e.Message, filePath);
                        }
                        else
                        {
#if DEBUG
                            App.AppContext.NotificationHandler.MessageBoxError(res.Strings.IO_SaveFailed + "\n" + e.Message + "\n" + e.StackTrace, filePath);
#else
                                App.AppContext.NotificationHandler.MessageBoxError(res.Strings.IO_SaveFailed + "\n" + e.Message, filePath);
#endif
                        }
                    }

                    if (written[i])
                    {
                        AfterDocumentSaved(documents[i]);
                        OnDocumentSaved(EventArgs.Empty);
                        savedDocuments.Add(documents[i]);
                    }
                }
            }

            // 保存後コマンド
            PostSaveCommand(savedDocuments.Select(x => x.FilePath).ToArray());

            return savedDocuments.Any();
        }

        private void AfterDocumentSaved(Document document)
        {
            // 保存済みデータを更新します。
            document.UpdateSavedData();

            document.SetContentsNotModified();

            document.UpdateWatcher();

            var saved = new DocumentContentsChangedArgs(document, null);
            App.AppContext.NotifyPropertyChanged(null, saved);
            App.AppContext.NotifyDocumentSaved(document);
        }

        /// <summary>
        /// 保存(1ファイル)。
        /// </summary>
        public bool BinarySave(Document document, List<string> errMsgs, TeamConfig.PlatformPreset selectedPlatformPreset)
        {
            string outPath;
            var doc = document as IntermediateFileDocument;
            if (doc == null)
            {
                return false;	// まぁ、こんなことは無いはず。
            }

            if (!doc.ConfirmBeforeBinarySave())
            {
                return false;
            }

            {
                var filename = App.AppContext.NotificationHandler.SaveFileDialog(
                    string.Format("{0} (*.{1})|*.{1}", Strings.FileFilter_Bfres, Const.Bfres),
                    Const.Bfres,
                    Strings.DlgTitle_BinarySave);

                if (string.IsNullOrEmpty(filename))
                {
                    return true;
                }

                outPath = filename;
            }

            // 保存
            using (var wait = new WaitCursor())
            {
                if (!PreSaveCommand(new[] { outPath }))
                {
                    return false;
                }
                string filePath = BinaryConverterManager.ConvertToBinary(doc, doc.FilePath, outPath, Const.Bfres, errMsgs, false, selectedPlatformPreset);
                if (string.IsNullOrEmpty(filePath))
                {
                    // バイナリ化失敗エラーメッセージ
                    DebugConsole.WriteLine("Binarize failed  : {0}", doc.FilePath);
                    return false;
                }
                PostSaveCommand(new[] { outPath });
            }

            return true;
        }

        /// <summary>
        /// 保存(複数ファイル)。
        /// </summary>
        public bool BinarySave(List<Document> documents, string BinaryfilePath, List<string> errMsgs, TeamConfig.PlatformPreset platformPreset)
        {
            using (WaitCursor wait = new WaitCursor())
            {
                List<IntermediateFileDocument> docs = new List<IntermediateFileDocument>();
                foreach(var document in documents)
                {
                    IntermediateFileDocument doc = document as IntermediateFileDocument;
                    if (doc != null)
                    {
                        docs.Add(doc);
                    }
                }

                string filePath = App.BinaryConverterManager.ConvertToBinary(docs, BinaryfilePath, errMsgs, false, platformPreset);
                if (string.IsNullOrEmpty(filePath) || errMsgs.Any())
                {
                    return false;
                }
                PostSaveCommand(new[] { BinaryfilePath });
            }

            return true;
        }

        [Serializable]
        private class FileNotOutputException : Exception
        {
            public FileNotOutputException(string message)
                : base(message)
            {}
        }
    }

    /// <summary>
    /// ドキュメントセーバステータスインジケータクラス。
    /// </summary>
    public sealed class DocumentSaverStatusIndicator : IDisposable
    {
        private readonly bool nested = false;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public DocumentSaverStatusIndicator(DocumentSaver saver, int stepCount, int init = 0)
        {
            if (!TheApp.MainFrame.IsHandleCreated) return;

            saver.DocumentSaved += (s, e) => ForwardProgress();

            // プログレスバー表示
            UIToolStripProgressBar progress = TheApp.MainFrame.StatusProgressBar;
            if (progress.Visible)
            {
                nested = true;
            }
            else
            {
                progress.Maximum = stepCount;
                progress.Value = init;
                progress.Visible = true;
            }
        }

        /// <summary>
        /// 破棄。
        /// </summary>
        public void Dispose()
        {
            if (!TheApp.MainFrame.IsHandleCreated) return;

            // プログレスバー非表示
            if (!nested)
            {
                UIToolStripProgressBar progress = TheApp.MainFrame.StatusProgressBar;
                progress.Visible = false;
                progress.Maximum = 1;
                progress.Value = 0;
            }
        }

        /// <summary>
        /// ステップを進める。
        /// </summary>
        private void ForwardProgress()
        {
            UIToolStripProgressBar progress = TheApp.MainFrame.StatusProgressBar;
            progress.PerformStep();
        }
    }
}
