﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Xml;
using System.Xml.Serialization;
using App.Utility;
using nw.g3d.nw4f_3dif;
using nw.g3d.iflib;
using nw.g3d.iflib.nw3de;
namespace App.Data
{
    public class ToolData
    {
        public ToolData()
        {
            EditComment = "";
            CustomUI = new CustomUI();
            SearchPaths = new List<nw3de_SearchPath.SearchPath>();
        }

        public Color? EditColor	{	get; set;	}
        public string EditComment{	get; set;	}

        public CustomUI CustomUI { get; set; }

        public List<nw3de_SearchPath.SearchPath> SearchPaths;

        // マテリアル参照情報
        public nw3de_MaterialReference MaterialReference { get; set; }

        // コンバイナーオプション情報
        public nw3de_CombinerOptionInfo CombinerOptionInfo { get; set; }

        private tool_dataType orgData_;

        [Serializable]
        public class nw3de_editdata
        {
            public nw3de_editdata()
            {
                comment = "";
            }

            [XmlAttribute]
            public string color	{	get; set;	}

            [XmlAttribute]
            public string comment{	get; set;	}

            // XML出力制御用
            [XmlIgnore]	public bool colorSpecified{   get { return string.IsNullOrEmpty(color)   == false; } }
            [XmlIgnore]	public bool commentSpecified{ get { return string.IsNullOrEmpty(comment) == false; } }
        };

        public class nw3de_CustomUI
        {
            [XmlIgnore]
            public bool option_varsSpecified { get { return option_vars.Any(); } }
            public List<CustomItem> option_vars = new List<CustomItem>();

            [XmlIgnore]
            public bool attrib_varsSpecified { get { return attrib_vars.Any(); } }
            public List<CustomItem> attrib_vars  = new List<CustomItem>();

            [XmlIgnore]
            public bool sampler_varsSpecified { get { return sampler_vars.Any(); } }
            public List<CustomItem> sampler_vars  = new List<CustomItem>();

            [XmlIgnore]
            public bool uniform_varsSpecified { get { return uniform_vars.Any(); } }
            public List<CustomItem> uniform_vars  = new List<CustomItem>();

            [XmlIgnore]
            public bool render_info_slotsSpecified { get { return render_info_slots.Any(); } }
            public List<CustomItem> render_info_slots  = new List<CustomItem>();

            [XmlIgnore]
            public bool groupsSpecified { get { return groups.Any(); } }
            public List<CustomItem> groups = new List<CustomItem>();

            [XmlIgnore]
            public bool colorControlsSpecified { get { return colorControls.Any(); } }
            public List<string> colorControls = new List<string>();

            public bool AnyElements()
            {
                return option_varsSpecified ||
                    attrib_varsSpecified ||
                    sampler_varsSpecified ||
                    uniform_varsSpecified ||
                    render_info_slotsSpecified ||
                    groupsSpecified ||
                    colorControlsSpecified;
            }
        }

        public void Load(tool_dataType data, string materialName = null)
        {
            // 元のデータを保存する
            orgData_ = data;

            // 各データを読み込む
            if (data != null)
            {
                foreach (var elem in data.Any)
                {
                    // 編集データ
                    if (elem.Name == typeof(nw3de_editdata).Name)
                    {
                        var editData = IfToolData.ConvertToEditData<nw3de_editdata>(elem);

                        EditColor   = editData.colorSpecified ? (Color?)ColorTranslator.FromHtml(editData.color) : (Color?)null;
                        EditComment = editData.comment;
                    }
                    else if (elem.Name == typeof(nw3de_CustomUI).Name)
                    {
                        var customUI = IfToolData.ConvertToEditData<nw3de_CustomUI>(elem);
                        CustomUI = convertToCustomUI(customUI);
                    }
                    else if (elem.Name == typeof(nw3de_SearchPath).Name)
                    {
                        var searchPath = IfToolData.ConvertToEditData<nw3de_SearchPath>(elem);
                        SearchPaths = searchPath.SearchPaths;
                    }
                    else if (elem.Name == typeof(nw3de_CombinerOptionInfo).Name)
                    {
                        CombinerOptionInfo = IfToolData.ConvertToEditData<nw3de_CombinerOptionInfo>(elem);
                    }
                    else if (elem.Name == typeof(nw3de_MaterialReference).Name)
                    {
                        MaterialReference = IfToolData.ConvertToEditData<nw3de_MaterialReference>(elem);
                    }
                }
            }
        }

        public static CustomUI convertToCustomUI(nw3de_CustomUI customUI)
        {
            return new CustomUI()
            {
                option_vars = customUI.option_vars.ToDictionary(x => x.id, x => x.label),
                attrib_vars = customUI.attrib_vars.ToDictionary(x => x.id, x => x.label),
                sampler_vars = customUI.sampler_vars.ToDictionary(x => x.id, x => x.label),
                uniform_vars = customUI.uniform_vars.ToDictionary(x => x.id, x => x.label),
                render_info_slots = customUI.render_info_slots.ToDictionary(x => x.id, x => x.label),
                groups = customUI.groups.ToDictionary(x => x.id, x => x.label),
                colorControls = new HashSet<string>(customUI.colorControls),
            };
        }

        public List<XmlElement> GetElements()
        {
            return (orgData_ != null) ?
                orgData_.Any.ToList() :
                new List<XmlElement>();
        }

        public void UpdateEditData(List<XmlElement> elements)
        {
            // editdataTagを削除
            elements.RemoveAll(x => x.Name == typeof(nw3de_editdata).Name);

            if ((EditColor != null) || (string.IsNullOrEmpty(EditComment) == false))
            {
                // デシリアライズ用のオブジェクトを作る
                var editData = new nw3de_editdata();
                {
                    if (EditColor != null)
                    {
                        editData.color = ColorTranslator.ToHtml((Color)EditColor);
                    }
                    editData.comment = EditComment;
                }

                // リストに加える
                elements.Add(IfToolData.ConvertToXmlElement(editData, 1));
            }
        }

        /// <summary>
        /// そのままSave
        /// </summary>
        public tool_dataType SaveDefault()
        {
            var elements = GetElements();
            return
                elements.Any() ?
                    new tool_dataType()
                    {
                        Any = elements.ToArray()
                    } :
                    null;
        }
    }

    public class CustomUI
    {
        public Dictionary<string, string> option_vars = new Dictionary<string, string>();
        public Dictionary<string, string> attrib_vars = new Dictionary<string, string>();
        public Dictionary<string, string> sampler_vars = new Dictionary<string, string>();
        public Dictionary<string, string> uniform_vars = new Dictionary<string, string>();
        public Dictionary<string, string> render_info_slots = new Dictionary<string, string>();
        public Dictionary<string, string> groups = new Dictionary<string, string>();
        public HashSet<string> colorControls = new HashSet<string>();

        public bool hasElement()
        {
            return option_vars.Any() ||
                attrib_vars.Any() ||
                sampler_vars.Any() ||
                uniform_vars.Any() ||
                render_info_slots.Any() ||
                groups.Any() ||
                colorControls.Any();
        }

        public CustomUI Clone()
        {
            return new CustomUI()
            {
                option_vars = option_vars.ToDictionary(x => x.Key, x => x.Value),
                attrib_vars = attrib_vars.ToDictionary(x => x.Key, x => x.Value),
                sampler_vars = sampler_vars.ToDictionary(x => x.Key, x => x.Value),
                uniform_vars = uniform_vars.ToDictionary(x => x.Key, x => x.Value),
                render_info_slots = render_info_slots.ToDictionary(x => x.Key, x => x.Value),
                groups = groups.ToDictionary(x => x.Key, x => x.Value),
                colorControls = new HashSet<string>(colorControls),
            };
        }

        public static CustomUI DummyCustomUI = new CustomUI();
    }

    public class CustomItem
    {
        [XmlAttribute]
        public string id { get; set; }

        [XmlAttribute]
        public string label { get; set; }
    }
}
