﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Xml.Serialization;
using App.Command;
using App.PropertyEdit;
using App.res;
using App.Utility;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace App.Data
{
    public sealed class ProjectDocument : Document
    {
        public const string ProjectVersion = "3.6.0.0";
        public const string DefaultName = "Project";
        public const string DefaultDotExt = ".f3pj";
        public ProjectDocument() : base(GuiObjectID.Project)
        {
            FileDotExt = DefaultDotExt;
            Name = DefaultName;
            SetFileLocation();
            FileInfo = new FileInfo();
            UpdateSavedData();
        }

        public FileInfo FileInfo { get; set; }

        public void UpdateFileInfo()
        {
            if (FileInfo == null)
            {
                return;
            }

            var log = new EditLog()
                {
                    user = Environment.UserName,
                    host = Environment.MachineName,
                    date_time = new DateTime(
                        DateTime.Now.Year, DateTime.Now.Month, DateTime.Now.Day,
                        DateTime.Now.Hour, DateTime.Now.Minute, DateTime.Now.Second,
                        DateTimeKind.Local),
                    tool_name = System.Windows.Forms.Application.ProductName,
                    tool_version = System.Windows.Forms.Application.ProductVersion,
                };
            if (FileInfo.create == null && FileInfo.modify == null)
            {
                FileInfo.create = log;
            }
            else
            {
                FileInfo.modify = log;
            }
        }

        // 保存確認対象でないプロジェクトかどうか
        public bool DefaultProject;
        public EditCommand DefaultProjectModifiedCommand;

        public void LoadToolData(nw.g3d.nw4f_3dif.tool_dataType toolData)
        {
            ToolData.Load(toolData);
        }

        public nw.g3d.nw4f_3dif.tool_dataType SaveToolData()
        {
            //return ToolData.SaveDefault();
            var elements = ToolData.GetElements();
            ToolData.UpdateEditData(elements);
            return
                elements.Any() ?
                    new tool_dataType()
                    {
                        Any = elements.ToArray()
                    } :
                    null;
        }

        public override IEnumerable<GuiObject> ContentObjects
        {
            get
            {
                return base.ContentObjects
                    .Concat(DocumentManager.AllSceneAnimationSets)
                    .Concat(DocumentManager.Models.SelectMany(x => x.AnimationSetsWithDefault));
            }
        }

        // テンポラリファイル(TGA)がプロジェクトに含まれている場合はセーブできないようにする
        public override bool IsFileOutputable
        {
            get { return DocumentManager.Textures.All(x => x.IsTemporary == false); }
        }

        public override string FileNotOutputErrorMessage
        {
            get
            {
                Debug.Assert(IsFileOutputable == false);
                return Strings.ProjectDocument_FileNotOutputErrorMessage;
            }
        }

        #region savedData
        public Project savedProject;
        public override void UpdateSavedData()
        {
            base.UpdateSavedData();
        }

        public override bool EqualsToSavedData()
        {
            if (!base.EqualsToSavedData())
            {
                return false;
            }

            if (savedProject == null)
            {
                return false;
            }

            if (string.IsNullOrEmpty(FileLocation))
            {
                return false;
            }

            if (DocumentManager.DocumentsWithoutProject.Any(x => string.IsNullOrEmpty(x.FileLocation)))
            {
                return false;
            }

            if (DocumentManager.DocumentsWithoutProject.Count() != savedProject.Items.Count)
            {
                return false;
            }

            foreach (var doc in DocumentManager.DocumentsWithoutProject)
            {
                if (string.IsNullOrEmpty(doc.FileLocation))
                {
                    return false;
                }

                string path;
                if (doc.BaseName != null)
                {
                    string basePath = ConfigData.ApplicationConfig.GetSearchPaths().FirstOrDefault(x => x.Name == doc.BaseName).path + "\\";
                    path = PathUtility.MakeRelativePath(basePath, doc.FilePath);
                }
                else if (doc.BasePath != null)
                {
                    var expanded = Environment.ExpandEnvironmentVariables(doc.BasePath) + "\\";
                    path = PathUtility.MakeRelativePath(expanded, doc.FilePath);
                }
                else
                {
                    path = PathUtility.MakeRelativePath(FilePath, doc.FilePath);
                }

                var item = savedProject.Items.FirstOrDefault(x => x.Path == path && x.Base == doc.BaseName && x.BasePath == doc.BasePath);
                if (item == null)
                {
                    return false;
                }

                if (doc is Model)
                {
                    var model = (Model)doc;
                    if ((model.PreviewAnimSet == model.DefaultAnimationSet) != item.PreviewDefaultAnimationSet)
                    {
                        return false;
                    }

                    if (ModelPreviewPage.IsModified(model)
                        || model.IsPreviewInfoValueChanged(x => x.Visible)
                        || model.IsPreviewInfoValueChanged(x => x.ShowInObjView))
                    {
                        return false;
                    }

                    if (model.AnimationSets.Count != item.AnimationSetItems.Count)
                    {
                        return false;
                    }

                    foreach (var animationSet in model.AnimationSets)
                    {
                        var animationSetItem = item.AnimationSetItems.FirstOrDefault(x => x.Name == animationSet.Name);
                        if (animationSetItem == null)
                        {
                            return false;
                        }

                        if ((model.PreviewAnimSet == animationSet) != animationSetItem.Preview)
                        {
                            return false;
                        }

                        if (!animationSetItem.AnimationItems.OrderBy(x => x.Name).ThenBy(x => x.Directory).Select(x => new AnimationSetItem(x.Name, x.Directory)).SequenceEqual(
                            animationSet.Animations.OrderBy(x => x.Name).ThenBy(x => x.Directory)))
                        {
                            return false;
                        }

                        foreach (var animationItem in animationSetItem.AnimationItems)
                        {
                            var animation = DocumentManager.GetAnimation(animationItem.Name, animationItem.Directory);
                            if (animation == null)
                            {
                                return false;
                            }

                            if (animation.Pause.InvisibleBinds.Contains(new KeyValuePair<object, string>(model.ModelId, animationSet.Name)) == animationItem.Preview)
                            {
                                return false;
                            }
                        }
                    }

                    // 既定アニメーションセット分の比較
                    foreach (var animationItem in item.AnimationItems)
                    {
                        var animation = DocumentManager.GetAnimation(animationItem.Name, animationItem.Directory);
                        if (animation == null)
                        {
                            return false;
                        }

                        if (animation.Pause.InvisibleBinds.Contains(new KeyValuePair<object, string>(model.ModelId, model.DefaultAnimationSet.Name)) == animationItem.Preview)
                        {
                            return false;
                        }
                    }

                    foreach (var material in model.Materials)
                    {
                        var materialItem = item.MaterialItems.FirstOrDefault(x => x.Name == material.Name);
                        if (materialItem == null)
                        {
                            return false;
                        }

                        if (MaterialPreviewPage.IsModified(material))
                        {
                            return false;
                        }
                    }
                }
                else if (doc is AnimationDocument)
                {
                    if (((AnimationDocument)doc).RetargetingHostModelName != item.RetargetingHostModelName)
                    {
                        return false;
                    }
                }
            }

            // シーンの比較
            if ((DocumentManager.PreviewSceneAnimSet == DocumentManager.DefaultSceneAnimationSet) != savedProject.Scene.PreviewDefaultAnimationSet)
            {
                return false;
            }

            if (DocumentManager.SceneAnimationSets.Count != savedProject.Scene.AnimationSetItems.Count)
            {
                return false;
            }

            foreach (var animationSet in DocumentManager.SceneAnimationSets)
            {
                var animationSetItem = savedProject.Scene.AnimationSetItems.FirstOrDefault(x => x.Name == animationSet.Name);
                if (animationSetItem == null)
                {
                    return false;
                }

                if ((DocumentManager.PreviewSceneAnimSet == animationSet) != animationSetItem.Preview)
                {
                    return false;
                }

                // TODO: AnimationSet
                if (!animationSetItem.AnimationItems.OrderBy(x => x.Name).ThenBy(x => x.Directory).Select(x => new AnimationSetItem(x.Name, x.Directory)).SequenceEqual(
                    animationSet.Animations.OrderBy(x => x.Name).ThenBy(x => x.Directory)))
                {
                    return false;
                }

                // アニメーションセット内のアニメーションの表示・非表示を比較。
                foreach (var animSetItem in animationSet.Animations)
                {
                    var animation = DocumentManager.GetAnimation(animSetItem.Name, animSetItem.Directory);
                    if (animation == null)
                    {
                        return false;
                    }

                    if (animation is SceneAnimation)
                    {
                        var animItem = animationSetItem.AnimationItems.FirstOrDefault(x => x.Name == animation.FileName);
                        if (animItem == null)
                        {
                            return false;
                        }

                        var sceneAnim = (SceneAnimation)animation;
                        if (sceneAnim.InvisibleBinds.Contains(animationSet) == animItem.Preview)
                        {
                            return false;
                        }
                    }
                }
            }

            // 既定アニメーションセット内のアニメーションの表示・非表示を比較。
            foreach (var animSetItem in DocumentManager.DefaultSceneAnimationSet.Animations)
            {
                var animation = DocumentManager.GetAnimation(animSetItem.Name, animSetItem.Directory);
                if (animation == null)
                {
                    return false;
                }

                if (animation is SceneAnimation)
                {
                    var animItem = savedProject.Scene.AnimationItems.FirstOrDefault(x => x.Name == animation.FileName);
                    if (animItem == null)
                    {
                        return false;
                    }

                    var sceneAnim = (SceneAnimation)animation;
                    if (sceneAnim.InvisibleBinds.Contains(DocumentManager.DefaultSceneAnimationSet) == animItem.Preview)
                    {
                        return false;
                    }
                }
            }

            return true;
        }

        public Model.PreviewInfoType GetSavedModelPreviewInfo(Model model)
        {
            var previewInfo = new Model.PreviewInfoType();
            if (savedProject != null)
            {
                var item = savedProject.Items.FirstOrDefault(x => Path.GetFileName(x.Path) == model.FileName);
                if (item != null && item.PreviewItem != null)
                {
                    item.PreviewItem.Load(previewInfo);
                }
            }
            return previewInfo;
        }

        public Project.MaterialItem GetSavedMaterialItem(Material material)
        {
            var materialOwnerDoc = material.OwnerDocument;
            if (savedProject != null)
            {
                var item = savedProject.Items.FirstOrDefault(x => Path.GetFileName(x.Path) == materialOwnerDoc.FileName);
                if (item != null)
                {
                    return item.MaterialItems.FirstOrDefault(x => x.Name == material.Name) ?? Project.DefaultMaterialItem;
                }
            }

            return Project.DefaultMaterialItem;
        }

        public string GetSavedSkeletalAnimationHostModelName(AnimationDocument animation)
        {
            if (savedProject != null)
            {
                var item = savedProject.Items.FirstOrDefault(x => x.Directory == animation.FileLocation && x.FileName == animation.FileName);
                if (item != null)
                {
                    return item.RetargetingHostModelName;
                }
            }

            return null;
        }

        public List<AnimationSetItem> GetSavedAnimationSet(AnimationSet animationSet)
        {
            if (savedProject == null)
            {
                return null;
            }

            if (animationSet.IsSceneAnimationSet)
            {
                if (animationSet.IsDefaultAnimationSet)
                {
                    return savedProject.Scene.AnimationItems.Select(x => new AnimationSetItem(x.Name, x.Directory)).ToList();
                }

                var item = savedProject.Scene.AnimationSetItems.FirstOrDefault(x => x.Name == animationSet.Name);
                if (item != null)
                {
                    return item.AnimationItems.Select(x => new AnimationSetItem(x.Name, x.Directory)).ToList();
                }
            }
            else
            {
                var model = DocumentManager.Models.FirstOrDefault(x => x.AnimationSetsWithDefault.Contains(animationSet));

                if (model == null)
                {
                    // 描画のタイミングによっては以前の animationSet に対する問い合わせが来ることがある
                    return null;
                }

                var modelItem = savedProject.Items.FirstOrDefault(x => Path.GetFileName(x.Path) == model.FileName);
                if (modelItem != null)
                {
                    if (animationSet.IsDefaultAnimationSet)
                    {
                        return modelItem.AnimationItems.Select(x => new AnimationSetItem(x.Name, x.Directory)).ToList();
                    }
                    else
                    {
                        var item = modelItem.AnimationSetItems.FirstOrDefault(x => x.Name == animationSet.Name);
                        if (item != null)
                        {
                            return item.AnimationItems.Select(x => new AnimationSetItem(x.Name, x.Directory)).ToList();
                        }
                    }
                }
            }

            return null;
        }

        #endregion

        public Project.Item CreateDocumentItem(string filePath, Document doc)
        {
            // ファイルパスが未確定
            if (string.IsNullOrEmpty(doc.FileLocation))
            {
                return null;
            }

            Project.RelativeType relative;
            string path = doc.GetProjectPath(out relative);
            /*
            if (doc.BaseName != null)
            {
                string basePath = ConfigData.ApplicationConfig.GetSearchPaths().FirstOrDefault(x => x.Name == doc.BaseName).path + "\\";
                path = PathUtility.MakeRelativePath(basePath, doc.FilePath);
            }
            else if (doc.BasePath != null)
            {
                var expanded = Environment.ExpandEnvironmentVariables(doc.BasePath) + "\\";
                path = PathUtility.MakeRelativePath(expanded, doc.FilePath);
            }
            else
            {
                path = PathUtility.MakeRelativePath(filePath, doc.FilePath);
            }

            // スタートアップフォルダ内のファイルか？
            var isInnerStartupFolder = Path.GetFullPath(doc.FilePath).Contains(Path.GetFullPath(DocumentManager.StartupFolderPath));

            // ・スタートアップフォルダ以下のファイルの場合は「StartUpFolder」
            // ・Base指定なしのファイルの場合は「Project」
            // ・Base指定ありのファイルの場合は「SearchPath」

            // スタートアップフォルダ内のファイルであればパスを調整する
            if (isInnerStartupFolder)
            {
                path = PathUtility.MakeRelativePath(Path.GetFullPath(DocumentManager.StartupFolderPath) + "\\", Path.GetFullPath(doc.FilePath));
            }*/

            bool previewDefaultAnimationSet = false;
            bool previewDefaultAnimationSetSpecified = false;
            string retargetingHostModel = null;
            if (doc.ObjectID == GuiObjectID.Model)
            {
                var model = (Model)doc;
                previewDefaultAnimationSet = model.PreviewAnimSet == model.DefaultAnimationSet;
                previewDefaultAnimationSetSpecified = true;
            }
            else if (doc is AnimationDocument)
            {
                retargetingHostModel = ((AnimationDocument)doc).RetargetingHostModelName;
            }

            return new Project.Item()
            {
                Path		= path,
                Base		= doc.BaseName,
                BasePath = doc.BasePath,
                Relative	= relative,
                PreviewDefaultAnimationSet = previewDefaultAnimationSet,
                PreviewDefaultAnimationSetSpecified = previewDefaultAnimationSetSpecified,
                Directory = doc.FileLocation,
                FileName = doc.FileName,
                RetargetingHostModelName = retargetingHostModel,
            };
        }

        // TODO: AnimationSet
        public Project GetProject(string filePath)
        {
            // プロジェクトファイルの保存
            var data = new Project();
            data.Version = ProjectVersion;
            DocumentManager.ProjectDocument.UpdateFileInfo();
            data.FileInfo = DocumentManager.ProjectDocument.FileInfo;
            data.tool_data = DocumentManager.ProjectDocument.SaveToolData();
            data.comment = GetComment();

            // シーン
            foreach (var animationSet in DocumentManager.SceneAnimationSets)
            {
                var sceneAnims = DocumentManager.GetAnimations(animationSet.Animations).Cast<SceneAnimation>().ToArray();

                data.Scene.AnimationSetItems.Add(
                    new Project.AnimationSetItem()
                    {
                        Name = animationSet.Name,
                        Preview = animationSet == DocumentManager.PreviewSceneAnimSet,
                        AnimationItems = new List<Project.AnimationItem>(
                            animationSet.Animations
                            .Select((x, i) => new Project.AnimationItem()
                            {
                                Name = x.Name,
                                Preview = !sceneAnims[i].InvisibleBinds.Contains(animationSet),
                                Directory = x.Directory
                            }))
                    });

                animationSet.UpdateSavedData();
                animationSet.SetNotModified();
            }

            {
                var sceneAnims = DocumentManager.GetAnimations(DocumentManager.DefaultSceneAnimationSet.Animations).Cast<SceneAnimation>().ToArray();

                data.Scene.AnimationItems = new List<Project.AnimationItem>(
                    DocumentManager.DefaultSceneAnimationSet.Animations
                    .Select((x, i) => new Project.AnimationItem()
                    {
                        Name = x.Name,
                        Preview = !sceneAnims[i].InvisibleBinds.Contains(DocumentManager.DefaultSceneAnimationSet),
                        Directory = x.Directory
                    }));
            }

            data.Scene.PreviewDefaultAnimationSet = DocumentManager.PreviewSceneAnimSet == DocumentManager.DefaultSceneAnimationSet;

            // ファイル
            foreach (var doc in DocumentManager.DocumentsWithoutProject)
            {
                Project.Item item = CreateDocumentItem(filePath, doc);

                if (item == null)
                {
                    return null;
                }

                // アニメーション
                if (doc.ObjectID == GuiObjectID.Model)
                {
                    Model model = (Model)doc;

                    // アニメーションセット
                    {
                        foreach (var animationSet in model.AnimationSets)
                        {
                            bool preview = (model.PreviewAnimSet == animationSet);
                            item.AnimationSetItems.Add(
                                new Project.AnimationSetItem()
                                {
                                    Name = animationSet.Name,
                                    Preview = preview,
                                    AnimationItems = new List<Project.AnimationItem>(
                                        animationSet.Animations.
                                        Select(x =>
                                            {
                                                var animation = DocumentManager.GetAnimation(x.Name, x.Directory);
                                                return new Project.AnimationItem()
                                                {
                                                    Name = x.Name,
                                                    Directory = x.Directory,
                                                    Preview = animation != null ?
                                                        !animation.Pause.InvisibleBinds.Contains(new KeyValuePair<object, string>(model.ModelId, animationSet.Name)) :
                                                        true,
                                                };
                                            })),
                                });

                            //animationSet.UpdateSavedData();
                            //animationSet.SetNotModified();
                        }
                    }

                    // アニメーション
                    {
                        item.AnimationItems = new List<Project.AnimationItem>(
                            model.DefaultAnimationSet.Animations.
                            Select(x =>
                                {
                                    var animation = DocumentManager.GetAnimation(x.Name, x.Directory);
                                    return new Project.AnimationItem()
                                {
                                    Name = x.Name,
                                    Directory = x.Directory,
                                    Preview = animation != null ?
                                        !animation.Pause.InvisibleBinds.Contains(new KeyValuePair<object, string>(model.ModelId, model.DefaultAnimationSet.Name)) :
                                        true,
                                };
                                }));
                    }

                    // プレビュ
                    {
                        if (model.PreviewInfo.IsEdited)
                        {
                            item.PreviewItem = new Project.PreviewItem();
                            item.PreviewItem.Store(model.PreviewInfo);
                        }
                    }

                    // マテリアル
                    foreach (var material in model.Materials)
                    {
                        item.MaterialItems.Add(
                            new Project.MaterialItem
                            {
                                Name = material.Name,
                                OptimizeShader = material.OptimizeShader,
                                OptimizeShaderOnReload = material.OptimizeShaderOnReload,
                            });
                    }
                }
                data.Items.Add(item);
            }

            // ItemIndex の設定
            foreach (var animationItem in data.Scene.AnimationItems.Concat(data.Scene.AnimationSetItems.SelectMany(x => x.AnimationItems))
                .Concat(data.Items.SelectMany(x => x.AnimationItems.Concat(x.AnimationSetItems.SelectMany(y => y.AnimationItems)))))
            {
                animationItem.ItemIndex = data.Items.FindIndex(x => Path.GetFileName(x.Path) == animationItem.Name && x.Directory == animationItem.Directory);
            }

            return data;
        }
    }

    public class Project
    {
        // Ver3.1.0
        //	1361: Startup フォルダから開かれたファイルをプロジェクトファイルに保存するときはパスを Startup フォルダからの相対にする
        //	http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1361

        [XmlAttribute]
        //      public string Version { get { return "2.3.0"; } set { } }
        //      public string Version { get { return "3.1.0"; } set { } }
        public string Version { get; set; }

        [XmlElement("file_info")]
        public FileInfo FileInfo;

        [XmlElement("Scene")]
        public SceneItem Scene;

        [XmlElement("Item")]
        public List<Item> Items;

        [XmlElement("tool_data")]
        public nw.g3d.nw4f_3dif.tool_dataType tool_data;

        [XmlElement("comment")]
        public nw.g3d.nw4f_3dif.commentType comment;

        public Project()
        {
            Scene = new SceneItem();
            Items = new List<Item>();
        }

        public class SceneItem
        {
            [XmlAttribute]
            public bool PreviewDefaultAnimationSet;

            [XmlElement("AnimationSet")]
            public List<AnimationSetItem> AnimationSetItems;

            [XmlElement("Animation")]
            public List<AnimationItem> AnimationItems;

            public SceneItem()
            {
                AnimationItems = new List<AnimationItem>();
                AnimationSetItems = new List<AnimationSetItem>();
                PreviewDefaultAnimationSet = true;
            }
        }

        public enum RelativeType
        {
            StartUpFolder, // スタートアップフォルダからの相対
            Project, // プロジェクトファイルからの相対
            SearchPath, // 追加参照パスからの相対
            StartUp, // (互換性のため残されている)追加参照パスからの相対
        }

        public class Item
        {
            public Item()
            {
                AnimationSetItems = new List<AnimationSetItem>();
                AnimationItems = new List<AnimationItem>();
                MaterialItems = new List<MaterialItem>();

                PreviewItem = null;						// モデル保存時に作る
                Base = null;
                BasePath = null;
                Relative = RelativeType.Project;
                PreviewDefaultAnimationSet = true;
                Directory = "";
                RetargetingHostModelName = null;

            }

            [XmlAttribute]
            public RelativeType Relative;

            [XmlAttribute]
            public string Base;

            [XmlAttribute]
            public string BasePath;

            [XmlAttribute]
            public string Path;

            [XmlAttribute]
            public bool PreviewDefaultAnimationSet;

            // スケルタルアニメーションのリターゲッティングホストモデル
            [XmlAttribute]
            public string RetargetingHostModelName;

            // PreviewDefaultAnimationSet のシリアル化制御
            [XmlIgnore]
            public bool PreviewDefaultAnimationSetSpecified;

            // 内部でファイルを識別するためのファイル名
            [XmlIgnore]
            public string FileName;

            // 内部でファイルで識別するための絶対パス
            [XmlIgnore]
            public string Directory;

            [XmlElement("AnimationSet")]
            public List<AnimationSetItem> AnimationSetItems;

            [XmlElement("Animation")]
            public List<AnimationItem> AnimationItems;

            [XmlElement("Preview")]
            public PreviewItem PreviewItem;

            [XmlElement("Material")]
            public List<MaterialItem> MaterialItems;
        }

        public class AnimationSetItem
        {
            public AnimationSetItem()
            {
                AnimationItems = new List<AnimationItem>();
            }

            [XmlAttribute]
            public string Name;

            [XmlAttribute]
            public bool Preview;

            [XmlElement("Animation")]
            public List<AnimationItem> AnimationItems;
        }

        public class AnimationItem
        {
            [XmlAttribute]
            public string Name;

            [XmlAttribute]
            public int ItemIndex = -1;

            [XmlAttribute]
            public bool Preview = true;

            [XmlIgnore]
            public string Directory;
        }

        public class MaterialItem
        {
            public string Name;
            public bool OptimizeShader;
            public bool OptimizeShaderOnReload;
        }

        public static MaterialItem DefaultMaterialItem
        {
            get
            {
                return new MaterialItem
                {
                    OptimizeShader = true,
                    OptimizeShaderOnReload = ConfigData.ApplicationConfig.DefaultValue.OptimizeShaderAfterMaterialChanged,
                };
            }
        }

        public class PreviewItem
        {
            [XmlAttribute]
            public string	BindModelName;

            [XmlAttribute]
            public string	BindBoneName;

            [XmlAttribute]
            public float	ScaleX;

            [XmlAttribute]
            public float	ScaleY;

            [XmlAttribute]
            public float	ScaleZ;

            [XmlAttribute]
            public float	RotateX;

            [XmlAttribute]
            public float	RotateY;

            [XmlAttribute]
            public float	RotateZ;

            [XmlAttribute]
            public float	TranslateX;

            [XmlAttribute]
            public float	TranslateY;

            [XmlAttribute]
            public float	TranslateZ;

            // 古いプロジェクトとの互換性のためにデフォルト値を設定しておく
            [XmlAttribute]
            public bool Visible = true;
            public bool ShowInObjView = true;

            public void Store(Model.PreviewInfoType src)
            {
                BindModelName	= src.BindModelName;
                BindBoneName	= src.BindBoneName;
                ScaleX			= src.Scale.X;
                ScaleY			= src.Scale.Y;
                ScaleZ			= src.Scale.Z;
                RotateX			= src.Rotate.X;
                RotateY			= src.Rotate.Y;
                RotateZ			= src.Rotate.Z;
                TranslateX		= src.Translate.X;
                TranslateY		= src.Translate.Y;
                TranslateZ		= src.Translate.Z;
                Visible         = src.Visible;
                ShowInObjView    = src.ShowInObjView;
            }

            public void Load(Model.PreviewInfoType dst)
            {
                dst.BindModelName	= BindModelName;
                dst.BindBoneName	= BindBoneName;
                dst.Scale			= new Vector4(ScaleX,     ScaleY,     ScaleZ);
                dst.Rotate			= new Vector4(RotateX,    RotateY,    RotateZ);
                dst.Translate		= new Vector4(TranslateX, TranslateY, TranslateZ);
                dst.Visible         = Visible;
                dst.ShowInObjView    = ShowInObjView;
            }
        }
    }

    public class FileInfo
    {
        public EditLog create { get; set; }
        public EditLog modify { get; set; }
    }

    public class EditLog
    {
        [XmlAttribute]
        public string user { get; set; }

        [XmlAttribute]
        public string host { get; set; }

        [XmlAttribute]
        public System.DateTime date_time { get; set; }

        [XmlAttribute]
        public string tool_name { get; set; }

        [XmlAttribute]
        public string tool_version { get; set; }
    }
}
