﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Windows.Forms;
using System.IO;
using System.Xml;
using System.Xml.Serialization;
using App.Command;
using App.ConfigData;
using App.PropertyEdit;
using App.Utility;
using ConfigCommon;
using nw.g3d.nw4f_3dif;
using Viewer;
using nw.g3d.iflib;
using nw.g3d.iflib.nw3de;

namespace App.Data
{
    public class Model :
        IntermediateFileDocument,
        IHasUserData,
        NintendoWare.G3d.Edit.IEditModelTarget,
        IReferTexture,
        IMaterialOwner
    {
        public modelType Data
        {
            get;
            private set;
        }

        public override object nw4f_3difItem
        {
            get { return Data; }
        }

        public override process_log_arrayType process_log_array
        {
            get { return Data.process_log_array; }
        }

        public ReadOnlyList<Material>	Materials{	get { return new ReadOnlyList<Material>(materials_);	} }
        public ReadOnlyList<Bone>		Bones{		get { return new ReadOnlyList<Bone>(bones_);			} }
        public vertexType[]		Vertices {
            get {
                if (Data.vertex_array == null)
                {
                    return new vertexType[0];
                }
                else
                {
                    return Data.vertex_array.vertex;
                }
            }
        }

        public ReadOnlyList<Shape>		Shapes{		get { return new ReadOnlyList<Shape>(shapes_);			} }

        private List<Material>	materials_	= new List<Material>();
        private List<Bone>		bones_		= new List<Bone>();
        private List<Shape>		shapes_		= new List<Shape>();

        // モデルを識別するためのキー (Model が GC されても ModelId は残りうる)
        public object ModelId = new object();

        #region LOD 情報
        public LodInfo[] LodInfoArray
        {
            get;
            private set;
        }

        public sealed class LodInfo
        {
            /// <summary>
            /// 所属するモデル。
            /// ProcessVertexCount, ProcessVertexCountPerTriangle の取得時に参照する。
            /// </summary>
            private readonly Model Owner;

            /// <summary>
            /// LOD レベル。
            /// ProcessVertexCount, ProcessVertexCountPerTriangle の取得時に参照する。
            /// </summary>
            private readonly int Level;

            /// <summary>
            /// ポリゴン数。
            /// </summary>
            public readonly int PolygonCount;

            /// <summary>
            /// 頂点数。
            /// </summary>
            public readonly int VertexCount;

            /// <summary>
            /// 処理頂点数を持っているかどうか。
            /// </summary>
            public bool HasProcessVertexCount
            {
                get
                {
                    return Owner.HasProcessVertexCount;
                }
            }

            /// <summary>
            /// 処理頂点数を取得する。
            /// 処理頂点数はプラットフォーム毎に異なる。
            /// プラットフォーム選択はツール上で任意のタイミングで変更でき、
            /// 選択処理もコマンドに積まれないため、取得時に算出しているため変数として保持していない。
            /// 処理頂点数を取得できないプラットフォームでは -1 を返す。
            /// 0 を返すことによって、取得可能なプラットフォームの 0 と比較不能になることを避けるため。
            /// </summary>
            public int ProcessVertexCount
            {
                get
                {
                    return HasProcessVertexCount ? Owner.Shapes.Where(x => Level < x.ProcessVertexCountInMesh.Count).Select(x => x.ProcessVertexCountInMesh[Level]).Sum() : -1;
                }
            }

            /// <summary>
            /// トライアングルあたりの処理頂点数を取得する。
            /// 処理頂点数はプラットフォーム毎に異なる。
            /// プラットフォーム選択はツール上で任意のタイミングで変更でき、
            /// 選択処理もコマンドに積まれないため、取得時に算出しているため変数として保持していない。
            /// 処理頂点数を取得できないプラットフォームでは -1 を返す。
            /// 0 を返すことによって、取得可能なプラットフォームの 0 と比較不能になることを避けるため。
            /// </summary>
            public float ProcessVertexCountPerTriangle
            {
                get
                {
                    return HasProcessVertexCount ? ((PolygonCount != 0) ? ((float)ProcessVertexCount / PolygonCount) : 0.0f) : -1.0f;
                }
            }

            /// <summary>
            /// LOD 情報クラスのコンストラクタ。
            /// 処理頂点数とトライアングルあたりの処理頂点数は変数として保持しない。
            /// これらは owner と level から算出する。
            /// </summary>
            /// <param name="owner">所属するモデル</param>
            /// <param name="level">LOD レベル</param>
            /// <param name="polygonCount">ポリゴン数</param>
            /// <param name="vertexCount">頂点数</param>
            public LodInfo(Model owner, int level, int polygonCount, int vertexCount)
            {
                Owner = owner;
                Level = level;
                PolygonCount = polygonCount;
                VertexCount = vertexCount;
            }
        }

        public sealed class ConstantLodInfo
        {
            public int PolygonCount
            {
                get;
                private set;
            }

            public int VertexCount
            {
                get;
                private set;
            }

            public int ProcessVertexCount
            {
                get;
                private set;
            }

            public float ProcessVertexCountPerTriangle
            {
                get;
                private set;
            }

            public ConstantLodInfo(LodInfo info)
            {
                PolygonCount = info.PolygonCount;
                VertexCount = info.VertexCount;
                ProcessVertexCount = info.ProcessVertexCount;
                ProcessVertexCountPerTriangle = info.ProcessVertexCountPerTriangle;
            }
        }
        #endregion
        #region ReferTexture
        private Dictionary<string, string> referenceTexturePaths = new Dictionary<string, string>();

        // 参照しているテクスチャ
        public Dictionary<string, string> ReferenceTexturePaths
        {
            get
            {
                return referenceTexturePaths;
            }
            set
            {
                referenceTexturePaths = value;
            }
        }

        public void UpdateReferenceTexturePaths()
        {
            // マテリアル中間ファイル対応のために実装を拡張メソッドに移動。
            // 移動によって Model.UpdateReferenceTexturePaths() は不要となったので削除してもよい。
            MaterialOwnerExtensions.UpdateReferenceTexturePaths(this);
        }
        #endregion

        private IEnumerable<KeyValuePair<string, string>> UnresolvedTexturesOnReload(string path)
        {
            var location = Path.GetFullPath(Path.GetDirectoryName(path));
            var textures = new Dictionary<string, string>();
            foreach (var tex in DocumentManager.Textures)
            {
                if (!string.IsNullOrEmpty(tex.FileLocation))
                {
                    // DocumentManager.MakeReferenceTextureFilenames() での辞書値比較には
                    // 拡張子を除いた名前が用いられるので、名前を指定する。
                    textures.Add(tex.FilePath.ToLower(), tex.Name);
                }
            }

            var baseName = string.Equals(location, FileLocation, StringComparison.OrdinalIgnoreCase) ? BaseName : null;
            foreach (var item in ReferenceTexturePaths)
            {
                if (Path.IsPathRooted(item.Value))
                {
                    var pathWithName = DocumentManager.MakeReferenceTextureFilenames(
                        location,
                        this,
                        baseName,
                        item.Key,
                        additionalLowerCaseFilePaths: textures).FirstOrDefault();

                    if (pathWithName == null)
                    {
                        // 参照解決として使用できるテクスチャが存在しない。
                        // アプリケーションが管理してないという意味だけでなく
                        // 参照解決優先度の全パターンでも解決できない。
                        yield return item;
                    }
                    else
                    {
                        // アプリケーションが管理しているテクスチャを参照しているかチェックする。
                        // ディレクトリパスは大文字小文字を区別しないが、ファイル名は区別する。
                        // 詳細は MakeReferenceTextureFilenames() を参照。
                        string texName = null;
                        if (!textures.TryGetValue(pathWithName.path.ToLower(), out texName) || (texName != Path.GetFileNameWithoutExtension(pathWithName.path)))
                        {
                            // 参照中のテクスチャよりも優先度の高いものが保存先にあるため、
                            // リロード時に異なるもの (優先度の高いもの) に切り替わる。
                            yield return item;
                        }
                    }
                }
                else
                {
                    yield return item;
                }
            }
        }

        public override bool NeedToAskBeforeSave(string path)
        {
            return UnresolvedTexturesOnReload(path).Any();
        }

        public override bool ConfirmPathBeforeSave(string path)
        {
            var unresolvedTexturePaths = UnresolvedTexturesOnReload(path);
            if (unresolvedTexturePaths.Any())
            {
                var unresolvedTextures = new List<Tuple<string, System.Drawing.Image>>();
                foreach (var unresolvedTexturePath in unresolvedTexturePaths)
                {
                    unresolvedTextures.Add(new Tuple<string, System.Drawing.Image>(unresolvedTexturePath.Key, GuiObject.GetObjectIcon(GuiObjectID.Texture)));
                }

                if (unresolvedTextures.Any())
                {
                    return App.AppContext.NotificationHandler.OkCancelListBoxDialog(
                        MessageContext.IO_TextureReferenceCanNotResolved,
                        res.Strings.IO_ConfirmSave,
                        string.Format(res.Strings.IO_TextureReferenceCanNotResolved, FileName + DocumentManager.GetSameNameIndexText(this, false)),
                        App.AppContext.NotificationHandler.StringFormat.UnresolvedTexture,
                        unresolvedTextures);
                }
            }
            return true;
        }
        #region ランタイム連携
        /// <summary>
        /// IEditModelTarget
        /// </summary>
        public uint ResModelKey { get; set; }

        /// <summary>
        /// IEditModelTarget
        /// </summary>
        public uint ModelObjKey { get; set; }

        /// <summary>
        /// PreviewLodLevel
        /// </summary>
        public int PreviewLodLevel { get; set; }

        /// <summary>
        /// DisableVertexQuantize
        /// </summary>
        public bool DisableVertexQuantize { get; set; }


        private bool isSendAttached_ = false;

        public static event Action<Document, bool> IsSendAttachedChanged = null;
        /// <summary>
        /// IEditModelTarget
        /// </summary>
        public bool IsSendAttached
        {
        	get
        	{
        		return isSendAttached_;
        	}

        	set
        	{
                DebugConsole.WriteLine("IsSendAttached.set() : {0}, {1} ---------------------------------------------", value, isSendAttached_);
                bool changed = isSendAttached_ != value;
        		isSendAttached_ = value;

                if (changed)
                {
                    // [モデル]-[プレビュー]ページの更新のため
                    TheApp.MainFrame.BeginInvoke(new MethodInvoker(() => App.AppContext.NotifyPropertyChanged(null, new DocumentContentsChangedArgs(this, null))));
                }

                if (changed && IsSendAttachedChanged != null)
                {
                    IsSendAttachedChanged(this, value);
                }
        	}
        }

        public bool IsSendAttached_BeforeResetStatus{ get; set; }		// HACK

        public static event Action<Document, bool> IsAttachedChanged = null;
        public override bool IsAttached
        {
            get
            {
                return base.IsAttached;
            }
            set
            {
                bool changed = base.IsAttached != value;
                base.IsAttached = value;
                if (changed && IsAttachedChanged != null)
                {
                    IsAttachedChanged(this, value);
                }
            }
        }
        /// <summary>
        /// IEditTarget
        /// </summary>
        public override void ResetStatus()
        {
            DebugConsole.WriteLine("Model.ResetStatus() ---------------------------------------------");

            base.ResetStatus();

            lock (Viewer.Connecter.ModelFileLoadedActions)
            {
                Viewer.Connecter.ModelFileLoadedActions.Remove(ModelObjKey);
            }

            ResModelKey = 0;
            ModelObjKey = 0;
            //PreviewLodLevel = -1;
            IsSendAttached = false;
            SetLayoutActionOnLoad = null;
        }

        public Action SetLayoutActionOnLoad;

        private object _WaitingQueryModelLayoutLock = new object();
        private bool _WaitingQueryModelLayout;

        public bool WaitingQueryModelLayout
        {
            set
            {
                lock (_WaitingQueryModelLayoutLock)
                {
                    _WaitingQueryModelLayout = value;
                    System.Threading.Monitor.Pulse(_WaitingQueryModelLayoutLock);
                }
            }
        }

        public bool WaitQueryModelLayout(int timeout)
        {
            lock (_WaitingQueryModelLayoutLock)
            {
                var stopWatch = new Stopwatch();
                stopWatch.Start();
                int remain = timeout;
                while (_WaitingQueryModelLayout)
                {
                    if (remain <= 0)
                    {
                        return false;
                    }
                    System.Threading.Monitor.Wait(_WaitingQueryModelLayoutLock, remain);
                    remain = (int)(timeout - stopWatch.ElapsedMilliseconds);
                }
            }

            return true;
        }

        public override void UnloadedFromHio()
        {
            DebugConsole.WriteLine("UnloadFromHio model");
            base.UnloadedFromHio();
            PreviewingAnimations.Clear();
            RetargetHostingAnimations.Clear();
            lock (DocumentManager.PreviewingModels)
            {
                DocumentManager.PreviewingModels.Remove(this);
            }

            foreach (var material in Materials)
            {
                material.RenderInfoQueried = false;
            }

            loadedTextures.Clear();

            if (TheApp.MainFrame != null && TheApp.MainFrame.IsHandleCreated)
            {
                TheApp.MainFrame.BeginInvoke((Action)(() => AfterModelReloaded(new List<ShaderOptimizeData>())));
            }
        }

        public class ShaderOptimizeData
        {
            public string name;
            public ShaderDefinition document;
            public shader_definitionType definition;
            public List<G3dStream> streams;

            public bool optimize;
            public byte[] _shader_variation;
            public byte[] shader_variation
            {
                get
                {
                    return _shader_variation;
                }
                set
                {
                    _shader_variation = value;
                    shadingModelSets.Clear();
                    if (_shader_variation != null)
                    {
                        // 例外処理は呼び出し側に任せる
                        using (var stream = new MemoryStream(_shader_variation))
                        {
                            var nw4f_3dif = (nw4f_3difType)(new XmlSerializer(typeof(nw4f_3difType))).Deserialize(stream);
                            var variation = (shader_variationType)nw4f_3dif.Item;
                            foreach (var item in variation.target_shader_array.GetItems())
                            {
                                shadingModelSets[item.shading_model_name] = new HashSet<shader_programType>(
                                    item.shader_program_array.GetItems(),
                                    ShaderTypeUtility.ShaderProgramComparer.TheComparer);
                            }
                        }
                    }
                }
            }

            // HashSet の Comparer は ShaderDefinitionUtility.ShaderProgramComparer.TheComparer;
            public Dictionary<string, HashSet<shader_programType>> shadingModelSets = new Dictionary<string,HashSet<shader_programType>>();

            public bool DeviceTargeted;
            public TeamConfig.PlatformPreset platform;

            // ShaderOptimizeData 生成時のランタイムが 32 ビットかどうか?
            // 注意: 比較のみに使う。コンバート時には直接この値は参照していない
            public bool useConverter32;

            public MaterialData[] materials;

            public byte[] lastModelData;
            public List<byte[]> lastMaterialData;
            public List<G3dStream> lastModelStream;
            public bool runScript;

            public class MaterialData
            {
                public int index;
                public bool optimizeEnabled;
                public bool optimizeOnReloadEnabled;
                public bool forceOptimize;
                public bool optimize;
                public Tuple<ShaderOptimizeData, string> reusedData;

                public string shadingModelName;
                public int modifiedCount;
                public string bfshaPath;
                public MaterialData(Material material, bool forceOptimize, bool optimize, ShaderDefinition definition)
                {
                    index = material.Index;
                    optimizeEnabled = material.OptimizeShader;
                    optimizeOnReloadEnabled = DocumentManager.GetMaterialOptimizeShaderOnReload(material);
                    this.forceOptimize = forceOptimize || optimizeOnReloadEnabled;
                    this.optimize = optimize;
                    shadingModelName = material.MaterialShaderAssign.ShaderName;
                    modifiedCount = definition.shadingModelModifiedCounts.ContainsKey(shadingModelName) ? definition.shadingModelModifiedCounts[shadingModelName] : 0;
                }
            }
        }


        // 最適化シェーダー
        public Tuple<ShaderOptimizeData, string>[] lastMaterialOptimizeData;
        public Tuple<ShaderOptimizeData, string>[] MaterialOptimizeDataCache;

        public Tuple<ShaderOptimizeData, string>[] deviceFirstOptimizedData = new Tuple<ShaderOptimizeData, string>[0];
        public Tuple<ShaderOptimizeData, string>[] pcFirstOptimizedData = new Tuple<ShaderOptimizeData, string>[0];

        /// <summary>
        /// ロード時に送信したテクスチャの情報
        /// </summary>
        public Dictionary<string, Tuple<byte[], List<G3dStreamCachedComparer>>> loadedTextures =
            new Dictionary<string, Tuple<byte[], List<G3dStreamCachedComparer>>>();

        // メインスレッドで呼ばれる
        public void AfterModelReloaded(List<ShaderOptimizeData> shaderOptimizeData)
        {
            var materialData = new ShaderOptimizeData.MaterialData[Materials.Count];
            if (IsAttached)
            {
                foreach (var item in shaderOptimizeData.SelectMany(x => x.materials))
                {
                    materialData[item.index] = item;
                }
            }

            bool changed = false;
            foreach (var pair in materialData.Zip(Materials, (item, material) => new { item, material }))
            {
                changed |= pair.material.AfterModelReloaded(pair.item);
            }

            // 変更通知
            if (changed)
            {
                App.AppContext.NotifyPropertyChanged(this, new DocumentContentsChangedArgs(this, null));
            }
        }
        #endregion

        #region プレビュ情報
        [Serializable]
        public class PreviewInfoType
        {
            public bool Visible = true;
            public bool ShowInObjView = true;

            public string	BindModelName	= null;
            public string	BindBoneName	= null;
            //
            public Vector4	Scale			= new Vector4(1.0f, 1.0f, 1.0f);
            public Vector4	Rotate			= new Vector4(0.0f, 0.0f, 0.0f);
            public Vector4	Translate		= new Vector4(0.0f, 0.0f, 0.0f);
            //
            public bool		IsBind { get { return (BindModelName != null) && (BindBoneName != null); } }

            // 編集されているか？
            public bool		IsEdited
            {
                get
                {
                    return
                        (BindModelName	!= null) ||
                        (BindBoneName	!= null) ||
                        (Scale.X		!= 1.0f) ||
                        (Scale.Y		!= 1.0f) ||
                        (Scale.Z		!= 1.0f) ||
                        (Rotate.X		!= 0.0f) ||
                        (Rotate.Y		!= 0.0f) ||
                        (Rotate.Z		!= 0.0f) ||
                        (Translate.X	!= 0.0f) ||
                        (Translate.Y	!= 0.0f) ||
                        (Translate.Z	!= 0.0f) ||
                        Visible != true ||
                        ShowInObjView != true;
                }
            }
        };

        // ランタイム連携用に表示するか？
        public bool IsVisible
        {
            get
            {
                return PreviewInfo.Visible || IsSendAttached;
            }
        }

        // エディタに表示するか？
        public bool IsShowInObjView
        {
            get
            {
                return PreviewInfo.ShowInObjView;
            }
        }
        public PreviewInfoType PreviewInfo { get; set; }

        public void SendEditModelLayout(bool afterModelLoad, bool sendIdentity = true, bool sendChildren = false)
        {
            Viewer.EditModelLayout.Send(
                this,
                PreviewInfo.IsBind,
                Convert(PreviewInfo.Scale,     false),
                Convert(PreviewInfo.Rotate,    true),
                Convert(PreviewInfo.Translate, false),
                afterModelLoad,
                sendIdentity
            );

            if (sendChildren)
            {
                // このモデルのボーンにバインドしているモデル（小モデル）も送信する
                var children = DocumentManager.Models.Where(x => x.PreviewInfo.BindModelName == this.Name);
                foreach (var child in children)
                {
                    child.SendEditModelLayout(afterModelLoad, sendIdentity, sendChildren);
                }
            }
        }

        public void SendEditBoneBind()
        {
            var targetModel =
                (PreviewInfo.BindModelName == null) ?
                    null :
                    DocumentManager.Models.FirstOrDefault(x => x.Name == PreviewInfo.BindModelName);

            int boneIndex = -1;
            {
                if (targetModel != null)
                {
                    var bone =
                        (PreviewInfo.BindBoneName == null) ?
                            null :
                            targetModel.Bones.FirstOrDefault(x => x.Name == PreviewInfo.BindBoneName);

                    boneIndex =
                        (bone == null) ?
                            -1 :
                            targetModel.Bones.IndexOf(bone);
                }
            }

            Viewer.EditBoneBind.Send(this, targetModel, boneIndex);
        }

        // モデルをプレビューのバインドの深さ順にソートして返す
        public static IEnumerable<Model> SortByPreviewDepth(IEnumerable<Model> models, bool addChildren = false)
        {
            var modelList = new List<Model>();
            if (addChildren)
            {
                foreach (var model in models)
                {
                    modelList.AddRange(GetPreviewChildren(model));
                }
            }
            modelList = modelList.Distinct().ToList();

            var modelDepth = new List<Tuple<Model, int>>();
            foreach (var model in modelList)
            {
                if (!model.PreviewInfo.IsBind)
                {
                    modelDepth.Add(new Tuple<Model, int>(model, 0));
                }
                else
                {
                    // 循環参照防止(depthは不正確)
                    var parentList = new List<Model>(){model};
                    var depth = 0;
                    var parent = DocumentManager.Models.FirstOrDefault(x => x.Name == model.PreviewInfo.BindModelName);
                    while (parent != null)
                    {
                        depth++;
                        parentList.Add(parent);
                        parent = DocumentManager.Models.Where(x => !parentList.Contains(x)).FirstOrDefault(x => x.Name == parent.PreviewInfo.BindModelName);
                    }
                    modelDepth.Add(new Tuple<Model, int>(model, depth));
                }
            }
            modelDepth.Sort((tuple1, tuple2) => tuple1.Item2.CompareTo(tuple2.Item2));
            return modelDepth.Select(x => x.Item1);

        }

        // モデルにバインドされた子モデルを返す（再帰）
        public static List<Model> GetPreviewChildren(Model model, List<Model> models = null)
        {
            models = models ?? new List<Model>();
            models.Add(model);
            var children = DocumentManager.Models.Where(x => !models.Contains(x)).Where(x => x.PreviewInfo.BindModelName == model.Name).ToList();
            foreach (var child in children)
            {
                models = GetPreviewChildren(child, models);
            }
            return models;
        }

        private static NintendoWare.G3d.Edit.Math.Vector3 Convert(Data.Vector4 src, bool toRadian)
        {
            return
                new NintendoWare.G3d.Edit.Math.Vector3()
                {
                    X = toRadian ? MathUtility.ToRadian(src.X) : src.X,
                    Y = toRadian ? MathUtility.ToRadian(src.Y) : src.Y,
                    Z = toRadian ? MathUtility.ToRadian(src.Z) : src.Z
                };
        }

        #endregion

        // バインドされているアニメーション
        public IEnumerable<AnimationDocument> AllAnimations
        {
            get
            {
                return DocumentManager.GetAnimations(DefaultAnimationSet.Animations.Concat(AnimationSets.SelectMany(x => x.Animations)).Distinct());
            }
        }

        public bool ContainsAnimation(AnimationDocument animation)
        {
            return DefaultAnimationSet.Animations.Concat(AnimationSets.SelectMany(x => x.Animations)).Distinct().Any(
                x => x.Name == animation.FileName && animation.FileLocation.Equals(x.Directory, StringComparison.OrdinalIgnoreCase));
        }

        public UserDataArray UserDataArray { get; set; }
        public UserDataArray SavedUserDataArray { get; set; }
        public bool UserDataArrayChanged { get; set; }

        public List<AnimationSet> AnimationSets { get; set; }
        public AnimationSet DefaultAnimationSet { get; set; }
        public Dictionary<string, string> OptimizedLogFolders { get; set; }
        public bool ForcedOptimized { get; set; }
        private Object OptimizedLogLock { get; set; }

        public IEnumerable<AnimationSet> AnimationSetsWithDefault
        {
            get
            {
                return Enumerable.Repeat(DefaultAnimationSet, 1).Concat(AnimationSets);
            }
        }

        public void SendBindAnimations()
        {
            using (var vdsb = new Viewer.ViewerDrawSuppressBlock(null))
            {
                Viewer.HioUtility.AddViewerDrawSuppressBlockBindAnim(this);
            }
        }

        // プレビュー対象のアニメーション
        public List<Tuple<string, AnimationDocument>> PreviewingAnimations = new List<Tuple<string, AnimationDocument>>();

        // モデルをリターゲットホストとするアニメーション
        public List<AnimationDocument> RetargetHostingAnimations = new List<AnimationDocument>();

        private AnimationSet previewAnimSet_;
        public AnimationSet PreviewAnimSet
        {
            get
            {
                return previewAnimSet_;
            }
            set
            {
                previewAnimSet_ = value;
                PreviewAnimSetUpdated();
            }
        }

        public void PreviewAnimSetUpdated()
        {
            DocumentManager.ProjectDocument.SetMaybeModified();
            Viewer.ViewerUtility.SendAnimationSet(this);
            App.AppContext.NotifyPropertyChanged(this, new PreviewAnimSetUpdatedArg());
        }

        public Model(modelType model, List<G3dStream> streamArray)
            : base(GuiObjectID.Model, streamArray)
        {
            ModelObjKey = 0;
            ResModelKey = 0;
            PreviewLodLevel = -1;
            Data = model;
            DefaultAnimationSet = new AnimationSet(false)
            {
                Name = res.Strings.FileTreeView_DefaultAnimationSet,
                IsDefaultAnimationSet = true
            };
            DefaultAnimationSet.UpdateSavedData();
            DefaultAnimationSet.SetNotModified();

            DefaultAnimationSet.Animations = new List<AnimationSetItem>();
            AnimationSets = new List<AnimationSet>();
            previewAnimSet_ = DefaultAnimationSet;

            PreviewInfo = new PreviewInfoType();

            ForcedOptimized = false;
            OptimizedLogFolders = new Dictionary<string, string>();
            OptimizedLogLock = new Object();

            Initialize();
        }

        // stream は参照をコピーして保持します。
        public void Initialize()
        {
            ToolData.Load(Data.tool_data);
            MakeComment(Data.comment);

            materials_.Clear();
            bones_.Clear();
            shapes_.Clear();

            smoothSkinningCount_ = -1;
            rigidSkinningCount_ = -1;
            rigidBodyCount_ = -1;

            int index = 0;
            if (Data.material_array != null)
            {
                foreach (var x in Data.material_array.material) { materials_.Add(new Material(x, this, index)); index++; }
                index = 0;
            }
            foreach (var x in Data.skeleton.bone_array.bone) { bones_.Add(new Bone(x, this, index)); index++; }
            index = 0;
            if (Data.shape_array != null)
            {
                foreach (var x in Data.shape_array.shape) { shapes_.Add(new Shape(x, this, index)); index++; }
            }

            // ボーンの親子関係を設定します。
            Dictionary<string, Bone> nameToBone = new Dictionary<string, Bone>();
            foreach (var bone in bones_)
            {
                nameToBone.Add(bone.Name, bone);
            }

            foreach (var bone in bones_)
            {
                if (bone.Data.parent_name != string.Empty)
                {
                    bone.Parent = nameToBone[bone.Data.parent_name];
                    bone.Parent.Children.Add(bone);
                }
            }

            // ユーザーデータ作成
            this.MakeUserData(Data.user_data_array, BinaryStreams);

            // LOD 情報作成
            {
                // 処理頂点数を強制更新する。
                foreach (var shape in Shapes)
                {
                    shape.UpdateProcessVertexCount(App.AppContext.SelectedPlatformPreset, true);
                }

                var itemCount = Shapes.Any() ? Shapes.Max(x => x.VertexCountInMesh.Length) : 0;
                LodInfoArray = new LodInfo[itemCount];
                for (int level = 0; level < itemCount; level++)
                {
                    int polygonCount = 0;
                    int vertexCount = 0;
                    int processVertexCount = 0;
                    foreach (var shape in Shapes)
                    {
                        if (level < shape.VertexCountInMesh.Length)
                        {
                            polygonCount += shape.PolygonCountInMesh(level);
                            vertexCount += shape.VertexCountInMesh[level];
                            processVertexCount += shape.ProcessVertexCountInMesh[level];
                        }
                    }

                    LodInfoArray[level] = new LodInfo(this, level, polygonCount, vertexCount);
                }
            }
        }

        #region Bridge
        // モデルデータ差し替え用
        public class SwapData
        {
            public List<nw3de_SearchPath.SearchPath> SearchPaths;
            public List<G3dStream> streams;
            public string Comment;
            public string Label;
            public RgbaColor? EditColor;
            public int smoothSkinningCount;
            public int rigidSkinningCount;
            public int rigidBodyCount;
            public modelType Data;
            public List<Material> materials;
            public List<Shape> shapes;
            public List<Bone> bones;
            public UserDataArray userData;
            public object ModelId;
            public LodInfo[] LodInfoArray;
        }

        public SwapData CreateSwapData(modelType data, List<G3dStream> streams, out bool isSameStructure)
        {
            var current = GetSwapData();
            Data = data;
            BinaryStreams = streams;
            Initialize();

            // 保存済みデータの設定
            foreach (var content in Materials.Concat<GuiObject>(Shapes).Concat(Bones))
            {
                content.UpdateSavedData();
            }

            // プレビュー情報の引き継ぎ
            foreach (var material in Materials)
            {
                var currentMaterial = current.materials.FirstOrDefault(x => x.Name == material.Name);
                if (currentMaterial != null)
                {
                    material.OptimizeShader = currentMaterial.OptimizeShader;
                    material.OptimizeShaderOnReload = currentMaterial.OptimizeShaderOnReload;
                }
            }

            isSameStructure = Materials.Count == current.materials.Count && Materials.Zip(current.materials, (x, y) => x.Name == y.Name).All(x => x) &&
                Bones.Count == current.bones.Count && Bones.Zip(current.bones, (x, y) => x.Name == y.Name && (x.Parent == null ? null : x.Parent.Name) == (y.Parent == null ? null : y.Parent.Name)).All(x => x) &&
                Shapes.Count == current.shapes.Count && Shapes.Zip(current.shapes, (x, y)=> x.Name == y.Name && (x.Data.shape_info.bone_name == y.Data.shape_info.bone_name) && (x.Data.shape_info.mat_name == y.Data.shape_info.mat_name)).All(x => x) &&
                ToolData.SearchPaths.Count == current.SearchPaths.Count && ToolData.SearchPaths.Zip(current.SearchPaths, (x, y) => x.Path == y.Path).All(x => x);
            return Swap(current, false);
        }

        public SwapData GetSwapData()
        {
            return new SwapData()
            {
                Data = Data,
                SearchPaths = ToolData.SearchPaths.ToList(),
                streams = BinaryStreams,
                Comment = Comment,
                Label = Label,
                EditColor = EditColor,
                smoothSkinningCount = smoothSkinningCount_,
                rigidSkinningCount = rigidSkinningCount_,
                rigidBodyCount = rigidBodyCount_,
                materials = materials_.ToList(),
                shapes = shapes_.ToList(),
                bones = bones_.ToList(),
                userData = UserDataArray,
                ModelId = ModelId,
                LodInfoArray = LodInfoArray,
            };
        }

        public SwapData Swap(SwapData data, bool copyHistory)
        {
            var current = GetSwapData();
            Data = data.Data;
            ToolData.SearchPaths = data.SearchPaths;
            BinaryStreams = data.streams;
            Comment = data.Comment;
            Label = data.Label;
            EditColor = data.EditColor;
            smoothSkinningCount_ = data.smoothSkinningCount;
            rigidSkinningCount_ = data.rigidSkinningCount;
            rigidBodyCount_ = data.rigidBodyCount;
            materials_ = data.materials;
            shapes_ = data.shapes;
            bones_ = data.bones;
            UserDataArray = data.userData;
            ModelId = data.ModelId;
            LodInfoArray = data.LodInfoArray;

            if (copyHistory)
            {
                UserDataArrayChanged = !SavedUserDataArray.IsSame(UserDataArray);

                foreach (var old in current.materials)
                {
                    var mat = Materials.FirstOrDefault(x => x.Name == old.Name);
                    if (mat != null)
                    {
                        SavedContents.Remove(old);
                        SavedContents.Add(mat);
                        mat.CopySavedData(old);
                        mat.MaterialShaderAssignPool = old.MaterialShaderAssignPool;
                        mat.LastRenderInfo = old.LastRenderInfo;
                    }
                }

                foreach (var old in current.shapes)
                {
                    var shape = Shapes.FirstOrDefault(x => x.Name == old.Name);
                    if (shape != null)
                    {
                        SavedContents.Remove(old);
                        SavedContents.Add(shape);
                        shape.CopySavedData(old);
                    }
                }

                foreach (var old in current.bones)
                {
                    var bone = Bones.FirstOrDefault(x => x.Name == old.Name);
                    if (bone != null)
                    {
                        SavedContents.Remove(old);
                        SavedContents.Add(bone);
                        bone.CopySavedData(old);
                    }
                }
            }

            return current;
        }
        #endregion

        // マテリアル適用用のモデルのコンストラクタ
        public Model(materialType material, List<G3dStream> streamArray)
            : base(GuiObjectID.Model, streamArray)
        {
            ModelObjKey = 0;
            ResModelKey = 0;
            PreviewLodLevel = -1;
            materials_.Add(new Material(material, this, 0));

            DefaultAnimationSet = new AnimationSet(false)
            {
                Name = res.Strings.FileTreeView_DefaultAnimationSet,
                IsDefaultAnimationSet = true
            };

            DefaultAnimationSet.Animations = new List<AnimationSetItem>();
            AnimationSets = new List<AnimationSet>();
            previewAnimSet_ = DefaultAnimationSet;
        }

        public override IEnumerable<GuiObject> ContentObjects
        {
            get
            {
                return base.ContentObjects.
                    Concat<GuiObject>(materials_).
                    Concat<GuiObject>(bones_).
                    Concat<GuiObject>(shapes_);
            }
        }

        public override nw4f_3difType Create_nw4f_3difType(bool viewer)
        {
            lock (this)
            {
                foreach (var material in Materials)
                {
                    material.UpdateData();
                }

                foreach (var shape in Shapes)
                {
                    shape.UpdateData();
                }

                foreach (var bone in Bones)
                {
                    bone.UpdateData();
                }

                UpdateData();

                //			nw4f_3difType nw4f_3dif = new nw4f_3difType();
                nw4f_3dif_.Item = Data;
                nw4f_3dif_.file_info = null;

                return nw4f_3dif_;
            }
        }

        /// <summary>
        /// 参照しているドキュメント
        /// </summary>
        public override IEnumerable<Document> ReferenceDocuments
        {
            get
            {
                var textures = DocumentManager.Textures.Where(x => ReferenceTexturePaths.ContainsKey(x.Name) &&
                                                                     string.Equals(x.FilePath, ReferenceTexturePaths[x.Name], StringComparison.OrdinalIgnoreCase) &&
                                                                     Materials.SelectMany(y => y.ResolvedSamplerTextureNames).Any(z => z == x.Name));
                var shaderDefinitions = DocumentManager.ShaderDefinitions.Where(
                    x =>
                        DocumentManager.Materials.Any(
                            y => y.MaterialShaderAssign.ShaderDefinitionFileName == x.Name));
                var parentMaterialOwnerDocs =
                    Materials.SelectMany(x => x.ParentMaterials).Where(x => x != null && x.Referrers.All(y => y != this)).Select(x => x.OwnerDocument);

                return
                    textures
                        .Concat<Document>(shaderDefinitions)
                        .Concat<Document>(AllAnimations)
                        .Concat<Document>(parentMaterialOwnerDocs)
                        .Distinct();
            }
        }

        public IEnumerable<string> UnresolvedTextures()
        {
            // マテリアル中間ファイル対応のために実装を拡張メソッドに移動。
            // 移動によって Model.UnresolvedTextures() は不要となったので削除してもよい。
            return MaterialOwnerExtensions.UnresolvedTextures(this);
        }

        public int VertexCount{			get{ return Shapes.Sum(x => x.VertexCount);			} }
        public int PolygonCount{		get{ return Shapes.Sum(x => x.PolygonCount);		} }
        public bool HasProcessVertexCount { get { return Shape.IsProcessVertexCountComputable(); } }
        public int ProcessVertexCount{	get{ return Shapes.Sum(x => x.ProcessVertexCount);	} }
        public float ProcessVertexPerTriangleCount
        {
            get
            {
                var n = PolygonCount;
                return (n > 0) ? ((float)ProcessVertexCount / (float)n) : 0.0f;
            }
        }

        public bool HasSkelton{ get { return Data.skeleton != null; } }

        private int smoothSkinningCount_ = -1;
        private int rigidSkinningCount_ = -1;
        private int rigidBodyCount_ = -1;

        public int SmoothSkinningCount
        {
            get
            {
                CalcSmoothSkinningCount();
                return smoothSkinningCount_;
            }
        }

        public int RigidSkinningCount
        {
            get
            {
                CalcSmoothSkinningCount();
                return rigidSkinningCount_;
            }
        }

        public int RigidBodyCount
        {
            get
            {
                CalcSmoothSkinningCount();
                return rigidBodyCount_;
            }
        }

        private void CalcSmoothSkinningCount()
        {
            if ((smoothSkinningCount_ == -1) ||
                (rigidSkinningCount_  == -1) ||
                (rigidBodyCount_      == -1))
            {
                smoothSkinningCount_ = 0;
                rigidSkinningCount_ = 0;
                rigidBodyCount_ = 0;
                {
                    foreach (var shape in Shapes)
                    {
                        smoothSkinningCount_ += (shape.Data.shape_info.vertex_skinning_count >= 2) ? 1 : 0;
                        rigidSkinningCount_  += (shape.Data.shape_info.vertex_skinning_count == 1) ? 1 : 0;
                        rigidBodyCount_      += (shape.Data.shape_info.vertex_skinning_count == 0) ? 1 : 0;
                    }
                }
            }
        }

        // すべてのマテリアルにシェーダーが割り当てられているか？
        public bool IsAllMaterialShaderAssigned
        {
            get
            {
                return Materials.Any(x => string.IsNullOrEmpty(x.MaterialShaderAssign.ShaderName)) == false;
            }
        }

        public void UpdateData()
        {
            Data.tool_data = GetToolData(ToolData);
            Data.comment =GetComment();

            // ユーザーデータのシリアライズ
            if (UserDataArray == null ||
                UserDataArray.Data == null ||
                UserDataArray.Data.Count == 0)
            {
                Data.user_data_array = null;
            }
            else
            {
                Data.user_data_array = new user_data_arrayType();
                this.MakeSerializeData(Data.user_data_array, UserDataArray, BinaryStreams);
            }

            // ストリームの設定
            {
                // ストリームのソート(や削除等)を行う。
                nw.g3d.iflib.StreamUtility.SortStream(Data, BinaryStreams);

                Data.stream_array = null;
            }
        }

        // 参照されているオブジェクト名変更コマンドを返す
        public override ICommand CreateReferenceObjectRenameCommand(Document srcObject, string oldName, string oldDirectory, string newName, string newExt, string newDirectory)
        {
            var commandSet = new EditCommandSet();
            {
                if (srcObject.ObjectID == GuiObjectID.Texture)
                {
                    foreach(var material in Materials)
                    {
                        foreach(var sampler in material.sampler_array.sampler)
                        {
                            if (sampler.tex_name == oldName)
                            {
                                // TODO: 確認
                                commandSet.Add(PropertyEdit.MaterialSamplerPage.CreateEditCommand_tex_name(new GuiObjectGroup(material), sampler.name, newName/*, false*/));
                            }
                        }
                    }
                }
                else if (srcObject is AnimationDocument)
                {
                    // アニメーションセットが保持しているアニメーション
                    {
                        foreach(var animSet in AnimationSetsWithDefault)
                        {
                            var command = animSet.CreateReferenceObjectRenameCommand(srcObject, oldName, oldDirectory, newName, newExt, newDirectory);
                            if (command != null)
                            {
                                commandSet.Add(command);
                            }
                        }
                    }
                }
                else if (srcObject is Model)
                {
                    // 親マテリアル
                    {
                        var srcFilePath = srcObject.FilePath;
                        foreach (var material in Materials)
                        {
                            if (material?.MaterialReference?.ParentMaterials?.FirstOrDefault() == null)
                            {
                                continue;
                            }
                            var materialReference = ObjectUtility.Clone(material.MaterialReference);
                            var modified = false;
                            foreach (var parentMaterial in materialReference.ParentMaterials)
                            {
                                var parentModelPath = string.IsNullOrEmpty(FilePath)
                                                          ? parentMaterial.Path
                                                          : IfApplyBaseMaterialUtility.GetParentModelFullPath(
                                                              parentMaterial.Path,
                                                              FilePath);
                                if (string.Equals(
                                    parentModelPath,
                                    srcFilePath,
                                    StringComparison.OrdinalIgnoreCase))
                                {
                                    modified = true;
                                    var directory = string.IsNullOrEmpty(newDirectory) ? oldDirectory : newDirectory;
                                    var newPath = PathUtility.MakeRelativePath(
                                        FileLocation + "/",
                                        Path.ChangeExtension(Path.Combine(directory, newName), newExt));
                                    parentMaterial.Path = newPath;
                                }
                            }
                            if (modified)
                            {
                                commandSet.Add(
                                    Material.CreateEditCommand_MaterialReference(new GuiObjectGroup(material), materialReference));
                            }
                        }
                    }
                }
                else
                {
                    // 未実装
                    Debug.Assert(false);
                }
            }
            return (commandSet.CommandCount > 0) ? commandSet : null;
        }

        // Preview用アニメーションセットの更新コマンド
        public EditCommand CreateUpdatePreviewAnimationCommand()
        {
            return
                new GeneralGroupValueEditCommand<int>(
                    DummyObject.TheDummyObjectGroup,
                    GuiObjectID.DummyObject,
                    0, // ダミー
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        PreviewAnimSetUpdated();
                    }
                );
        }

        #region SavedData
        public model_infoType savedModelInfo;
        public skeletonInfoType savedSkeletonInfo;
        public int savedMaterialCount;
        public int savedBoneCount;
        public int savedShapeCount;
        public ConstantLodInfo savedLodInfo0;
        public int savedSmoothSkinningCount;
        public int savedRigidSkinningCount;
        public int savedRigidBodyCount;
        public Project savedPreviewInfoProject;

        public override void UpdateSavedData()
        {
            base.UpdateSavedData();
            SavedUserDataArray = ObjectUtility.Clone(UserDataArray);
            UserDataArrayChanged = false;

            savedMaterialCount = Materials.Count;
            savedBoneCount = Bones.Count;
            savedShapeCount = Shapes.Count;
            savedLodInfo0 = (LodInfoArray.Length > 0) ? new ConstantLodInfo(LodInfoArray[0]) : null;
            savedSmoothSkinningCount = SmoothSkinningCount;
            savedRigidSkinningCount = RigidSkinningCount;
            savedRigidBodyCount = RigidBodyCount;
            savedSkeletonInfo = ObjectUtility.Clone(Data.skeleton.skeleton_info);
            savedModelInfo = ObjectUtility.Clone(Data.model_info);
            savedSearchPaths = ObjectUtility.Clone(SearchPaths);
            foreach (var content in ContentObjects.Where(x => x != this))
            {
                content.UpdateSavedData();
            }
        }

        public override bool EqualsToSavedData()
        {
            if (!base.EqualsToSavedData())
            {
                return false;
            }

            // 以前は SavedContents と ContentObjects の数が一致しない場合は同じデータでないと判断しましたが、
            // 現在は数が違っても同じデータの場合もあるので判定条件から削除。

            if (Materials.Any(MaterialParentsPage.IsModified))
            {
                return false;
            }

            return !ModelGeneralPage.IsModified(this) &&
                   !UserDataPage.IsModified(this) &&
                   !SearchPathPage.IsModified(this);
        }

        public bool IsLodInfo0ValueChanged<T>(Func<ConstantLodInfo, T> select) where T : struct
        {
            if (savedLodInfo0 == null)
            {
                return LodInfoArray.Length > 0;
            }
            else if (LodInfoArray.Length == 0)
            {
                return true;
            }
            else
            {
                return !select(savedLodInfo0).Equals(select(new ConstantLodInfo(LodInfoArray[0])));
            }
        }

        public bool IsModelInfoValueChanged<T>(Func<model_infoType, T> select) where T : struct
        {
            return !select(savedModelInfo).Equals(select(Data.model_info));
        }

        public bool IsModelInfoStringChanged(Func<model_infoType, string> select)
        {
            return select(savedModelInfo) != select(Data.model_info);
        }

        public bool IsSkeletonInfoValueChanged<T>(Func<skeletonInfoType, T> select) where T : struct
        {
            return !select(savedSkeletonInfo).Equals(select(Data.skeleton.skeleton_info));
        }

        public bool IsPreviewInfoValueChanged<T>(Func<PreviewInfoType, T> select) where T : struct
        {
            var savedPreviewInfo = DocumentManager.ProjectDocument.GetSavedModelPreviewInfo(this);
            return !select(savedPreviewInfo).Equals(select(PreviewInfo));
        }

        public bool IsPreviewInfoValueChanged(Func<PreviewInfoType, string> select)
        {
            var savedPreviewInfo = DocumentManager.ProjectDocument.GetSavedModelPreviewInfo(this);
            return select(savedPreviewInfo) != select(PreviewInfo);
        }

        /// <summary>
        /// 保存データのコピー
        /// </summary>
        public void CopySavedData(Model source)
        {
            CopyDocumentSavedData(source);
            SavedUserDataArray = source.SavedUserDataArray;
            UserDataArrayChanged = !source.SavedUserDataArray.IsSame(UserDataArray);

            savedMaterialCount = source.savedMaterialCount;
            savedBoneCount = source.savedBoneCount;
            savedShapeCount = source.savedShapeCount;
            savedLodInfo0 = source.savedLodInfo0;
            savedSmoothSkinningCount = source.savedSmoothSkinningCount;
            savedRigidSkinningCount = source.savedRigidSkinningCount;
            savedRigidBodyCount = source.savedRigidBodyCount;
            savedSkeletonInfo = source.savedSkeletonInfo;
            savedModelInfo = source.savedModelInfo;
            savedSearchPaths = source.savedSearchPaths;
            var materialPairs = from t in Materials
                                from s in source.Materials
                                where t.Name == s.Name
                                select new { s, t };
            foreach (var p in materialPairs)
            {
                p.t.CopySavedData(p.s);
            }

            var bonePairs = from t in Bones
                            from s in source.Bones
                            where t.Name == s.Name
                            select new { s, t };
            foreach (var p in bonePairs)
            {
                p.t.CopySavedData(p.s);
            }

            var shapePairs = from t in Shapes
                             from s in source.Shapes
                             where t.Name == s.Name
                             select new { s, t};
            foreach (var p in shapePairs)
            {
                p.t.CopySavedData(p.s);
            }
        }

        public override void OnDocumentSwapAdd(Document old)
        {
            CopySavedData((Model)old);
            base.OnDocumentSwapAdd(old);
        }

        #endregion

        public void AddOptimizedLogFolder(string bfshaPath, string optimizedLogFolder)
        {
            lock (OptimizedLogLock)
            {
                OptimizedLogFolders[bfshaPath] = optimizedLogFolder;
            }
        }

        public string GetOptimizedLogFolder(string bfshaPath)
        {
            lock (OptimizedLogLock)
            {
                string folder = "";
                OptimizedLogFolders.TryGetValue(bfshaPath, out folder);
                return folder;
            }
        }
    }
}
