﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.IO;
using System.Text;
using System.Threading.Tasks;
using ConfigCommon;
using App.PropertyEdit;
using App.Utility;
using nw.g3d.nw4f_3dif;
using nw.g3d.iflib;
using nw.g3d.iflib.nw3de;

namespace App.Data
{
    public class SeparateMaterial :
        IntermediateFileDocument,
        IHasUserData,
        IMaterialOwner
    {
        public materialType Data
        {
            get;
            private set;
        }

        public override object nw4f_3difItem
        {
            get { return Data; }
        }

        public override process_log_arrayType process_log_array
        {
            get { return Data.process_log_array; }
        }

        public ReadOnlyList<Material> Materials
        {
            get
            {
                return new ReadOnlyList<Material>(materials_);
            }
        }
        private List<Material> materials_ = new List<Material>();

        public IEnumerable<AnimationDocument> AllAnimations
        {
            get
            {
                return DocumentManager.GetAnimations(DefaultAnimationSet.Animations.Concat(AnimationSets.SelectMany(x => x.Animations)).Distinct());
            }
        }

        public List<AnimationSet> AnimationSets { get; set; }
        public AnimationSet DefaultAnimationSet { get; set; }

        public IEnumerable<AnimationSet> AnimationSetsWithDefault
        {
            get
            {
                return Enumerable.Repeat(DefaultAnimationSet, 1).Concat(AnimationSets);
            }
        }

        /// <summary>
        /// ドキュメント内のマテリアルを直接参照しているモデルを取得する (マテリアル参照による間接参照は含まない)
        /// </summary>
        public System.Collections.ObjectModel.ReadOnlyDictionary<Material, List<Model>> Referrers
        {
            get
            {
                return new System.Collections.ObjectModel.ReadOnlyDictionary<Material, List<Model>>(models_);
            }
        }
        private Dictionary<Material, List<Model>> models_ = new Dictionary<Material, List<Model>>();

        public UserDataArray UserDataArray { get; set; }
        public UserDataArray SavedUserDataArray { get; set; }
        public bool UserDataArrayChanged { get; set; }

        public override nw4f_3difType Create_nw4f_3difType(bool viewer)
        {
            lock (this)
            {
                UpdateData();

                //			nw4f_3difType nw4f_3dif = new nw4f_3difType();
                nw4f_3dif_.Item = Data;
                nw4f_3dif_.file_info = null;

                return nw4f_3dif_;
            }
        }

        /// <summary>
        /// 参照しているドキュメント
        /// </summary>
        public override IEnumerable<Document> ReferenceDocuments
        {
            get
            {
                var textures = DocumentManager.Textures.Where(x => ReferenceTexturePaths.ContainsKey(x.Name) &&
                                                                     string.Equals(x.FilePath, ReferenceTexturePaths[x.Name], StringComparison.OrdinalIgnoreCase) &&
                                                                     Materials.SelectMany(y => y.ResolvedSamplerTextureNames).Any(z => z == x.Name));
                var shaderDefinitions = DocumentManager.ShaderDefinitions.Where(
                    x =>
                        DocumentManager.Materials.Any(
                            y => y.MaterialShaderAssign.ShaderDefinitionFileName == x.Name));
                var parentMaterialOwnerDocs =
                    Materials.SelectMany(x => x.ParentMaterials).Where(x => x != null && x.OwnerDocument != this).Select(x => x.OwnerDocument);

                return
                    textures
                        .Concat<Document>(shaderDefinitions)
                        .Concat<Document>(AllAnimations)
                        .Concat<Document>(parentMaterialOwnerDocs)
                        .Distinct();
            }
        }

        public void UpdateData()
        {
            Data.tool_data = GetToolData(ToolData);
            Data.comment = GetComment();

            // ユーザーデータのシリアライズ
            if (UserDataArray == null ||
                UserDataArray.Data == null ||
                UserDataArray.Data.Count == 0)
            {
                Data.user_data_array = null;
            }
            else
            {
                Data.user_data_array = new user_data_arrayType();
                this.MakeSerializeData(Data.user_data_array, UserDataArray, BinaryStreams);
            }

            foreach (var material in Materials)
            {
                material.UpdateData();
            }

            // ストリームの設定
            {
                // ストリームのソート(や削除等)を行う。
                nw.g3d.iflib.StreamUtility.SortStream(Data, BinaryStreams);

                Data.stream_array = null;
            }
        }

        public override IEnumerable<GuiObject> ContentObjects
        {
            get
            {
                return base.ContentObjects.
                    Concat<GuiObject>(materials_);
            }
        }

        public int savedMaterialCount;
        public override void UpdateSavedData()
        {
            base.UpdateSavedData();
            SavedUserDataArray = ObjectUtility.Clone(UserDataArray);
            UserDataArrayChanged = false;

            savedMaterialCount = Materials.Count;
            savedSearchPaths = ObjectUtility.Clone(SearchPaths);
            foreach (var content in ContentObjects.Where(x => x != this))
            {
                content.UpdateSavedData();
            }
        }

        public override bool EqualsToSavedData()
        {
            if (!base.EqualsToSavedData())
            {
                return false;
            }

            if (ContentObjects.Count() != SavedContents.Count)
            {
                return false;
            }

            if (Materials.Any(MaterialParentsPage.IsModified))
            {
                return false;
            }

            return
                !UserDataPage.IsModified(this) &&
                !SearchPathPage.IsModified(this);
        }

        #region ReferTexture
        // 参照しているテクスチャ
        public Dictionary<string, string> ReferenceTexturePaths
        {
            get
            {
                return referenceTexturePaths;
            }
            set
            {
                referenceTexturePaths = value;
            }
        }
        private Dictionary<string, string> referenceTexturePaths = new Dictionary<string, string>();
        #endregion

        public SeparateMaterial(materialType material, List<G3dStream> streamArray)
            : base(GuiObjectID.SeparateMaterial, streamArray)
        {
            Data = material;
            ToolData.Load(Data.tool_data);

            DefaultAnimationSet = new AnimationSet(false)
            {
                Name = res.Strings.FileTreeView_DefaultAnimationSet,
                IsDefaultAnimationSet = true
            };
            DefaultAnimationSet.UpdateSavedData();
            DefaultAnimationSet.SetNotModified();

            DefaultAnimationSet.Animations = new List<AnimationSetItem>();
            AnimationSets = new List<AnimationSet>();

            materials_.Add(new Material(Data, this, 0));
            foreach (var m in materials_)
            {
                models_.Add(m, new List<Model>());
            }
        }
    }
}
