﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App.ConfigData;
using ConfigCommon;
using nw.g3d.iflib;
using nw.g3d.iflib.nw3de;
using nw.g3d.nw4f_3dif;

namespace App.Data
{
    // 中間ファイル
    public abstract class IntermediateFileDocument : Document
    {
        protected IntermediateFileDocument(GuiObjectID obj, List<G3dStream> stream) : base(obj) {
            if (stream != null)
            {
                BinaryStreams = stream;
            }

            // ファイル読み込み時のnull 設定と区別するために必要です。
            file_info = new file_infoType();
        }

        /// <summary>
        /// 出力データ生成
        /// </summary>
        public abstract nw4f_3difType Create_nw4f_3difType(bool viewer = false);

        public file_infoType file_info { get; set; }
        public abstract process_log_arrayType process_log_array { get; }

        protected nw4f_3difType nw4f_3dif_ = new nw4f_3difType();
        public abstract object nw4f_3difItem { get; }

        private List<G3dStream> BinaryStreams_;
        public List<G3dStream> BinaryStreams
        {
            get
            {
                if (BinaryStreams_ == null)
                {
                    var nw4f_3dif = new nw4f_3difType();
                    nw4f_3dif.Item = nw4f_3difItem;
                    BinaryStreams_ = new List<G3dStream>();

                    // ストリームがあれば保持する
                    if (G3dStreamUtility.HasStreamArray(nw4f_3dif))
                    {
                        G3dStreamUtility.ToStreams(
                            BinaryStreams_, nw4f_3dif.RootElement.stream_array);

                        // 改行をバイナリファイルを読み込んだ時に合わせる
                        foreach (var stream in BinaryStreams_)
                        {
                            switch (stream.type)
                            {
                                case stream_typeType.@string:
                                case stream_typeType.wstring:
                                    stream.StringData = stream.StringData.Replace("\n", "\r\n");
                                    break;
                            }
                        }
                    }
                }

                return BinaryStreams_;
            }

            set
            {
                BinaryStreams_ = value;

                OnChangeBinaryStreams();
            }
        }

        public List<nw.g3d.iflib.nw3de.nw3de_SearchPath.SearchPath> SearchPaths
        {
            get
            {
                Debug.Assert(ObjectID == GuiObjectID.Model || ObjectID == GuiObjectID.SeparateMaterial || ObjectID == GuiObjectID.MaterialAnimation || ObjectID == GuiObjectID.TexturePatternAnimation);
                return ToolData.SearchPaths;
            }
            set
            {
                Debug.Assert(ObjectID == GuiObjectID.Model || ObjectID == GuiObjectID.SeparateMaterial || ObjectID == GuiObjectID.MaterialAnimation || ObjectID == GuiObjectID.TexturePatternAnimation);
                ToolData.SearchPaths = value;
            }
        }

        protected virtual void OnChangeBinaryStreams()
        {
            ;
        }

        public override void OnDocumentSwapAdd(Document old)
        {
            // ファイル情報は引き継ぐ
            file_info = ((IntermediateFileDocument)old).file_info;
            base.OnDocumentSwapAdd(old);
        }

        public override bool EqualsToSavedData()
        {
            if (!base.EqualsToSavedData()) return false;
            if (   IsFileInfoCreateSrcpathModified()
                || IsFileInfoCreateToolnameModified()
                || IsFileInfoCreateToolversionModified()
                || IsFileInfoModifyToolnameModified()
                || IsFileInfoModifyToolversionModified()
                )
            {
                return false;
            }
            return true;
        }

        #region savedData

        private file_infoType savedFileInfo;
        public List<nw3de_SearchPath.SearchPath> savedSearchPaths;

        public override void UpdateSavedData()
        {
            base.UpdateSavedData();
            savedFileInfo = file_info;
        }
        private static bool StringEqual(string a, string b)
        {
            return (string.IsNullOrEmpty(a) && string.IsNullOrEmpty(a)) || a == b;
        }

        private bool IsValidFileinfoCreate(file_infoType info)
        {
            return info != null && info.create != null;
        }

        private bool IsValidBothFileinfoCreate()
        {
            return IsValidFileinfoCreate(savedFileInfo) && IsValidFileinfoCreate(file_info);
        }

        private bool IsValidFileinfoModify(file_infoType info)
        {
            return info != null && info.create != null && info.modify != null;
        }

        private bool IsValidBothFileinfoModify()
        {
            return IsValidFileinfoModify(savedFileInfo) && IsValidFileinfoModify(file_info);
        }

        private bool IsNullBothFileinfoCreate()
        {
            return ((savedFileInfo == null || savedFileInfo.create == null)
                       && (file_info == null || file_info.create == null));
        }
        private bool IsNullBothFileinfoModify()
        {
            return ((savedFileInfo == null || savedFileInfo.modify == null)
                       && (file_info == null ||  file_info.modify== null));
        }

        public bool IsFileInfoCreateSrcpathModified()
        {
            if (IsValidBothFileinfoCreate()) return !StringEqual(savedFileInfo.create.src_path ,file_info.create.src_path);
            return !IsNullBothFileinfoCreate();
        }

        public bool IsFileInfoCreateToolnameModified()
        {
            if (IsValidBothFileinfoCreate()) return !StringEqual(savedFileInfo.create.tool_name ,file_info.create.tool_name);
            return !IsNullBothFileinfoCreate();
        }

        public bool IsFileInfoCreateToolversionModified()
        {
            if (IsValidBothFileinfoCreate()) return !StringEqual(savedFileInfo.create.tool_version, file_info.create.tool_version);
            return !IsNullBothFileinfoCreate();
        }

        public bool IsFileInfoModifyToolnameModified()
        {
            if (IsValidBothFileinfoModify()) return !StringEqual(savedFileInfo.modify.tool_name,file_info.modify.tool_name);
            return !IsNullBothFileinfoModify();
        }

        public bool IsFileInfoModifyToolversionModified()
        {
            if (IsValidBothFileinfoModify()) return !StringEqual(savedFileInfo.modify.tool_version, file_info.modify.tool_version);
            return !IsNullBothFileinfoModify();
        }

        #endregion

        internal static tool_dataType GetToolData(ToolData toolData)
        {
            var elements = toolData.GetElements();
            elements.RemoveAll(x => x.Name == typeof(nw3de_SearchPath).Name);
            if (toolData.SearchPaths.Any())
            {
                var item = new nw3de_SearchPath()
                {
                    SearchPaths = toolData.SearchPaths,
                };
                elements.Add(IfToolData.ConvertToXmlElement(item, 1));
            }

            return elements.Any()
                ? new tool_dataType()
                {
                    Any = elements.ToArray()
                }
                : null;
        }

        public bool IsSearchPathsChanged()
        {
            return savedSearchPaths.Count != SearchPaths.Count ||
                   Enumerable.Any<bool>(savedSearchPaths.Zip(SearchPaths, (x, y) => x.Path == y.Path && x.Recursive == y.Recursive), x => !x);
        }
    }
}
