﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App.Utility;
using ConfigCommon;

namespace App.Data
{
    /// <summary>
    /// ＧＵＩオブジェクトグループクラス。
    /// </summary>
    public sealed class GuiObjectGroup
    {
        // オブジェクトリスト
        private readonly List<GuiObject>			objects;
        private readonly ReadOnlyList<GuiObject>	objectsReadOnly;

        // アクティブオブジェクト
        private GuiObject active;

        /// <summary>内容変更イベント。</summary>
        public event EventHandler Changed;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public GuiObjectGroup()
        {
            objects			= new List<GuiObject>();
            objectsReadOnly	= new ReadOnlyList<GuiObject>(objects);
            invokeBlockCounter.ReleasedEvent += InvokeChanged;
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public GuiObjectGroup(GuiObjectGroup src)
        {
            objects			= new List<GuiObject>(src.objects);
            objectsReadOnly	= new ReadOnlyList<GuiObject>(objects);
            active			= src.active;
            invokeBlockCounter.ReleasedEvent += InvokeChanged;
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public GuiObjectGroup(GuiObject guiObject) : this()
        {
            Add(guiObject, true);
            invokeBlockCounter.ReleasedEvent += InvokeChanged;
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public GuiObjectGroup(IEnumerable<GuiObject> src)
        {
            objects = new List<GuiObject>(src);
            objectsReadOnly = new ReadOnlyList<GuiObject>(objects);
            active = src.FirstOrDefault();
            invokeBlockCounter.ReleasedEvent += InvokeChanged;
        }
        /// <summary>
        /// アクティブオブジェクト。
        /// </summary>
        public GuiObject Active
        {
            get { return active; }
            set
            {
                Debug.Assert(value != null);
                Debug.Assert(objects.Contains(value));

                if (active != value)
                {
                    active = value;
                    InvokeChanged();
                }
            }
        }

        /// <summary>
        /// 未設定かどうか。
        /// </summary>
        public bool IsEmpty
        {
            get { return objects.Count == 0; }
        }

        /// <summary>
        /// 複数設定かどうか。
        /// </summary>
        public bool IsMulti
        {
            get { return objects.Count >= 2; }
        }

        /// <summary>
        /// オブジェクトリストを取得。
        /// </summary>
        public ReadOnlyList<GuiObject> Objects
        {
            get
            {
                return objectsReadOnly;
            }
        }

        /// <summary>
        /// 指定したＩＤのオブジェクトリストを取得。
        /// </summary>
        public ReadOnlyList<GuiObject> GetObjects(GuiObjectID objectID)
        {
            return new ReadOnlyList<GuiObject>(objects.FindAll(x => x.ObjectID == objectID));
        }

        public ReadOnlyList<GuiObject> GetObjects(IEnumerable<GuiObjectID> objectIDs)
        {
            var list = new List<GuiObject>();
            {
                foreach(var objectID in objectIDs)
                {
                    list.AddRange(GetObjects(objectID));
                }
            }
            return new ReadOnlyList<GuiObject>(list);
        }

        /// <summary>
        /// オブジェクト数を取得。
        /// </summary>
        public int GetCount(GuiObjectID objectID)
        {
            return objects.Count(x => x.ObjectID == objectID);
        }

        /// <summary>
        /// オブジェクトが含まれているか。
        /// </summary>
        public bool Contains(GuiObject obj)
        {
            Debug.Assert(obj != null);
            return objects.Contains(obj);
        }

        /// <summary>
        /// クリア。
        /// </summary>
        public void Clear()
        {
            if (objects.Count > 0)
            {
                objects.Clear();
                active = null;
                InvokeChanged();
            }
        }

        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(GuiObject obj, bool forceInvokeChanged = false)
        {
            if (obj == null)
            {
                Clear();
            }
            else
            {
                if (!(objects.Count == 1 && active == obj))
                {
                    objects.Clear();
                    objects.Add(obj);
                    active = obj;
                    InvokeChanged();
                }
                else if (forceInvokeChanged)
                {
                    InvokeChanged();
                }
            }
        }

        public void Set(IEnumerable<GuiObject> objs, GuiObject activeObj, bool forceInvokeChanged = false)
        {
            Debug.Assert(objs != null);

            if (!(objects.SequenceEqual(objs) && active == activeObj))
            {
                objects.Clear();
                objects.AddRange(objs);
                active = objects.Contains(activeObj) ? activeObj : objects.LastOrDefault();

                InvokeChanged();
            }
            else if (forceInvokeChanged)
            {
                InvokeChanged();
            }
        }
        public void AddRange(IEnumerable<GuiObject> objs)
        {
            var newobjs = objs.Where(x => !Contains(x));
            if (!newobjs.Any())
            {
                return;
            }
            objects.AddRange(newobjs);
            if (active == null)
            {
                active = newobjs.FirstOrDefault();
            }
        }

        /// <summary>
        /// 追加。
        /// </summary>
        public void Add(GuiObject obj)
        {
            Add(obj, true);
        }

        /// <summary>
        /// 追加。
        /// </summary>
        public void Add(GuiObject obj, bool activate)
        {
            Debug.Assert(obj != null);
            Debug.Assert(!objects.Contains(obj));

            objects.Add(obj);

            if (active == null || activate)
            {
                active = obj;
            }
            InvokeChanged();
        }

        /// <summary>
        /// 削除。
        /// </summary>
        public void Remove(GuiObject obj)
        {
            Debug.Assert(obj != null);
            Debug.Assert(objects.Contains(obj));

            objects.Remove(obj);

            if (objects.Count > 0 && active == obj)
            {
                active = objects[0];
            }
            else if (objects.Count == 0)
            {
                active = null;
            }
            InvokeChanged();
        }

        public void Remove(GuiObjectID id)
        {
            // 削除しようとしているオブジェクトがあるか？
            var isExists = objects.Exists(x => x.ObjectID == id);

            if (isExists)
            {
                // 削除しようとしているものがアクティブか？
                var isActiveRemove = (active != null) && (active.ObjectID == id);

                objects.RemoveAll(x => x.ObjectID == id);

                if (objects.Count == 0)
                {
                    active = null;
                }
                else if (isActiveRemove)
                {
                    active = objects[0];
                }

                InvokeChanged();
            }
        }

        public bool Swap(GuiObject Old, GuiObject New)
        {
            int index = objects.IndexOf(Old);
            if (index >= 0)
            {
                objects[index] = New;
                if (active == Old)
                {
                    active = New;
                }
                InvokeChanged();
                return true;
            }
            return false;
        }

        #region 比較
        /// <summary>
        /// 等値比較。
        /// </summary>
        public bool Equals(GuiObjectGroup src)
        {
            // TODO: 実装間違ってるから Equals とともに削除
            Debug.Assert(false);
            if (src == null) return false;
            if (src == this) return true;

            // 比較処理
            if (active == src.active)
            {
                if (objects.Count == src.objects.Count)
                {
                    foreach (GuiObject obj in objects)
                    {
                        if (!src.objects.Contains(obj))
                        {
                            return false;
                        }
                    }
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override bool Equals(object obj)
        {
            // TODO: 実装間違ってるから Equals とともに削除
            Debug.Assert(false);
            return Equals(obj as GuiObjectGroup);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override int GetHashCode()
        {
            // TODO: 実装間違ってるから Equals とともに削除
            Debug.Assert(false);
            return base.GetHashCode();
        }
        #endregion

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override string ToString()
        {
            // TODO: 文字列化
            return base.ToString();
        }

        /// <summary>
        /// 内容変更イベント発行。
        /// </summary>
        private void InvokeChanged()
        {
            if (!invokeBlockCounter.CheckBlock)
            {
                if (Changed != null)
                {
                    Changed(this, EventArgs.Empty);
                }
            }
        }

        public SuppressBlockCounter InvokeBlockCounter
        {
            get
            {
                return invokeBlockCounter;
            }
        }

        private readonly SuppressBlockCounter invokeBlockCounter = new SuppressBlockCounter();

    }
}
