﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using App.Command;
using App.PropertyEdit;
using App.Utility;
using ConfigCommon;
using nw.g3d.iflib;
using nw.g3d.iflib.nw3de;
using nw.g3d.nw4f_3dif;

namespace App.Data
{
    public sealed class TexturePatternAnimation : AnimationDocument, IHasUserData, IReferTexture, IHasTexturePatternAnimation
    {
        public tex_pattern_animType Data{ get; private set; }
        public override object nw4f_3difItem
        {
            get { return Data; }
        }

        public override int FrameCount
        {
            get{ return Data.tex_pattern_anim_info.frame_count; }
            set{ Data.tex_pattern_anim_info.frame_count = value; }
        }

        public override bool Loop
        {
            get { return Data.tex_pattern_anim_info.loop; }
            set { Data.tex_pattern_anim_info.loop = value; }
        }

        public override bool IsFrameCountModified
        {
            get { return savedData.tex_pattern_anim_info.frame_count != Data.tex_pattern_anim_info.frame_count; }
        }

        public override bool IsLoopModified
        {
            get { return savedData.tex_pattern_anim_info.loop != Data.tex_pattern_anim_info.loop; }
        }

        // ファイル出力できるか？
        public override bool IsFileOutputable
        {
            get
            {
                return FileNotOutputErrorMessage == null;
            }
        }

        public override string FileNotOutputErrorMessage
        {
            get
            {
                string error = null;
                Action<TexPatternMatAnim, PatternAnimTarget> errorHandler = delegate(TexPatternMatAnim texPatternMatAnim, PatternAnimTarget patternAnim)
                {
                    error = string.Format(res.Strings.FileNotOutputError_TexturePatternAnimation, texPatternMatAnim.mat_name, patternAnim.sampler_name);
                };
                foreach (var tuple in QuantizeAnalyse(errorHandler))
                {
                    if (error != null)
                    {
                        return error;
                    }
                }
                return null;
            }
        }

        public override bool HasAnimation
        {
            get
            {
                return TexPatternMatAnims != null &&
                       TexPatternMatAnims
                            .Where(x => x.PatternAnimTargets != null)
                            .Any(x => x.PatternAnimTargets.Any());
            }
        }


        public override IEnumerable<Tuple<AnimTarget, IfQuantizationAnalysisResult>> QuantizeAnalyse()
        {
            return QuantizeAnalyse(null);
        }

        private IEnumerable<Tuple<AnimTarget, IfQuantizationAnalysisResult>> QuantizeAnalyse(Action<TexPatternMatAnim, PatternAnimTarget> errorHandler)
        {
            foreach (var texPatternMatAnim in TexPatternMatAnims)
            {
                foreach (var patternAnim in texPatternMatAnim.PatternAnimTargets)
                {
                    var curve = new TexturePatternAnimationCurveTreeNodeInfo(this, texPatternMatAnim.mat_name, patternAnim.sampler_name);

                    IfQuantizationAnalysisResult result = null;
                    try
                    {
                        result = AnimationCurveEditCommand.MakeQuantizationAnalysis(this, ObjectID, curve, false);
                    }
                    catch (Exception)
                    {
                        if (errorHandler != null)
                        {
                            errorHandler(texPatternMatAnim, patternAnim);
                        }
                    }
                    yield return new Tuple<AnimTarget, IfQuantizationAnalysisResult>(patternAnim, result);
                }
            }
        }

        public override bool IsEditable
        {
            get
            {
                // 注意：
                // TexturePatternAnimationCurveEditPage の UpdateTreeView() をもとに実装しました
                // 方針：
                // なるべく元のソースをいじらないように実装

                bool IsBound = true;
                var ActiveTarget = this;

                IEnumerable<Model> parentModels = ParentModels;

                foreach (var texPatternMatAnims in ActiveTarget.TexPatternMatAnims)
                {
                    var materialName = texPatternMatAnims.mat_name;
                    var materials = from model in parentModels
                                    from material in model.Materials
                                    where material.Name == materialName
                                    select material;

                    {
                        var addedKey = new Dictionary<string, bool>();

                        // 親のモデルから構築する
                        var query = new List<string>();
                        foreach (var sampler in materials.SelectMany(x => x.sampler_array.sampler))
                        {
                            // データで追加していないものを追加する
                            if (addedKey.ContainsKey(sampler.name) == false)
                            {
                                query.Add(sampler.name);
                                addedKey.Add(sampler.name, true);
                            }
                        }

                        // データから構築する
                        foreach (var patternAnimTarget in texPatternMatAnims.PatternAnimTargets)
                        {
                            // データで追加していないものを追加する
                            if (addedKey.ContainsKey(patternAnimTarget.sampler_name) == false)
                            {
                                query.Add(patternAnimTarget.sampler_name);
                                addedKey.Add(patternAnimTarget.sampler_name, false);
                            }
                        }

                        foreach (var sampler in query)
                        {
                            IsBound = addedKey[sampler];
                            if (IsBound == false)
                            {
                                return IsBound;
                            }
                        }
                    }
                }

                Debug.Assert(IsBound);
                return IsBound;
            }
        }

        private IEnumerable<Model> ParentModels
        {
            get
            {
                var ActiveTarget = this;

                return
                    from model in DocumentManager.Models
                    from anim in model.AllAnimations
                    where anim == ActiveTarget
                    select model;
            }
        }

        public UserDataArray UserDataArray { get; set; }
        public UserDataArray SavedUserDataArray { get; set; }
        public bool UserDataArrayChanged { get; set; }

        public override process_log_arrayType process_log_array
        {
            get { return Data.process_log_array; }
        }

        [Serializable]
        public class TexPattern
        {
            public string	tex_name{ get; set; }
        }

        [Serializable]
        public class TexPatternMatAnim : ITexPatternMatAnim
        {
            public TexPatternMatAnim()
            {
                PatternAnimTargets = new List<PatternAnimTarget>();
            }

            public string					mat_name{			get; set; }				// tex_pattern_mat_anim.mat_name
            public List<PatternAnimTarget>	PatternAnimTargets{	get; set; }
        }

        [Serializable]
        public class PatternAnimTarget : AnimTarget
        {
            public PatternAnimTarget()
            {
                KeyFrames = new List<KeyFrame>();
                QuantizationInfo = new QuantizationInfo();
            }

            //public bool					IsTemporary { get; set; }			// IsTemporary && (KeyFrames.Any() == false) の時はシリアライズしない

            public string				ParentMatName{	get; set; }				// 親のmat_name

            public string				sampler_name{	get; set; }				// pattern_anim_target.sampler_name
            public string				hint{			get; set; }				// pattern_anim_target.hint
            //public float				base_value{		get; set; }				// pattern_anim_target.base_value

            //public QuantizationInfo		QuantizationInfo{ get; set; }
            //public curve_wrapType		pre_wrap		{ get; set; }	// step_curveType.pre_wrap;
            //public curve_wrapType		post_wrap		{ get; set; }	// step_curveType.post_wrap;
            //public bool				 	baked			{ get; set; }	// step_curveType.baked;

            //public List<KeyFrame>		KeyFrames{		get; set; }

            public pattern_anim_targetType ConvertTo_pattern_anim_targetType(out G3dStream stream, bool viewer = false)
            {
                step_curveType curve;
                Convert(out curve, out stream, viewer);
                var target = new pattern_anim_targetType
                {
                    base_value = GetBaseValue(),
                    Curve = curve,
                    sampler_name = sampler_name,
                    hint = hint,
                };
                return target;
            }
        }

        public List<TexPattern>			TexPatterns{		get; set; }
        public List<TexPatternMatAnim>	TexPatternMatAnims{	get; set; }
        public IEnumerable<ITexPatternMatAnim> ITexPatternMatAnims { get { return TexPatternMatAnims; } }

        #region 圧縮率
        public bool	IsAllFrameCurveConstant { get { return TexPatternMatAnims.SelectMany(x => x.PatternAnimTargets).Any(x => x.KeyFrames.Any()) == false; } }
        public int	FrameCompressDataSize   { get { return GetFrameDataSize(false); } }
        public int	FrameUncompressDataSize { get { return GetFrameDataSize(true);  } }
        #endregion

        // 編集コマンド用
        [Serializable]
        public class QuantizeToleranceData
        {
            public float					QuantizeToleranceFrame{ get; set; }
            public List<TexPatternMatAnim>	TexPatternMatAnims{ get; set; }
        }

        public TexturePatternAnimation(tex_pattern_animType animation, List<G3dStream> streams) : base(GuiObjectID.TexturePatternAnimation, streams)
        {
            Data = animation;
            Initialize();
        }

        public void Initialize()
        {
            ToolData.Load(Data.tool_data);
            MakeComment(Data.comment);

            // ユーザーデータ作成
            this.MakeUserData(Data.user_data_array, BinaryStreams);

            MakeTexPatterns();
            MakeTexPatternMatAnims();
        }

        #region Bridge
        public class TexPatternAnimSwapData : SwapData
        {
            public ToolData ToolData;
            public List<G3dStream> streams;
            public string Comment;
            public RgbaColor? EditColor;
            public tex_pattern_animType Data;
            public List<TexPattern> TexPatterns;
            public List<TexPatternMatAnim> TexPatternMatAnims;
            public UserDataArray userData;
        }

        public override SwapData CreateSwapData(object data, List<G3dStream> streams)
        {
            var current = GetSwapData();
            Data = (tex_pattern_animType)data;
            BinaryStreams = streams;
            Initialize();

            return Swap(current, false);
        }

        public override SwapData GetSwapData()
        {
            return new TexPatternAnimSwapData()
            {
                Data = Data,
                streams = BinaryStreams,
                Comment = Comment,
                EditColor = EditColor,
                TexPatterns = TexPatterns,
                TexPatternMatAnims = TexPatternMatAnims,
                userData = UserDataArray,
            };
        }

        public override SwapData Swap(SwapData swapData, bool copyHistory)
        {
            var current = (TexPatternAnimSwapData)GetSwapData();
            var data = (TexPatternAnimSwapData)swapData;
            Data = data.Data;
            BinaryStreams = data.streams;
            Comment = data.Comment;
            EditColor = data.EditColor;
            TexPatterns = data.TexPatterns;
            TexPatternMatAnims = data.TexPatternMatAnims;
            UserDataArray = data.userData;

            if (copyHistory)
            {
                UserDataArrayChanged = !SavedUserDataArray.IsSame(UserDataArray);
                UpdateIsModifiedAnimTargetAll();
            }

            return current;
        }
        #endregion

        public override nw4f_3difType Create_nw4f_3difType(bool viewer)
        {
            lock (this)
            {
                UpdateData(viewer);

                //			nw4f_3difType nw4f_3dif = new nw4f_3difType();
                nw4f_3dif_.Item = Data;
                nw4f_3dif_.file_info = null;

                return nw4f_3dif_;
            }
        }

        public void Optimize()
        {
            // ストリームを更新する。
            UpdateData();
            // <hermite_curve>、<linear_curve>、<step_curve> の frame_type と key_type を
            // none に設定しないと、再分析が行われません。
            if (Data.tex_pattern_mat_anim_array != null)
            {
                foreach (var anim in Data.tex_pattern_mat_anim_array.tex_pattern_mat_anim)
                {
                    foreach (var target in anim.pattern_anim_target)
                    {
                        if (target.step_curve != null)
                        {
                            target.step_curve.frame_type = curve_frame_typeType.none;
                            target.step_curve.key_type = curve_key_typeType.none;
                        }
                    }
                }
            }

            try
            {
                // アニメーションに合わせたAnimQuantizationAnalysisOptimizerを呼び出す。
                IfTexPatternAnimQuantizationAnalysisOptimizer optimizer = new IfTexPatternAnimQuantizationAnalysisOptimizer();
                optimizer.Optimize(Data, BinaryStreams);
            }
            catch(Exception e)
            {
                DebugConsole.WriteLine("Exception Catch : IfTexPatternAnimQuantizationAnalysisOptimizer.Optimize() : {0}", e.Message);
            }

            // 最適化後に、データを再作成する。
            this.MakeUserData(Data.user_data_array, BinaryStreams);
            MakeTexPatterns();
            MakeTexPatternMatAnims();
        }

        /// <summary>
        /// 参照しているドキュメント(シェーダーは除いてあります)
        /// </summary>
        public override IEnumerable<Document> ReferenceDocuments
        {
            get
            {
                return from tex in DocumentManager.Textures
                       where ReferenceTexturePaths.ContainsKey(tex.Name) &&
                             string.Compare(ReferenceTexturePaths[tex.Name], tex.FilePath, true) == 0 &&
                             TexPatterns.Any(x => x.tex_name == tex.Name)
                       select tex;
            }
        }

        public IEnumerable<string> UnresolvedTextures()
        {
            return (from pattern in TexPatterns
                    where !ReferenceTexturePaths.ContainsKey(pattern.tex_name)
                    select pattern.tex_name).Distinct();
        }

        private int GetFrameDataSize(bool isBase)
        {
            int size = 0;
            {
                foreach(var texPatternMatAnim in TexPatternMatAnims)
                {
                    foreach(var patternAnimTarget in texPatternMatAnim.PatternAnimTargets)
                    {
                        size += patternAnimTarget.KeyFrames.Count * (isBase ? 4 : IfUtility.GetByteSizeFromCurveFrameType(patternAnimTarget.QuantizationInfo.frame_type));
                    }
                }
            }
            return size;
        }

        public PatternAnimTarget GetParamAnimFromId(string materialName, string sampler_name)
        {
            var texPatternMatAnim = TexPatternMatAnims.FirstOrDefault(x => x.mat_name == materialName);
            if (texPatternMatAnim != null)
            {
                return texPatternMatAnim.PatternAnimTargets.FirstOrDefault(x => x.sampler_name == sampler_name);
            }
            else
            {
                return null;
            }
        }

        private void MakeTexPatterns()
        {
            TexPatterns = new List<TexPattern>();

            if ((Data.tex_pattern_array == null) ||
                (Data.tex_pattern_array.tex_pattern == null))
            {
                return;
            }

            foreach (var texPattern in Data.tex_pattern_array.tex_pattern)
            {
                TexPatterns.Add(
                    new TexPattern()
                    {
                        tex_name		= texPattern.tex_name
                    }
                );
            }
        }

        private void MakeTexPatternMatAnims()
        {
            TexPatternMatAnims = new List<TexPatternMatAnim>();

            if ((Data.tex_pattern_mat_anim_array == null) ||
                (Data.tex_pattern_mat_anim_array.tex_pattern_mat_anim == null))
            {
                return;
            }

            foreach (var texPatternMatAnim in Data.tex_pattern_mat_anim_array.tex_pattern_mat_anim)
            {
                var newTexPatternMatAnim = new TexPatternMatAnim()
                {
                    mat_name		= texPatternMatAnim.mat_name,
                };

                foreach (var patternAnimTarget in texPatternMatAnim.pattern_anim_target)
                {
                    var newPatternAnimTarget = new PatternAnimTarget()
                    {
                        ParentMatName	= newTexPatternMatAnim.mat_name,
                        sampler_name	= patternAnimTarget.sampler_name,
                        hint			= patternAnimTarget.hint,
                    };

                    // キーフレームがあれば保存する
                    if (patternAnimTarget.step_curve != null)
                    {
                        var curve = patternAnimTarget.step_curve;

                        bool exist = true;
                        if (curve == null)
                        {
                            curve = new step_curveType();
                            exist = false;
                        }

                        newPatternAnimTarget.QuantizationInfo.frame_type = curve.frame_type;
                        newPatternAnimTarget.QuantizationInfo.key_type = curve.key_type;
                        newPatternAnimTarget.QuantizationInfo.scale = curve.scale;
                        newPatternAnimTarget.QuantizationInfo.offset = curve.offset;
                        newPatternAnimTarget.pre_wrap = curve.pre_wrap;
                        newPatternAnimTarget.post_wrap = curve.post_wrap;
                        newPatternAnimTarget.baked = curve.baked;

                        if (exist)
                        {
                            const int dataElementCount = 2;		// frame, value

                            var stream = BinaryStreams[curve.stream_index];
                            var data = stream.FloatData;
                            Debug.Assert(data != null);
                            Debug.Assert(data.Count == curve.count * dataElementCount);

                            for (int i = 0; i != curve.count; ++i)
                            {
                                newPatternAnimTarget.KeyFrames.Add(
                                    new KeyFrame()
                                    {
                                        Frame = data[i * dataElementCount + 0],
                                        Value = data[i * dataElementCount + 1],
                                    }
                                );
                            }
                        }
                    }
                    else
                    {
                        // コンスタントカーブの場合、中間ファイルにはカーブは存在しないので、
                        // base_value を使用して、0フレームにキーフレームを生成する。
                        newPatternAnimTarget.KeyFrames.Add(
                            new KeyFrame()
                            {
                                Frame = 0.0f,
                                Value = patternAnimTarget.base_value
                            }
                        );
                    }

                    newTexPatternMatAnim.PatternAnimTargets.Add(newPatternAnimTarget);
                }

                TexPatternMatAnims.Add(newTexPatternMatAnim);
            }
        }

        public void UpdateData(bool viewer = false)
        {
            Data.tool_data = Model.GetToolData(ToolData);
            Data.comment = GetComment();

            if (Data.tex_pattern_array == null)
            {
                Data.tex_pattern_array = new tex_pattern_arrayType();
            }

            if (Data.tex_pattern_mat_anim_array == null)
            {
                Data.tex_pattern_mat_anim_array = new tex_pattern_mat_anim_arrayType();
            }

            var	tempTexPatterns			= new List<tex_patternType>();
            var	tempTexPatternMatAnims	= new List<tex_pattern_mat_animType>();
            var	tempStreams				= new List<G3dStream>();

            {
                int index = 0;

                foreach (var texPattern in TexPatterns)
                {
                    tempTexPatterns.Add(
                        new tex_patternType()
                        {
                            pattern_index = index,
                            tex_name = texPattern.tex_name
                        }
                    );

                    ++ index;
                }
            }

            {
                int targetIndex = 0;	// stream_indexとして使用します。
                foreach (var texPatternMatAnim in TexPatternMatAnims)
                {
                    // データ
                    {
                        var tempTexPatternMatAnim = new tex_pattern_mat_animType()
                        {
                            mat_name	= texPatternMatAnim.mat_name,
                        };

                        var tempPatternAnimTarget = new List<pattern_anim_targetType>();

                        foreach (var patternAnimTarget in texPatternMatAnim.PatternAnimTargets)
                        {
                            // キーフレームが無いものは削除するように修正します。
                            if (patternAnimTarget.KeyFrames.Any() == false)
                            {
                                continue;
                            }

                            // 通常のカーブのはず
                            step_curveType curve = null;
                            if (patternAnimTarget.KeyFrames.Any())
                            {
                                // カーブを作っちゃいます。
                                curve = new step_curveType()
                                {
                                    count			= patternAnimTarget.KeyFrames.Count,
                                    frame_type		= patternAnimTarget.QuantizationInfo.frame_type,
                                    key_type		= patternAnimTarget.QuantizationInfo.key_type,
                                    scale			= patternAnimTarget.QuantizationInfo.scale,
                                    offset			= patternAnimTarget.QuantizationInfo.offset,
                                    pre_wrap		= patternAnimTarget.pre_wrap,
                                    post_wrap		= patternAnimTarget.post_wrap,
                                    baked			= patternAnimTarget.baked,
                                    stream_index	= targetIndex
                                };
                            }
                            else
                            {
                                // コンスタントカーブの場合、カーブは作成しない。
                            }

                            tempPatternAnimTarget.Add(
                                new pattern_anim_targetType()
                                {
                                    sampler_name = patternAnimTarget.sampler_name,
                                    hint = patternAnimTarget.hint,
                                    base_value = patternAnimTarget.GetBaseValue(),
                                    step_curve = curve
                                }
                            );


                            // ストリーム
                            {
                                // コンスタントカーブかどうかをチェックする
                                {
                                    float base_value = 0.0f;
                                    if (patternAnimTarget.KeyFrames.IsConstantCurve(out base_value))
                                    {
                                        if (viewer)
                                        {
                                            // ランタイム連携用に水平なカーブを出力する
                                            tempPatternAnimTarget.Last().step_curve = new step_curveType()
                                            {
                                                count = 2,
                                                frame_type = curve_frame_typeType.none,
                                                key_type = curve_key_typeType.none,
                                                scale = 1,
                                                offset = 0,
                                                pre_wrap = curve_wrapType.clamp,
                                                post_wrap = curve_wrapType.clamp,
                                                baked = false,
                                                stream_index = targetIndex,
                                            };
                                            var stream = new G3dStream()
                                            {
                                                type = stream_typeType.@float,
                                                column = 2,
                                            };

                                            stream.FloatData.Add(0); // フレーム
                                            stream.FloatData.Add(base_value);

                                            stream.FloatData.Add(1); // フレーム
                                            stream.FloatData.Add(base_value);
                                            tempStreams.Add(stream);
                                            ++targetIndex;
                                        }
                                        else
                                        {
                                            // コンスタントカーブなのでベース値を保存し、カーブを削除する
                                            tempPatternAnimTarget.Last().step_curve = null;
                                            tempPatternAnimTarget.Last().base_value = base_value;
                                        }
                                        continue;
                                    }
                                }

                                var tempStream = new G3dStream()
                                {
                                    type = stream_typeType.@float,
                                    column = 2
                                };

                                foreach (var keyFrame in patternAnimTarget.KeyFrames)
                                {
                                    tempStream.FloatData.Add(keyFrame.Frame);
                                    tempStream.FloatData.Add(keyFrame.Value);
                                }

                                tempStreams.Add(tempStream);
                                ++ targetIndex;
                            }
                        }

                        // iflibのフォーマットの仕様に合わせる。
                        if (tempPatternAnimTarget.Any())
                        {
                            tempTexPatternMatAnim.pattern_anim_target = tempPatternAnimTarget.ToArray();
                            tempTexPatternMatAnims.Add(tempTexPatternMatAnim);
                        }
                    }
                }
            }

            //  リストを配列に変換して設定
            {
                {
                    Data.tex_pattern_array.length = tempTexPatterns.Count;
                    Data.tex_pattern_array.tex_pattern = tempTexPatterns.ToArray();

                    if (Data.tex_pattern_array.length == 0)
                    {
                        Data.tex_pattern_array = null;
                    }
                }

                {
                    Data.tex_pattern_mat_anim_array.length = tempTexPatternMatAnims.Count;
                    Data.tex_pattern_mat_anim_array.tex_pattern_mat_anim = tempTexPatternMatAnims.Any() ? tempTexPatternMatAnims.ToArray() : null;

                    if (Data.tex_pattern_mat_anim_array.length == 0)
                    {
                        Data.tex_pattern_mat_anim_array = null;
                    }
                }
            }

            // ユーザーデータのシリアライズ
            if (UserDataArray.Data == null ||
                UserDataArray.Data.Count == 0)
            {
                Data.user_data_array = null;
            }
            else
            {
                Data.user_data_array = new user_data_arrayType();
                this.MakeSerializeData(Data.user_data_array, UserDataArray, tempStreams);
            }

            // ストリームの設定
            {
                BinaryStreams = tempStreams;

                // ストリームのソート(や削除等)を行う。
                nw.g3d.iflib.StreamUtility.SortStream(Data, BinaryStreams);

                Data.stream_array = null;
            }
        }

        // 参照されているオブジェクト名変更コマンドを返す
        public override ICommand CreateReferenceObjectRenameCommand(Document srcObject, string oldName, string oldDirectory, string newName, string newExt, string newDirectory)
        {
            var commandSet = new EditCommandSet();
            {
                var newTexPatterns = new List<TexPattern>();
                {
                    foreach(var texPattern in TexPatterns)
                    {
                        newTexPatterns.Add(
                            new TexPattern()
                            {
                                tex_name = (texPattern.tex_name == oldName) ? newName : texPattern.tex_name
                            }
                        );
                    }
                }

                commandSet.Add(
                    PropertyEdit.TexturePatternAnimationPatternPage.CreateEditCommand_TexPatterns(
                        new GuiObjectGroup(this),
                        newTexPatterns,
                        false
                    )
                );
            }
            return (commandSet.CommandCount > 0) ? commandSet : null;
        }

        public override EditCommand CreateUpdateBindCommand()
        {
            bool updateTexPatternMatAnims = false;
            var newTexPatternMatAnims = new List<TexPatternMatAnim>();

            // バインドされたマテリアルのグループ
            var materials = (from model in DocumentManager.Models
                             where model.AllAnimations.Contains(this)
                             from material in model.Materials
                             select material).ToArray();

            var materialTable = new Dictionary<string, List<Material>>();
            var materialNames = new HashSet<string>();
            foreach (var material in materials)
            {
                List<Material> list;
                if (materialTable.TryGetValue(material.Name, out list))
                {
                    list.Add(material);
                }
                else
                {
                    materialTable.Add(material.Name, new List<Material>() { material });
                    materialNames.Add(material.Name);
                }
            }

            var oldTexPatternMatAnimDictionary = new Dictionary<string, TexPatternMatAnim>();
            // 既存のTexPatternMatAnim
            foreach (var texPatternMatAnim in TexPatternMatAnims)
            {
                if (materialTable.ContainsKey(texPatternMatAnim.mat_name) ||
                    texPatternMatAnim.PatternAnimTargets.Any(x => x.ExportType != CurveExportType.Ignored))
                {
                    oldTexPatternMatAnimDictionary.Add(texPatternMatAnim.mat_name, texPatternMatAnim);
                    materialNames.Add(texPatternMatAnim.mat_name);
                }
                else
                {
                    updateTexPatternMatAnims = true;
                }
            }

            // マテリアル名順に並べる
            foreach (var materialName in materialNames.OrderBy(x => x))
            {
                if (oldTexPatternMatAnimDictionary.ContainsKey(materialName))
                {
                    newTexPatternMatAnims.Add(oldTexPatternMatAnimDictionary[materialName]);
                }
                else
                {
                    newTexPatternMatAnims.Add(new TexPatternMatAnim()
                    {
                        mat_name = materialName,
                    });
                    updateTexPatternMatAnims = true;
                }
            }

            // ParamAnims の更新
            var patternAnims = new List<PatternAnimsSwapData>();
            foreach (var texPatternMatAnim in newTexPatternMatAnims)
            {
                List<Material> materialList;
                var paramNames = new List<string>();
                HashSet<string> nameSet = null;
                if (materialTable.TryGetValue(texPatternMatAnim.mat_name, out materialList))
                {
                    paramNames.AddRange(materialList.SelectMany(x => x.sampler_array.sampler).Select(x => x.name).Distinct());
                    nameSet = new HashSet<string>(paramNames);
                }
                else
                {
                    nameSet = new HashSet<string>();
                }

                bool updatePatternAnimTargets = false;
                var newPatternAnimTargets = new List<PatternAnimTarget>();
                var oldPatternAnimTargets = new Dictionary<string, PatternAnimTarget>();
                // 既存のもの
                foreach (var patternAnimTarget in texPatternMatAnim.PatternAnimTargets)
                {
                    if (patternAnimTarget.ExportType != CurveExportType.Ignored)
                    {
                        oldPatternAnimTargets.Add(patternAnimTarget.sampler_name, patternAnimTarget);
                        if (nameSet.Add(patternAnimTarget.sampler_name))
                        {
                            // バインドされたものになければ末尾に追加
                            paramNames.Add(patternAnimTarget.sampler_name);
                        }
                    }
                    else
                    {
                        if (nameSet.Contains(patternAnimTarget.sampler_name))
                        {
                            oldPatternAnimTargets.Add(patternAnimTarget.sampler_name, patternAnimTarget);
                        }
                        else
                        {
                            updatePatternAnimTargets = true;
                        }
                    }
                }

                // 新しいものと古いものを合わせる
                foreach (var paramName in paramNames)
                {
                    if (oldPatternAnimTargets.ContainsKey(paramName))
                    {
                        newPatternAnimTargets.Add(oldPatternAnimTargets[paramName]);
                    }
                    else
                    {
                        newPatternAnimTargets.Add(new PatternAnimTarget()
                        {
                            sampler_name = paramName,
                            hint = ""
                        });
                        updatePatternAnimTargets = true;
                    }
                }

                if (updatePatternAnimTargets)
                {
                    patternAnims.Add(new PatternAnimsSwapData()
                    {
                        texPatternMatAnim = texPatternMatAnim,
                        PatternAnimTargets = newPatternAnimTargets,
                    });
                }
            }

            if (updateTexPatternMatAnims || patternAnims.Count > 0)
            {
                // 編集フラグを更新しない
                return new GeneralGroupReferenceEditCommand<DummyObject>(
                    DummyObject.TheDummyObjectGroup,
                    GuiObjectID.DummyObject,
                    Enumerable.Repeat<DummyObject>(DummyObject.TheDummyObject, 1),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        if (updateTexPatternMatAnims)
                        {
                            var temp = TexPatternMatAnims;
                            TexPatternMatAnims = newTexPatternMatAnims;
                            newTexPatternMatAnims = temp;
                        }

                        foreach (var swapData in patternAnims)
                        {
                            var temp = swapData.texPatternMatAnim.PatternAnimTargets;
                            swapData.texPatternMatAnim.PatternAnimTargets = swapData.PatternAnimTargets;
                            swapData.PatternAnimTargets = temp;
                        }

                        // 変更マークを更新する
                        UpdateIsModifiedAnimTargetAll();

                        // シリアライズされるデータに変更はないので Viewer に再転送する必要はない
                    });
            }

            return null;
        }

        public Texture GetTextureFromPatternIndex(int index)
        {
            if ((index >= 0) &&
                (index <  TexPatterns.Count))
            {
                var texName = TexPatterns[index].tex_name;

                return this.GetReferenceTexture(texName);
            }
            else
            {
                return null;
            }
        }

        #region ReferTexture
        private Dictionary<string, string> referenceTexturePaths = new Dictionary<string, string>();

        // 参照しているテクスチャ
        public Dictionary<string, string> ReferenceTexturePaths
        {
            get
            {
                return referenceTexturePaths;
            }
            set
            {
                referenceTexturePaths = value;
            }
        }
        #endregion


        private class PatternAnimsSwapData
        {
            public TexPatternMatAnim texPatternMatAnim;
            public List<PatternAnimTarget> PatternAnimTargets;
        };

        #region 範囲チェック
        public static IEnumerable<KeyFrame> AllKeys(IEnumerable<TexPatternMatAnim> TexPatternMatAnims)
        {
            var keys = from texPatternMatAnim in TexPatternMatAnims
                       from patternAnimTarget in texPatternMatAnim.PatternAnimTargets
                       from key in patternAnimTarget.KeyFrames
                       select key;

            return keys;
        }

        public IEnumerable<KeyFrame> AllKeys()
        {
            return AllKeys(TexPatternMatAnims);
        }

        public bool IsValueOutOfRange()
        {
            var keys = from texPatternMatAnim in TexPatternMatAnims
                       from patternAnimTarget in texPatternMatAnim.PatternAnimTargets
                       from key in patternAnimTarget.KeyFrames
                       select key;

            return keys.Any(x => x.Value < 0 || TexPatterns.Count <= x.Value);
        }


        private IEnumerable<KeyValuePair<string, string>> UnresolvedTexturesOnReload(string path)
        {
            var location = Path.GetFullPath(Path.GetDirectoryName(path));
            var textures = new Dictionary<string, string>();
            foreach (var tex in DocumentManager.Textures)
            {
                if (!string.IsNullOrEmpty(tex.FileLocation))
                {
                    textures.Add(tex.FilePath.ToLower(), tex.FileName);
                }
            }

            var baseName = string.Equals(location, FileLocation, StringComparison.OrdinalIgnoreCase) ? BaseName: null;
            var models = DocumentManager.Models.Where(x => x.ContainsAnimation(this)).DefaultIfEmpty().ToArray();
            foreach (var item in ReferenceTexturePaths)
            {
                if (Path.IsPathRooted(item.Value))
                {
                    // 一つでも参照解決に失敗するモデルがあれば警告
                    foreach (var model in models)
                    {
                        var pathWithName = DocumentManager.MakeReferenceTextureFilenames(
                            location,
                            this,
                            baseName,
                            item.Key,
                            model != null ? model.FileLocation : null,
                            model,
                            model != null ? model.BaseName : null,
                            textures).FirstOrDefault();

                        if (pathWithName == null || !string.Equals(pathWithName.path, item.Value, StringComparison.OrdinalIgnoreCase))
                        {
                            yield return item;
                            break;
                        }
                    }
                }
                else
                {
                    yield return item;
                }
            }
        }

        public override bool NeedToAskBeforeSave(string path)
        {
            return UnresolvedTexturesOnReload(path).Any() || IsValueOutOfRange();
        }

        public override bool ConfirmBeforeSave()
        {
            if (IsValueOutOfRange())
            {
                return App.AppContext.NotificationHandler.MessageBoxYesNo(MessageContext.TexturePatternAnimation_OutOfRange, res.Strings.TexturePatternAnimation_OutOfRange, FileName);
            }

            return true;
        }

        public override bool ConfirmPathBeforeSave(string path)
        {
            var unresolvedTextures = UnresolvedTexturesOnReload(path)
                .Select(y => DocumentManager.Textures.FirstOrDefault(x => x.Name == y.Key && string.Compare(x.FilePath, y.Value, true) == 0))
                .Where(x => x != null).ToArray();
            if (unresolvedTextures.Any())
            {
                return App.AppContext.NotificationHandler.OkCancelListBoxDialog(
                    MessageContext.IO_TextureReferenceCanNotResolved,
                    TheApp.AssemblyName,
                    string.Format(res.Strings.IO_TextureReferenceCanNotResolved, FileName + DocumentManager.GetSameNameIndexText(this, false)),
                    App.AppContext.NotificationHandler.StringFormat.UnresolvedTexture,
                    unresolvedTextures);
            }

            return true;
        }

        public override bool ConfirmBeforeBinarySave()
        {
            if (IsValueOutOfRange())
            {
                return App.Controls.UIMessageBox.YesNo(res.Strings.TexturePatternAnimation_OutOfRange, FileName);
            }

            return true;
        }

        public bool IsInvalidTextureReference()
        {
            return UnresolvedTextures().Any();
        }

        public override void CheckAndDisConnect()
        {
            // 不正なデータの場合、ランタイム連携を切断する。
            if (Viewer.Manager.Instance.IsConnected)
            {
                if (IsValueOutOfRange())
                {
                    // 先に切断してから
                    TheApp.MainFrame.ConnectToHio(false);
                    // メッセージ表示する。
                    App.AppContext.NotificationHandler.MessageBoxError(res.Strings.TexturePatternAnimation_OutOfRange_ViewerDisconnect, FileName);
                }
                else if (IsInvalidTextureReference())
                {
                    // 先に切断してから
                    TheApp.MainFrame.ConnectToHio(false);
                    // メッセージ表示する。
                    App.AppContext.NotificationHandler.MessageBoxError(res.Strings.TexturePatternAnimation_InvalidTextureReference_ViewerDisconnect, FileName);
                }
            }
        }
        #endregion

        #region savedData
        private tex_pattern_animType savedData;
        public List<TexPatternMatAnim> savedTexPatternMatAnims;

        /// <summary>
        /// 空以外の保存されたカーブ
        /// </summary>
        public int SavedCurveCount { get; private set; }

        public override void UpdateSavedData()
        {
            base.UpdateSavedData();
            savedData = ObjectUtility.Clone(Data);
            SavedUserDataArray = ObjectUtility.Clone(UserDataArray);
            UserDataArrayChanged = false;

            SavedCurveCount = 0;
            foreach (var texPatternMatAnim in TexPatternMatAnims)
            {
                foreach (var patternAnim in texPatternMatAnim.PatternAnimTargets)
                {
                    patternAnim.IsModified = false;
                    if (NotEmpty(patternAnim))
                    {
                        patternAnim.IsSaved = true;
                        SavedCurveCount++;
                    }
                    else
                    {
                        patternAnim.IsSaved = false;
                    }
                }
            }

            savedTexPatternMatAnims = ObjectUtility.Clone(TexPatternMatAnims);
            savedSearchPaths = ObjectUtility.Clone(SearchPaths);
        }

        public void CopySavedData(TexturePatternAnimation source)
        {
            CopyDocumentSavedData(source);
            savedData = source.savedData;

            SavedUserDataArray = source.SavedUserDataArray;
            UserDataArrayChanged = !SavedUserDataArray.IsSame(UserDataArray);

            SavedCurveCount = source.SavedCurveCount;
            savedTexPatternMatAnims = source.savedTexPatternMatAnims;
            savedSearchPaths = source.savedSearchPaths;

            UpdateIsModifiedAnimTargetAll();
        }

        public override bool EqualsToSavedData()
        {
            if (!base.EqualsToSavedData())
            {
                return false;
            }

            return !TexturePatternAnimationGeneralPage.IsModified(this) &&
                !TexturePatternAnimationPatternPage.IsModified(this) &&
                !IsCurvesModified() &&
                !UserDataPage.IsModified(this) &&
                !SearchPathPage.IsModified(this);
        }

        public void UpdateIsModifiedAnimTargetAll()
        {
            foreach (var vertexShapeAnim in TexPatternMatAnims)
            {
                var savedVertexShapeAnim = savedTexPatternMatAnims.FirstOrDefault(x => x.mat_name == vertexShapeAnim.mat_name);
                foreach (var shapeAnim in vertexShapeAnim.PatternAnimTargets)
                {
                    var savedShapeAnim = savedVertexShapeAnim != null ? savedVertexShapeAnim.PatternAnimTargets.FirstOrDefault(x => x.sampler_name == shapeAnim.sampler_name) : null;
                    UpdateIsModifiedAnimTarget(shapeAnim, savedShapeAnim);
                }
            }
        }

        public bool NotEmpty(PatternAnimTarget current)
        {
            return current.KeyFrames.Count > 0;
        }

        /// <summary>
        /// カーブの変更フラグを更新する
        /// </summary>
        public void UpdateIsModifiedAnimTarget(PatternAnimTarget current, PatternAnimTarget saved)
        {
            if (saved == null)
            {
                current.IsModified = NotEmpty(current);
                current.IsSaved = false;
            }
            else
            {
                current.IsModified = !current.IsSame(saved);
                current.IsSaved = saved.IsSaved;
            }
        }

        public bool IsValueChanged<T>(Func<tex_pattern_animType, T> select) where T : struct
        {
            return !select(savedData).Equals(select(Data));
        }

        public bool IsPatternChanged()
        {
            var patterns = savedData.tex_pattern_array != null ? savedData.tex_pattern_array.tex_pattern: new tex_patternType[0];
            return TexPatterns.Count != patterns.Length ||
                !TexPatterns.Zip(patterns, (x, y) => x.tex_name == y.tex_name).All(x => x);
        }

        public bool IsSequenceValueChanged<T>(Func<tex_pattern_animType, IEnumerable<T>> select)
        {
            return !select(savedData).SequenceEqual(select(Data));
        }

        public bool IsCurvesModified()
        {
            int savedCount = 0;
            foreach (var texPatternMatAnim in TexPatternMatAnims)
            {
                foreach (var patternAnim in texPatternMatAnim.PatternAnimTargets)
                {
                    if (patternAnim.IsModified)
                    {
                        return true;
                    }
                    if (patternAnim.IsSaved)
                    {
                        savedCount++;
                    }
                }
            }

            return savedCount != SavedCurveCount;
        }
        #endregion

        public override void OnDocumentSwapAdd(Document old)
        {
            CopySavedData((TexturePatternAnimation)old);
            base.OnDocumentSwapAdd(old);
        }

        public bool IsTexturePatternType()
        {
            return true;
        }
    }
}
