﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App.PropertyEdit;
using App.Utility;
using ConfigCommon;
using nw.g3d.nw4f_3dif;
using nw.g3d.iflib;

namespace App.Data
{
    sealed public class CameraAnimation : GuiObject, IHasUserData, ISceneAnimationObject, IQuantizeAnalysis
    {
        public camera_animType Data { get; private set; }
        public SceneAnimation Owner { get; private set; }
        public override Document OwnerDocument
        {
            get { return Owner; }
        }

        public UserDataArray UserDataArray { get; set; }
        public UserDataArray SavedUserDataArray { get; set; }
        public bool UserDataArrayChanged { get; set; }

        public bool HasAnimation
        {
            get
            {
                return CameraAnimTargets != null && CameraAnimTargets.Any();
            }
        }
        // ファイル出力できるか？
        public string FileNotOutputMessage
        {
            get
            {
                // 量子化エラー
                string error = null;
                Action<CameraAnimationTarget> errorHandler = delegate(CameraAnimationTarget animTarget)
                {
                    error = string.Format(res.Strings.FileNotOutputError_CameraAnimation, Name, animTarget.targetType.ToString());
                };

                foreach (var tuple in QuantizeAnalyse(errorHandler))
                {
                    if (error != null)
                    {
                        return error;
                    }
                }

                return null;
            }
        }

        public IEnumerable<Tuple<AnimTarget, IfQuantizationAnalysisResult>> QuantizeAnalyse()
        {
            return QuantizeAnalyse(null);
        }

        public IEnumerable<Tuple<AnimTarget, IfQuantizationAnalysisResult>> QuantizeAnalyse(Action<CameraAnimationTarget> errorHandler)
        {
            foreach (var animTarget in CameraAnimTargets)
            {
                var curve = new CameraAnimationCurveTreeNodeInfo(this, animTarget.targetType);

                IfQuantizationAnalysisResult result = null;
                try
                {
                    result = AnimationCurveEditCommand.MakeQuantizationAnalysis(this, ObjectID, curve, false);
                }
                catch (Exception)
                {
                    if (errorHandler != null)
                    {
                        errorHandler(animTarget);
                    }
                }
                yield return new Tuple<AnimTarget, IfQuantizationAnalysisResult>(animTarget, result);
            }
        }

        public CameraAnimation(camera_animType camera_anim, SceneAnimation owner)
            : base(GuiObjectID.CameraAnimation)
        {
            Name = camera_anim.camera_name;
            Data = camera_anim;
            Owner = owner;
            MakeComment(Data.comment);

            CameraAnimTargets = new List<CameraAnimationTarget>();
            var targetTypes = Enum.GetValues(typeof(camera_anim_target_targetType)).OfType<camera_anim_target_targetType>().ToArray();
            var cameraAnimTargets = camera_anim.camera_anim_target ?? Enumerable.Empty<camera_anim_targetType>();
            // カメラタイプの切り替えを出来るようにしたので、ここのAssertは無効にする。
            //Debug.Assert(cameraAnimTargets.Select(x => x.target).OrderBy(x => x).SequenceEqual(targetTypes.OrderBy(x => x)));

            // 全てのtargettypeに対してデータにcamera_anim_targetTypeがない場合は作成する
            var allCameraTargets =
                targetTypes.Select(
                    targetType =>
                    cameraAnimTargets.FirstOrDefault(anim => anim.target == targetType)
                    ?? new camera_anim_targetType() { target = targetType });

            foreach (var cameraAnimTarget in allCameraTargets.OrderBy(x => x.target))
            {
                var target = new CameraAnimationTarget()
                {
                    targetType = cameraAnimTarget.target,
                };

                if (cameraAnimTarget.Curve != null)
                {
                    target.QuantizationInfo = new AnimationDocument.QuantizationInfo()
                    {
                        frame_type = cameraAnimTarget.Curve.frame_type,
                        key_type = cameraAnimTarget.Curve.key_type,
                        offset = cameraAnimTarget.Curve.offset,
                        scale = cameraAnimTarget.Curve.scale,
                    };
                }
                target.Initialize(cameraAnimTarget.Curve, cameraAnimTarget.base_value, owner.BinaryStreams, InterpolationType.Hermite);

                CameraAnimTargets.Add(target);
            }

            // ユーザーデータ作成
            this.MakeUserData(Data.user_data_array, Owner.BinaryStreams);

            UpdateSavedData();
        }

        public CameraAnimationTarget GetTarget(camera_anim_target_targetType targetType)
        {
            return CameraAnimTargets.FirstOrDefault(x => x.targetType == targetType);
        }

        public CameraAnimationTarget GetAnimTarget(CameraAnimationCurveTreeNodeInfo info)
        {
            return GetTarget((camera_anim_target_targetType)info.ComponentIndex);
        }
        public List<CameraAnimationTarget> CameraAnimTargets;

        public static Dictionary<camera_anim_target_targetType, float> DefaultBases = new Dictionary<camera_anim_target_targetType, float>()
        {
            {camera_anim_target_targetType.position_x, 0},
            {camera_anim_target_targetType.position_y, 0},
            {camera_anim_target_targetType.position_z, 0},
            {camera_anim_target_targetType.aim_x, 0},
            {camera_anim_target_targetType.aim_y, 0},
            {camera_anim_target_targetType.aim_z, -1},
            {camera_anim_target_targetType.twist, 0},
            {camera_anim_target_targetType.rotate_x, 0},
            {camera_anim_target_targetType.rotate_y, 0},
            {camera_anim_target_targetType.rotate_z, 0},
            {camera_anim_target_targetType.aspect, 1.777778f},
            {camera_anim_target_targetType.near, 1},
            {camera_anim_target_targetType.far, 10000},
            {camera_anim_target_targetType.ortho_height, 720},
            {camera_anim_target_targetType.persp_fovy, 37.8493f},
        };

        #region savedData
        private camera_animType savedData;
        public List<CameraAnimationTarget> savedCameraAnimTargets;

        /// <summary>
        /// 空以外の保存されたカーブ
        /// </summary>
        public int SavedCurveCount { get; private set; }

        public override void UpdateSavedData()
        {
            base.UpdateSavedData();
            savedData = ObjectUtility.Clone(Data);
            SavedUserDataArray = ObjectUtility.Clone(UserDataArray);
            UserDataArrayChanged = false;

            SavedCurveCount = 0;
            foreach (var animTarget in CameraAnimTargets)
            {
                animTarget.IsModified = false;
                if (NotEmpty(animTarget))
                {
                    animTarget.IsSaved = true;
                    SavedCurveCount++;
                }
                else
                {
                    animTarget.IsSaved = false;
                }
            }
            savedCameraAnimTargets = ObjectUtility.Clone(CameraAnimTargets);
        }

        public void CopySavedData(CameraAnimation source)
        {
            CopyGuiObjectSavedData(source);
            savedData = source.savedData;
            SavedUserDataArray = source.SavedUserDataArray;
            UserDataArrayChanged = !source.SavedUserDataArray.IsSame(UserDataArray);

            SavedCurveCount = source.SavedCurveCount;
            savedCameraAnimTargets = source.savedCameraAnimTargets;

            UpdateIsModifiedAnimTargetAll();
        }

        public override bool EqualsToSavedData()
        {
            if (!base.EqualsToSavedData())
            {
                return false;
            }

            return !CameraAnimationGeneralPage.IsModified(this) &&
                !IsCurvesModified() &&
                !UserDataPage.IsModified(this);
        }

        public void UpdateIsModifiedAnimTargetAll()
        {
            foreach (var animTarget in CameraAnimTargets)
            {
                var savedAnimTarget = savedCameraAnimTargets.FirstOrDefault(x => x.targetType == animTarget.targetType);
                UpdateIsModifiedAnimTarget(animTarget, savedAnimTarget);
            }
        }

        /// <summary>
        /// カーブの変更フラグを更新する
        /// </summary>
        public void UpdateIsModifiedAnimTarget(CameraAnimationTarget current, CameraAnimationTarget saved)
        {
            if (saved == null)
            {
                current.IsModified = NotEmpty(current);
                current.IsSaved = false;
            }
            else
            {
                current.IsModified = !current.IsSame(saved);
                current.IsSaved = saved.IsSaved;
            }
        }


        public bool NotEmpty(CameraAnimationTarget current)
        {
            return current.KeyFrames.Count > 0;
        }

        public bool IsValueChanged<T>(Func<camera_animType, T> select) where T : struct
        {
            return !select(savedData).Equals(select(Data));
        }

        public bool IsCurvesModified()
        {
            int savedCount = 0;
            foreach (var animTarget in CameraAnimTargets)
            {
                if (animTarget.IsModified)
                {
                    return true;
                }
                if (animTarget.IsSaved)
                {
                    savedCount++;
                }
            }

            return savedCount != SavedCurveCount;
        }
        #endregion
    }


    [Serializable]
    public class CameraAnimationTarget : AnimTarget
    {
        public camera_anim_target_targetType targetType;
        public camera_anim_targetType ConvertTo_camera_animType(out G3dStream stream, bool viewer = false)
        {
            IG3dQuantizedCurve curve;
            Convert(out curve, out stream, viewer);
            var target = new camera_anim_targetType()
            {
                base_value = GetBaseValue(),
                Curve = curve,
                target = targetType,
            };
            return target;
        }

        public override float GetBaseValue()
        {
            if (KeyFrames.Any())
            {
                return GetFrameValue(0);
            }
            return CameraAnimation.DefaultBases[targetType];
        }

    }
}
