﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading;
using System.Windows.Forms;
using App.ConfigData;
using App.res;

namespace App.Controls
{
    public sealed partial class UserCommandLog : UIForm
    {
        // インスタンス
        private static UserCommandLog _instance = null;
        // worker
        private static BackgroundWorker _bgWorker = null;
        // lock
        public static object WorkLock = new object();
        // thread
        public static Thread _thread = null;
        // Exitcode
        private static int _exitcode = 0;
        //

        // コンストラクタ
        private UserCommandLog()
        {
            InitializeComponent();
            DesktopLocation = new Point(ApplicationConfig.Setting.UserCommandLog.X, ApplicationConfig.Setting.UserCommandLog.Y);
            Size = new Size(ApplicationConfig.Setting.UserCommandLog.Width, ApplicationConfig.Setting.UserCommandLog.Height);
            SizeChanged += (sender, args) => SaveLocation();

            cbxAutoClose.Checked = ApplicationConfig.UserSetting.ExternalProgram.AutoCloseUserCommandLog;
            _bgWorker = new BackgroundWorker();
            _bgWorker.DoWork += _bgWorker_DoWork;
            _bgWorker.RunWorkerCompleted += _bgWorker_RunWorkerCompleted;
        }

        private void SaveLocation()
        {
            var wp = new Win32.WINDOWPLACEMENT();
            wp.length = Marshal.SizeOf(wp);
            Win32.NativeMethods.GetWindowPlacement(Handle, ref wp);
            ApplicationConfig.Setting.UserCommandLog.X = wp.rcNormalPosition.left;
            ApplicationConfig.Setting.UserCommandLog.Y = wp.rcNormalPosition.top;
            ApplicationConfig.Setting.UserCommandLog.Width = wp.rcNormalPosition.right - wp.rcNormalPosition.left;
            ApplicationConfig.Setting.UserCommandLog.Height = wp.rcNormalPosition.bottom - wp.rcNormalPosition.top;
        }

        private static DateTime _startWorkTime;
        private static DateTime _lastIdleTime;

        // BackgroundWorker 開始イベント
        static void _bgWorker_DoWork(object sender, DoWorkEventArgs e)
        {
            var info = e.Argument as ProcessStartInfo;
            Debug.Assert(info != null);
            Process process = null;
            try
            {
                _startWorkTime = _lastIdleTime= DateTime.Now;
                process = Process.Start(info);
                Debug.Assert(process != null, "process != null");

                // OutputDataReceived, ErrorDataReceivedが最後まで出力されるのを待つ
                var cde = new CountdownEvent(2);
                Action<string> report = s =>
                {
                    if (s == null)
                    {
                        cde.Signal();
                        return;
                    }

                    WriteLineIf(!string.IsNullOrEmpty(s), s);
                    if ((DateTime.Now - _lastIdleTime).TotalMilliseconds > 1000)
                    {
                        _lastIdleTime = DateTime.Now;
                        Thread.Sleep(10);
                    }
                };

                process.OutputDataReceived += (s, a) => report(a.Data);
                process.ErrorDataReceived += (s, a) => report(a.Data);
                process.BeginErrorReadLine();
                process.BeginOutputReadLine();

                while (!process.WaitForExit(10000)) { }

                cde.Wait(5000);
                _exitcode = process.ExitCode;
            }
            catch (Exception ex)
            {
                if (process != null)
                {
                    try
                    {
                        // プロセスが走っているかもしれないので止める
                        process.Kill();
                    }
                    catch
                    {
                        // 何もしない
                        DebugConsole.WriteLine("Failed to Kill process");
                    }
                }
                WriteLine("UserCommand Exception.\n{0}", ex.Message);
            }
        }

        // BackgroundWorker 終了イベント
        static void _bgWorker_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
            _instance.Invoke(new Action(()=> {
                lock (WorkLock) { Monitor.Pulse(WorkLock); }
            }));
            _instance.Invoke(new Action(() => _instance.lblProgress.Text = Strings.UserCommandLog_UserCommandDone));
            if (_instance.cbxAutoClose.Checked)
            {
                _instance.BeginInvoke(new Action(() => Terminate(_instance)));
            }
        }

        /// <summary>
        /// 初期化処理。
        /// </summary>
        public static void Initialize()
        {
            if (_instance == null)
            {
                while (_instance != null && !_instance.IsDisposed)
                {
                    Thread.Sleep(10);
                }
                _instance = new UserCommandLog();
            }
            _lastWrite = DateTime.MinValue;
        }


        /// <summary>
        /// 終了処理。
        /// </summary>
        public static void Terminate(UserCommandLog form)
        {
            if (form != null)
            {
                form.Close();
            }

        }

        /// <summary>
        /// 新規スレッドを立ち上げそこで起動
        /// </summary>
        /// <param name="info"></param>
        /// <returns></returns>
        public static int Execute(ProcessStartInfo info)
        {
            Debug.Assert(File.Exists(info.FileName));
            ThreadStart threadfunc = () =>
            {
                _bgWorker.RunWorkerAsync(info);
                _instance.Text = info.FileName;
                _instance.lblProgress.Text = Strings.UserCommandLog_UserCommandStart;
                _instance.tbxOutput.Text = string.Format("Execute : {0}\r\n", info.FileName);

                // ステータス更新用タイマー
                var statusTimer = new System.Windows.Forms.Timer { Interval = 500 };
                statusTimer.Tick += (s, a) =>
                {
                    if (_bgWorker.IsBusy)
                    {
                        var mes = string.Format(Strings.UserCommandLog_UserCommandInProgress,
                            (DateTime.Now - _startWorkTime).TotalSeconds.ToString("N0"));
                        _instance.Invoke(new Action(() => _instance.lblProgress.Text = mes));
                    }
                };
                statusTimer.Start();

                // テキスト追加用タイマー
                // テキストの追加漏れや遅れが出ないように適当なタイミングで呼び出す
                var writeTimer = new System.Windows.Forms.Timer { Interval = 50 };
                writeTimer.Tick += (s, a) =>
                {
                    if (_instance!=null && _instance.Visible)
                    {
                        _instance.Invoke(new Action(_Write));
                    }
                };
                writeTimer.Start();

                Application.Run(_instance);
            };
            Initialize();


            lock (WorkLock)
            {
                if (_thread == null || !_thread.IsAlive) // 初回のみ
                {
                    _thread = new Thread(threadfunc);
                    TheApp.MainFrame.Closed += (sender, args) => _thread.Abort();
                    _thread.Start();
                }
                else
                {
                    _instance.Invoke(new Action(() =>
                    {
                        //Monitor.Enter(WorkLock);
                        WriteLine("");
                        WriteLine("Execute : {0}", info.FileName);
                        _instance.Activate();
                        _instance.lblProgress.Text = Strings.UserCommandLog_UserCommandStart;
                        _instance.Text = info.FileName;
                        _bgWorker.RunWorkerAsync(info);
                    }));
                }

                // タスクが完了するのを待つ
                Monitor.Wait(WorkLock);
            }

            return _exitcode;
        }

        protected override void OnClosing(CancelEventArgs e)
        {
            Debug.Assert(_bgWorker != null);
            if (_bgWorker.IsBusy)
            {
                e.Cancel = true;
                return;
            }
            base.OnClosing(e);
        }

        protected override void OnClosed(EventArgs e)
        {
            SaveLocation();
            base.OnClosed(e);
            _instance = null;
        }

        // 出力文字
        public static Queue<string> Buffer = new Queue<string>();
        public static int BufferedLength = 0;
        private static DateTime _lastWrite;

        // 書き込みメソッド
        private static void _Write()
        {
            lock (Messages)
            {
                // テキストの追加は重いため、ある程度時間が経つか指定行数分貯まってから追加する。
                if ((DateTime.Now - _lastWrite).TotalMilliseconds < 100 && Messages.Count < 10) return;

                if (Messages.Count > 0)
                {
                    _lastWrite = DateTime.Now;
                    string message = "";
                    while (Messages.Count > 0)
                    {
                        message += Messages.Dequeue();
                    }

                    // 通常の呼び出し用
                    TextBox tbx = _instance.tbxOutput;
                    string output = message;
                    Buffer.Enqueue(output);
                    BufferedLength += output.Length;

                    if (BufferedLength > tbx.MaxLength)
                    {
                        tbx.Clear();
                        // 2/3にする
                        while (BufferedLength > 2 * tbx.MaxLength / 3)
                        {
                            BufferedLength -= Buffer.Peek().Length;
                            Buffer.Dequeue();
                        }
                        var builder = new StringBuilder();
                        foreach (var item in Buffer)
                        {
                            builder.Append(item);
                        }
                        tbx.AppendText(builder.ToString());
                        Debug.Assert(tbx.Text.Length == BufferedLength);
                    }
                    else
                    {
                        tbx.AppendText(output);
                    }

                }
            }
        }

        private static readonly Queue<string> Messages = new Queue<string>();
        /// <summary>
        /// 書き込み。
        /// </summary>
        public static void Write(string message)
        {
            if (_instance != null && _instance.IsHandleCreated)
            {
                lock (Messages)
                {
                    Messages.Enqueue(message);
                }

                // 別スレッドからの呼び出し用
                if (Thread.CurrentThread != TheApp.MainThread)
                {
                    _instance.BeginInvoke(new Action(_Write));
                    return;
                }

                _Write();
            }
        }

        /// <summary>
        /// 書式指定付き書き込み。
        /// </summary>
        public static void Write(string format, params object[] args)
        {
            Write(string.Format(format, args));
        }

        /// <summary>
        /// 書き込み＋改行。
        /// </summary>
        public static void WriteLine(string message)
        {
            Write(message + "\r\n");
        }

        /// <summary>
        /// 書式指定付き書き込み＋改行。
        /// </summary>
        public static void WriteLine(string format, params object[] args)
        {
            Write(string.Format(format, args) + "\r\n");
        }

        /// <summary>
        /// 条件付き書き込み。
        /// </summary>
        public static void WriteIf(bool condition, string message)
        {
            if (condition)
            {
                Write(message);
            }
        }

        /// <summary>
        /// 条件付き書式指定付き書き込み。
        /// </summary>
        public static void WriteIf(bool condition, string format, params object[] args)
        {
            if (condition)
            {
                Write(format, args);
            }
        }

        /// <summary>
        /// 条件付き書き込み＋改行。
        /// </summary>
        public static void WriteLineIf(bool condition, string message)
        {
            if (condition)
            {
                WriteLine(message);
            }
        }

        /// <summary>
        /// 条件付き書式指定付き書き込み＋改行。
        /// </summary>
        public static void WriteLineIf(bool condition, string format, params object[] args)
        {
            if (condition)
            {
                WriteLine(format, args);
            }
        }

        private void cbxAutoClose_CheckedChanged(object sender, EventArgs e)
        {
            ApplicationConfig.UserSetting.ExternalProgram.AutoCloseUserCommandLog = cbxAutoClose.Checked;
            ApplicationConfig.SaveUserSetting();
        }
    }
}
