﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Windows.Forms;

namespace App.Controls
{
    /// <summary>
    /// ＵＩラジオボタンクラス。
    /// </summary>
    [DefaultEvent("RadioChecked")]
    public class UIRadioButton : RadioButton
    {
        // OnEnter処理中フラグ
        private bool _processOnEnter = false;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIRadioButton()
        {
        }

        /// <summary>
        /// Checked プロパティを Tag 値比較で設定。
        /// </summary>
        public void SetCheckedByTag(object tag)
        {
            if (Tag != null)
            {
                Checked = Tag.Equals(tag);
            }
        }

        #region イベント
        //---------------------------------------------------------------------
        private static readonly object EVENT_RADIOCHECKED = new object();

        /// <summary>
        /// ラジオチェックイベント。
        /// </summary>
        [Category(UIControlHelper.OriginalEventCategoryName)]
        [Description("Checked プロパティが True になった時に発生します。")]
        public event EventHandler RadioChecked
        {
            add { base.Events.AddHandler(EVENT_RADIOCHECKED, value); }
            remove { base.Events.RemoveHandler(EVENT_RADIOCHECKED, value); }
        }

        /// <summary>
        /// ラジオチェックハンドラ。
        /// </summary>
        protected virtual void OnRadioChecked(EventArgs e)
        {
            EventHandler handler = (EventHandler)base.Events[EVENT_RADIOCHECKED];
            if (handler != null) { handler(this, e); }
        }
        #endregion

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnEnter(EventArgs e)
        {
            // フォーカス時に勝手にチェックが入るのを防ぐ
            _processOnEnter = true;
            {
                base.OnEnter(e);
            }
            _processOnEnter = false;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnClick(EventArgs e)
        {
            // OnEnter処理中はクリック処理を行なわない
            if (!_processOnEnter)
            {
                base.OnClick(e);
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnCheckedChanged(EventArgs e)
        {
            // ラジオチェックイベント
            if (Checked)
            {
                OnRadioChecked(e);
            }
            base.OnCheckedChanged(e);
        }
        #endregion

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new FlatStyle FlatStyle
        {
            get { return base.FlatStyle; }
            set { base.FlatStyle = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public override RightToLeft RightToLeft
        {
            get { return base.RightToLeft; }
            set { base.RightToLeft = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool TabStop
        {
            get { return base.TabStop; }
            set { base.TabStop = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool UseCompatibleTextRendering
        {
            get { return base.UseCompatibleTextRendering; }
            set { base.UseCompatibleTextRendering = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool UseVisualStyleBackColor
        {
            get { return base.UseVisualStyleBackColor; }
            set { base.UseVisualStyleBackColor = value; }
        }
        #endregion
    }

    /// <summary>
    /// ラジオボタンアダプタクラス。
    /// </summary>
    [DefaultEvent("SelectionChanged")]
    public sealed class RadioButtonAdapter : Component
    {
        // ラジオボタンリスト
        private readonly List<RadioButton> _radioButtons = new List<RadioButton>();

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public RadioButtonAdapter()
        {
        }

        /// <summary>
        /// コンストラクタ（デザイナ生成用）。
        /// </summary>
        public RadioButtonAdapter(IContainer container)
        {
            container.Add(this);
        }

        #region プロパティ
        /// <summary>
        /// 選択ラジオボタン。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public RadioButton SelectedRadioButton
        {
            get
            {
                foreach (RadioButton radioButton in _radioButtons)
                {
                    if (radioButton.Checked)
                    {
                        return radioButton;
                    }
                }
                return null;
            }
            set
            {
                foreach (RadioButton radioButton in _radioButtons)
                {
                    radioButton.Checked = (radioButton == value);
                }
            }
        }

        /// <summary>
        /// 選択ラジオボタンデータ。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object SelectedRadioButtonData
        {
            get
            {
                RadioButton radioButton = SelectedRadioButton;
                if (radioButton != null)
                {
                    return radioButton.Tag;
                }
                return null;
            }
            set
            {
                foreach (RadioButton radioButton in _radioButtons)
                {
                    if (radioButton.Tag != null)
                    {
                        radioButton.Checked = radioButton.Tag.Equals(value);
                    }
                    else
                    {
                        radioButton.Checked = false;
                    }
                }
            }
        }
        #endregion

        #region イベント
        //---------------------------------------------------------------------
        private static readonly object EVENT_SELECTIONCHANGED = new object();

        /// <summary>
        /// 選択変更イベント。
        /// </summary>
        [Category(UIControlHelper.OriginalEventCategoryName)]
        [Description("選択項目が変更された時に発生します。")]
        public event EventHandler SelectionChanged
        {
            add { base.Events.AddHandler(EVENT_SELECTIONCHANGED, value); }
            remove { base.Events.RemoveHandler(EVENT_SELECTIONCHANGED, value); }
        }

        /// <summary>
        /// 選択変更ハンドラ。
        /// </summary>
        private void OnSelectionChanged(EventArgs e)
        {
            EventHandler handler = (EventHandler)base.Events[EVENT_SELECTIONCHANGED];
            if (handler != null) { handler(this, e); }
        }
        #endregion

        /// <summary>
        /// ラジオボタンを設定。
        /// </summary>
        public void SetRadioButton(RadioButton radioButton, object tag)
        {
            SetRadioButton(radioButton, tag, null);
        }

        /// <summary>
        /// ラジオボタンを設定。
        /// </summary>
        public void SetRadioButton(RadioButton radioButton, object tag, string text)
        {
            if (!_radioButtons.Contains(radioButton))
            {
                // テキストを設定
                if (text != null)
                {
                    radioButton.Text = text;
                }

                // タグを設定
                radioButton.Tag = tag;

                // イベントを設定
                radioButton.CheckedChanged += new EventHandler(radioButton_CheckedChanged);

                // リストに登録
                _radioButtons.Add(radioButton);
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }

        //---------------------------------------------------------------------
        // ラジオボタンのチェック状態変更
        private void radioButton_CheckedChanged(object sender, EventArgs e)
        {
            RadioButton radioButton = (RadioButton)sender;
            if (radioButton.Checked)
            {
                OnSelectionChanged(EventArgs.Empty);
            }
        }
    }
}
