﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Drawing;
using System.Threading;
using System.Windows.Forms;

namespace App.Controls
{
    // 概要:
    //     System.Windows.Forms.MessageBox に表示するボタンを定義する定数を指定します。
    public enum UIMessageBoxButtons
    {
        // 概要:
        //     メッセージ ボックスに [OK] ボタンを含めます。
        OK = 0,
        //
        // 概要:
        //     メッセージ ボックスに [OK] ボタンと [キャンセル] ボタンを含めます。
        OKCancel = 1,
        //
        // 概要:
        //     メッセージ ボックスに [中止]、[再試行]、および [無視] の各ボタンを含めます。
        AbortRetryIgnore = 2,
        //
        // 概要:
        //     メッセージ ボックスに [はい]、[いいえ]、および [キャンセル] の各ボタンを含めます。
        YesNoCancel = 3,
        //
        // 概要:
        //     メッセージ ボックスに [はい] ボタンと [いいえ] ボタンを含めます。
        YesNo = 4,
        //
        // 概要:
        //     メッセージ ボックスに [再試行] ボタンと [キャンセル] ボタンを含めます。
        RetryCancel = 5,

        /// <summary>
        /// 報告するとキャンセルボタンを含めます。
        /// </summary>
        MailCancel,
    }

    /// <summary>
    /// ＵＩメッセージボックスダイアログクラス。
    /// </summary>
    public sealed partial class UIMessageBox : ModalDialog
    {
        public class Option : IDisposable
        {
            private static int  depth_ = 0;
            private static bool isSameOpForAll_ = false;
            private static bool isVisibleSameOpForAll_ = false;

            private readonly bool oldIsVisibleSameOpForAll_;

            private static bool IsEnabled
            {
                get
                {
                    return depth_ > 0;
                }
            }

            public Option()
            {
                ++ depth_;

                oldIsVisibleSameOpForAll_ = IsVisibleSameOpForAll;
            }

            public void Dispose()
            {
                IsVisibleSameOpForAll = oldIsVisibleSameOpForAll_;

                -- depth_;
            }

            // [以降すべてに同じ操作を適用する]チェックボックスの表示
            public bool IsVisibleSameOpForAll
            {
                get
                {
                    Debug.Assert(IsEnabled);
                    return isVisibleSameOpForAll_;
                }

                set
                {
                    Debug.Assert(IsEnabled);
                    isVisibleSameOpForAll_ = value;
                }
            }

            // [以降すべてに同じ操作を適用する]
            public bool Result_IsSameOpForAll
            {
                get
                {
                    Debug.Assert(IsEnabled);
                    return isSameOpForAll_;
                }

                set
                {
                    Debug.Assert(IsEnabled);
                    isSameOpForAll_ = value;
                }
            }
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIMessageBox(
            string					message,
            UIMessageBoxButtons		mbButtons,
            MessageBoxIcon			mbIcon,
            MessageBoxDefaultButton	mbDefaultButton,
            bool					isVisibleSameOpForAll
        )
        {
            InitializeComponent();
            Debug.Assert(message != null);

            cbxSameOpForAll.Visible = isVisibleSameOpForAll;

            // タイトル
            Text = Application.ProductName;

            // アイコン＆サウンド効果
            switch (mbIcon)
            {
                // 情報
                case MessageBoxIcon.Information:
                    pctIcon.Image = SystemIcons.Information.ToBitmap();
                    SoundEffect = DialogSoundEffect.Information;
                    break;
                // 警告
                case MessageBoxIcon.Warning:
                    pctIcon.Image = SystemIcons.Warning.ToBitmap();
                    SoundEffect = DialogSoundEffect.Warning;
                    break;
                // エラー
                case MessageBoxIcon.Error:
                    pctIcon.Image = SystemIcons.Error.ToBitmap();
                    SoundEffect = DialogSoundEffect.Error;
                    break;
                // 問い合わせ
                case MessageBoxIcon.Question:
                    pctIcon.Image = SystemIcons.Question.ToBitmap();
                    SoundEffect = DialogSoundEffect.Question;
                    break;
                default: break;
            }

            // メッセージ
            // ２行以下の場合は上部にマージンを設定する
            lblMessage.Text = message;
            {
                string[] split = message.Split('\n');
                if (split.Length <= 2)
                {
                    Padding margin = lblMessage.Margin;
                    margin.Top = split.Length == 1 ? 8 : 4;
                    lblMessage.Margin = margin;
                }
            }

            // ボタン
            switch (mbButtons)
            {
                case UIMessageBoxButtons.OK:
                    btnResult1.Text = App.res.Strings.DialogResult_Ok;
                    btnResult1.DialogResult = DialogResult.OK;
                    btnResult2.Visible = false;
                    btnResult2.Enabled = false;
                    btnResult3.Visible = false;
                    btnResult3.Enabled = false;
                    break;
                case UIMessageBoxButtons.OKCancel:
                    btnResult1.Text = App.res.Strings.DialogResult_Ok;
                    btnResult1.DialogResult = DialogResult.OK;
                    btnResult2.Text = App.res.Strings.DialogResult_Cancel;
                    btnResult2.DialogResult = DialogResult.Cancel;
                    btnResult3.Visible = false;
                    btnResult3.Enabled = false;
                    break;
                case UIMessageBoxButtons.YesNo:
                    btnResult1.Text = App.res.Strings.DialogResult_Yes;
                    btnResult1.DialogResult = DialogResult.Yes;
                    btnResult2.Text = App.res.Strings.DialogResult_No;
                    btnResult2.DialogResult = DialogResult.No;
                    btnResult3.Visible = false;
                    btnResult3.Enabled = false;
                    break;
                case UIMessageBoxButtons.YesNoCancel:
                    btnResult1.Text = App.res.Strings.DialogResult_Yes;
                    btnResult1.DialogResult = DialogResult.Yes;
                    btnResult2.Text = App.res.Strings.DialogResult_No;
                    btnResult2.DialogResult = DialogResult.No;
                    btnResult3.Text = App.res.Strings.DialogResult_Cancel;
                    btnResult3.DialogResult = DialogResult.Cancel;
                    break;
                case UIMessageBoxButtons.RetryCancel:
                    btnResult1.Text = App.res.Strings.DialogResult_Retry;
                    btnResult1.DialogResult = DialogResult.Retry;
                    btnResult2.Text = App.res.Strings.DialogResult_Cancel;
                    btnResult2.DialogResult = DialogResult.Cancel;
                    btnResult3.Visible = false;
                    btnResult3.Enabled = false;
                    break;
                case UIMessageBoxButtons.AbortRetryIgnore:
                    btnResult1.Text = App.res.Strings.DialogResult_About;
                    btnResult1.DialogResult = DialogResult.Abort;
                    btnResult2.Text = App.res.Strings.DialogResult_Retry;
                    btnResult2.DialogResult = DialogResult.Retry;
                    btnResult3.Text = App.res.Strings.DialogResult_Ignore;
                    btnResult3.DialogResult = DialogResult.Ignore;
                    break;
                case UIMessageBoxButtons.MailCancel:
                    btnResult1.Text = App.res.Strings.DialogResult_Report;
                    btnResult1.DialogResult = DialogResult.OK;
                    btnResult2.Text = App.res.Strings.DialogResult_Cancel;
                    btnResult2.DialogResult = DialogResult.Cancel;
                    btnResult3.Visible = false;
                    btnResult3.Enabled = false;
                    break;
                default: break;
            }

            // デフォルトボタン
            switch (mbDefaultButton)
            {
                case MessageBoxDefaultButton.Button1:
                    AcceptButton = btnResult1;
                    break;
                case MessageBoxDefaultButton.Button2:
                    AcceptButton = btnResult2.Visible ? btnResult2 : btnResult1;
                    break;
                case MessageBoxDefaultButton.Button3:
                    AcceptButton = btnResult3.Visible ? btnResult3 : btnResult1;
                    break;
                default: break;
            }

            // キャンセルボタン
            if      (btnResult3.Enabled) { CancelButton = btnResult3; }
            else if (btnResult2.Enabled) { CancelButton = btnResult2; }
            else                         { CancelButton = btnResult1; }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnShown(EventArgs e)
        {
            // ボタンの横方向中央配置
            // 全てのレイアウト調整が完了した後で行う
            if (pnlClient.Width > pnlButtons.Width)
            {
                Padding margin = pnlButtons.Margin;
                margin.Left = (pnlClient.Width - pnlButtons.Width) / 2;
                pnlButtons.Margin = margin;
            }
            base.OnShown(e);
        }

        #region 情報メッセージ
        /// <summary>
        /// 情報メッセージ。
        /// </summary>
        public static void Information(string message)
        {
            Show(
                message,
                UIMessageBoxButtons.OK,
                MessageBoxIcon.Information,
                MessageBoxDefaultButton.Button1
            );
        }

        /// <summary>
        /// 情報メッセージ。
        /// </summary>
        public static void Information(string format, params object[] args)
        {
            Information(string.Format(format, args));
        }
        #endregion

        #region 警告メッセージ
        /// <summary>
        /// 警告メッセージ。
        /// </summary>
        public static void Warning(string message)
        {
            Show(
                message,
                UIMessageBoxButtons.OK,
                MessageBoxIcon.Warning,
                MessageBoxDefaultButton.Button1
            );
        }

        /// <summary>
        /// 警告メッセージ。
        /// </summary>
        public static void Warning(string format, params object[] args)
        {
            Warning(string.Format(format, args));
        }
        #endregion

        #region エラーメッセージ
        /// <summary>
        /// エラーメッセージ。
        /// </summary>
        public static void Error(string message)
        {
            Show(
                message,
                UIMessageBoxButtons.OK,
                MessageBoxIcon.Error,
                MessageBoxDefaultButton.Button1
            );
        }

        /// <summary>
        /// エラーメッセージ。
        /// </summary>
        public static void Error(string format, params object[] args)
        {
            Error(string.Format(format, args));
        }
        #endregion

        #region ＯＫ、キャンセルメッセージ
        /// <summary>
        /// ＯＫ、キャンセルメッセージ。
        /// </summary>
        public static bool OkCancel(string message)
        {
            DialogResult result = Show(
                message,
                UIMessageBoxButtons.OKCancel,
                MessageBoxIcon.Question,
                MessageBoxDefaultButton.Button2
            );
            return result == DialogResult.OK;
        }

        /// <summary>
        /// ＯＫ、キャンセルメッセージ。
        /// </summary>
        public static bool OkCancel(string format, params object[] args)
        {
            return OkCancel(string.Format(format, args));
        }
        #endregion

        #region 報告する、キャンセルメッセージ
        /// <summary>
        /// ＯＫ、キャンセルメッセージ。
        /// </summary>
        public static bool MailCancel(string message)
        {
            DialogResult result = Show(
                message,
                UIMessageBoxButtons.MailCancel,
                MessageBoxIcon.Error,
                MessageBoxDefaultButton.Button2
            );
            return result == DialogResult.OK;
        }

        /// <summary>
        /// ＯＫ、キャンセルメッセージ。
        /// </summary>
        public static bool MailCancel(string format, params object[] args)
        {
            return MailCancel(string.Format(format, args));
        }
        #endregion

        #region はい、いいえメッセージ
        /// <summary>
        /// はい、いいえメッセージ。
        /// </summary>
        public static bool YesNo(string message)
        {
            DialogResult result = Show(
                message,
                UIMessageBoxButtons.YesNo,
                MessageBoxIcon.Question,
                MessageBoxDefaultButton.Button2
            );
            return result == DialogResult.Yes;
        }

        /// <summary>
        /// はい、いいえメッセージ。
        /// </summary>
        public static bool YesNo(string format, params object[] args)
        {
            return YesNo(string.Format(format, args));
        }
        #endregion

        #region はい、いいえ、キャンセルメッセージ
        /// <summary>
        /// はい、いいえ、キャンセルメッセージ。
        /// </summary>
        public static DialogResult YesNoCancel(string message)
        {
            DialogResult result = Show(
                message,
                UIMessageBoxButtons.YesNoCancel,
                MessageBoxIcon.Question,
                MessageBoxDefaultButton.Button3
            );
            return result;
        }

        /// <summary>
        /// はい、いいえ、キャンセルメッセージ。
        /// </summary>
        public static DialogResult YesNoCancel(string format, params object[] args)
        {
            return YesNoCancel(string.Format(format, args));
        }

        /// <summary>
        /// はい、いいえ、キャンセルメッセージ。
        /// </summary>
        public static DialogResult YesNoCancelWithSameOpForAll(string format, params object[] args)
        {
            return YesNoCancel(string.Format(format, args));
        }

        #endregion

        /// <summary>
        /// メッセージボックスを表示。
        /// </summary>
        private static DialogResult Show(
            string					message,
            UIMessageBoxButtons		mbButtons,
            MessageBoxIcon			mbIcon,
            MessageBoxDefaultButton	mbDefaultButton
        )
        {
            var result = DialogResult.OK;
            {
                // 別スレッドからの呼び出し用
                if (Thread.CurrentThread != TheApp.MainThread && TheApp.MainFrame != null && TheApp.MainFrame.IsHandleCreated)
                {
                    TheApp.MainFrame.BeginInvoke(
                        new MethodInvoker(
                            () =>
                            {
                                result = ShowProc(message, mbButtons, mbIcon, mbDefaultButton);
                            }
                        )
                    ).AsyncWaitHandle.WaitOne();
                }
                else
                {
                    result = ShowProc(message, mbButtons, mbIcon, mbDefaultButton);
                }
            }
            return result;
        }

        /// <summary>
        /// メッセージボックス表示処理
        /// </summary>
        private static DialogResult ShowProc(
            string					message,
            UIMessageBoxButtons		mbButtons,
            MessageBoxIcon			mbIcon,
            MessageBoxDefaultButton	mbDefaultButton
        )
        {
            var isMainFrameCreated = TheApp.MainFrame != null && TheApp.MainFrame.IsHandleCreated;

            if (isMainFrameCreated)
            {
                Win32.Utility.WakeupWindow(TheApp.MainFrame.Handle);
            }
            // カーソルを保存
            //Cursor cursor = Cursor.Current;

            // ダイアログ作成
            var option = new Option();

            using (var dialog = new UIMessageBox(message, mbButtons, mbIcon, mbDefaultButton, option.IsVisibleSameOpForAll))
            {
                // アクティブフォームがなければ画面中央に表示する
                var activeForm = Form.ActiveForm;
                if (activeForm == null)
                {
                    dialog.StartPosition = FormStartPosition.CenterScreen;
                }

                DeactivatedForm = ActiveForm;

                if (isMainFrameCreated == false)
                {
                    dialog.StartPosition = FormStartPosition.CenterScreen;
                }

                DialogResult result;
                // 表示
                using (new App.Utility.WaitCursor(false))
                {
                    result = dialog.ShowDialog();
                }

                option.Result_IsSameOpForAll = dialog.cbxSameOpForAll.Checked;

                DeactivatedForm = null;

                // 表示前のカーソルに戻す
                //Cursor.Current = cursor;

                return result;
            }
        }

        /// <summary>
        /// ダイアログ表示によりDeactive になるフォーム
        /// フォームがポップアップの場合、閉じるのを回避する必要がある。
        /// </summary>
        public static Form DeactivatedForm { get; set; }

        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            if (keyData == (Keys.Control | Keys.C))
            {
                App.Utility.ClipboardUtility.SetDataObject(this.Text + "\r\n" + this.lblMessage.Text, true);
                return true;
            }
            return base.ProcessCmdKey(ref msg, keyData);
        }
    }
}
