﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.ComponentModel;
using System.Windows.Forms;

namespace App.Controls
{
    /// <summary>
    /// ＵＩチェックリストボックスクラス。
    /// </summary>
    public class UICheckedListBox : CheckedListBox, IUIListControl
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UICheckedListBox()
        {
            base.IntegralHeight = false;
        }

        /// <summary>
        /// イメージリスト。
        /// UICheckedListBox は何もしません。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public ImageList ImageList
        {
            get { return null; }
            set { }
        }

        /// <summary>
        /// 選択項目。
        /// UIListControlItem 型で再定義しています。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public new UIListControlItem SelectedItem
        {
            get { return (UIListControlItem)base.SelectedItem; }
            set { base.SelectedItem = value; }
        }

        /// <summary>
        /// 選択項目データ。
        /// 各項目は UIListControlItem 型である必要があります。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object SelectedItemData
        {
            get
            {
                UIListControlItem item = (UIListControlItem)base.SelectedItem;
                if (item != null)
                {
                    return item.Data;
                }
                return null;
            }
            set
            {
                // データが null の項目を探す
                if (value == null)
                {
                    foreach (UIListControlItem item in base.Items)
                    {
                        if (item.Data == null)
                        {
                            base.SelectedItem = item;
                            return;
                        }
                    }
                }
                // データが等しい項目を探す
                else
                {
                    foreach (UIListControlItem item in Items)
                    {
                        if (item.Data != null && item.Data.Equals(value))
                        {
                            base.SelectedItem = item;
                            return;
                        }
                    }
                }

                // 該当する項目が見つからない
                base.SelectedItem = null;
            }
        }

        /// <summary>
        /// 項目追加。
        /// </summary>
        public void AddItem(string text, object data, bool isChecked)
        {
            AddItem(new UIListControlItem(text, data), isChecked);
        }

        /// <summary>
        /// 項目追加（文字列リソース指定）。
        /// </summary>
        public void AddItemRes(string stringID, object data, bool isChecked)
        {
            AddItem(new UIListControlItem(stringID, data), isChecked);
        }

        /// <summary>
        /// 項目追加。
        /// </summary>
        public void AddItem(UIListControlItem item, bool isChecked)
        {
            base.Items.Add(item, isChecked);
        }

        /// <summary>
        /// 項目データ取得。
        /// </summary>
        public object GetItemData(int index)
        {
            UIListControlItem item = (UIListControlItem)base.Items[index];
            return item.Data;
        }

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }
        #endregion

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new int ColumnWidth
        {
            get { return base.ColumnWidth; }
            set { base.ColumnWidth = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new int HorizontalExtent
        {
            get { return base.HorizontalExtent; }
            set { base.HorizontalExtent = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool HorizontalScrollbar
        {
            get { return base.HorizontalScrollbar; }
            set { base.HorizontalScrollbar = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new ImeMode ImeMode
        {
            get { return base.ImeMode; }
            set { base.ImeMode = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        [DefaultValue(false)]
        public new bool IntegralHeight
        {
            get { return base.IntegralHeight; }
            set { base.IntegralHeight = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [Browsable(false)]
        public new CheckedListBox.ObjectCollection Items
        {
            get { return base.Items; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool MultiColumn
        {
            get { return base.MultiColumn; }
            set { base.MultiColumn = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public override RightToLeft RightToLeft
        {
            get { return base.RightToLeft; }
            set { base.RightToLeft = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new SelectionMode SelectionMode
        {
            get { return base.SelectionMode; }
            set { base.SelectionMode = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool Sorted
        {
            get { return base.Sorted; }
            set { base.Sorted = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool TabStop
        {
            get { return base.TabStop; }
            set { base.TabStop = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        public new bool UseCompatibleTextRendering
        {
            get { return base.UseCompatibleTextRendering; }
            set { base.UseCompatibleTextRendering = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool UseTabStops
        {
            get { return base.UseTabStops; }
            set { base.UseTabStops = value; }
        }
        #endregion
    }
}
