﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;

namespace App.Controls
{
    /// <summary>
    /// ＵＩチェックボックスクラス。
    /// </summary>
    public class UICheckBox : CheckBox
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UICheckBox()
        {
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new FlatStyle FlatStyle
        {
            get { return base.FlatStyle; }
            set { base.FlatStyle = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public override RightToLeft RightToLeft
        {
            get { return base.RightToLeft; }
            set { base.RightToLeft = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool UseCompatibleTextRendering
        {
            get { return base.UseCompatibleTextRendering; }
            set { base.UseCompatibleTextRendering = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool UseVisualStyleBackColor
        {
            get { return base.UseVisualStyleBackColor; }
            set { base.UseVisualStyleBackColor = value; }
        }
        #endregion
    }

    /// <summary>
    /// フォーカス切り替え時にボタンの背景を変えるチェックボックス
    /// UICheckBox と違い Text が空でもフォーカスを識別できる。
    /// </summary>
    public class UICheckBox2 : UICheckBox
    {
        protected override void OnEnter(EventArgs e)
        {
            base.OnEnter(e);
            base.OnMouseEnter(e);
        }

        protected override void OnLeave(EventArgs e)
        {
            base.OnLeave(e);
            base.OnMouseLeave(e);
        }

        protected override void OnMouseLeave(EventArgs eventargs)
        {
            if (!Focused)
            {
                base.OnMouseLeave(eventargs);
            }
        }
    }

    public class UIModifiedMarkCheckBox : UICheckBox
    {
        private bool modified = false;
        public bool IsModified
        {
            get
            {
                return modified;
            }
            set
            {
                if (modified != value)
                {
                    modified = value;
                    Invalidate();
                }
            }
        }

        public override bool AutoSize
        {
            get { return false; }
            set { ; }		// 無視
        }

        protected override void OnTextChanged(EventArgs e)
        {
            using (var g = CreateGraphics())
            {
                // サイズ、位置を計算し直す
                var textStringSize = TextRenderer.MeasureText(g, Text, Font);

                MarkPosX = minCheckBoxWidth_ + textStringSize.Width;

                Size = new Size(
                    minCheckBoxWidth_ + textStringSize.Width + ModifiedMark.StarMark.Width,
                    Math.Max(textStringSize.Height, minCheckBoxHeight_)
                );
            }

            Invalidate();
        }

        protected int MarkPosX = 16;

        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            if (IsModified || DesignMode)
            {
                e.Graphics.DrawImage(ModifiedMark.StarMark, MarkPosX, 0);
            }
        }

        // Text が空文字列の時のサイズ
        protected static readonly int minCheckBoxWidth_  = 16;
        protected static readonly int minCheckBoxHeight_ = 16;
    }
    public class UIModifiedMarkAndImageCheckBox : UIModifiedMarkCheckBox
    {
        // ラベル右に表示される画像を保持
        // ラベル、画像、変更マークの順で表示される
        private List<Bitmap> _tailImages = new List<Bitmap>();

        public List<Bitmap> TailImages
        {
            get { return _tailImages; }
            set
            {
                _tailImages = value;
                ResetSize();
            }
        }

        protected override void OnTextChanged(EventArgs e)
        {
            ResetSize();
        }

        protected override void OnFontChanged(EventArgs e)
        {
            ResetSize();
        }

        private int _imagePosX = 0;

        protected internal void ResetSize()
        {
            using (var g = CreateGraphics())
            {
                // サイズ、位置を計算し直す
                var textStringSize = TextRenderer.MeasureText(g, Text, Font);

                var rightPos = minCheckBoxWidth_ + textStringSize.Width;
                var height = textStringSize.Height;
                if (_tailImages != null && _tailImages.Any())
                {
                    _imagePosX = rightPos;
                    height = Math.Max(_tailImages.Max(x => x.Height), height);
                    rightPos = _imagePosX + _tailImages.Sum(x => x.Width) + (_tailImages.Count - 1) * 2;
                }

                MarkPosX = rightPos;

                Size = new Size(rightPos + ModifiedMark.StarMark.Width, Math.Max(minCheckBoxHeight_, height));
            }

            Invalidate();
        }

        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);
            if (_tailImages != null && _tailImages.Any())
            {
                var x = _imagePosX;
                foreach (var image in _tailImages)
                {
                    e.Graphics.DrawImage(image, x, 0, image.Width, image.Height);
                    x += image.Width + 2;
                }
            }
        }
    }
}
