﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Drawing;
using App.Data;
using App.res;

namespace App.Controls
{
    [ToolboxItem(true)]
    public partial class TextureImageViewPanel : UIUserControl
    {
        public TextureImageViewPanel()
        {
            InitializeComponent();

            SetControlLayout();
            UpdateDepth();
        }

        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Texture Target
        {
            get
            {
                return tipColor.Target;
            }

            set
            {
                tipColor.Target = value;
                tipAlpha.Target = value;

                UpdateDepth();
            }
        }

        public enum LayoutType
        {
            Horizontal,
            Vertical
        }

        public LayoutType ImageLayout { get; set; }

        protected override void OnResize(EventArgs e)
        {
            SetControlLayout();
            base.OnResize(e);
        }

        private void SetControlLayout()
        {
            int margin = 6;

            if (ImageLayout == LayoutType.Horizontal)
            {
                int imageWidth = (Width - margin * 3) / 2;

                lblColor.Size     = new Size(imageWidth, 16);
                lblAlpha.Size     = new Size(imageWidth, 16);
                lblColor.Location = new Point(margin, 2);
                lblAlpha.Location = new Point(Width / 2 + margin / 2, 2);

                tipColor.Size     = new Size(imageWidth, imageWidth);
                tipAlpha.Size     = new Size(imageWidth, imageWidth);
                tipColor.Location = new Point(margin, 20);
                tipAlpha.Location = new Point(Width / 2 + margin / 2, 20);

                cbxDepth.Location = new Point(tipAlpha.Right - cbxDepth.Width + 1, tipAlpha.Bottom + margin / 2);
                lblDepth.Location = new Point(cbxDepth.Left - lblDepth.Width, cbxDepth.Top + 5);
            }
            else
            if (ImageLayout == LayoutType.Vertical)
            {
                int imageWidth = Width - margin * 2;

                lblColor.Size     = new Size(imageWidth, 16);
                lblColor.Location = new Point(margin, 2);
                tipColor.Size     = new Size(imageWidth, imageWidth);
                tipColor.Location = new Point(margin, 20);

                lblAlpha.Size     = new Size(imageWidth, 16);
                lblAlpha.Location = new Point(margin, tipColor.Bottom + margin * 2);
                tipAlpha.Size     = new Size(imageWidth, imageWidth);
                tipAlpha.Location = new Point(margin, lblAlpha.Bottom + 2);

                cbxDepth.Location = new Point(tipAlpha.Right - cbxDepth.Width + 1, tipAlpha.Bottom + margin);
                lblDepth.Location = new Point(cbxDepth.Left - lblDepth.Width, cbxDepth.Top + 5);
            }
        }

        private void UpdateDepth()
        {
            cbxDepth.Items.Clear();

            if ((Target == null) || (Target.IsArray == false))
            {
                cbxDepth.Enabled = false;
                lblDepth.Enabled = false;
            }
            else
            {
                cbxDepth.AddItem(Strings.TextureImageViewPanel_AllView, -1);

                var maxDepth = Target.IsCube ? Target.DepthCount / 6 : Target.DepthCount;
                for(int i = 0;i != maxDepth;++ i)
                {
                    cbxDepth.AddItem(i.ToString(), i);
                }

                cbxDepth.SelectedItemData = -1;

                cbxDepth.Enabled = true;
                lblDepth.Enabled = true;
            }
        }

        private void cbxDepth_SelectedIndexChanged(object sender, EventArgs e)
        {
            var depthIndex = (int)cbxDepth.SelectedItemData;

            tipColor.DepthIndex = depthIndex;
            tipAlpha.DepthIndex = depthIndex;
        }
    }
}
