﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;
using App.Data;
using App.Utility;
using System;

namespace App.Controls
{
    public sealed class TextureImageListBox : UIListBox
    {
        public TextureImageListBox()
        {
            base.ItemHeight = 32;
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(32)]
        public override int ItemHeight
        {
            get { return base.ItemHeight; }
            set { base.ItemHeight = value; }
        }

        /// <summary>
        /// 項目追加。
        /// </summary>
        public void AddItem(Texture item)
        {
            base.AddItem(item.Name, item);
        }

        public void AddDummyItem(string name)
        {
            base.AddItem(name, null);
        }

        public HashSet<Texture> DisabledTextures = new HashSet<Texture>();

        public Func<Texture, string> GetErrorMessage = x => App.res.Strings.TextureImageListBox_SameNameTexture;

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnCustomDrawItem(CustomDrawListControlItemEventArgs e)
        {
            Texture image = null;
            if (e.Index >= 0)
            {
                image = (Texture)((UIListControlItem)Items[e.Index]).Data;
                if (image == null)
                {
                    base.OnCustomDrawItem(e);
                    return;
                }
            }

            // 背景
            if (Enabled)
            {
                if (DisabledTextures.Contains(image))
                {
                    if (e.CheckState(DrawItemState.Selected))
                    {
                        e.Graphics.FillRectangle(Brushes.Pink, e.Bounds);
                    }
                    else
                    {
                        e.Graphics.FillRectangle(SystemBrushes.Control, e.Bounds);
                    }
                }
                else
                {
                    e.DrawBackground();
                }
            }
            else
            {
                e.Graphics.FillRectangle(SystemBrushes.Window, e.Bounds);
            }

            // 項目
            if (image != null)
            {
                // イメージ枠
                Rectangle rcFrame = e.Bounds;
                rcFrame.Inflate(-1, -1);
                rcFrame.Width = rcFrame.Height;
                if (Enabled)
                {
                    GraphicsUtility.DrawRectangle(e.Graphics, Pens.Black, rcFrame);
                }
                else
                {
                    GraphicsUtility.DrawRectangle(e.Graphics, SystemPens.GrayText, rcFrame);
                }

                // イメージ
                Rectangle rcImage = rcFrame;
//                rcImage.Inflate(-1, -1);
                rcImage.Offset(1,1);
                image.DrawColorThumbnail(e.Graphics, rcImage, Enabled, true);

                // テキスト
                Rectangle rcText = e.Bounds;
                RectangleUtility.OffsetLeft(ref rcText, rcText.Height);
                using (StringFormat sf = new StringFormat())
                {
                    sf.Alignment = StringAlignment.Near;
                    sf.LineAlignment = StringAlignment.Center;
                    sf.Trimming = StringTrimming.EllipsisCharacter;
                    sf.FormatFlags |= StringFormatFlags.NoWrap;

                    string fileName = image.FileName;

                    if (Enabled)
                    {
                        if (e.CheckState(DrawItemState.Selected))
                        {
                            e.Graphics.DrawString(image.FileName, e.Font, SystemBrushes.HighlightText, rcText, sf);
                        }
                        else
                        {
                            e.Graphics.DrawString(image.FileName, e.Font, SystemBrushes.WindowText, rcText, sf);
                        }
                    }
                    else
                    {
                        e.Graphics.DrawString(image.FileName, e.Font, SystemBrushes.GrayText, rcText, sf);
                    }

                    rcText.X += (int)e.Graphics.MeasureString("  " + image.FileName, e.Font).Width;

                    string indexText = DocumentManager.GetSameNameIndexText(image, false);
                    if (!string.IsNullOrEmpty(indexText))
                    {
                        Brush brush = Enabled && e.CheckState(DrawItemState.Selected) ?
                            SystemBrushes.HighlightText :
                            SystemBrushes.GrayText;
                        e.Graphics.DrawString(indexText, e.Font, brush, rcText, sf);

                        rcText.X += (int)e.Graphics.MeasureString("  " + indexText, e.Font).Width;
                    }

                    if (DisabledTextures.Contains(image) && GetErrorMessage != null)
                    {
                        var errorText = GetErrorMessage(image);
                        Brush brush = Enabled && e.CheckState(DrawItemState.Selected) ?
                            SystemBrushes.HighlightText :
                            SystemBrushes.GrayText;
                        e.Graphics.DrawString(errorText, e.Font, brush, rcText, sf);

                        rcText.X += (int)e.Graphics.MeasureString("  " + errorText, e.Font).Width;
                    }
                }
            }

            // フォーカス枠
            e.DrawFocusRectangle();

            // 処理した
            e.DoDefaultProcess = false;

            // base.OnCustomDrawItem は呼ばない
        }
    }
}
