﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.ComponentModel;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Windows.Forms.Design;
using App.Data;
using App.Utility;

namespace App.Controls
{
    /// <summary>
    /// ＧＵＩオブジェクトボックスクラス。
    /// </summary>
    [ToolboxItem(true)]
    public sealed class GuiObjectBox : UIControl
    {
        // オブジェクト
        private GuiObjectGroup objects_ = null;

        private HintToolTip hint_ = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public GuiObjectBox()
        {
            base.SetStyle(ControlStyles.Opaque, true);
            base.SetStyle(ControlStyles.ResizeRedraw, true);
            base.SetStyle(ControlStyles.Selectable, false);
            base.TabStop = false;
            DoubleBuffered = true;
        }

        /// <summary>
        /// オブジェクト。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public GuiObjectGroup Objects
        {
            get { return objects_; }
            set
            {
                objects_ = value;

                UpdateToolTip();

                Invalidate();
            }
        }

        public enum Border
        {
            ObjectView,
            ObjectPropertyDialog,
        };

        private Border borderType_ = Border.ObjectPropertyDialog;
        public Border BorderType
        {
            get
            {
                return borderType_;
            }

            set
            {
                borderType_ = value;
                Invalidate();
            }
        }

        #region オーバーライド

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override Size DefaultSize
        {
            get { return new Size(80, 22); }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnPaint(PaintEventArgs e)
        {
            // 背景
            e.Graphics.Clear(BackColor);

            if (DesignMode)
            {
                using (StringFormat sf = new StringFormat())
                {
                    sf.Alignment     = StringAlignment.Near;
                    sf.LineAlignment = StringAlignment.Center;
                    sf.FormatFlags  |= StringFormatFlags.NoWrap;

                    e.Graphics.DrawString(GetType().Name, Font, SystemBrushes.ControlDark, ClientRectangle, sf);
                }
                return;
            }

            switch(BorderType)
            {
                case Border.ObjectView:
                {
                    e.Graphics.DrawLine(SystemPens.ControlDark, ClientRectangle.Left, ClientRectangle.Top, ClientRectangle.Left, ClientRectangle.Bottom);
                    e.Graphics.DrawLine(SystemPens.ControlDark, ClientRectangle.Right - 1, ClientRectangle.Top, ClientRectangle.Right - 1, ClientRectangle.Bottom);
                    break;
                }

                case Border.ObjectPropertyDialog:
                {
                    e.Graphics.DrawLine(SystemPens.ControlDark, ClientRectangle.Left, ClientRectangle.Top, ClientRectangle.Left, ClientRectangle.Bottom);
                    e.Graphics.DrawLine(SystemPens.ControlDark, ClientRectangle.Right - 1, ClientRectangle.Top, ClientRectangle.Right - 1, ClientRectangle.Bottom);
                    e.Graphics.DrawLine(SystemPens.ControlDark, ClientRectangle.Left, ClientRectangle.Top, ClientRectangle.Right, ClientRectangle.Top);
                    e.Graphics.DrawLine(SystemPens.ControlDark, ClientRectangle.Left, ClientRectangle.Bottom - 1, ClientRectangle.Right, ClientRectangle.Bottom - 1);

                    break;
                }
            }

            // オブジェクト
            Rectangle rcClient = ClientRectangle;
            if (objects_ != null && !objects_.IsEmpty)
            {
                GuiObject active = objects_.Active;

                // アイコン
                Rectangle rcIcon = rcClient;
                rcIcon.X   += 3;
                rcIcon.Y   += (rcIcon.Height - 16) / 2;
                rcIcon.Size = new Size(16, 16);

                e.Graphics.DrawImage(active.ObjectIcon, rcIcon);

                // テキスト
                Rectangle rcText = rcClient;
                RectangleUtility.OffsetLeft(ref rcText, rcIcon.Width + 3);
                using (StringFormat sf = new StringFormat())
                {
                    sf.Alignment     = StringAlignment.Near;
                    sf.LineAlignment = StringAlignment.Center;
                    sf.Trimming      = StringTrimming.EllipsisCharacter;
                    sf.FormatFlags  |= StringFormatFlags.NoWrap;

                    string text = active.Name;
                    if (objects_.IsMulti)
                    {
                        text += string.Format(" ... ({0})", objects_.GetCount(active.ObjectID));
                    }
                    e.Graphics.DrawString(text, TheApp.GuiFontBold, SystemBrushes.WindowText, rcText, sf);
                    if (!objects_.IsMulti && active.OwnerDocument != null)
                    {
                        var size = e.Graphics.MeasureString(text, TheApp.GuiFontBold, rcText.Size, sf);
                        var rect = new RectangleF(rcText.X + size.Width, rcText.Y, rcText.Width - size.Width, rcText.Height);

                        var text2 = active.OwnerDocument.FileName + " " + DocumentManager.GetSameNameIndexText(active.OwnerDocument, false);

                        e.Graphics.DrawString(text2, TheApp.GuiFont, Brushes.Gray, rect, sf);
                    }
                }
            }
            else
            {
                // 未選択テキスト
                Rectangle rcText = rcClient;
                RectangleUtility.OffsetLeft(ref rcText, 3);
                using (StringFormat sf = new StringFormat())
                {
                    sf.Alignment     = StringAlignment.Near;
                    sf.LineAlignment = StringAlignment.Center;
                    sf.Trimming      = StringTrimming.EllipsisCharacter;
                    sf.FormatFlags  |= StringFormatFlags.NoWrap;

                    e.Graphics.DrawString(res.Strings.ObjectView_SelectedEmpty, Font, Brushes.Gray, rcText, sf);
                }
            }
        }
        #endregion

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        [DefaultValue(false)]
        public new bool TabStop
        {
            get { return base.TabStop; }
            set { base.TabStop = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public override string Text
        {
            get { return base.Text; }
            set { base.Text = value; }
        }
        #endregion

        private void UpdateToolTip()
        {
            if ((objects_ == null) || (objects_.Active == null))
            {
                if (hint_ != null)
                {
                    hint_.SetToolTip(this, string.Empty);
                    hint_ = null;
                }
            }
            else
            {
                var sb = new StringBuilder();
                {
                    foreach(var obj in objects_.GetObjects(objects_.Active.ObjectID))
                    {
                        if (obj.OwnerDocument != null)
                        {
                            sb.AppendFormat("{0} - {1}", obj.Name, obj.OwnerDocument.FileName + DocumentManager.GetSameNameIndexText(obj.OwnerDocument, true));
                        }
                        else
                        {
                            sb.Append(obj.Name);
                        }
                        sb.Append("\n");
                    }
                }

                if (hint_ == null)
                {
                    hint_ = new HintToolTip();
                }

                hint_.SetToolTip(this, sb.ToString());
            }
        }
    }

    //----------------------------------------------------------------------------
    // ツールストリップ用
    #region ToolStripGuiObjectBox
    /// <summary>
    /// ツールストリップ項目用ＧＵＩオブジェクトボックスクラス。
    /// </summary>
    [ToolStripItemDesignerAvailability(ToolStripItemDesignerAvailability.ToolStrip)]
    public sealed class ToolStripGuiObjectBox : ToolStripControlHost
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ToolStripGuiObjectBox()
            : base(CreateControlInstance())
        {
            base.AutoSize = false;
        }

        /// <summary>
        /// ホストしているコントロール。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public GuiObjectBox HostControl
        {
            get { return (GuiObjectBox)base.Control; }
        }

        public GuiObjectBox.Border BorderType
        {
            get { return HostControl.BorderType; }
            set { HostControl.BorderType = value; }
        }

        /// <summary>
        /// コントロールインスタンス作成。
        /// </summary>
        private static Control CreateControlInstance()
        {
            GuiObjectBox control = new GuiObjectBox();
            control.Height = 20;
            return control;
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(false)]
        public new bool AutoSize
        {
            get { return base.AutoSize; }
            set { base.AutoSize = value; }
        }


    }
    #endregion
}
