﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Windows.Forms;
using App.Data;
using App.Utility;
using nw.g3d.iflib;

namespace App.Controls
{
    /// <summary>
    /// ドキュメントを選択して閉じるダイアログクラス。
    /// </summary>
    public sealed partial class FixShaderAssignDialog : OkCancelDialog
    {
        private readonly List<string> Messages = new List<string>();

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public FixShaderAssignDialog(IEnumerable<Material> materials)
        {
            InitializeComponent();
            lvwFileList.SetItemCount(materials.Count());

            int index = 0;
            // ファイルリスト構築
            foreach (Material material in materials)
            {
                if (material.Referrers.Any())
                {
                    foreach (Model model in material.Referrers)
                    {
                        ListViewItem item = lvwFileList.Items[index];
                        item.Tag = material;
                        item.Checked = true;
                        int count;
                        var message = ShaderAssignUtility.InconsistentMessage(model, material, out count, IfShaderAssignUtility.InconsistentMessageType.Plan);

                        item.SubItems[clhMaterial.Index].Text = material.Name;
                        item.SubItems[clhModel.Index].Text = model.Name;
                        item.SubItems[clhShaderDefinition.Index].Text = material.MaterialShaderAssign.ShaderDefinitionFileName;
                        item.SubItems[clhShadingModel.Index].Text = material.MaterialShaderAssign.ShaderName;
                        item.SubItems[clhItem.Index].Text = count.ToString();
                        Messages.Add(message);
                        Debug.Assert(!string.IsNullOrEmpty(message));
                        index++;
                    }
                }
                else if (material.OwnerDocument is SeparateMaterial)
                {
                    ListViewItem item = lvwFileList.Items[index];
                    item.Tag = material;
                    item.Checked = true;
                    int count = 0;

                    item.SubItems[clhMaterial.Index].Text = material.Name;
                    item.SubItems[clhModel.Index].Text = string.Empty;
                    item.SubItems[clhShaderDefinition.Index].Text = material.MaterialShaderAssign.ShaderDefinitionFileName;
                    item.SubItems[clhShadingModel.Index].Text = material.MaterialShaderAssign.ShaderName;
                    item.SubItems[clhItem.Index].Text = count.ToString();
                    index++;
                }
            }
            lvwFileList.CheckBoxes = true;
        }

        public class MaterialInconsistentInfo
        {
            public string MaterialName{ get; set; }
            public string MaterialOwnerName{ get; set; }
            public string ShaderDefinitionFileName{ get; set; }
            public string ShaderName{ get; set; }
            public string Count{ get; set; }
            public string Message{ get; set; }
        }

        /// <summary>
        /// コンストラクタ。リードオンリー。情報表示のみ
        /// </summary>
        private readonly bool isInfomationShowOnly_ = false;

        public FixShaderAssignDialog(IEnumerable<MaterialInconsistentInfo> infos)
        {
            isInfomationShowOnly_ = true;

            InitializeComponent();
            lvwFileList.SetItemCount(infos.Count());

            int index = 0;
            // ファイルリスト構築
            foreach (var info in infos)
            {
                ListViewItem item = lvwFileList.Items[index];
                item.Tag = null;
                item.Checked = true;

                item.SubItems[clhMaterial.Index].Text = info.MaterialName;
                item.SubItems[clhModel.Index].Text = info.MaterialOwnerName;
                item.SubItems[clhShaderDefinition.Index].Text = info.ShaderDefinitionFileName;
                item.SubItems[clhShadingModel.Index].Text = info.ShaderName;
                item.SubItems[clhItem.Index].Text = info.Count;

                Messages.Add(info.Message);
                Debug.Assert(!string.IsNullOrEmpty(info.Message));
                index++;
            }

            // 編集系コントーロールを無効化
            {
                cbxAutoFix.Visible = false;
                btnCancel.Visible = false;
                btnOK.Location = btnCancel.Location;
                btnOK.Enabled = true;

                pnlBottom.Height -= cbxAutoFix.Height;
            }

            // ウィンドウ上部のラベルの文面を差し替える
            {
                lblDocumentsTitle.Text = res.Strings.FixShaderAssignDialog_AutoFix;
            }
        }

        public bool IsAutoFix
        {
            get{ return cbxAutoFix.Checked; }
        }

        /// <summary>
        /// 閉じるドキュメントリスト。
        /// </summary>
        public IEnumerable<Material> CheckedMaterials
        {
            get { return lvwFileList.CheckedItems.OfType<ListViewItem>().Select(x => (Material)x.Tag); }
        }

        public IEnumerable<Material> NotCheckedMaterials
        {
            get { return lvwFileList.Items.OfType<ListViewItem>().Where(x => !x.Checked).Select(x => (Material)x.Tag); }
        }

        private void lvwFileList_ItemChecked(object sender, ItemCheckedEventArgs e)
        {
            if (isInfomationShowOnly_ == false)
            {
                // チェックされていなければ OK ボタンは有効にならない
                btnOK.Enabled = lvwFileList.CheckedItems.Count > 0;
            }
        }

        private void lvwFileList_SelectionChanged(object sender, EventArgs e)
        {
            var index = lvwFileList.SelectedIndex;
            if (0 <= index && index < Messages.Count)
            {
                lblDescription.Text = Messages[index];
            }
            else
            {
                lblDescription.Text = "";
            }
        }
    }
}
