﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;

namespace App.Controls
{
    #region ListViewUtility
    /// <summary>
    /// リストビューユーティリティクラス。
    /// </summary>
    public static class ListViewUtility
    {
        /// <summary>
        /// LVM_GETHEADER。
        /// </summary>
        public static IntPtr GetHeader(ListView lv)
        {
            return Win32.NativeMethods.SendMessage(lv.Handle, Win32.LVM.LVM_GETHEADER, IntPtr.Zero, IntPtr.Zero);
        }

        /// <summary>
        /// LVM_GETSUBITEMRECT。
        /// </summary>
        public static Rectangle GetSubItemRect(ListView lv, int item, int subItem, int lvir)
        {
            Win32.RECT rect = new Win32.RECT();
            rect.top  = subItem;
            rect.left = lvir;
            Win32.NativeMethods.SendMessage(lv.Handle, Win32.LVM.LVM_GETSUBITEMRECT, (IntPtr)item, ref rect);
            return rect.ToRectangle();
        }

        /// <summary>
        /// LVM_SETCOLUMNORDERARRAY。
        /// </summary>
        public static void SetColumnOrderArray(ListView lv, int[] array)
        {
            Win32.NativeMethods.SendMessage(lv.Handle, Win32.LVM.LVM_SETCOLUMNORDERARRAY, (IntPtr)array.Length, array);
        }

        /// <summary>
        /// LVM_GETCOLUMNORDERARRAY。
        /// </summary>
        public static int[] GetColumnOrderArray(ListView lv)
        {
            int[] array = new int[lv.Columns.Count];
            Win32.NativeMethods.SendMessage(lv.Handle, Win32.LVM.LVM_GETCOLUMNORDERARRAY, (IntPtr)array.Length, array);
            return array;
        }

        /// <summary>
        /// 列順序を取得。
        /// </summary>
        public static int GetColumnOrder(ListView lv, int column)
        {
            if (lv.AllowColumnReorder)
            {
                int[] orderArray = GetColumnOrderArray(lv);
                for (int i = 0; i < orderArray.Length; i++)
                {
                    if (orderArray[i] == column)
                    {
                        return i;
                    }
                }
            }
            return column;
        }

        /// <summary>
        /// 列幅配列を設定。
        /// </summary>
        public static void SetColumnWidthArray(ListView lv, int[] array)
        {
            for (int i = 0; i < lv.Columns.Count && i < array.Length; i ++)
            {
                lv.Columns[i].Width = array[i];
            }
        }

        /// <summary>
        /// 列幅配列を取得。
        /// </summary>
        public static int[] GetColumnWidthArray(ListView lv)
        {
            int[] array = new int[lv.Columns.Count];
            for (int i = 0; i < array.Length; i ++)
            {
                array[i] = lv.Columns[i].Width;
            }
            return array;
        }
    }
    #endregion

    #region TreeViewUtility
    /// <summary>
    /// ツリービューユーティリティクラス。
    /// </summary>
    public static class TreeViewUtility
    {
        /// <summary>
        /// TVM_GETITEMRECT。
        /// </summary>
        public static Rectangle GetItemRect(TreeView tv, TreeNode item, bool textOnly)
        {
            Win32.RECT rect = new Win32.RECT();
            rect.left = (int)item.Handle;
            Win32.NativeMethods.SendMessage(tv.Handle, Win32.TVM.TVM_GETITEMRECT, textOnly ? (IntPtr)1 : IntPtr.Zero, ref rect);
            return rect.ToRectangle();
        }

        public static List<TreeNode> MakeNodeList(TreeNodeCollection srcNodes)
        {
            var nodes = new List<TreeNode>();
            {
                foreach (TreeNode node in srcNodes)
                {
                    nodes.Add(node);
                    nodes.AddRange(MakeNodeList(node.Nodes));
                }
            }
            return nodes;
        }

        public static IEnumerable<TreeNode> AllNodes(TreeView treeView)
        {
            return treeView.Nodes.OfType<TreeNode>().SelectMany(x => Descendents(x));
        }

        /// <summary>
        /// 自分自身も含めた子孫
        /// </summary>
        public static IEnumerable<TreeNode> Descendents(TreeNode treeNode)
        {
            return Enumerable.Repeat(treeNode, 1).Concat(treeNode.Nodes.OfType<TreeNode>().SelectMany(x => Descendents(x)));
        }
    }
    #endregion

    #region TrackBarUtility
    /// <summary>
    /// トラックバーユーティリティクラス。
    /// </summary>
    public static class TrackBarUtility
    {
        /// <summary>
        /// TBM_GETTHUMBLENGTH。
        /// </summary>
        public static int GetThumbLength(TrackBar tb)
        {
            return Win32.NativeMethods.SendMessage(tb.Handle, Win32.TBM.TBM_GETTHUMBLENGTH, IntPtr.Zero, IntPtr.Zero).ToInt32();
        }

        /// <summary>
        /// TBM_GETTHUMRECT。
        /// </summary>
        public static Rectangle GetThumbRect(TrackBar tb)
        {
            Win32.RECT rect = new Win32.RECT();
            Win32.NativeMethods.SendMessage(tb.Handle, Win32.TBM.TBM_GETTHUMBRECT, IntPtr.Zero, ref rect);
            return rect.ToRectangle();
        }

        /// <summary>
        /// TBM_GETCHANNELRECT。
        /// </summary>
        public static Rectangle GetChannelRect(TrackBar tb)
        {
            Win32.RECT rect = new Win32.RECT();
            Win32.NativeMethods.SendMessage(tb.Handle, Win32.TBM.TBM_GETCHANNELRECT, IntPtr.Zero, ref rect);
            return rect.ToRectangle();
        }

        /// <summary>
        /// TBM_SETPOS。
        /// </summary>
        public static void SetPos(TrackBar tb, int position)
        {
            Win32.NativeMethods.SendMessage(tb.Handle, Win32.TBM.TBM_SETPOS, (IntPtr)1, (IntPtr)position);
        }
    }
    #endregion

    #region HeaderUtility
    /// <summary>
    /// ヘッダユーティリティクラス。
    /// </summary>
    public static class HeaderUtility
    {
        /// <summary>
        /// HDM_GETITEMRECT。
        /// </summary>
        public static Rectangle GetItemRect(IntPtr handle, int index)
        {
            Win32.RECT rect = new Win32.RECT();
            Win32.NativeMethods.SendMessage(handle, Win32.HDM.HDM_GETITEMRECT, (IntPtr)index, ref rect);
            return rect.ToRectangle();
        }
    }
    #endregion

    #region ControlUtility
    /// <summary>
    /// コントロールユーティリティクラス。
    /// </summary>
    public static class ControlUtility
    {
        /// <summary>
        /// スクリーン上の位置を取得。
        /// </summary>
        public static Point GetScreenLocation(Control control)
        {
            Rectangle bounds = GetScreenBounds(control);
            return bounds.Location;
        }

        /// <summary>
        /// スクリーン上の領域を取得。
        /// </summary>
        public static Rectangle GetScreenBounds(Control control)
        {
            if (control is Form)
            {
                return ((Form)control).Bounds;
            }
            else
            {
                return control.Parent.RectangleToScreen(control.Bounds);
            }
        }

        /// <summary>
        /// デスクトップ上の位置を取得。
        /// </summary>
        public static Point GetDesktopLocation(Control control)
        {
            Rectangle bounds = GetDesktopBounds(control);
            return bounds.Location;
        }

        /// <summary>
        /// デスクトップ上の領域を取得。
        /// </summary>
        public static Rectangle GetDesktopBounds(Control control)
        {
            if (control is Form)
            {
                return ((Form)control).DesktopBounds;
            }
            else
            {
                // コントロールのスクリーン領域
                Rectangle controlBounds = GetScreenBounds(control);

                // フォームのスクリーン領域
                Form form = control.FindForm();
                Rectangle formBounds = form.Bounds;

                // 差分
                Point offset = new Point();
                offset.X = controlBounds.X - formBounds.X;
                offset.Y = controlBounds.Y - formBounds.Y;

                // 正しいデスクトップ領域と差分から正確な値を求める
                Rectangle formDesctopBounds = form.DesktopBounds;
                controlBounds.X = formDesctopBounds.X + offset.X;
                controlBounds.Y = formDesctopBounds.Y + offset.Y;

                return controlBounds;
            }
        }

        /// <summary>
        /// トラッキング中の自動スクロールを適用。
        /// ScrollableControl 派生クラスの OnScroll ハンドラで呼んでください。
        /// </summary>
        public static bool ApplyTrackingAutoScroll(ScrollableControl control, ScrollEventArgs e)
        {
            // つまみをトラッキング中
            if (e.Type == ScrollEventType.ThumbTrack)
            {
                // 値が変更された
                if (e.OldValue != e.NewValue)
                {
                    Point ptScroll = control.AutoScrollPosition;
                    ptScroll.X *= -1;
                    ptScroll.Y *= -1;

                    // スクロール位置再設定
                    switch (e.ScrollOrientation)
                    {
                        case ScrollOrientation.HorizontalScroll:
                            ptScroll.X = e.NewValue;
                            break;
                        case ScrollOrientation.VerticalScroll:
                            ptScroll.Y = e.NewValue;
                            break;
                    }
                    control.AutoScrollPosition = ptScroll;
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// 自分自身も含めた子孫
        /// </summary>
        public static IEnumerable<Control> Descendents(Control control)
        {
            return Enumerable.Repeat(control, 1).Concat(control.Controls.OfType<Control>().SelectMany(x => Descendents(x)));
        }

        public static Control GetFocusedActiveControl(ContainerControl cc)
        {
            var ctrl = cc.ActiveControl;
            if ((ctrl == null) || (ctrl.Focused)) return ctrl;

            // 仮定: ActiveControl であってフォーカスが当たってなければコンテナと見做せる。
            if(typeof(ContainerControl).IsInstanceOfType(ctrl))
            {
                return GetFocusedActiveControl((ContainerControl)ctrl);
            }
            else
            {
                return null;
            }
        }

        // キャプチャしているコントロールの取得
        public static Control GetCapturingControl()
        {
            IntPtr handle = App.Win32.NativeMethods.GetCapture();

            // msdn によれば Control.FromHandle よりも信用できるらしい
            return Control.FromChildHandle(handle);
        }

        /// <summary>
        /// マウスをキャプチャしているか
        /// </summary>
        public static bool IsAnyCapturing()
        {
            return GetCapturingControl() != null;
        }
    }
    #endregion

    public sealed class UpdateBlock : IDisposable
    {
        private readonly dynamic control_;

        public UpdateBlock(dynamic control)
        {
            control_ = control;
            control_.BeginUpdate();
        }

        public void Dispose()
        {
            control_.EndUpdate();
        }
    }

    public sealed class LockWindowUpdate : IDisposable
    {
        public LockWindowUpdate(Control control)
        {
            try
            {
                Win32.NativeMethods.LockWindowUpdate(control.Handle);
            }
            catch
            {
                ;
            }
        }

        public void Dispose()
        {
            Win32.NativeMethods.LockWindowUpdate(IntPtr.Zero);
        }
    }

    // キー入力によって編集するコントロールか？
    // ショートカットキーを実行させるかの判断を行う
    public interface KeyEditableControl
    {
        bool CanKeyEdit{ get; }
    }
}
