﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Linq;
using System.Windows.Forms;
using App.ConfigData;
using App.Data;
using App.Utility;
using App.res;
using ConfigCommon;
using TeamConfig;
using System.Collections.Generic;
using UserConfig;
using Microsoft.WindowsAPICodePack.Dialogs;

namespace App.Controls
{
    public sealed partial class UserConfigSettingDialog : OkCancelDialog
    {
        private readonly UIPanel[] configPanels_;

        private string preEditSearchPath;

        private string preEditStartUpPath;

        public UserConfig.UserSetting UserSetting{ get; private set; }

        public UserConfigSettingDialog()
        {
            InitializeComponent();

            DialogUtility.SetHelp(this, HelpUtility.PageKey.p_tools_menu_a_user_config);

            // 現在を保存する
            UserSetting = ObjectUtility.Clone(ApplicationConfig.UserSetting);

            ltbUserConfigPath.Text = ApplicationConfig.UserSettingFilePath;

            configPanels_ = new []
            {
                pnlSearchPath,
                pnlStartUpPath,
                pnlReloadOptions,
                pnlEditHistory,
                pnlExternalProgram,
                pnlIO,
                pnlUI,
            };

            foreach(var panel in configPanels_)
            {
                panel.Parent = null;
            }

            lbxMenu.AddItem(Strings.UserConfigSettingDialog_SearchPath,				pnlSearchPath);
            lbxMenu.AddItem(Strings.UserConfigSettingDialog_StartUpPath,			pnlStartUpPath);
            lbxMenu.AddItem(Strings.UserConfigSettingDialog_ReloadOptions,			pnlReloadOptions);
            lbxMenu.AddItem(Strings.UserConfigSettingDialog_EditHistory,			pnlEditHistory);
            lbxMenu.AddItem(Strings.UserConfigSettingDialog_ExternalProgram,		pnlExternalProgram);
            lbxMenu.AddItem(Strings.UserConfigSettingDialog_IO,						pnlIO);
            lbxMenu.AddItem(Strings.UserConfigSettingDialog_UI, pnlUI);
            lbxMenu.SelectedItemData = pnlSearchPath;
        }

        private void lbxMenu_SelectedValueChanged(object sender, EventArgs e)
        {
            foreach (var panel in configPanels_.Where(x => x != lbxMenu.SelectedItemData))
            {
                panel.Parent = null;
            }

            (lbxMenu.SelectedItemData as UIPanel).Parent = pnlConfigPanelBase;
        }

        private void pnlSearchPath_ParentChanged(object sender, EventArgs e)
        {
            if ((pnlSearchPath.Tag == null) && (pnlSearchPath.Parent != null))
            {
                UpdateSearchPath();
                this.SearchPathEditEnable(false);
                this.StartUpPathEditEnable(false);
                pnlSearchPath.Tag = true;
            }
        }

        private void pnlStartUpPath_ParentChanged(object sender, EventArgs e)
        {
            if ((pnlStartUpPath.Tag == null) && (pnlStartUpPath.Parent != null))
            {
                UpdateStartUpPath();
                pnlStartUpPath.Tag = true;
            }
        }

        private void pnlReloadOptions_ParentChanged(object sender, EventArgs e)
        {
            if ((pnlReloadOptions.Tag == null) && (pnlReloadOptions.Parent != null))
            {
                UpdateReloadOptions();
                UpdateReloadCheckBoxes();
                pnlReloadOptions.Tag = true;
            }
        }

        private void pnlEditHistory_ParentChanged(object sender, EventArgs e)
        {
            if ((pnlEditHistory.Tag == null) && (pnlEditHistory.Parent != null))
            {
                UpdateEditHistory();
                pnlEditHistory.Tag = true;
            }
        }

        private void pnlExternalProgram_ParentChanged(object sender, EventArgs e)
        {
            if ((pnlExternalProgram.Tag == null) && (pnlExternalProgram.Parent != null))
            {
                UpdateExternalProgram();
                pnlExternalProgram.Tag = true;
            }
        }

        private void pnlIO_ParentChanged(object sender, EventArgs e)
        {
            if ((pnlIO.Tag == null) && (pnlIO.Parent != null))
            {
                UpdateIO();
                pnlIO.Tag = true;
            }
        }

#region フォーム更新
        private void UpdateSearchPath()
        {
            using(var ub = new UpdateBlock(lvwSearchPath))
            {
                lvwSearchPath.SetItemCount(UserSetting.SearchPaths.Count);

                int index = 0;
                foreach(var searchPath in UserSetting.SearchPaths)
                {
                    var item = lvwSearchPath.Items[index];
                    item.Tag = index;

                    this.UpdateSearchPathItem(item, searchPath);

                    ++ index;
                }
            }
        }

        private void UpdateSearchPathItem(ListViewItem item, FileIo.SearchPath searchPath)
        {
            item.SubItems[this.clhFileIoSearchPath_Name.Index].Text = searchPath.Name;
            item.SubItems[this.clhFileIoSearchPath_Recursive.Index].Text = UIText.FlagYesNo(searchPath.Recursive);
            item.SubItems[this.clhFileIoSearchPath_Path.Index].Text = searchPath.pathXml;
            item.SubItems[this.clhFileIoSearchPath_FullPath.Index].Text = searchPath.path;
        }

        private void UpdateStartUpPath()
        {
            using(var ub = new UpdateBlock(this.lvwStartUpPath))
            {
                this.lvwStartUpPath.SetItemCount(this.UserSetting.StartUpPaths.Count);

                int index = 0;
                foreach(var startUpPath in this.UserSetting.StartUpPaths)
                {
                    this.UpdateStartUpPathItem(this.lvwStartUpPath.Items[index], startUpPath);
/*
                    item.SubItems[clhStartUpPath_Base.Index].Text		= startUpPath.Base;
                    item.SubItems[clhStartUpPath_Path.Index].Text		= startUpPath.pathXml;
                    item.SubItems[clhStartUpPath_FullPath.Index].Text	= startUpPath.FullPath();

*/
                    ++ index;
                }
            }
        }

        private void UpdateStartUpPathItem(ListViewItem item, StartUp startUpPath)
        {
            item.SubItems[this.clhStartUpPath_Base.Index].Text = startUpPath.Base;
            item.SubItems[this.clhStartUpPath_Path.Index].Text = startUpPath.pathXml;
            item.SubItems[this.clhStartUpPath_FullPath.Index].Text = startUpPath.FullPath();
        }


        private void UpdateReloadOptions()
        {
            using(var ub = new UpdateBlock(lvwReloadOptions))
            {
                Action<ListViewItem, GuiObjectID> setReloadOption = (dst, objectID) =>
                {
                    var option = UserSetting.ReloadOptions.GetOption(objectID);

                    dst.SubItems[clhReloadOptions_Type.Index].Text			= UIText.EnumValue(objectID);
                    dst.SubItems[clhReloadOptions_Reload.Index].Text		= UIText.FlagYesNo(option.Reload);
                    dst.SubItems[clhReloadOptions_ShowDialog.Index].Text	= UIText.FlagYesNo(option.ShowDialog);
                    dst.Tag = objectID;
                };

                var ids = new[] {
                    GuiObjectID.Model,
                    GuiObjectID.Texture,
                    GuiObjectID.SkeletalAnimation,
                    GuiObjectID.MaterialAnimation,
                    GuiObjectID.ShaderParameterAnimation,
                    GuiObjectID.ColorAnimation,
                    GuiObjectID.TextureSrtAnimation,
                    GuiObjectID.TexturePatternAnimation,
                    GuiObjectID.BoneVisibilityAnimation,
                    GuiObjectID.MaterialVisibilityAnimation,
                    GuiObjectID.ShapeAnimation,
                    GuiObjectID.SceneAnimation,
                    GuiObjectID.ShaderDefinition,
                };
                lvwReloadOptions.SetItemCount(ids.Length);
                for (int i = 0; i < ids.Length; i++)
                {
                    setReloadOption(lvwReloadOptions.Items[i], ids[i]);
                }
            }
        }

        private void UpdateEditHistory()
        {
            using (var block = new UIControlEventSuppressBlock())
            {
                cbxClearAfterFileClosed.Checked		= UserSetting.EditHistory.ClearAfterFileClosed;
                cbxClearAfterFileReloaded.Checked	= UserSetting.EditHistory.ClearAfterFileReloaded;
                cbxClearAfterFileSaved.Checked		= UserSetting.EditHistory.ClearAfterFileSaved;
                iudMaximumItemCount.Value			= UserSetting.EditHistory.MaximumItemCount;
            }
        }

        private void UpdateExternalProgram()
        {
            using (var block = new UIControlEventSuppressBlock())
            {
                tbxTextEditor.Text					= UserSetting.ExternalProgram.TextEditor;
                tbxTextEditorFileOpenOption.Text	= UserSetting.ExternalProgram.TextEditorFileOpenOption;
                cbxAutoCloseUserCommandLog.Checked  = UserSetting.ExternalProgram.AutoCloseUserCommandLog;
                cbxRuntimeDebugLog.Checked = UserSetting.ExternalProgram.EnableRuntimeDebugLog;
            }
        }

        private void UpdateIO()
        {
            using (var block = new UIControlEventSuppressBlock())
            {
                cbxMaterialAutoFix.Checked	= UserSetting.IO.AutoFix;
                cbxLightAnimAutoFix.Checked	= UserSetting.IO.AutoFixLightAnimation;
                cbxAutoReloadOnLoad.Checked = UserSetting.IO.AutoReloadUpdatedOnLoad;
                cbxDumpShaderSource.Checked = UserSetting.IO.DumpShaderSource;
                iudMaximumRecentlyUsedFileCount.Value = UserSetting.IO.MaximumRecentlyUsedFileCount;
                iudMaximumRecentlyUsedProjectCount.Value = UserSetting.IO.MaximumRecentlyUsedProjectCount;
            }
        }

        private void UpdateUI()
        {
            using (var block = new UIControlEventSuppressBlock())
            {
                cbxFixedSlider.Checked = UserSetting.UI.FixedSlider;
            }
        }

        private void pnlUI_ParentChanged(object sender, EventArgs e)
        {
            if ((pnlUI.Tag == null) && (pnlUI.Parent != null))
            {
                UpdateUI();
                pnlUI.Tag = true;
            }
        }
#endregion

        private void btnTextEditor_Click(object sender, EventArgs e)
        {
            if (ofdTextEditor.ShowDialog() == DialogResult.OK)
            {
                UserSetting.ExternalProgram.TextEditor = ofdTextEditor.FileName;
                UpdateExternalProgram();
            }
        }


        private void tbxTextEditor_TextChanged(object sender, EventArgs e)
        {
            UserSetting.ExternalProgram.TextEditor = tbxTextEditor.Text;
            UpdateExternalProgram();
        }

        private void tbxTextEditorFileOpenOption_TextChanged(object sender, EventArgs e)
        {
            UserSetting.ExternalProgram.TextEditorFileOpenOption = tbxTextEditorFileOpenOption.Text;
            UpdateExternalProgram();
        }

        private void cbxAutoCloseUserCommandLog_CheckedChanged(object sender, EventArgs e)
        {
            UserSetting.ExternalProgram.AutoCloseUserCommandLog = cbxAutoCloseUserCommandLog.Checked;
            UpdateExternalProgram();
        }

        private void cbxRuntimeDebugLog_CheckedChanged(object sender, EventArgs e)
        {
            UserSetting.ExternalProgram.EnableRuntimeDebugLog = cbxRuntimeDebugLog.Checked;
            UpdateExternalProgram();
        }

        private void cbxClearAfterFileClosed_CheckedChanged(object sender, EventArgs e)
        {
            UserSetting.EditHistory.ClearAfterFileClosed = cbxClearAfterFileClosed.Checked;
            UpdateEditHistory();
        }

        private void cbxClearAfterFileReloaded_CheckedChanged(object sender, EventArgs e)
        {
            UserSetting.EditHistory.ClearAfterFileReloaded = cbxClearAfterFileReloaded.Checked;
            UpdateEditHistory();
        }

        private void cbxClearAfterFileSaved_CheckedChanged(object sender, EventArgs e)
        {
            UserSetting.EditHistory.ClearAfterFileSaved = cbxClearAfterFileSaved.Checked;
            UpdateEditHistory();
        }

        private void iudMaximumItemCount_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            UserSetting.EditHistory.MaximumItemCount = iudMaximumItemCount.Value;
            UpdateEditHistory();
        }

        private void cbxMaterialAutoFix_CheckedChanged(object sender, EventArgs e)
        {
            UserSetting.IO.AutoFix = cbxMaterialAutoFix.Checked;
            UpdateIO();
        }

        private void cbxLightAnimAutoFix_CheckedChanged(object sender, EventArgs e)
        {
            UserSetting.IO.AutoFixLightAnimation = cbxLightAnimAutoFix.Checked;
            UpdateIO();
        }

        private void cbxAutoReload_CheckedChanged(object sender, EventArgs e)
        {
            UserSetting.IO.AutoReloadUpdatedOnLoad = cbxAutoReloadOnLoad.Checked;
            UpdateIO();
        }

        private void cbxFixedSlider_CheckedChanged(object sender, EventArgs e)
        {
            UserSetting.UI.FixedSlider = cbxFixedSlider.Checked;
            UpdateUI();
        }

        private void cbxDumpShaderSource_CheckedChanged(object sender, EventArgs e)
        {
            UserSetting.IO.DumpShaderSource = cbxDumpShaderSource.Checked;
            UpdateIO();
        }

        private void iudMaximumRecentlyUsedFileCount_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            UserSetting.IO.MaximumRecentlyUsedFileCount = iudMaximumRecentlyUsedFileCount.Value;
            UpdateIO();
        }

        private void iudMaximumRecentlyUsedProjectCount_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            UserSetting.IO.MaximumRecentlyUsedProjectCount = iudMaximumRecentlyUsedProjectCount.Value;
            UpdateIO();
        }

        private void cbxLogShaderConvert_CheckedChanged(object sender, EventArgs e)
        {
            UpdateIO();
        }

        #region 検索パスの編集
        private void lvwSearchPath_SelectionChanged(object sender, EventArgs e)
        {
            var id = this.lvwSearchPath.SelectedIndex;
            if (id >= 0 && id < this.UserSetting.SearchPaths.Count)
            {
                var searchPath = this.UserSetting.SearchPaths[id];
                this.tbxSearchPathName.Text = searchPath.Name;
                this.tbxSearchPath.Text = searchPath.pathXml;
                this.cbxSearchPathRecursive.Checked = searchPath.Recursive;
                this.SearchPathEditEnable(true);
            }
            else
            {
                this.SearchPathEditEnable(false);
            }
        }

        private void SearchPathEditEnable(bool enable)
        {
            this.gbxSearchPathEdit.Enabled = enable;
            this.btnSearchPathRemove.Enabled = enable;
            this.btnSearchPathUp.Enabled = enable;
            this.btnSearchPathDown.Enabled = enable;
        }

        private void btnSearchPathAppend_Click(object sender, EventArgs e)
        {
            var searchPath = new TeamConfig.FileIo.SearchPath
                                 {
                                     Name = "name",
                                     pathXml =
                                         string.IsNullOrEmpty(this.tbxSearchPath.Text)
                                             ? "./"
                                             : this.tbxSearchPath.Text,
                                     Recursive = false
                                 };
            var id = this.lvwSearchPath.SelectedIndex;
            if (id >= 0 && id < this.UserSetting.SearchPaths.Count)
            {
                this.UserSetting.SearchPaths.Insert(id, searchPath);
            }
            else
            {
                this.UserSetting.SearchPaths.Add(searchPath);
            }
            this.UpdateSearchPath();
        }

        private void btnSearchPathRemove_Click(object sender, EventArgs e)
        {
            var id = this.lvwSearchPath.SelectedIndex;
            if (id >= 0 && id < this.UserSetting.SearchPaths.Count)
            {
                this.UserSetting.SearchPaths.RemoveAt(id);
                this.UpdateSearchPath();
            }
        }

        private void SearchPathNameChanged(object sender, EventArgs e)
        {
            var id = this.lvwSearchPath.SelectedIndex;
            if (id >= 0 && id < this.UserSetting.SearchPaths.Count)
            {
                var searchPath = this.UserSetting.SearchPaths[id];
                if (searchPath.Name != this.tbxSearchPathName.Text)
                {
                    searchPath.Name = this.tbxSearchPathName.Text;
                    this.UpdateSearchPathItem(this.lvwSearchPath.SelectedItems[0], searchPath);
                }
            }
        }

        private void SearchPathPathChanged(object sender, EventArgs e)
        {
            var id = this.lvwSearchPath.SelectedIndex;
            if (id >= 0 && id < this.UserSetting.SearchPaths.Count)
            {
                var searchPath = this.UserSetting.SearchPaths[id];
                if (searchPath.pathXml != this.tbxSearchPath.Text)
                {
                    searchPath.pathXml = this.tbxSearchPath.Text;
                    this.UpdateSearchPathItem(this.lvwSearchPath.SelectedItems[0], searchPath);
                }
            }
        }

        private void SearchPathRecursiveChanged(object sender, EventArgs e)
        {
            var id = this.lvwSearchPath.SelectedIndex;
            if (id >= 0 && id < this.UserSetting.SearchPaths.Count)
            {
                var searchPath = this.UserSetting.SearchPaths[id];
                if (searchPath.Recursive != this.cbxSearchPathRecursive.Checked)
                {
                    searchPath.Recursive = this.cbxSearchPathRecursive.Checked;
                    this.UpdateSearchPathItem(this.lvwSearchPath.SelectedItems[0], searchPath);
                }
            }
        }

        private void tbxSearchPath_Validating(object sender, System.ComponentModel.CancelEventArgs e)
        {
            if (string.IsNullOrEmpty(this.tbxSearchPath.Text))
            {
                e.Cancel = true;
                UIMessageBox.Warning(Strings.UserConfigSettingDialog_DoNotOmitPath);
                this.tbxSearchPath.Text = this.preEditSearchPath;
            }
            else
            {
                e.Cancel = false;
            }
        }

        private void tbxSearchPath_Enter(object sender, EventArgs e)
        {
            this.preEditSearchPath = this.tbxSearchPath.Text;
        }

        private void btnSearchPathUp_Click(object sender, EventArgs e)
        {
            var id = this.lvwSearchPath.SelectedIndex;
            if (id >= 1 && id < this.UserSetting.SearchPaths.Count)
            {
                var searchPath = this.UserSetting.SearchPaths[id];
                this.UserSetting.SearchPaths.RemoveAt(id);
                this.UserSetting.SearchPaths.Insert(id - 1, searchPath);
                this.UpdateSearchPath();
                this.lvwSearchPath.SetSelectedItem(id - 1);
            }
        }

        private void btnSearchPathDown_Click(object sender, EventArgs e)
        {
            var id = this.lvwSearchPath.SelectedIndex;
            if (id >= 0 && id < (this.UserSetting.SearchPaths.Count - 1))
            {
                var searchPath = this.UserSetting.SearchPaths[id];
                this.UserSetting.SearchPaths.RemoveAt(id);
                this.UserSetting.SearchPaths.Insert(id + 1, searchPath);
                this.UpdateSearchPath();
                this.lvwSearchPath.SetSelectedItem(id + 1);
            }
        }

        private void btnAdditionalSearchDirectory_Click(object sender, EventArgs e)
        {
            using (var dialog = new CommonOpenFileDialog())
            {
                dialog.Title = res.Strings.IO_SpecifySearchDirectory;
                dialog.IsFolderPicker = true;

                if (!string.IsNullOrEmpty(tbxSearchPath.Text))
                {
                    dialog.InitialDirectory = tbxSearchPath.Text;
                }

                if (dialog.ShowDialog(this.Handle) == CommonFileDialogResult.Ok)
                {
                    tbxSearchPath.Text = dialog.FileName;
                }
            }
        }
        #endregion //検索パスの編集

        #region スタートアップパスの編集
        private void lvwStartUpPath_SelectionChanged(object sender, EventArgs e)
        {
            var id = this.lvwStartUpPath.SelectedIndex;
            if (id >= 0 && id < this.UserSetting.StartUpPaths.Count)
            {
                var startUpPath = this.UserSetting.StartUpPaths[id];
                this.tbxStartUpPathBase.Text = startUpPath.Base;
                this.tbxStartUpPath.Text = startUpPath.pathXml;
                this.StartUpPathEditEnable(true);
            }
            else
            {
                this.StartUpPathEditEnable(false);
            }
        }

        private void StartUpPathEditEnable(bool enable)
        {
            this.gbxStartUpPathEdit.Enabled = enable;
            this.btnStartUpPathRemove.Enabled = enable;
            this.btnStartUpPathUp.Enabled = enable;
            this.btnStartUpPathDown.Enabled = enable;
        }

        private void btnStartUpPathAppend_Click(object sender, EventArgs e)
        {
            var startUpPath = new ConfigCommon.StartUp
                                  {
                                      Base = "name",
                                      pathXml =
                                          string.IsNullOrEmpty(this.tbxStartUpPath.Text)
                                              ? "./"
                                              : this.tbxStartUpPath.Text,
                                  };
            var id = this.lvwStartUpPath.SelectedIndex;
            if (id >= 0 && id < this.UserSetting.StartUpPaths.Count)
            {
                this.UserSetting.StartUpPaths.Insert(id, startUpPath);
            }
            else
            {
                this.UserSetting.StartUpPaths.Add(startUpPath);
            }
            this.UpdateStartUpPath();
        }

        private void btnStartUpPathRemove_Click(object sender, EventArgs e)
        {
            var id = this.lvwStartUpPath.SelectedIndex;
            if (id >= 0 && id < this.UserSetting.StartUpPaths.Count)
            {
                this.UserSetting.StartUpPaths.RemoveAt(id);
                this.UpdateStartUpPath();
            }
        }

        private void StartUpPathBaseChanged(object sender, EventArgs e)
        {
            var id = this.lvwStartUpPath.SelectedIndex;
            if (id >= 0 && id < this.UserSetting.StartUpPaths.Count)
            {
                var startUpPath = this.UserSetting.StartUpPaths[id];
                if (startUpPath.Base != this.tbxStartUpPathBase.Text)
                {
                    startUpPath.Base = this.tbxStartUpPathBase.Text;
                    this.UpdateStartUpPathItem(this.lvwStartUpPath.SelectedItems[0], startUpPath);
                }
            }
        }

        private void StartUpPathPathChanged(object sender, EventArgs e)
        {
            var id = this.lvwStartUpPath.SelectedIndex;
            if (id >= 0 && id < this.UserSetting.StartUpPaths.Count)
            {
                var startUpPath = this.UserSetting.StartUpPaths[id];
                if (startUpPath.pathXml != this.tbxStartUpPath.Text)
                {
                    startUpPath.pathXml = this.tbxStartUpPath.Text;
                    this.UpdateStartUpPathItem(this.lvwStartUpPath.SelectedItems[0], startUpPath);
                }
            }

        }

        private void tbxStartUpPath_Validating(object sender, System.ComponentModel.CancelEventArgs e)
        {
            if (string.IsNullOrEmpty(this.tbxStartUpPath.Text))
            {
                e.Cancel = true;
                UIMessageBox.Warning(Strings.UserConfigSettingDialog_DoNotOmitPath);
                this.tbxStartUpPath.Text = this.preEditStartUpPath;
            }
            else
            {
                e.Cancel = false;
            }
        }

        private void tbxStartUpPath_Enter(object sender, EventArgs e)
        {
            this.preEditStartUpPath = this.tbxStartUpPath.Text;
        }

        private void btnStartUpPathUp_Click(object sender, EventArgs e)
        {
            var id = this.lvwStartUpPath.SelectedIndex;
            if (id >= 1 && id < this.UserSetting.StartUpPaths.Count)
            {
                var startUpPath = this.UserSetting.StartUpPaths[id];
                this.UserSetting.StartUpPaths.RemoveAt(id);
                this.UserSetting.StartUpPaths.Insert(id - 1, startUpPath);
                this.UpdateStartUpPath();
                this.lvwStartUpPath.SetSelectedItem(id - 1);
            }
        }

        private void btnStartUpPathDown_Click(object sender, EventArgs e)
        {
            var id = this.lvwStartUpPath.SelectedIndex;
            if (id >= 0 && id < (this.UserSetting.StartUpPaths.Count - 1))
            {
                var startUpPath = this.UserSetting.StartUpPaths[id];
                this.UserSetting.StartUpPaths.RemoveAt(id);
                this.UserSetting.StartUpPaths.Insert(id + 1, startUpPath);
                this.UpdateStartUpPath();
                this.lvwStartUpPath.SetSelectedItem(id + 1);
            }
        }

        private void btnStartUpSearchDirectory_Click(object sender, EventArgs e)
        {
            using (var dialog = new CommonOpenFileDialog())
            {
                dialog.Title = res.Strings.IO_SpecifySearchDirectory;
                dialog.IsFolderPicker = true;

                if (!string.IsNullOrEmpty(tbxStartUpPath.Text))
                {
                    dialog.InitialDirectory = tbxStartUpPath.Text;
                }

                if (dialog.ShowDialog(this.Handle) == CommonFileDialogResult.Ok)
                {
                    tbxStartUpPath.Text = dialog.FileName;
                }
            }
        }
        #endregion スタートアップパスの編集

        #region 再読み込み設定の編集
        private void lvwReloadOptions_SelectionChanged(object sender, EventArgs e)
        {
            UpdateReloadCheckBoxes();
        }

        private void UpdateReloadCheckBoxes()
        {
            if (lvwReloadOptions.SelectedItems.Count > 0)
            {
                var options = new List<ReloadOptions.Option>();
                for (int i = 0; i < this.lvwReloadOptions.SelectedItems.Count; i++)
                {
                    var item = this.lvwReloadOptions.SelectedItems[i];
                    var objectID = (GuiObjectID)item.Tag;
                    var option = this.UserSetting.ReloadOptions.GetOption(objectID);
                    if (option != null)
                    {
                        options.Add(option);
                    }
                }

                if (options.All(x => x.Reload))
                {
                    cbxReload.CheckState = CheckState.Checked;
                }
                else if (options.All(x => !x.Reload))
                {
                    cbxReload.CheckState = CheckState.Unchecked;
                }
                else
                {
                    cbxReload.CheckState = CheckState.Indeterminate;
                }

                if (options.All(x => x.ShowDialog))
                {
                    cbxReloadShowDialog.CheckState = CheckState.Checked;
                }
                else if (options.All(x => !x.ShowDialog))
                {
                    cbxReloadShowDialog.CheckState = CheckState.Unchecked;
                }
                else
                {
                    cbxReloadShowDialog.CheckState = CheckState.Indeterminate;
                }

                if (this.lvwReloadOptions.SelectedItems.Count == 1)
                {
                    var item = this.lvwReloadOptions.SelectedItems[0];
                    this.lblReloadType.Text = UIText.EnumValue((GuiObjectID)item.Tag) + @": ";
                }
                else
                {
                    this.lblReloadType.Text = "";
                }
                this.ReloadEditEnable(true);
            }
            else
            {
                this.ReloadEditEnable(false);
            }
        }

        private void ReloadEditEnable(bool enable)
        {
            this.gbxReloadEdit.Enabled = enable;
            if (!enable)
            {
                lblReloadType.Text = "";
            }
        }

        private void cbxReload_CheckedChanged(object sender, EventArgs e)
        {
            if (this.cbxReload.CheckState != CheckState.Indeterminate && lvwReloadOptions.SelectedItems.Count > 0)
            {
                for (var i = 0; i < this.lvwReloadOptions.SelectedItems.Count; i++)
                {
                    var item = this.lvwReloadOptions.SelectedItems[i];
                    var objectID = (GuiObjectID)item.Tag;
                    var option = this.UserSetting.ReloadOptions.GetOption(objectID);
                    option.Reload = this.cbxReload.Checked;
                }
                this.UpdateReloadOptions();
            }
        }

        private void cbxReloadConfirm_CheckedChanged(object sender, EventArgs e)
        {
            if (this.cbxReloadShowDialog.CheckState != CheckState.Indeterminate && lvwReloadOptions.SelectedItems.Count > 0)
            {
                for (var i = 0; i < this.lvwReloadOptions.SelectedItems.Count; i++)
                {
                    var item = this.lvwReloadOptions.SelectedItems[i];
                    var objectID = (GuiObjectID)item.Tag;
                    var option = this.UserSetting.ReloadOptions.GetOption(objectID);
                    option.ShowDialog = this.cbxReloadShowDialog.Checked;
                }
                this.UpdateReloadOptions();
            }
        }

        #endregion 再読み込み設定の編集
    }
}
