﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Drawing;
using System.Text.RegularExpressions;
using System.Windows.Forms;

namespace ColorPicker
{
    /// <summary>
    /// テキストボックススライダーコントロールクラス。
    /// </summary>
    internal sealed class TextSliderControl : UserControl
    {
        // . の入力を確実に受け取るために UITextBox を使う
        private App.Controls.UITextBox m_txtValue;
        private ColorPicker.RGBSlider m_bar;
        private System.Windows.Forms.Label m_label;
        //	装飾用
        private Rectangle m_BarWakuOut	= new Rectangle(32, 8, 256, 24);
        private Rectangle m_BarWakuIn	= new Rectangle(32, 8, 256, 24);

        private ConfigCommon.ValueMode m_mode = ConfigCommon.ValueMode.Float;
        private ConfigCommon.ColorMode m_colorMode = ConfigCommon.ColorMode.RGB;
        private bool		m_editRight	= false;

        // 外部から設定された HDR 上限値 (読み取り専用)。
        private float m_hdrUpperBound = float.MaxValue;

        // 外部から設定されたカラースケール値 (読み取り専用)。
        private float m_colorScale = 1.0f;

        /// <summary>
        /// HDR の上限値を設定する。
        /// カラースケールと組み合わせて値の調整に使用する。
        /// </summary>
        public float HDRUpperBound
        {
            set
            {
                m_hdrUpperBound = value;
            }
        }

        /// <summary>
        /// カラースケールを設定する。
        /// HDR の上限値と組み合わせて値の調整に使用する。
        /// </summary>
        public float ColorScale
        {
            set
            {
                m_colorScale = value;
            }
        }

        public float SliderMaxValue
        {
            get
            {
                return m_bar.MaxValue;
            }
            set
            {
                // SliderMaxValueChanged イベントは
                // m_bar.MaxValueChanged -> bar_MaxValueChanged -> SliderMaxValueChanged で呼び出されるので
                // ここでのイベント呼び出しは行わない。
                m_bar.MaxValue = value;
            }
        }

        /// <summary>
        /// 必要なデザイナ変数です。
        /// </summary>
        private readonly System.ComponentModel.Container components = null;

        public TextSliderControl()
        {
            // この呼び出しは、Windows.Forms フォーム デザイナで必要です。
            InitializeComponent();

            // イベント、プロパティ設定
            DoubleBuffered = true;
            m_bar.ColorSelected += new ColorSlider.ColorSelectedHandler(ColorSlider_ColorSelected);
            m_bar.MaxValueChanged += new EventHandler(bar_MaxValueChanged);

            InitializeLine();
            var color = m_bar.GetSelectColor();
            m_bar.UpdateColor(color, HSV.FromRGB(color), true, true);
            Invalidate(true);
        }

        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                if (components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose(disposing);
        }

        #region コンポーネント デザイナで生成されたコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            this.m_txtValue = new App.Controls.UITextBox();
            this.m_label = new System.Windows.Forms.Label();
            this.m_bar = new ColorPicker.RGBSlider();
            this.SuspendLayout();
            //
            // m_txtValue
            //
            this.m_txtValue.AcceptsReturn = true;
            this.m_txtValue.ImeMode = System.Windows.Forms.ImeMode.Disable;
            this.m_txtValue.Location = new System.Drawing.Point(16, 4);
            this.m_txtValue.Name = "m_txtValue";
            this.m_txtValue.Size = new System.Drawing.Size(48, 19);
            this.m_txtValue.TabIndex = 1;
            this.m_txtValue.Text = "0";
            this.m_txtValue.TextAlign = System.Windows.Forms.HorizontalAlignment.Right;
            this.m_txtValue.KeyDown += new System.Windows.Forms.KeyEventHandler(this.TextBox_KeyDown);
            this.m_txtValue.KeyPress += new System.Windows.Forms.KeyPressEventHandler(this.TextBox_KeyPress);
            this.m_txtValue.Leave += new System.EventHandler(this.TextBox_Leave);
            //
            // m_label
            //
            this.m_label.Location = new System.Drawing.Point(0, 4);
            this.m_label.Name = "m_label";
            this.m_label.Size = new System.Drawing.Size(16, 19);
            this.m_label.TabIndex = 0;
            this.m_label.Text = "R";
            this.m_label.TextAlign = System.Drawing.ContentAlignment.MiddleCenter;
            //
            // m_bar
            //
            this.m_bar.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left)
            | System.Windows.Forms.AnchorStyles.Right)));
            this.m_bar.ColorValue = 0F;
            this.m_bar.ComponentType = ColorPicker.ColorType.R;
            this.m_bar.Location = new System.Drawing.Point(72, 6);
            this.m_bar.MaxValue = 255F;
            this.m_bar.Name = "m_bar";
            this.m_bar.Size = new System.Drawing.Size(256, 15);
            this.m_bar.TabIndex = 2;
            this.m_bar.TabStop = false;
            //
            // TextSliderControl
            //
            this.Controls.Add(this.m_label);
            this.Controls.Add(this.m_bar);
            this.Controls.Add(this.m_txtValue);
            this.Name = "TextSliderControl";
            this.Size = new System.Drawing.Size(336, 24);
            this.ResumeLayout(false);
            this.PerformLayout();

        }
        #endregion

        #region イベント

        /// <summary>
        /// カラーが変更されたときに発生します。
        /// </summary>
        internal delegate void ColorSelectedHandler( object sender, ColorChangedEventArgs e );
        internal event ColorSelectedHandler ColorSelected;
        internal event SliderMaxValueChangedEventHandler SliderMaxValueChanged;
        internal event EventHandler IsDefaultLinearChanged;

        #endregion

        #region テキストボックスのキーやフォーカスのイベント処理
        /// <summary>
        /// 数字のみ入力許可
        /// </summary>
        private void TextBox_KeyPress(object sender, KeyPressEventArgs e)
        {
            TextBox obj = (TextBox)sender;
            if ((e.KeyChar < '0' || e.KeyChar > '9') && e.KeyChar != '\b')
            {
                if (m_mode != ConfigCommon.ValueMode.Float ||
                    (e.KeyChar != '.' && e.KeyChar != 'E' && e.KeyChar != 'e' && e.KeyChar != '-' && e.KeyChar != '+'))
                {
                    e.Handled = true;
                }
            }
        }

        /// <summary>
        /// リターンキーなら入力確定
        /// </summary>
        private void TextBox_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter || e.KeyCode == Keys.Return)
            {
                TextBox_Leave(sender, EventArgs.Empty);
                m_txtValue.SelectAll();
            }
        }

        /// <summary>
        /// 数値の確定処理
        /// </summary>
        private void TextBox_Leave(object sender, EventArgs e)
        {
            TextBox obj = (TextBox)sender;
            if (obj.Modified)
            {
                float colValue;
                if (m_mode == ConfigCommon.ValueMode.Uchar)
                {
                    int v;
                    if (int.TryParse(obj.Text, out v))
                    {
                        switch (m_bar.ComponentType)
                        {
                            case ColorType.R:
                            case ColorType.G:
                            case ColorType.B:
                            case ColorType.A:
                                colValue = v / 255.0f;
                                break;
                            default:
                            colValue = v;
                            break;
                        }
                    }
                    else
                    {
                        colValue = m_bar.ColorValue;
                    }
                }
                else
                {
                    float ff;
                    if (float.TryParse(obj.Text, out ff))
                    {
                        colValue = ff;
                    }
                    else
                    {
                        colValue = m_bar.ColorValue;
                    }
                }

                switch (m_bar.ComponentType)
                {
                    case ColorType.R:
                    case ColorType.G:
                    case ColorType.B:
                        // m_bar.MaxValue より大きな値も受け付ける。
                        // m_bar.MaxValue の調整は m_bar.ColorValue への代入直前に行っている。
                        colValue = Math.Max(0.0f, colValue);
                        if ((colValue * m_colorScale) > m_hdrUpperBound)
                        {
                            colValue = m_hdrUpperBound / m_colorScale;
                        }
                        break;
                    case ColorType.V:
                        // m_bar.MaxValue より大きな値も受け付ける。
                        // m_bar.MaxValue の調整は m_bar.ColorValue への代入直前に行っている。
                        colValue = Math.Max(0.0f, colValue);
                        if ((colValue * m_colorScale / 100.0f) > m_hdrUpperBound)
                        {
                            colValue = m_hdrUpperBound / m_colorScale * 100.0f;
                        }
                        break;
                    default:
                        colValue = Math.Max(0, Math.Min(m_bar.MaxValue, colValue));
                        break;
                }

                if (m_bar.ColorValue != colValue)
                {
                    RGB color = m_bar.GetSelectColor();
                    HSV hsv = m_bar.GetHsv(); // RGBSlider で保存されている値を使用する

                    switch (m_bar.ComponentType)
                    {
                        case ColorType.R:
                            color = new RGB(color.A, colValue, color.G, color.B);
                            hsv = HSV.FromRGB(color);
                            break;
                        case ColorType.G:
                            color = new RGB(color.A, color.R, colValue, color.B);
                            hsv = HSV.FromRGB(color);
                            break;
                        case ColorType.B:
                            color = new RGB(color.A, color.R, color.G, colValue);
                            hsv = HSV.FromRGB(color);
                            break;
                        case ColorType.A:
                            color = new RGB(colValue, color.R, color.G, color.B);
                            hsv = HSV.FromRGB(color);
                            break;
                        case ColorType.H:
                            hsv = new HSV(colValue, hsv.S, hsv.V);
                            color = RGB.FromHSV(hsv);
                            break;
                        case ColorType.S:
                            hsv = new HSV(hsv.H, colValue, hsv.V);
                            color = RGB.FromHSV(hsv);
                            break;
                        default:
                            hsv = new HSV(hsv.H, hsv.S, colValue);
                            color = RGB.FromHSV(hsv);
                            break;
                    }

                    // m_bar.ColorValue への代入前に m_bar.MaxValue を更新しておく。
                    // m_bar.MaxValue の変更時に bar_MaxValueChanged() が呼び出される。
                    switch (m_bar.ComponentType)
                    {
                        case ColorType.R:
                        case ColorType.G:
                        case ColorType.B:
                            var maxValue = Math.Max(Math.Max(color.R, color.G), color.B);
                            if (maxValue > m_bar.MaxValue)
                            {
                                // 最大値を拡大。
                                m_bar.MaxValue = maxValue;
                            }
                            else if (maxValue <= 1.0f)
                            {
                                // 最大値を正規化範囲に戻す。
                                m_bar.MaxValue = 1.0f;
                            }
                            break;
                        case ColorType.V:
                            if (hsv.V > m_bar.MaxValue)
                            {
                                // 最大値を拡大。
                                m_bar.MaxValue = hsv.V;
                            }
                            else if (hsv.V <= 100.0f)
                            {
                                // 最大値を正規化範囲に戻す。
                                m_bar.MaxValue = 100.0f;
                            }
                            break;
                        default:
                            break;
                    }

                    m_bar.ColorValue = colValue;
                    m_bar.UpdateColor(color, hsv, true, false);
                    ColorSlider_ColorSelected(this, new ColorChangedEventArgs(color, true, false /* 両方ありえるけど呼び出し側で判断 */));
                }

                WriteText();
            }

            obj.Invalidate();
        }
#endregion

#region プロパティ
        /// <summary>
        /// 選択カラー。
        /// </summary>
        public float ColorValue
        {
            get { return m_bar.ColorValue; }
        }

        /// <summary>
        /// 選択カラー。
        /// </summary>
        [Category("カスタム")]
        [Description("選択カラー。")]
        [DefaultValue(typeof(Color), "Black")]
        public RGB Color
        {
            get { return m_bar.GetSelectColor(); }
            /*
            set
            {
                HSV hsv = HSV.FromRGB(value);
                WriteText();
                m_bar.UpdateColor(value, hsv, true, false);
            }*/
        }

        /// <summary>
        /// RGBAのいずれか？
        /// </summary>
        [Category("カスタム")]
        [Description("RGBHSVのいずれか。")]
        [DefaultValue(ColorType.R)]
        public ColorType ComponentType
        {
            get { return m_bar.ComponentType; }
            set
            {
                m_bar.ComponentType = value;
                switch (value)
                {
                    case ColorType.R:
                        m_label.Text = "R";
                        break;
                    case ColorType.G:
                        m_label.Text = "G";
                        break;
                    case ColorType.B:
                        m_label.Text = "B";
                        break;
                    case ColorType.A:
                        m_label.Text = "A";
                        break;
                    case ColorType.H:
                        m_label.Text = "H";
                        break;
                    case ColorType.S:
                        m_label.Text = "S";
                        break;
                    default:
                        m_label.Text = "V";
                        break;
                }
            }
        }

        /// <summary>
        /// カラー値の表示方式。
        /// </summary>
        [Category("カスタム")]
        [Description("カラー値の表示方式。")]
        [DefaultValue(ConfigCommon.ValueMode.Float)]
        public ConfigCommon.ValueMode ValueMode
        {
            get { return m_mode; }
            set
            {
                m_mode = value;
                if (m_mode == ConfigCommon.ValueMode.Uchar)
                {
                    m_bar.Precision = 0;
                }
                else
                {
                    m_bar.Precision = 2;
                }

                if (Created)
                {
                    WriteText();
                }
            }
        }

        public ConfigCommon.ColorMode ColorMode
        {
            get { return m_colorMode; }
            set
            {
                m_colorMode = value;
                if (ComponentType != ColorType.A)
                {
                    if (IsLinear)
                    {
                        m_bar.Gamma = 1 / 2.2f;
                    }
                    else
                    {
                        m_bar.Gamma = 1;
                    }
                }
            }
        }

        public bool IsLinear
        {
            get
            {
                return ColorMode == ConfigCommon.ColorMode.LinearRgba || ColorMode == ConfigCommon.ColorMode.LinearRgb;
            }
        }

        private bool isDefaultLinear;
        public bool IsDefaultLinear
        {
            get
            {
                return isDefaultLinear;
            }
            set
            {
                isDefaultLinear = value;
                if (ComponentType == ColorType.A)
                {
                    if (isDefaultLinear)
                    {
                        m_bar.Gamma = 1 / 2.2f;
                    }
                    else
                    {
                        m_bar.Gamma = 1;
                    }
                }

                if (IsDefaultLinearChanged != null)
                {
                    IsDefaultLinearChanged(this, EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// テキストボックスの表示位置を右側に設定する。
        /// </summary>
        [Category("カスタム")]
        [Description("テキストボックスの表示位置を右側に設定する。")]
        [DefaultValue(false)]
        public bool TextBoxIsRight
        {
            get { return m_editRight; }
            set
            {
                m_editRight = value;
                SuspendLayout();
                if (value)
                {
                    m_txtValue.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
                    m_bar.Location = new System.Drawing.Point(20, 6);
                    m_txtValue.Location = new System.Drawing.Point(Size.Width - 52, 4);
                    InitializeLine();
                }
                else
                {
                    m_txtValue.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left)));
                    m_bar.Location = new System.Drawing.Point(72, 6);
                    m_txtValue.Location = new System.Drawing.Point(16, 4);
                    InitializeLine();
                }
                ResumeLayout();
                Invalidate();
            }
        }

        /// <summary>
        /// テキストボックスの前景色
        /// </summary>
        [Category("カスタム")]
        [Description("テキストボックスの前景色を取得または設定します。")]
        public Color TextBoxForeColor
        {
            get { return m_txtValue.ForeColor; }
            set { m_txtValue.ForeColor = value; }
        }

        /// <summary>
        /// テキストボックスの背景色
        /// </summary>
        [Category("カスタム")]
        [Description("テキストボックスの背景色を取得または設定します。")]
        public Color TextBoxBackColor
        {
            get { return m_txtValue.BackColor; }
            set { m_txtValue.BackColor = value; }
        }

        /// <summary>
        /// テキストボックスのフォント
        /// </summary>
        [Category("カスタム")]
        [Description("テキストボックスのフォントを取得または設定します。")]
        public Font TextBoxFont
        {
            get { return m_txtValue.Font; }
            set { m_txtValue.Font = value; }
        }
#endregion

#region オーバーライド
        /// <summary>
        /// OnPaintのオーバーライド（ボーダーの装飾）。
        /// </summary>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            if (DesignMode)
            {
                InitializeLine();
            }

            Pen pen = Enabled ? Pens.Black : Pens.DarkGray;
            Graphics g = e.Graphics;
            ControlPaint.DrawBorder3D(g, m_BarWakuOut);
            g.DrawRectangle(pen, m_BarWakuIn);
        }

        /// <summary>
        /// デザインモード時は親を再描画
        /// </summary>
        protected override void OnSizeChanged(EventArgs e)
        {
            base.OnSizeChanged(e);
            InitializeLine();
            if (DesignMode)
            {
                var color = m_bar.GetSelectColor();
                m_bar.UpdateColor(color, HSV.FromRGB(color), true, false);
                Invalidate(true);
            }
        }
#endregion

        /// <summary>
        /// RGBスライダーでカラー選択されたときに実行される
        /// </summary>
        private void ColorSlider_ColorSelected(object sender, ColorChangedEventArgs e)
        {
            Invalidate(true);
            if (sender == m_bar)
            {
                WriteText();
            }
            // 更新
            if (ColorSelected != null)
            {
                ColorSelected(this, e);
            }
        }

        private void bar_MaxValueChanged(object sender, EventArgs e)
        {
            if (SliderMaxValueChanged != null)
            {
                SliderMaxValueChanged(this, new SliderMaxValueChangedEventArgs(m_bar.ComponentType, m_bar.MaxValue));
            }
        }

        public const int DigitPrecision = 2;

        // TODO: - は必要? あっても害はないはず。
        private static readonly Regex onlyDigits = new Regex(@"^-?[0-9]+$");

        private string FormatFloat(float value)
        {
            var r = value.ToString("R");
            var f3 = value.ToString("F3");
            if (f3.StartsWith(r))
            {
                return f3;
            }

            return r;
        }

        /// <summary>
        /// テキストボックスの変更
        /// </summary>
        private void WriteText()
        {
            switch (ComponentType)
            {
                case ColorType.H: case ColorType.S: case ColorType.V:
                    if (ValueMode == ConfigCommon.ValueMode.Uchar)
                    {
                        m_txtValue.Text = ((int)Math.Round(m_bar.ColorValue)).ToString();
                    }
                    else
                    {
                        m_txtValue.Text = FormatFloat(m_bar.ColorValue);
                    }
                    break;
                default:
                    if (ValueMode == ConfigCommon.ValueMode.Uchar)
                    {
                        m_txtValue.Text = ((int)Math.Round(255 * m_bar.ColorValue)).ToString();
                    }
                    else
                    {
                        m_txtValue.Text = FormatFloat(m_bar.ColorValue);
                    }
                    break;
            }
        }

        /// <summary>
        /// ラインの再設定
        /// </summary>
        private void InitializeLine()
        {
            int x = m_bar.Location.X;
            int y = m_bar.Location.Y;
            int w = m_bar.Size.Width;
            int h = m_bar.Size.Height;
            m_BarWakuOut = new Rectangle(x - 3, y - 3, w + 7, h + 7);
            m_BarWakuIn = new Rectangle(x - 1, y - 1, w + 1, h + 1);
        }

        /// <summary>
        /// 指定されたカラーで描画するビットマップとカラーを更新する
        /// </summary>
        public void UpdateColor(RGB color, HSV hsv, bool resetPoint, bool flag)
        {
            m_bar.UpdateColor(color, hsv, resetPoint, flag);
            WriteText();
            Invalidate();
        }
    }
}
