﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using ColorPicker;

namespace App.Controls
{
    /// <summary>
    /// 拡張カラーピッカーコントロールクラス。
    /// </summary>
    public sealed partial class ColorPickerControlEx : ColorPickerControl
    {
        /// <summary>
        /// 現在のカラーモードにコメントをつけるか。
        /// </summary>
        private bool showCurrentColorComment = true;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ColorPickerControlEx()
        {
            InitializeComponent();

            // カラー表示方式
            this.UpdateColorModeItems();

            // 値表示方式
            cmbValueMode.AddItem("0 to 255", ConfigCommon.ValueMode.Uchar);
            cmbValueMode.AddItem("0.0 to 1.0", ConfigCommon.ValueMode.Float);
            cmbValueMode.SetSelectedItemData(base.ValueMode);
            cmbValueMode.SelectedIndexChanged += new EventHandler(Event_cmbValueMode_SelectedIndexChanged);

            // デフォルトリニアが変わったときカラー表示方式のアイテムを更新する
            this.IsDefaultLinearChanged += (s, e) => { this.UpdateColorModeItems(); };
        }

        /// <summary>
        /// カラー成分のモード（再定義）。
        /// </summary>
        [DefaultValue(ConfigCommon.ColorMode.RGBA)]
        public new ConfigCommon.ColorMode ColorMode
        {
            get { return base.ColorMode; }
            set
            {
                // 既定の処理
                base.ColorMode = value;

                // コンボボックス表示更新
                ColorModeOption modeOption = ColorModeToOption(value);
                cmbColorMode.SetSelectedItemData(modeOption);
            }
        }

        /// <summary>
        /// カラー値の表示方式（再定義）。
        /// </summary>
        [DefaultValue(ConfigCommon.ValueMode.Float)]
        public new ConfigCommon.ValueMode ValueMode
        {
            get { return base.ValueMode; }
            set
            {
                // 既定の処理
                base.ValueMode = value;

                // コンボボックス表示更新
                cmbValueMode.SetSelectedItemData(value);
            }
        }

        /// <summary>
        /// 現在のカラーモードにコメントをつけるか。
        /// </summary>
        public bool ShowCurrentColorComment
        {
            get { return showCurrentColorComment; }
            set
            {
                showCurrentColorComment = value;
                this.UpdateColorModeItems();
            }
        }

        /// <summary>
        /// カラー表示方式コンボボックスのアイテムを更新。
        /// </summary>
        private void UpdateColorModeItems()
        {
            // アイテムがなければ初期化する
            if (cmbColorMode.Items.Count == 0)
            {
                cmbColorMode.AddItem("sRGB", ColorModeOption.RGB);
                cmbColorMode.AddItem("Linear", ColorModeOption.LinearRgb);
                cmbColorMode.AddItem("HSV", ColorModeOption.HSV);
                cmbColorMode.SetSelectedItemData(ColorModeToOption(base.ColorMode));
                cmbColorMode.SelectedIndexChanged += new EventHandler(Event_cmbColorMode_SelectedIndexChanged);
            }

            string rgbModeText = "sRGB";
            string linearRgbModeText = "Linear";

            // ステータスに応じて現在のカラーモードにコメントを追加する
            if (ShowCurrentColorComment)
            {
                if (IsDefaultLinear)
                {
                    linearRgbModeText += res.Strings.ColorPicker_ColorMode_Current;
                }
                else
                {
                    rgbModeText += res.Strings.ColorPicker_ColorMode_Current;
                }
            }

            // コンボボックスアイテムの文字列を更新
            ((UIListControlItem)cmbColorMode.Items[0]).Text = rgbModeText;
            ((UIListControlItem)cmbColorMode.Items[1]).Text = linearRgbModeText;

            cmbColorMode.Invalidate();
        }

        /// <summary>
        /// ColorMode 値を対応する ColorModeOption に変換。
        /// </summary>
        private static ColorModeOption ColorModeToOption(ConfigCommon.ColorMode colorMode)
        {
            switch (colorMode)
            {
                case ConfigCommon.ColorMode.HSV:
                case ConfigCommon.ColorMode.HSVA:
                    return ColorModeOption.HSV;
                case ConfigCommon.ColorMode.LinearRgba:
                case ConfigCommon.ColorMode.LinearRgb:
                    return ColorModeOption.LinearRgb;
                default:
                    return ColorModeOption.RGB;
            }
        }

        #region イベントハンドラ
        //---------------------------------------------------------------------------
        // カラー表示方式
        private void Event_cmbColorMode_SelectedIndexChanged(object sender, EventArgs e)
        {
            ColorModeOption modeOption = (ColorModeOption)cmbColorMode.SelectedItemData;

            switch (modeOption)
            {
                case ColorModeOption.RGB:
                    ColorMode = HasAlpha(ColorMode) ? ConfigCommon.ColorMode.RGBA : ConfigCommon.ColorMode.RGB;
                    break;
                case ColorModeOption.LinearRgb:
                    ColorMode = HasAlpha(ColorMode) ? ConfigCommon.ColorMode.LinearRgba : ConfigCommon.ColorMode.LinearRgb;
                    break;
                case ColorModeOption.HSV:
                    ColorMode = HasAlpha(ColorMode) ? ConfigCommon.ColorMode.HSVA : ConfigCommon.ColorMode.HSV;
                    break;
            }
        }

        //---------------------------------------------------------------------------
        // 値表示方式
        private void Event_cmbValueMode_SelectedIndexChanged(object sender, EventArgs e)
        {
            base.ValueMode = (ConfigCommon.ValueMode)cmbValueMode.SelectedItemData;
        }
        #endregion

        //---------------------------------------------------------------------------
        /// <summary>
        /// カラーモード選択肢。
        /// </summary>
        private enum ColorModeOption
        {
            /// <summary>RGB方式。</summary>
            RGB,
            /// <summary>RGB方式。</summary>
            LinearRgb,
            /// <summary>HSV方式。</summary>
            HSV,
        }
    }
}
