﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using App.Data;
using App.Utility;

namespace App.Controls
{
    /// <summary>
    /// カラー編集パネルクラス。
    /// </summary>
    [ToolboxItem(true)]
    [DefaultEvent("SequentialValueChanged")]
    public sealed partial class ColorEditPanel : UIUserControl
    {
        // カラー
        private RgbaColor _color = new RgbaColor();//Color.White;

        // 編集モード
        private ColorEditMode _editMode = ColorEditMode.RGBA;
        // カラーピッカーアダプタ
        private readonly ColorPickerAdapter _colorPickerAdapter;
        private readonly ColorPickerAdapter _alphaPickerAdapter;

        private float dragBeginR;
        private float dragBeginG;
        private float dragBeginB;
        private Keys  dragModifierKeys;

        // コピペ用カラー保存値
        private static Nullable<RgbaColor> _copiedColor = null;
        public static Nullable<RgbaColor> CopiedColor
        {
            get
            {
                return _copiedColor;
            }
            set
            {
                if (_copiedColor != value)
                {
                    _copiedColor = value;
                    if (CopiedColorChanged != null)
                    {
                        CopiedColorChanged(null, EventArgs.Empty);
                    }
                }
            }
        }

        public static event EventHandler CopiedColorChanged;

        // コピペ用編集モード保存値
        private static ColorEditMode _copiedEditMode = ColorEditMode.RGBA;
        public static ColorEditMode CopiedEditMode
        {
            get
            {
                return _copiedEditMode;
            }
            set
            {
                if (_copiedEditMode != value)
                {
                    _copiedEditMode = value;
                    if (CopiedEditModeChanged != null)
                    {
                        CopiedEditModeChanged(null, EventArgs.Empty);
                    }
                }
            }
        }

        public static event EventHandler CopiedEditModeChanged;

        // 値変更モード
        public enum ValueChangedElementModeType
        {
            RGB_A,			// RGBとA
            R_G_B_A,		// R,G,B,A独立
        }

        private float[] rangeScales_ = new[] { 1.0f, 2.0f, 5.0f, 10.0f, 100.0f, 200.0f, 500.0f, 1000.0f, 2000.0f, 5000.0f, 10000.0f };

        public void SetDefaultMax(float value)
        {
            if (value > 0)
            {
                if (IsFreeRange)
                {
                    rangeScales_ = rangeScales_.Concat(Enumerable.Repeat(value, 1)).OrderBy(x => x).Distinct().ToArray();
                    RangeScaleIndex = Array.IndexOf(rangeScales_, value);
                }
                else
                {
                    float v = MaximumValue / value;
                    if (v >= 1)
                    {
                        rangeScales_ = rangeScales_.Concat(Enumerable.Repeat(v, 1)).OrderBy(x => x).Distinct().ToArray();
                        RangeScaleIndex = (rangeScales_.Length - 1) - Array.IndexOf(rangeScales_, v);
                    }
                }
            }
        }

        private int rangeScaleIndex_ = 0;
        private int RangeScaleIndex
        {
            get
            {
                return rangeScaleIndex_;
            }

            set
            {
                rangeScaleIndex_ = value;
                UpdateControlState();
            }
        }

        private bool _isFreeRange = true;
        public bool IsFreeRange
        {
            get { return _isFreeRange; }
            set
            {
                _isFreeRange = value;
                RangeScaleIndex = IsFreeRange ? 0 : (rangeScales_.Length - 1);

                UpdateControlState();
            }
        }

        public float MinimumValue { get { return 0.0f; } }
        public float ScaledMinimumValue { get { return 0.0f; } }

        private float _maximumValue = 1.0f;
        public float MaximumValue
        {
            get { return _maximumValue; }
            set
            {
                _maximumValue = value;

                if (_maximumValue < MinimumValue)
                {
                    _maximumValue = MinimumValue;
                }

                if (!IsFreeRange)
                {
                    updR.InputMax = _maximumValue;
                    updG.InputMax = _maximumValue;
                    updB.InputMax = _maximumValue;
                    updA.InputMax = _maximumValue;
                }

                UpdateControlState();
            }
        }

        private float ScaledMaximumValue
        {
            get
            {
                return IsFreeRange ?
                    MaximumValue * rangeScales_[RangeScaleIndex] :
                    MaximumValue / rangeScales_[(rangeScales_.Length - 1) - RangeScaleIndex];
            }
        }

        private static readonly Bitmap imgValueEditPanelRangeDown_ = App.Properties.Resources.Control_ValueEditPanelRangeDown;
        private static readonly Bitmap imgValueEditPanelRangeUp_   = App.Properties.Resources.Control_ValueEditPanelRangeUp;

        private static readonly Cursor cursorHand_ = System.Windows.Forms.Cursors.Hand;

        private UIContextMenuStrip cmsMenu = null;
        private UIToolStripMenuItem cmiCopy = null;
        private UIToolStripMenuItem cmiPaste = null;

        public bool IsClamp { get { return ConfigData.ApplicationConfig.DefaultValue.UIRange.Clamp; } }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ColorEditPanel()
        {
            Color = new RgbaColor();
            InitializeComponent();

            UpdateControlState();
            UpdateHdrLabel();

            btnRangeDown.Image = imgValueEditPanelRangeDown_;
            btnRangeUp.Image   = imgValueEditPanelRangeUp_;
            ctlAlphaSummary.Cursor = cursorHand_;
            ctlColorSummary.Cursor = cursorHand_;

            // アダプタ作成
            _colorPickerAdapter = new ColorPickerAdapter(ctlColorSummary);
            _colorPickerAdapter.ColorEdit += adapter_ColorEdit;
            _colorPickerAdapter.ConnectionStart += (s, e) => onConnectionStartColor_ = Color;

            _alphaPickerAdapter = new ColorPickerAdapter(ctlAlphaSummary);
            _alphaPickerAdapter.ColorEdit += adapter_ColorEdit;
            _alphaPickerAdapter.ConnectionStart += (s, e) => onConnectionStartColor_ = Color;

            // Ａ要素位置をタグとして保存
            lblA.Tag = lblA.Location;
            updA.Tag = updA.Location;
            ctlA.Tag = ctlA.Location;

            lblHdr.Tag			= lblHdr.Location;
            lblMinimum.Tag		= lblMinimum.Location;
            btnRangeDown.Tag	= btnRangeDown.Location;
            btnRangeUp.Tag		= btnRangeUp.Location;
            lblMaximum.Tag		= lblMaximum.Location;

            ctlColorSummary.ColorChange += (sender, args) => ctlAlphaSummary.Color = ctlColorSummary.Color;

            updR.InputMax = float.MaxValue;
            updG.InputMax = float.MaxValue;
            updB.InputMax = float.MaxValue;
            updA.InputMax = float.MaxValue;

            updR.EnableInputMinMax = true;
            updG.EnableInputMinMax = true;
            updB.EnableInputMinMax = true;
            updA.EnableInputMinMax = true;
        }

        private bool isFirstMenuVisible_ = true;

        private void BuildMenu()
        {
            if (cmsMenu == null)
            {
                var menu = new ColorEditPanelMenu();

                cmsMenu  = menu.cmsMenu;
                cmiCopy  = menu.cmiCopy;
                cmiPaste = menu.cmiPaste;

                cmsMenu.Opening += cmsMenu_Opening;
                cmiCopy.Click   += cmiCopy_Click;
                cmiPaste.Click  += cmiPaste_Click;

                cmsMenu.Closing += (s, e) =>
                {
                    // TODO:最初の一度目は勝手に閉じてしまうため。
                    if (isFirstMenuVisible_)
                    {
                        e.Cancel = true;
                        isFirstMenuVisible_ = false;
                    }
                };
            }
        }

        #region プロパティ
        public bool ReadOnly
        {
            get
            {
                return readOnly_;
            }

            set
            {
                readOnly_ = value;

                ctlColorSummary.ReadOnly = readOnly_;
                ctlAlphaSummary.ReadOnly = readOnly_;
                updR.ReadOnly = readOnly_;
                updG.ReadOnly = readOnly_;
                updB.ReadOnly = readOnly_;
                updA.ReadOnly = readOnly_;
                ctlR.ReadOnly = readOnly_;
                ctlG.ReadOnly = readOnly_;
                ctlB.ReadOnly = readOnly_;
                ctlA.ReadOnly = readOnly_;

                ctlColorSummary.Cursor = readOnly_ ? Cursors.Default : Cursors.Hand;
                ctlAlphaSummary.Cursor = readOnly_ ? Cursors.Default : Cursors.Hand;
            }
        }
        private bool readOnly_ = false;

        /// <summary>
        /// カラー。
        /// </summary>
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("カラー。")]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        [Browsable(false)]
        public RgbaColor Color
        {
            get { return _color; }
            set
            {
                if (_color != value)
                {
                    _color = value;

                    // フォーム更新
                    UpdateForm();

                    // 設定された値をアダプタに通知
                    _colorPickerAdapter.NotifyUpdate();
                    _alphaPickerAdapter.NotifyUpdate();
                }
            }
        }

        /// <summary>
        /// 編集モード。
        /// </summary>
        [DefaultValue(ColorEditMode.RGBA)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("編集モード。")]
        public ColorEditMode EditMode
        {
            get { return _editMode; }
            set
            {
                if (_editMode != value)
                {
                    _editMode = value;

                    switch (_editMode)
                    {
                        case ColorEditMode.RGBA:
                            lblR.Visible = true;
                            updR.Visible = true;
                            ctlR.Visible = true;
                            lblG.Visible = true;
                            updG.Visible = true;
                            ctlG.Visible = true;
                            lblB.Visible = true;
                            updB.Visible = true;
                            ctlB.Visible = true;
                            lblA.Visible = true;
                            updA.Visible = true;
                            ctlA.Visible = true;
                            lblA.Location = (Point)lblA.Tag;
                            updA.Location = (Point)updA.Tag;
                            ctlA.Location = (Point)ctlA.Tag;
                            ctlAlphaSummary.Visible = true;

                            lblHdr.Location			= (Point)lblHdr.Tag;
                            lblMinimum.Location		= (Point)lblMinimum.Tag;
                            btnRangeDown.Location	= (Point)btnRangeDown.Tag;
                            btnRangeUp.Location		= (Point)btnRangeUp.Tag;
                            lblMaximum.Location		= (Point)lblMaximum.Tag;

                            break;

                        case ColorEditMode.RGB:
                            lblR.Visible = true;
                            updR.Visible = true;
                            ctlR.Visible = true;
                            lblG.Visible = true;
                            updG.Visible = true;
                            ctlG.Visible = true;
                            lblB.Visible = true;
                            updB.Visible = true;
                            ctlB.Visible = true;
                            lblA.Visible = false;
                            updA.Visible = false;
                            ctlA.Visible = false;
                            lblA.Location = (Point)lblA.Tag;
                            updA.Location = (Point)updA.Tag;
                            ctlA.Location = (Point)ctlA.Tag;
                            ctlAlphaSummary.Visible = false;

                            lblHdr.Location			= new Point(lblHdr.Location.X,			((Point)lblHdr.Tag).Y       - (96 - 72));
                            lblMinimum.Location		= new Point(lblMinimum.Location.X,		((Point)lblMinimum.Tag).Y   - (96 - 72));
                            btnRangeDown.Location	= new Point(btnRangeDown.Location.X,	((Point)btnRangeDown.Tag).Y - (96 - 72));
                            btnRangeUp.Location		= new Point(btnRangeUp.Location.X,		((Point)btnRangeUp.Tag).Y   - (96 - 72));
                            lblMaximum.Location		= new Point(lblMaximum.Location.X,		((Point)lblMaximum.Tag).Y   - (96 - 72));

                            break;

                        case ColorEditMode.A:
                            lblR.Visible = false;
                            updR.Visible = false;
                            ctlR.Visible = false;
                            lblG.Visible = false;
                            updG.Visible = false;
                            ctlG.Visible = false;
                            lblB.Visible = false;
                            updB.Visible = false;
                            ctlB.Visible = false;
                            lblA.Visible = true;
                            updA.Visible = true;
                            ctlA.Visible = true;
                            lblA.Location = lblG.Location;
                            updA.Location = updG.Location;
                            ctlA.Location = ctlG.Location;
                            ctlAlphaSummary.Visible = false;

                            lblHdr.Location			= new Point(lblHdr.Location.X,			((Point)lblHdr.Tag).Y       - (96 - 72));
                            lblMinimum.Location		= new Point(lblMinimum.Location.X,		((Point)lblMinimum.Tag).Y   - (96 - 72));
                            btnRangeDown.Location	= new Point(btnRangeDown.Location.X,	((Point)btnRangeDown.Tag).Y - (96 - 72));
                            btnRangeUp.Location		= new Point(btnRangeUp.Location.X,		((Point)btnRangeUp.Tag).Y   - (96 - 72));
                            lblMaximum.Location		= new Point(lblMaximum.Location.X,		((Point)lblMaximum.Tag).Y   - (96 - 72));

                            break;
                    }

                    // サマリの設定を変更
                    ctlColorSummary.EditMode = _editMode;
                }
            }
        }

        /// <summary>
        /// カラーピッカーのタイトル
        /// ローカライズに対応していないので Browsable(false) にした
        /// </summary>
        [Browsable(false)]
        public string ColorPickerText
        {
            get
            {
                return _colorPickerAdapter.ColorPickerText;
            }
            set
            {
                _colorPickerAdapter.ColorPickerText = value;
            }
        }

        public ValueChangedElementModeType ValueChangedElementMode
        {
            get; set;
        }

        public bool IsDefaultLinear
        {
            set
            {
                ctlR.IsLinear = value;
                ctlG.IsLinear = value;
                ctlB.IsLinear = value;
                ctlA.IsLinear = value;
                ctlColorSummary.IsDefaultLinear = value;
                ctlAlphaSummary.IsDefaultLinear = value;
            }
        }

        #endregion

        #region イベント
        //---------------------------------------------------------------------
        private static readonly object EVENT_SEQUENTIALVALUECHANGED = new object();

        /// <summary>
        /// 値変更イベント。
        /// </summary>
        [Category(UIControlHelper.OriginalEventCategoryName)]
        [Description("値が変更された時に発生します。")]
        public event SequentialValueChangedEventHandler SequentialValueChanged
        {
            add { base.Events.AddHandler(EVENT_SEQUENTIALVALUECHANGED, value); }
            remove { base.Events.RemoveHandler(EVENT_SEQUENTIALVALUECHANGED, value); }
        }

        /// <summary>
        /// 値変更ハンドラ。
        /// </summary>
        private void OnSequentialValueChanged(SequentialValueChangedEventArgs e, uint elementBits)
        {
            SequentialValueChangedEventHandler handler = (SequentialValueChangedEventHandler)base.Events[EVENT_SEQUENTIALVALUECHANGED];

            UpdateHdrLabel();
//			FitRange();										!!!!!

            if (handler != null)
            {
                uint orgElementBits = e.ElementBits;
                e.ElementBits = elementBits;

                switch(ValueChangedElementMode)
                {
                    case ValueChangedElementModeType.RGB_A:
                    {
                        // R,G,Bのいずれかが立っていたらRGBとする
                        const uint rgbBit = (1<<0) | (1<<1) | (1<<2);
                        if ((e.ElementBits & rgbBit) != 0)
                        {
                            e.ElementBits |= rgbBit;
                        }

                        break;
                    }

                    case ValueChangedElementModeType.R_G_B_A:
                    {
                        // 何もしない
                        break;
                    }

                    default:
                    {
                        Debug.Assert(false);
                        break;
                    }
                }

                handler(this, e);

                e.ElementBits = orgElementBits;
            }
        }
        #endregion

        /// <summary>
        /// フォーム更新。
        /// </summary>
        private void UpdateForm()
        {
            float r = _color.R;
            float g = _color.G;
            float b = _color.B;
            float a = _color.A;

//			FitRange();							!!!!!!!!!!!!!!!!!!!!!!!!

            // コントロール同期
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                ctlColorSummary.Color = _color;

                updR.Value = r;
                updG.Value = g;
                updB.Value = b;
                updA.Value = a;

                ctlR.SuspendUpdateBitmap();
                ctlR.Value     = r;
                ctlR.RefValue0 = g;
                ctlR.RefValue1 = b;
                ctlR.ResumeUpdateBitmap();

                ctlG.SuspendUpdateBitmap();
                ctlG.Value     = g;
                ctlG.RefValue0 = r;
                ctlG.RefValue1 = b;
                ctlG.ResumeUpdateBitmap();

                ctlB.SuspendUpdateBitmap();
                ctlB.Value     = b;
                ctlB.RefValue0 = r;
                ctlB.RefValue1 = g;
                ctlB.ResumeUpdateBitmap();

                ctlA.Value = a;

                UpdateHdrLabel();
                FitRange();
            }
        }

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override Size DefaultSize
        {
            get { return new Size(248, 96); }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnLoad(EventArgs e)
        {
            // カラー初期値とコントロールの同期を取る
            if (!DesignMode)
            {
                UpdateForm();
            }
            base.OnLoad(e);
        }
        #endregion

        #region イベントハンドラ
        //---------------------------------------------------------------------
        // アップダウン
        private void updR_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            // 値更新
            float r = updR.Value;
            _color.R = r;

            // コントロール同期
            ctlR.Value     = r;
            ctlG.RefValue0 = r;
            ctlB.RefValue0 = r;
            ctlColorSummary.Color = _color;
            OnSequentialValueChanged(e, 1<<0);

            if (e.SourceEventArgs is UINumericUpDown2ValueChangeFromTextEventArgs)
            {
                FitRange();
            }
        }

        private void updG_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            // 値更新
            float g = updG.Value;
            _color.G = g;

            // コントロール同期
            ctlR.RefValue0 = g;
            ctlG.Value     = g;
            ctlB.RefValue1 = g;
            ctlColorSummary.Color = _color;
            OnSequentialValueChanged(e, 1<<1);

            if (e.SourceEventArgs is UINumericUpDown2ValueChangeFromTextEventArgs)
            {
                FitRange();
            }
        }

        private void updB_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            // 値更新
            float b = updB.Value;
            _color.B = b;

            // コントロール同期
            ctlR.RefValue1 = b;
            ctlG.RefValue1 = b;
            ctlB.Value     = b;
            ctlColorSummary.Color = _color;
            OnSequentialValueChanged(e, 1<<2);

            if (e.SourceEventArgs is UINumericUpDown2ValueChangeFromTextEventArgs)
            {
                FitRange();
            }
        }

        private void updA_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            // 値更新
            float a = updA.Value;
            _color.A = a;
            //_color = Color.FromArgb(a, _color.R, _color.G, _color.B);

            // コントロール同期
            ctlA.Value = a;
            ctlColorSummary.Color = _color;
            OnSequentialValueChanged(e, 1<<3);
        }

        //---------------------------------------------------------------------
        // スライダー
        private void ColorSliderValueChanged(SequentialValueChangedEventArgs e, uint elementsBits)
        {
            bool relativeRGB = (dragModifierKeys == Keys.Control);
            bool linkRGB = (relativeRGB == false)
                           && ((dragModifierKeys == Keys.Shift) || (e.MouseEventArgs.Button == MouseButtons.Middle));

            var isRChanged = false;
            var isGChanged = false;
            var isBChanged = false;

            if ((elementsBits & (1 << 0)) > 0)
            {
                isRChanged = true;
            }
            else if ((elementsBits & (1 << 1)) > 0)
            {
                isGChanged = true;
            }
            else if ((elementsBits & (1 << 2)) > 0)
            {
                isBChanged = true;
            }
            else
            {
                return;
            }

            // 値更新
            var r = isRChanged ? ctlR.Value : 0.0f;
            var g = isGChanged ? ctlG.Value : 0.0f;
            var b = isBChanged ? ctlB.Value : 0.0f;
            var val = isRChanged ? r : (isGChanged ? g : b);

            if (linkRGB)
            {
                _color.Set(val, val, val, _color.A);
            }
            else if (relativeRGB)
            {
                // なるべく比率を保ったまま連動して変更する
                // 変更カラーチャンネルがもともと0だった場合、他の連動カラーチャンネルは以下のようにする
                // 連動カラーがもともと0だった場合は、変更カラーと同じ値にする
                // 連動カラーがもともと0以外だった場合は、変更カラーの差分を追加する

                var diff = 0.0f;
                var ratio = 1.0;

                if (isRChanged)
                {
                    diff = r - dragBeginR;
                    ratio = IsNearZero(dragBeginR) ? double.NaN : r / dragBeginR;
                }
                else if (isGChanged)
                {
                    diff = g - dragBeginG;
                    ratio = IsNearZero(dragBeginG) ? double.NaN : g / dragBeginG;
                }
                else //if (isBChanged)
                {
                    diff = b - dragBeginB;
                    ratio = IsNearZero(dragBeginB) ? double.NaN : b / dragBeginB;
                }

                r = isRChanged ? r : (double.IsNaN(ratio) ? dragBeginR + diff : (float)(dragBeginR * ratio));
                g = isGChanged ? g : (double.IsNaN(ratio) ? dragBeginG + diff : (float)(dragBeginG * ratio));
                b = isBChanged ? b : (double.IsNaN(ratio) ? dragBeginB + diff : (float)(dragBeginB * ratio));
                r = isRChanged ? r : Math.Max(r, 0.0f);
                g = isGChanged ? g : Math.Max(g, 0.0f);
                b = isBChanged ? b : Math.Max(b, 0.0f);
                if (!IsFreeRange)
                {
                    r = isRChanged ? r : Math.Min(r, MaximumValue);
                    g = isGChanged ? g : Math.Min(g, MaximumValue);
                    b = isBChanged ? b : Math.Min(b, MaximumValue);
                }

                _color.Set(r, g, b, _color.A);
            }
            else
            {
                if (isRChanged)
                {
                    _color.R = r;
                }
                else if (isGChanged)
                {
                    _color.G = g;
                }
                else //if (isBChanged)
                {
                    _color.B = b;
                }
            }

            // コントロール同期
            if (e.Changing)
            {
                using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
                {
                    if (linkRGB)
                    {
                        updR.Value = val;
                        updG.Value = val;
                        updB.Value = val;

                        ctlR.SuspendUpdateBitmap();
                        if (!isRChanged)
                        {
                            ctlR.Value = val;
                        }
                        ctlR.RefValue0 = val;
                        ctlR.RefValue1 = val;
                        ctlR.ResumeUpdateBitmap();

                        ctlG.SuspendUpdateBitmap();
                        if (!isGChanged)
                        {
                            ctlG.Value = val;
                        }
                        ctlG.RefValue0 = val;
                        ctlG.RefValue1 = val;
                        ctlG.ResumeUpdateBitmap();

                        ctlB.SuspendUpdateBitmap();
                        if (!isBChanged)
                        {
                            ctlB.Value = val;
                        }
                        ctlB.RefValue0 = val;
                        ctlB.RefValue1 = val;
                        ctlB.ResumeUpdateBitmap();
                    }
                    else if (relativeRGB)
                    {
                        updR.Value = r;
                        updG.Value = g;
                        updB.Value = b;

                        ctlR.SuspendUpdateBitmap();
                        if (!isRChanged)
                        {
                            ctlR.Value = r;
                        }
                        ctlR.RefValue0 = g;
                        ctlR.RefValue1 = b;
                        ctlR.ResumeUpdateBitmap();

                        ctlG.SuspendUpdateBitmap();
                        if (!isGChanged)
                        {
                            ctlG.Value = g;
                        }
                        ctlG.RefValue0 = r;
                        ctlG.RefValue1 = b;
                        ctlG.ResumeUpdateBitmap();

                        ctlB.SuspendUpdateBitmap();
                        if (!isBChanged)
                        {
                            ctlB.Value = b;
                        }
                        ctlB.RefValue0 = r;
                        ctlB.RefValue1 = g;
                        ctlB.ResumeUpdateBitmap();
                    }
                    else
                    {
                        if (isRChanged)
                        {
                            updR.Value = r;
                            ctlG.RefValue0 = r;
                            ctlB.RefValue0 = r;
                        }
                        else if (isGChanged)
                        {
                            ctlR.RefValue0 = g;
                            updG.Value = g;
                            ctlB.RefValue1 = g;
                        }
                        else //if (isBChanged)
                        {
                            ctlR.RefValue1 = b;
                            ctlG.RefValue1 = b;
                            updB.Value = b;
                        }
                    }
                    ctlColorSummary.Color = _color;
                }
            }
            OnSequentialValueChanged(e, (linkRGB || relativeRGB) ? (uint)((1 << 2) | (1 << 1) | (1 << 0)) : elementsBits);
        }

        // カラースライダをRelative編集する場合の、変更チャンネル以外の値を計算する

        private static bool IsNearZero(float beginR)
        {
            return Math.Abs(beginR) < 0.000001;
        }

        private void ctlR_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            ColorSliderValueChanged(e, 1 << 0);
        }

        private void ctlG_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            ColorSliderValueChanged(e, 1 << 1);
        }

        private void ctlB_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            ColorSliderValueChanged(e, 1 << 2);
        }

        private void ctlA_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            // 値更新
            float a = ctlA.Value;
            _color.A = a;

            // コントロール同期
            if (e.Changing)
            {
                using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
                {
                    updA.Value = a;
                    ctlColorSummary.Color = _color;
                }
            }
            OnSequentialValueChanged(e, 1<<3);
        }

        //---------------------------------------------------------------------
        // コンテキストメニュー
        private void ctlSummary_MouseClick(object sender, MouseEventArgs e)
        {
            if (ReadOnly == false)
            {
                if (e.Button == MouseButtons.Right)
                {
                    BuildMenu();
                    cmsMenu.Show(Control.MousePosition);
                }
            }
        }

        public static bool CanPaste(ColorEditMode editMode)
        {
            // 重複する成分が一つもなければペースト不可
            return CopiedColor.HasValue && !((CopiedEditMode == ColorEditMode.RGB && editMode == ColorEditMode.A) ||
                    (CopiedEditMode == ColorEditMode.A && editMode == ColorEditMode.RGB));
        }

        private void cmsMenu_Opening(object sender, CancelEventArgs e)
        {
            cmiPaste.Enabled = CanPaste(_editMode);
        }

        private void cmiCopy_Click(object sender, EventArgs e)
        {
            // コピー用に保存
            CopiedColor = _color;
            CopiedEditMode = _editMode;

            isFirstMenuVisible_ = false;
            cmsMenu.Close();
        }

        private void cmiPaste_Click(object sender, EventArgs e)
        {
            bool pasteC = false;
            bool pasteA = false;

            switch (_editMode)
            {
                case ColorEditMode.RGBA:
                    switch (CopiedEditMode)
                    {
                        case ColorEditMode.RGBA:
                            pasteC = true;
                            pasteA = true;
                            break;

                        case ColorEditMode.RGB:
                            pasteC = true;
                            break;

                        case ColorEditMode.A:
                            pasteA = true;
                            break;
                    }
                    break;

                case ColorEditMode.RGB:
                    pasteC = true;
                    break;

                case ColorEditMode.A:
                    pasteA = true;
                    break;
            }

            // カラー更新
            RgbaColor color = _color;
            if (pasteC) { color.Set(CopiedColor.Value, color.A); }
            if (pasteA) { color.Set(color, CopiedColor.Value.A); }

            // 値が変更された場合のみ
            if (_color != color)
            {
                uint elementBits = 0;

                if (pasteC)
                {
                    elementBits |= (1<<2) | (1<<1) | (1<<0);
                }

                if (pasteA)
                {
                    elementBits |= (1<<3);
                }

                _color = color;

                UpdateForm();
                OnSequentialValueChanged(new SequentialValueChangedEventArgs(false), elementBits);
            }

            isFirstMenuVisible_ = false;
            cmsMenu.Close();
        }

        //---------------------------------------------------------------------
        // アダプタ
        private RgbaColor onConnectionStartColor_;

        private void adapter_ColorEdit(object sender, ColorEditEventArgs e)
        {
            // 内部値は常に更新
            _color.R = e.Color.R;
            _color.G = e.Color.G;
            _color.B = e.Color.B;
            _color.A = e.Color.A;

            // 変更後のみフォーム更新
            if (e.EditFixed)
            {
                UpdateForm();
            }

            uint elementBits = 0;

            if ((EditMode == ColorEditMode.RGBA) || (EditMode == ColorEditMode.RGB))
            {
                if (e.RedEdited){	elementBits |= (1<<0);	}
                if (e.GreenEdited){	elementBits |= (1<<1);	}
                if (e.BlueEdited){	elementBits |= (1<<2);	}
            }

            if ((EditMode == ColorEditMode.RGBA) || (EditMode == ColorEditMode.A))
            {
                if (e.AlphaEdited){	elementBits |= (1<<3);	}
            }

            OnSequentialValueChanged(new SequentialValueChangedEventArgs(!e.EditFixed), elementBits);
        }
        #endregion

        private void Bar_DragBegin(object sender, EventArgs e)
        {
            dragBeginR = ctlR.Value;
            dragBeginG = ctlG.Value;
            dragBeginB = ctlB.Value;

            dragModifierKeys = Control.ModifierKeys;
        }

        private void btnRange_Click(object sender, EventArgs e)
        {
            var dir = (sender == btnRangeUp) ? +1 : -1;

            RangeScaleIndex = Math.Min(rangeScales_.Length - 1, Math.Max(0, RangeScaleIndex + dir));
        }

        // UINumericUpDown.cs からコピペ
        private static readonly Regex onlyDigits = new Regex(@"^-?[0-9]+$");
        private static string Format(float value)
        {
            string text = value.ToString("R");

            // 整数の場合は小数表示に直す
            if (onlyDigits.IsMatch(text, 0))
            {
                text = value.ToString("F1");
            }
            if (text.Length > 10)
            {
                text = value.ToString("0.#####E+0");
            }
            return text;
        }

        private void UpdateControlState()
        {
#if false
            lblMinimum.Text = ScaledMinimumValue.ToString("#0.0########");
            lblMaximum.Text = ScaledMaximumValue.ToString("#0.0########");
#else
            lblMinimum.Text = Format(ScaledMinimumValue);
            lblMaximum.Text = Format(ScaledMaximumValue);
#endif

            var rangeScalerEnable = true;

            btnRangeDown.Enabled = rangeScalerEnable && (RangeScaleIndex != 0);
            btnRangeUp.Enabled   = rangeScalerEnable && (RangeScaleIndex != rangeScales_.Length - 1);

            using (var block = new UIControlEventSuppressBlock())
            {
                // edit
                updR.Minimum = ScaledMinimumValue;
                updG.Minimum = ScaledMinimumValue;
                updB.Minimum = ScaledMinimumValue;
                updA.Minimum = ScaledMinimumValue;
                updR.Maximum = ScaledMaximumValue;
                updG.Maximum = ScaledMaximumValue;
                updB.Maximum = ScaledMaximumValue;
                updA.Maximum = ScaledMaximumValue;

                // slider
                ctlR.Minimum = ScaledMinimumValue;
                ctlG.Minimum = ScaledMinimumValue;
                ctlB.Minimum = ScaledMinimumValue;
                ctlA.Minimum = ScaledMinimumValue;
                ctlR.Maximum = ScaledMaximumValue;
                ctlG.Maximum = ScaledMaximumValue;
                ctlB.Maximum = ScaledMaximumValue;
                ctlA.Maximum = ScaledMaximumValue;
            }

            ctlColorSummary.HDRUpperBound = IsFreeRange ? float.MaxValue : MaximumValue;
            ctlAlphaSummary.HDRUpperBound = IsFreeRange ? float.MaxValue : MaximumValue;
        }

        private void UpdateHdrLabel()
        {
            var isHdr = true;

            lblHdr.Enabled = isHdr;

            var hdr = Math.Max(Math.Max(Color.R, Color.G), Color.B);
            if (hdr > 1.0f)
            {
                lblHdr.Text = string.Format("HDR={0:0.##}", hdr);
                lblHdr.Visible = true;
            }
            else
            {
                lblHdr.Visible = false;
            }
        }

        // 現在のカラー値から適切な範囲に設定します。
        public void FitRange()
        {
            var max = Math.Max(Math.Max(Math.Max(Color.R, Color.G), Color.B), Color.A);

            // 自動で 1.0 より範囲を狭めてもうれしくないので 1.0 以上に設定する
            max = Math.Max(max, 1.0f);

            if (IsFreeRange)
            {
                if (max >= rangeScales_.Last() * MaximumValue)
                {
                    RangeScaleIndex = rangeScales_.Length - 1;
                }
                else
                {
                    for(int i = 0;i != rangeScales_.Length;++ i)
                    {
                        if (max <= rangeScales_[i] * MaximumValue)
                        {
                            RangeScaleIndex = i;
                            break;
                        }
                    }
                }
            }
            else
            {
                if (max >= MaximumValue / rangeScales_.First())
                {
                    RangeScaleIndex = rangeScales_.Length - 1;
                }
                else
                {
                    for(int i = 0;i != rangeScales_.Length;++ i)
                    {
                        if (max <= MaximumValue / rangeScales_[(rangeScales_.Length - 1) - i])
                        {
                            RangeScaleIndex = i;
                            break;
                        }
                    }
                }
            }
        }

        /// <summary>
        /// カラーピッカーとの接続を終了
        /// </summary>
        public void EndColorPickerConnection()
        {
            _colorPickerAdapter.EndConnection();
            _alphaPickerAdapter.EndConnection();
        }
    }

    /// <summary>
    /// カラー編集モード。
    /// </summary>
    public enum ColorEditMode
    {
        /// <summary>ＲＧＢＡ。</summary>
        RGBA,
        /// <summary>ＲＧＢ成分のみ。</summary>
        RGB,
        /// <summary>Ａ成分のみ。</summary>
        A,
    }

    #region ColorEditEvent
    /// <summary>
    /// カラー編集イベントハンドラデリゲート。
    /// </summary>
    public delegate void ColorEditEventHandler(object sender, ColorEditEventArgs e);

    /// <summary>
    /// カラー編集イベントデータクラス。
    /// </summary>
    public sealed class ColorEditEventArgs : EventArgs
    {
        // カラー
        private readonly RgbaColor _color;
        // 編集確定フラグ
        private readonly bool _editFixed;

        public bool RedEdited{   get; private set; }
        public bool GreenEdited{ get; private set; }
        public bool BlueEdited{  get; private set; }
        public bool AlphaEdited{ get; private set; }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ColorEditEventArgs(RgbaColor color, bool editFixed, bool redEdited, bool greenEdited, bool blueEdited, bool alphaEdited)
        {
            _color     = color;
            _editFixed = editFixed;

            RedEdited	= redEdited;
            GreenEdited	= greenEdited;
            BlueEdited	= blueEdited;
            AlphaEdited	= alphaEdited;
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ColorEditEventArgs(RgbaColor color, bool editFixed)
            :this(color, editFixed, true, true, true, true)
        {
        }

        /// <summary>
        /// カラー。
        /// </summary>
        public RgbaColor Color
        {
            get { return _color; }
        }

        /// <summary>
        /// 編集確定。
        /// </summary>
        public bool EditFixed
        {
            get { return _editFixed; }
        }
    }
    #endregion
}
