﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Windows.Forms;
using App.Command;
using App.ConfigData;
using App.Data;
using App.PropertyEdit;
using App.Utility;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace App.Controls
{
    partial class SkeletalAnimationSettingPopup : AnimationSettingPopupBase
    {
        private readonly Form				_parent;
        private readonly GuiObjectGroup _targetGroup;

        public override bool IsEditing()
        {
            return fepQuantizeScale.SliderShowing
                  || fepQuantizeRotate.SliderShowing
                  || fepQuantizeTranslate.SliderShowing;
        }


        public SkeletalAnimationSettingPopup(Form parent)
        {
            InitializeComponent();

            _parent = parent;
            _targetGroup = parent.Tag as GuiObjectGroup;
            Debug.Assert(_targetGroup != null);

            if (ApplicationConfig.DefaultValue.DisableAnimationQuantize)
            {
                gbxQuantize.Visible = false;
                gbxQuantizationReductionRate.Visible = false;
            }

            DocumentPropertyChangedEventHandler propertyChanged = (s, e) =>
            {
                // 表示中なら
                if (Visible)
                {
                    UpdateForm();
                }
            };

            App.AppContext.PropertyChanged += propertyChanged;
            Disposed += (s, e) =>
            {
                App.AppContext.PropertyChanged -= propertyChanged;
            };


            UpdateForm();
        }

        private void UpdateForm()
        {
            using (new UIControlEventSuppressBlock())
            {
                var target = _targetGroup.Active as SkeletalAnimation;
                if (target == null) return;

                cbxLoop.IsModified = target.IsValueChanged(x => x.loop);
                cbxLoop.Checked = target.Loop;

                lblQuantizeScale.IsModified = target.IsValueChanged(x => x.quantize_tolerance_scale);
                fepQuantizeScale.Value = target.Data.skeletal_anim_info.quantize_tolerance_scale;

                lblQuantizeRotate.IsModified = target.IsValueChanged(x => x.quantize_tolerance_rotate);
                fepQuantizeRotate.Value = target.Data.skeletal_anim_info.quantize_tolerance_rotate;

                lblQuantizeTranslate.IsModified = target.IsValueChanged(x => x.quantize_tolerance_translate);
                fepQuantizeTranslate.Value = target.Data.skeletal_anim_info.quantize_tolerance_translate;

                UpdateCompress(target);
            }
        }

        private void UpdateCompress(SkeletalAnimation target)
        {
            UpdateQuantizationResult(lstQuantizationResult,
                target.ScaleUncompressDataSize, target.ScaleCompressDataSize,
                target.RotateUncompressDataSize, target.RotateCompressDataSize,
                target.TranslateUncompressDataSize, target.TranslateCompressDataSize);
        }

        #region コマンド

        internal static GroupEditCommand CreateEditCommand_loop(GuiObjectGroup targets, bool value)
        {
            return
                new GeneralGroupValueEditCommand<bool>(
                    targets,
                    targets.Active.ObjectID,
                    value,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var anim = target as SkeletalAnimation;

                        Debug.Assert(data != null);

                        if (anim == null) return;

                        swap = anim.Data.skeletal_anim_info.loop;
                        anim.Data.skeletal_anim_info.loop = (bool)data;
                        Viewer.LoadOrReloadAnimation.Send(anim);
                    }
                );
        }

        internal static GroupEditCommand CreateEditCommand_quantize(
            GuiObjectGroup targets,
            //
            float scale,
            float rotate,
            float translate
        )
        {
            var quantizeData = new List<SkeletalAnimation.QuantizeToleranceData>();
            foreach (var targetObject in targets.GetObjects(GuiObjectID.SkeletalAnimation))
            {
                using (var currentTempFileName = TemporaryFileUtility.MakeDisposableFileName(".current" + G3dPath.SkeletalAnimBinaryExtension))
                {
                    (new DocumentSaver()).WriteDocument(targetObject as SkeletalAnimation, currentTempFileName.Path, false);

                    Document tempDoc = null;
                    {
                        bool isSuccess = DocumentManager.Read(new DocumentManager.PathWithName(currentTempFileName.Path, string.Empty), out tempDoc);
                        Debug.Assert(isSuccess && (tempDoc is SkeletalAnimation));
                    }

                    // テンポラリに出力したドキュメントの実体
                    var target = tempDoc as SkeletalAnimation;

                    // 最適化
                    target.Data.skeletal_anim_info.quantize_tolerance_scale = scale;
                    target.Data.skeletal_anim_info.quantize_tolerance_rotate = rotate;
                    target.Data.skeletal_anim_info.quantize_tolerance_translate = translate;
                    target.Optimize();

                    var quantizeToleranceData = new SkeletalAnimation.QuantizeToleranceData
                    {
                        BoneAnims = target.BoneAnims,
                        //
                        QuantizeToleranceScale = scale,
                        QuantizeToleranceRotate = rotate,
                        QuantizeToleranceTranslate = translate,
                    };
                    quantizeData.Add(quantizeToleranceData);
                }
            }

            return
                new GeneralGroupReferenceEditCommand<SkeletalAnimation.QuantizeToleranceData>(
                    targets,
                    GuiObjectID.SkeletalAnimation,
                    quantizeData,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        Debug.Assert(target is SkeletalAnimation);
                        var anim = target as SkeletalAnimation;

                        swap =
                            new SkeletalAnimation.QuantizeToleranceData
                            {
                                BoneAnims = ObjectUtility.Clone(anim.BoneAnims),
                                //
                                QuantizeToleranceScale = anim.Data.skeletal_anim_info.quantize_tolerance_scale,
                                QuantizeToleranceRotate = anim.Data.skeletal_anim_info.quantize_tolerance_rotate,
                                QuantizeToleranceTranslate = anim.Data.skeletal_anim_info.quantize_tolerance_translate
                            };

                        var qtd = data as SkeletalAnimation.QuantizeToleranceData;

                        Debug.Assert(qtd != null, "qtd != null");

                        anim.BoneAnims = ObjectUtility.Clone(qtd.BoneAnims);
                        //
                        anim.Data.skeletal_anim_info.quantize_tolerance_scale = qtd.QuantizeToleranceScale;
                        anim.Data.skeletal_anim_info.quantize_tolerance_rotate = qtd.QuantizeToleranceRotate;
                        anim.Data.skeletal_anim_info.quantize_tolerance_translate = qtd.QuantizeToleranceTranslate;

//                        anim.curveChanged = anim.IsCurveChanged();

                        AnimationCurveEditCommand.UpdateIsModifiedAnimTargetAll(target);

                        // Viewerへ再転送
                        Viewer.LoadOrReloadAnimation.Send(anim);
                    },
                    postEditDelegate: (editTargets, data) =>
                    {
                        ;	// リロードはまとめて送れないのでEditDelegateで送る
                    }
                );
        }

        #endregion



        #region イベント

        private void cbxLoop_CheckedChanged(object sender, EventArgs e)
        {
            TheApp.CommandManager.Execute(CreateEditCommand_loop(_targetGroup, cbxLoop.Checked));
        }

        private void fepQuantizeFrame_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            if(e.Changing)
            {
                using (var currentTempFileName = TemporaryFileUtility.MakeDisposableFileName(".current" + G3dPath.SkeletalAnimBinaryExtension))
                {
                    (new DocumentSaver()).WriteDocument(_targetGroup.Active as SkeletalAnimation, currentTempFileName.Path, false);

                    Document tempDoc = null;
                    {
                        bool isSuccess = DocumentManager.Read(new DocumentManager.PathWithName(currentTempFileName.Path, string.Empty), out tempDoc);
                        Debug.Assert(isSuccess && (tempDoc is SkeletalAnimation));
                    }

                    // テンポラリに出力したドキュメントの実体
                    var target = tempDoc as SkeletalAnimation;

                    // 最適化
                    target.Data.skeletal_anim_info.quantize_tolerance_scale = fepQuantizeScale.Value;
                    target.Data.skeletal_anim_info.quantize_tolerance_rotate = fepQuantizeRotate.Value;
                    target.Data.skeletal_anim_info.quantize_tolerance_translate = fepQuantizeTranslate.Value;
                    target.Optimize();

                    UpdateCompress(target);
                }
            }
            else
            {
                TheApp.CommandManager.Execute(
                    CreateEditCommand_quantize(
                        _targetGroup,
                        //
                        fepQuantizeScale.Value,
                        fepQuantizeRotate.Value,
                        fepQuantizeTranslate.Value
                    )
                );
            }

        }

        private void fepQuantizeScale_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            fepQuantizeFrame_SequentialValueChanged(sender, e);
        }

        private void fepQuantizeRotate_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            fepQuantizeFrame_SequentialValueChanged(sender, e);
        }

        private void fepQuantizeTranslate_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            fepQuantizeFrame_SequentialValueChanged(sender, e);
        }

        #endregion

    }
}
