﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using App.Command;
using App.ConfigData;
using App.Controls;
using App.Data;
using App.PropertyEdit;
using App.Utility;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace App.Controls
{
    partial class ShaderAnimationSettingPopup : AnimationSettingPopupBase
    {
        private readonly Form				_parent;
        private readonly GuiObjectGroup _targetGroup;
        private int _prevFrameCount = 0;

        public override bool IsEditing()
        {
            return iepFrameSize.SliderShowing || fepQuantizeTexScale.SliderShowing
                  || fepQuantizeTexRotate.SliderShowing || fepQuantizeTexTranslate.SliderShowing;
        }


        public ShaderAnimationSettingPopup(Form parent)
        {
            InitializeComponent();

            _parent = parent;
            _targetGroup = parent.Tag as GuiObjectGroup;
            Debug.Assert(_targetGroup != null);

            iepFrameSize.SetContextMenuStrip(uiContextMenuTotalFrames);
            iepFrameSize.SliderCloseHandler = (sender, ev) =>
            {
                // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1810
                // スライダが閉じられた時に、ポップアップがアクティブでない場合は
                // 閉じる処理を入れる
                var flag = _parent == Form.ActiveForm;
                if (flag) return;
                if (TheApp.MainFrame.CanFocus) return;

                EventHandler waitCanFocus = null;
                waitCanFocus = (s, e) =>
                {
                    if (TheApp.MainFrame.CanFocus)
                    {
                        Application.Idle -= waitCanFocus;
                        _parent.Close();
                    }
                };
                Application.Idle += waitCanFocus;
            };

            if (ApplicationConfig.DefaultValue.DisableAnimationQuantize)
            {
                gbxQuantize.Visible = false;
                gbxQuantizationReductionRate.Visible = false;
            }


            DocumentPropertyChangedEventHandler propertyChanged = (s, e) =>
            {
                // 表示中なら
                if (Visible)
                {
                    UpdateForm();
                }
            };

            App.AppContext.PropertyChanged += propertyChanged;
            Disposed += (s, e) =>
            {
                App.AppContext.PropertyChanged -= propertyChanged;
            };


            UpdateForm();
        }

        private void UpdateForm()
        {
            using (var block = new UIControlEventSuppressBlock())
            {
                cbxAutoScale.Checked = AnimationSettingPopupBase.AutoScale;

                var target = _targetGroup.Active as ShaderParameterAnimation;
                if (target == null) return;

                lblFrameCount.IsModified = target.IsValueChanged(x => x.frame_count);
                iepFrameSize.Value = target.Data.shader_param_anim_info.frame_count;
                _prevFrameCount = target.Data.shader_param_anim_info.frame_count;

                cbxLoop.IsModified = target.IsValueChanged(x => x.loop);
                cbxLoop.Checked = target.Data.shader_param_anim_info.loop;

                lblQuantizeTexScale.IsModified = target.IsValueChanged(x => x.quantize_tolerance_tex_scale);
                fepQuantizeTexScale.Value = target.Data.shader_param_anim_info.quantize_tolerance_tex_scale;

                lblQuantizeTexRotate.IsModified = target.IsValueChanged(x => x.quantize_tolerance_tex_rotate);
                fepQuantizeTexRotate.Value = target.Data.shader_param_anim_info.quantize_tolerance_tex_rotate;

                lblQuantizeTexTranslate.IsModified = target.IsValueChanged(x => x.quantize_tolerance_tex_translate);
                fepQuantizeTexTranslate.Value = target.Data.shader_param_anim_info.quantize_tolerance_tex_translate;

                UpdateControlEnabled();
                UpdateCompress(target);
            }
        }

        private void UpdateCompress(ShaderParameterAnimation target)
        {
            UpdateQuantizationResult(lstQuantizationResult,
                target.TextureScaleUncompressDataSize, target.TextureScaleCompressDataSize,
                target.TextureRotateUncompressDataSize, target.TextureRotateCompressDataSize,
                target.TextureTranslateUncompressDataSize, target.TextureTranslateCompressDataSize);
        }

        private void UpdateControlEnabled()
        {
            var id = _targetGroup.Active.ObjectID;
            bool isEnabledTexSrt = (id == GuiObjectID.ShaderParameterAnimation) || (id == GuiObjectID.TextureSrtAnimation);


            fepQuantizeTexScale.Enabled = isEnabledTexSrt;
            fepQuantizeTexRotate.Enabled = isEnabledTexSrt;
            fepQuantizeTexTranslate.Enabled = isEnabledTexSrt;


            lblQuantizeTex.Enabled = isEnabledTexSrt;
            lblQuantizeTexScale.Enabled = isEnabledTexSrt;
            lblQuantizeTexRotate.Enabled = isEnabledTexSrt;
            lblQuantizeTexTranslate.Enabled = isEnabledTexSrt;
        }


        #region コマンド
        private static GroupEditCommand CreateEditCommand_frame_count(GuiObjectGroup targets, int frameCount)
        {
            return
                new GeneralGroupValueEditCommand<int>(
                    targets,
                    targets.Active.ObjectID,
                    frameCount,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var anim = target as ShaderParameterAnimation;

                        Debug.Assert(data != null);

                        if (anim == null) return;

                        swap = anim.Data.shader_param_anim_info.frame_count;
                        anim.Data.shader_param_anim_info.frame_count = (int)data;

                        AnimationDocument.NotifyFrameCountChanged(anim, EventArgs.Empty);

                        Viewer.LoadOrReloadAnimation.Send(anim);
                    }
                );
        }

        private static GroupEditCommand CreateEditCommand_loop(GuiObjectGroup targets, bool value)
        {
            return
                new GeneralGroupValueEditCommand<bool>(
                    targets,
                    targets.Active.ObjectID,
                    value,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var anim = target as ShaderParameterAnimation;

                        Debug.Assert(data != null);

                        if (anim == null) return;

                        swap = anim.Data.shader_param_anim_info.loop;
                        anim.Data.shader_param_anim_info.loop = (bool)data;
                        Viewer.LoadOrReloadAnimation.Send(anim);
                    }
                );
        }

        public static EditCommand CreateEditCommand_quantize(
            GuiObjectGroup targets,
            GuiObjectID id,
            float texScale,
            float texRotate,
            float texTranslate,
            bool isLoadAnimation
        )
        {

            var quantizeData = (targets.GetObjects(id)
                .Cast<ShaderParameterAnimation>()
                .Select(targetObject => new ShaderParameterAnimation.QuantizeToleranceData()
                {
                    QuantizeToleranceTextureScale = texScale,
                    QuantizeToleranceTextureRotate = texRotate,
                    QuantizeToleranceTextureTranslate = texTranslate
                })).ToList();

            //return
            var commandSet = new EditCommandSet();
            commandSet.Add(new GeneralGroupReferenceEditCommand<ShaderParameterAnimation.QuantizeToleranceData>(
                    targets,
                    targets.Active.ObjectID,
                    quantizeData,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        Debug.Assert(target is ShaderParameterAnimation);
                        var anim = target as ShaderParameterAnimation;

                        Debug.Assert(data != null);

                        if (anim == null) return;

                        swap =
                            new ShaderParameterAnimation.QuantizeToleranceData
                            {
                                QuantizeToleranceTextureScale = anim.Data.shader_param_anim_info.quantize_tolerance_tex_scale,
                                QuantizeToleranceTextureRotate = anim.Data.shader_param_anim_info.quantize_tolerance_tex_rotate,
                                QuantizeToleranceTextureTranslate = anim.Data.shader_param_anim_info.quantize_tolerance_tex_translate
                            };

                        var qtd = data as ShaderParameterAnimation.QuantizeToleranceData;

                        Debug.Assert(qtd != null, "qtd != null");
                        anim.Data.shader_param_anim_info.quantize_tolerance_tex_scale = qtd.QuantizeToleranceTextureScale;
                        anim.Data.shader_param_anim_info.quantize_tolerance_tex_rotate = qtd.QuantizeToleranceTextureRotate;
                        anim.Data.shader_param_anim_info.quantize_tolerance_tex_translate = qtd.QuantizeToleranceTextureTranslate;
                    },
                    postEditDelegate: (editTargets, data) =>
                    {
                        // リロードはまとめて送れないのでEditDelegateで送る
                    }
                ));
            foreach (ShaderParameterAnimation tmp in targets.GetObjects(id))
            {
                // スコープを制限
                var targetObject = tmp;
                commandSet.Add(new LazyCommand(() => AnimationCurveEditCommand.CreateQuantizeAnalyseAllCommand(targetObject, false)));
            }

            if (isLoadAnimation)
            {
                var anims = targets.GetObjects(id).OfType<ShaderParameterAnimation>().ToArray();
                commandSet.OnPostEdit += (s, args)
                    =>
                    {
                        foreach (var anim in anims)
                        {
                            Viewer.LoadOrReloadAnimation.Send(anim);
                        }
                    };
            }
            return commandSet;
        }

        #endregion



        #region イベント
        private void iepFrameSize_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            if (e.Changing)
            {

            }
            else
            {
                //TheApp.CommandManager.Execute(CreateEditCommand_frame_count(_targetGroup, iepFrameSize.Value));

                var commandSet = new EditCommandSet();
                commandSet.Add(CreateEditCommand_frame_count(_targetGroup, iepFrameSize.Value));
                if (cbxAutoScale.Checked && _prevFrameCount > 0)
                {
                    var scale = (float)iepFrameSize.Value / (float)_prevFrameCount;
                    commandSet.Add(GeneralAnimationSettingPopup.CreateScaleAnimationEditCommand(_targetGroup, scale));

                    var target = _targetGroup.Active;
                    var targetId = target.ObjectID;
                    AnimationDocument animDoc = null;

                    if (targetId != GuiObjectID.SceneAnimation
                        && targetId != GuiObjectID.SkeletalAnimation
                        && target != null
                        )
                    {
                        animDoc = target as AnimationDocument;
                    }
                    else if (targetId == GuiObjectID.CameraAnimation)
                    {
                        var anim = target as CameraAnimation;
                        if (anim != null) animDoc = anim.Owner as AnimationDocument;
                    }
                    else if (targetId == GuiObjectID.FogAnimation)
                    {
                        var anim = target as FogAnimation;
                        if (anim != null) animDoc = anim.Owner as AnimationDocument;
                    }
                    else if (targetId == GuiObjectID.LightAnimation)
                    {
                        var anim = target as LightAnimation;
                        if (anim != null) animDoc = anim.Owner as AnimationDocument;
                    }
                    Debug.Assert(animDoc != null, "animDoc != null");
                    //commandSet.Add(new LazyCommand(() => animDoc.CreateUpdateBindCommand()));
                }

                TheApp.CommandManager.Add(commandSet.Execute());
            }

        }

        private void cbxLoop_CheckedChanged(object sender, EventArgs e)
        {
            TheApp.CommandManager.Execute(CreateEditCommand_loop(_targetGroup, cbxLoop.Checked));
        }

        private void fepQuantizeFrame_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            if (e.Changing)
            {
                try
                {
                    Document tempDoc;
                    using (var currentTempFileName = TemporaryFileUtility.MakeDisposableFileName(".current" + G3dPath.ShaderParamAnimBinaryExtension))
                    {
                        // 実質一つだけのとき
                        (new DocumentSaver()).WriteDocument(_targetGroup.Active.OwnerDocument, currentTempFileName.Path, false);
                        {
                            bool isSuccess = DocumentManager.Read(new DocumentManager.PathWithName(currentTempFileName.Path, string.Empty), out tempDoc);
                            Debug.Assert(isSuccess && (tempDoc is ShaderParameterAnimation));
                        }
                    }

                    // テンポラリに出力したドキュメントの実体
                    var target = tempDoc as ShaderParameterAnimation;

                    // 最適化
                    if (ApplicationConfig.DefaultValue.DisableAnimationQuantize)
                    {
                        target.Data.shader_param_anim_info.quantize_tolerance_tex_scale = 0.0f;
                        target.Data.shader_param_anim_info.quantize_tolerance_tex_rotate = 0.0f;
                        target.Data.shader_param_anim_info.quantize_tolerance_tex_translate = 0.0f;
                        target.Optimize();
                        UpdateCompress(target);
                        target.Data.shader_param_anim_info.quantize_tolerance_tex_scale = fepQuantizeTexScale.Value;
                        target.Data.shader_param_anim_info.quantize_tolerance_tex_rotate = fepQuantizeTexRotate.Value;
                        target.Data.shader_param_anim_info.quantize_tolerance_tex_translate = fepQuantizeTexTranslate.Value;
                    }
                    else
                    {
                        target.Data.shader_param_anim_info.quantize_tolerance_tex_scale = fepQuantizeTexScale.Value;
                        target.Data.shader_param_anim_info.quantize_tolerance_tex_rotate = fepQuantizeTexRotate.Value;
                        target.Data.shader_param_anim_info.quantize_tolerance_tex_translate = fepQuantizeTexTranslate.Value;
                        target.Optimize();
                        UpdateCompress(target);
                    }
                }
                catch
                {
                }
            }
            else
            {
                TheApp.CommandManager.Execute(
                        CreateEditCommand_quantize(
                            _targetGroup,
                            _targetGroup.Active.ObjectID,
                            fepQuantizeTexScale.Value,
                            fepQuantizeTexRotate.Value,
                            fepQuantizeTexTranslate.Value,
                            true
                        )
                    );
            }
        }

        private void cbxAutoScale_CheckedChanged(object sender, EventArgs e)
        {
            AnimationSettingPopupBase.AutoScale = cbxAutoScale.Checked;
        }

        private void fepQuantizeTexScale_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            fepQuantizeFrame_SequentialValueChanged(sender, e);
        }

        private void fepQuantizeTexRotate_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            fepQuantizeFrame_SequentialValueChanged(sender, e);
        }

        private void fepQuantizeTexTranslate_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            fepQuantizeFrame_SequentialValueChanged(sender, e);
        }

        private void cmiCopy_Click(object sender, EventArgs e)
        {
            App.Utility.ClipboardUtility.SetDataObject(iepFrameSize.Value.ToString());
        }

        private void cmiPaste_Click(object sender, EventArgs e)
        {
            var iData = App.Utility.ClipboardUtility.GetDataObject();
            if (iData == null || !iData.GetDataPresent(DataFormats.Text)) return;
            int val;
            if (StringUtility.TrySafeIntParse((String)iData.GetData(DataFormats.Text), out val))
            {
                iepFrameSize.Value = val;
            }
        }

        private void cmiMaxValue_Click(object sender, EventArgs e)
        {
            iepFrameSize.Value = iepFrameSize.MaxBound;
        }

        #endregion

    }
}
