﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App.Data;
using Viewer;

namespace App.Command
{
    /// <summary>
    /// 編集コマンドセット
    /// </summary>
    public sealed class EditCommandSet : EditCommand , ICommandSet
    {
        public int CommandCount{ get{ return _commands.Count; } }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public EditCommandSet() : this(true) {}

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public EditCommandSet(bool reversal)
        {
            _commands = new List<ICommand>();
            _reversal = reversal;
        }

        public override bool IsValid()
        {
            return _commands.Any(x => x.IsValid());
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// 追加
        /// </summary>
        public void Add(ICommand command)
        {
            Debug.Assert(command != null);
            _commands.Add(command);
        }

        /// <summary>
        /// 編集
        /// </summary>
        public override ICommand Edit(DocumentPropertyChangedEventArgs eventArgs)
        {
            using (var vdsb = new Viewer.ViewerDrawSuppressBlock(_viewerSuppresBlockDelegate))
            {
                if (OnPreEdit != null)
                {
                    OnPreEdit(this, EventArgs.Empty);
                }

                {
                    DocumentComponentChangedEventArgs changed = new DocumentComponentChangedEventArgs();
                    {
                        foreach (ICommand command in _commands)
                        {
                            EditCommand editCommand = command as EditCommand;
                            if (editCommand != null)
                            {
                                editCommand.Edit(eventArgs);
                                //results.Add(editCommand.Edit(eventArgs));

                                if (editCommand is IHasDocumentComponentChangedInfo)
                                {
                                    IHasDocumentComponentChangedInfo info = command as IHasDocumentComponentChangedInfo;
                                    foreach(Document doc in info.ChangeSource.Objects)
                                    {
                                        changed.AddClosedDocument(doc);
                                    }
                                }
                            }
                            else
                            {
                                // 古い
                                Debug.Assert(true);
                                command.Execute();
                            }
                        }
                    }
                }

                // Viewer用のdelegateを設定
                //results.SetViewerDrawSuppressBlockDelegate(_viewerSuppresBlockDelegate);

                // 編集前後イベントをコピー
                //results.OnPreEdit  = OnPreEdit;
                //results.OnPostEdit = OnPostEdit;
                //results.canRedo += CanRedo;
                //results.canUndo += CanUndo;

                //if (_reversal) { results.Reverse(); }
                if (_reversal) { this.Reverse(); }

                if (OnPostEdit != null)
                {
                    OnPostEdit(this, EventArgs.Empty);
                }

                return this;
            }
        }

        /// <summary>
        /// 反転
        /// </summary>
        public void Reverse() { _commands.Reverse(); }

        /// <summary>
        /// ビューアの描画抑制ブロック(ViewerDrawSuppressBlock)用のdelegate設定関数
        /// </summary>
        public void SetViewerDrawSuppressBlockDelegate(ViewerDrawSuppressBlock.DisposedDelegate delegateEvent) { _viewerSuppresBlockDelegate = delegateEvent; }
        ViewerDrawSuppressBlock.DisposedDelegate _viewerSuppresBlockDelegate = null;

        // 編集前・後イベント
        //	注意：アンドゥ・リドゥでスワップしません。
        public event EventHandler	OnPreEdit	= null;
        public event EventHandler	OnPostEdit	= null;

        //---------------------------------------------------------------------
        // コマンドリスト
        private readonly List<ICommand> _commands;
        // 反転
        private readonly bool _reversal;

        /// <summary>
        /// コマンドが実行されたか
        /// </summary>
        public override bool IsExecuted
        {
            get
            {
                if (_commands.Count == 0)
                {
                    // コマンドが未登録なら常にtrue
                    return true;
                }
                else
                {
                    // 全てのコマンドがExecuteされたか、コマンドセットがExecuteされたらtrue
                    // 二重にExecuteされたらfalse
                    return _commands.All(c => c.IsExecuted) ^ base.IsExecuted;
                }
            }
        }

        public override bool CanUndo()
        {
            foreach (var command in _commands)
            {
                if (!command.CanUndo())
                {
                    return false;
                }
            }

            return base.CanUndo();
        }

        public override bool CanRedo()
        {
            foreach (var command in _commands)
            {
                if (!command.CanRedo())
                {
                    return false;
                }
            }

            return base.CanRedo();
        }
    }
}
