﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "stdafx.h"

#include "msclr/marshal.h"
#include "msclr/marshal_windows.h"
#include "msclr/marshal_cppstd.h"
#include "msclr/marshal_atl.h"

#include "Utility.h"
#include "HIO.h"

#include "CommandUtility.h"

#include <nn/g3d/g3d_ResAnimCurve.h>
#include <msclr/marshal_cppstd.h>

using namespace msclr::interop;
using namespace nw::g3d::edit::detail;
using namespace NintendoWare::G3d::Edit;
using namespace System;

#include <nn/g3d/g3d_ResShader.h>

namespace {

class ScopedAllocator
{
public:
    ScopedAllocator(size_t byteSize)
        : m_pBuffer(NULL)
    {
        m_pBuffer = malloc(byteSize);
    }

    ~ScopedAllocator()
    {
        free(m_pBuffer);
    }

    void* GetAllocatedBuffer()
    {
        return m_pBuffer;
    }

private:
    void* m_pBuffer;
};

void GetShaderFileAlignment(size_t* pOutAlignment, const void* pShaderFileBuffer)
{
    *pOutAlignment = nn::g3d::detail::Alignment_Default;

    const nn::g3d::ResShaderFile* pResShaderFile = static_cast<const nn::g3d::ResShaderFile*>(pShaderFileBuffer);
    *pOutAlignment = pResShaderFile->GetFileHeader()->GetAlignment();
    if (!pResShaderFile->GetFileHeader()->IsEndianValid())
    {
        nn::util::SwapEndian(pOutAlignment);
    }
}

size_t ExamineFileSize(String^ fileName)
{
    std::string path = marshal_as<std::string>(fileName);
    fpos_t fileSize = 0;
    FILE *fp;
    errno_t error = fopen_s(&fp, path.c_str(), "rb");
    G3D_HTC_THROW_EXCEPTION_UNLESS(error == 0, System::String::Format("Could not open {0}", fileName));

    fseek(fp, 0, SEEK_END);
    fgetpos(fp, &fileSize);
    fclose(fp);

    return static_cast<size_t>(fileSize);
}

}

namespace nw { namespace g3d { namespace tool {

void StringUtility::CopyString(s8* pOut, String^ pSource, size_t destSizeMax)
{
    memset(pOut, 0, destSizeMax);
    std::string scriptTextAsStdString = marshal_as<std::string>(pSource);
    size_t copySize = scriptTextAsStdString.length();
    if (scriptTextAsStdString.length() >= destSizeMax)
    {
        // 格納先のサイズが足りないので入るところまで入れる
        copySize = destSizeMax - 1;
    }

    std::memcpy(pOut, scriptTextAsStdString.c_str(), copySize);
}

void Endian::Swap(nw::g3d::edit::detail::PacketHeader* data)
{
    Swap(&data->dataSize);
    Swap(&data->command);
}

void Endian::Swap(nn::g3d::viewer::detail::PacketHeader* data)
{
    Swap(&data->dataSize);
    Swap(&data->command);
}

void Endian::Swap(nw::g3d::edit::detail::FileDataBlock* data)
{
    Swap(&data->key);
    Swap(&data->resFileKey);
    Swap(&data->kind);
    Swap(&data->fileSize);
    Swap(&data->fileAlignment);
    Swap(&data->pathSize);
}

void Endian::Swap(nw::g3d::edit::detail::FileDataPacket* data)
{
    Swap(&data->header);
    Swap(&data->block);
}

void Endian::Swap(nw::g3d::edit::detail::RenderInfoEditPacket* data)
{
    Swap(&data->header);
    Swap(&data->info.type);
    Swap(&data->info.modelKey);
    Swap(&data->info.materialIndex);
    Swap(&data->info.labelOffset);
    Swap(&data->info.slotIndex);
    Swap(&data->info.itemIndex);
}

void Endian::Swap(nw::g3d::edit::detail::ModelLayoutRecvPacket* data)
{
    Swap(&data->header);
    Swap(&data->block.modelKey);
}

void Endian::Swap(nw::g3d::edit::detail::ModelLayoutEditPacket* data)
{
    Swap(&data->header);
    Swap(&data->block.modelKey);
    Swap(&data->block.isBind);

    for (int i = 0; i < 3; ++i)
    {
        Swap(&data->block.scale.a[i]);
        Swap(&data->block.rotate.a[i]);
        Swap(&data->block.translate.a[i]);
    }
}

void Endian::Swap(nw::g3d::edit::detail::BondBindEditPacket* data)
{
    Swap(&data->header);
    Swap(&data->block.isBind);
    Swap(&data->block.childModelKey);
    Swap(&data->block.parentModelKey);
    Swap(&data->block.parentBoneIndex);
}

void Endian::Swap(nw::g3d::edit::detail::ShapeLodLevelEditPacket* data)
{
    Swap(&data->header);
    Swap(&data->block.modelKey);
    Swap(&data->block.lodLevel);
}

void Endian::Swap(nw::g3d::edit::detail::EditValueInfoBlock* data)
{
    Swap(&data->key);

    Swap(&data->valueKind);
    Swap(&data->editTargetKind);

    Swap(&data->length);
    Swap(&data->indexSize);
}

void Endian::Swap(nw::g3d::edit::detail::EditValuePacket* data)
{
    Swap(&data->header);
    Swap(&data->block);
}

void Endian::Swap(nw::g3d::edit::detail::BindAnimInfoBlock* data)
{
    Swap(&data->info.modelKey);
    Swap(&data->info.attachModelFlag);
    Swap(&data->info.animationKeySize);
}

void Endian::Swap(nw::g3d::edit::detail::BindAnimPacket* data)
{
    Swap(&data->header);
    Swap(&data->block);
}

void Endian::Swap(nw::g3d::edit::detail::RenderInfoUpdatePacket* data)
{
    Swap(&data->header);
    Swap(&data->block);
}

void Endian::Swap(nw::g3d::edit::detail::RenderInfoRecvPacket* data)
{
    Swap(&data->header);
    Swap(&data->block);
}

void Endian::SwapBindAnimationKeys(nw::g3d::edit::detail::BindAnimInfoBlock* data, u32 keySize)
{
    for (u32 i = 0; i < keySize; ++i)
    {
        Swap(&data->animationKeys[i]);
    }
}

void Endian::Swap(nw::g3d::edit::detail::AnimCurveBlock* data)
{
    Swap(&data->info.animationKey);
    Swap(&data->info.animationKind);

    Swap(&data->value.materialIndex);
    Swap(&data->value.curveIndex);
    Swap(&data->value.curveSize);
}

void Endian::Swap(nw::g3d::edit::detail::AnimCurvePacket* data)
{
    Swap(&data->header);
    Swap(&data->block);
}

void Endian::SwapValue(nw::g3d::edit::detail::EditValueBlock* data, bool isBoolType)
{
    if (isBoolType)
    {
        return;
    }

    Swap(&data->value.uValue);
}

void Endian::SwapIndices(nw::g3d::edit::detail::EditValueBlock* data, u32 indexSize)
{
    for (u32 i = 0; i < indexSize; ++i)
    {
        Swap(&data->index[i]);
    }
}

void Endian::SwapValue(nw::g3d::edit::detail::EditVector4ValueBlock* data)
{
    for (int i = 0; i < 4; ++i)
    {
        Swap(&data->value.uValue[i]);
    }
}

void Endian::SwapIndices(nw::g3d::edit::detail::EditVector4ValueBlock* data, u32 indexSize)
{
    for (u32 i = 0; i < indexSize; ++i)
    {
        Swap(&data->index[i]);
    }
}

void Endian::SwapValue(nw::g3d::edit::detail::EditSamplerValueBlock* data)
{
    Swap(&data->value.samplerIndex);
    Swap(&data->value.uValue);
}

void Endian::SwapIndices(nw::g3d::edit::detail::EditSamplerValueBlock* data, u32 indexSize)
{
    for (u32 i = 0; i < indexSize; ++i)
    {
        Swap(&data->index[i]);
    }
}

void Endian::SwapValue(nw::g3d::edit::detail::EditShaderParamVectorValueBlock* data, bool isBoolType)
{
    if (isBoolType)
    {
        return;
    }

    for (int i = 0; i < 4; ++i)
    {
        Swap(&data->value.value.uValue[i]);
    }
}
void Endian::SwapIndices(nw::g3d::edit::detail::EditShaderParamVectorValueBlock* data, u32 indexSize)
{
    for (u32 i = 0; i < indexSize; ++i)
    {
        Swap(&data->index[i]);
    }
}

void Endian::SwapValue(nw::g3d::edit::detail::EditShaderParamMatrixValueBlock* data)
{
    for (int i = 0; i < 16; ++i)
    {
        Swap(&data->value.value.a[i]);
    }
}
void Endian::SwapIndices(nw::g3d::edit::detail::EditShaderParamMatrixValueBlock* data, u32 indexSize)
{
    for (u32 i = 0; i < indexSize; ++i)
    {
        Swap(&data->index[i]);
    }
}

void Endian::SwapIndices(nw::g3d::edit::detail::EditShadingModelValueBlock* data, u32 indexSize)
{
    for (u32 i = 0; i < indexSize; ++i)
    {
        Swap(&data->index[i]);
    }
}

void Endian::SwapResAnimCurveData(void* pResAnimCurveData)
{
    switch (HIO::GetInstance()->TargetDeviceType)
    {
    case HIOBase::TargetType::Cafe:
    case HIOBase::TargetType::OldPc:
        {
            Endian::Swap(static_cast<nw::g3d::res::ResAnimCurveData*>(pResAnimCurveData));
        }
        return;
    case HIOBase::TargetType::Htc:
        {
            Endian::Swap(static_cast<nn::g3d::ResAnimCurveData*>(pResAnimCurveData));
        }
        return;
    default:
        unexpected();
    }
}

void Endian::Swap(nw::g3d::res::ResAnimCurveData* data)
{
    NW_G3D_TABLE_FIELD int typeSize[] = { 4, 2, 1, 1, 1, 1, 1 };
    NW_G3D_TABLE_FIELD int keySize[] = { 4, 2, 1 };

    Swap(&data->targetOffset);
    Swap(&data->startFrame);
    Swap(&data->endFrame);
    Swap(&data->fScale);
    Swap(&data->fOffset);
    Swap(&data->fDelta);

    if ((data->flag & ResAnimCurve::FRAME_MASK) == ResAnimCurve::FRAME32)
    {
        SwapArray<f32>(&data->ofsFrameArray, data->numKey);
    }
    else if ((data->flag & ResAnimCurve::FRAME_MASK) == ResAnimCurve::FRAME16)
    {
        SwapArray<s16>(&data->ofsFrameArray, data->numKey);
    }
    else if ((data->flag & ResAnimCurve::FRAME_MASK) == ResAnimCurve::FRAME8)
    {
        SwapArray<u8>(&data->ofsFrameArray, data->numKey);
    }
    else
    {
        NW_G3D_NOT_IMPLEMENTED();
    }

    int curveType = (data->flag & ResAnimCurve::CURVE_MASK) >> ResAnimCurve::CURVE_SHIFT;
    int keyType = (data->flag & ResAnimCurve::KEY_MASK) >> ResAnimCurve::KEY_SHIFT;

    Swap(data->ofsKeyArray.to_ptr(), keySize[keyType], typeSize[curveType] * data->numKey);

    Swap(&data->ofsFrameArray);
    Swap(&data->ofsKeyArray);

    Swap(&data->flag);
    Swap(&data->numKey);
}


void Endian::Swap(nn::g3d::ResAnimCurveData* data)
{
    NW_G3D_TABLE_FIELD int typeSize[] = { 4, 2, 1, 1, 1, 1, 1 };
    NW_G3D_TABLE_FIELD int keySize[] = { 4, 2, 1 };

    Swap(&data->targetOffset);
    Swap(&data->startFrame);
    Swap(&data->endFrame);
    Swap(&data->fScale);
    Swap(&data->fOffset);
    Swap(&data->fDelta);

    if ((data->flag & ResAnimCurve::FRAME_MASK) == ResAnimCurve::FRAME32)
    {
        SwapArray<f32>(static_cast<float*>(data->pFrameArray.ToPtr(0)), data->keyCount);
    }
    else if ((data->flag & ResAnimCurve::FRAME_MASK) == ResAnimCurve::FRAME16)
    {
        SwapArray<s16>(static_cast<s16*>(data->pFrameArray.ToPtr(0)), data->keyCount);
    }
    else if ((data->flag & ResAnimCurve::FRAME_MASK) == ResAnimCurve::FRAME8)
    {
        SwapArray<u8>(static_cast<u8*>(data->pFrameArray.ToPtr(0)), data->keyCount);
    }
    else
    {
        NW_G3D_NOT_IMPLEMENTED();
    }

    int curveType = (data->flag & ResAnimCurve::CURVE_MASK) >> ResAnimCurve::CURVE_SHIFT;
    int keyType = (data->flag & ResAnimCurve::KEY_MASK) >> ResAnimCurve::KEY_SHIFT;

    Swap(data->pKeyArray.ToPtr(0), keySize[keyType], typeSize[curveType] * data->keyCount);

    Swap(&data->pFrameArray);
    Swap(&data->pKeyArray);

    Swap(&data->flag);
    Swap(&data->keyCount);
}

void Endian::Swap(nw::g3d::edit::detail::FrameCtrlPacket* data)
{
    Swap(&data->header);
    Swap(&data->block.frame);
    Swap(&data->block.frameStep);
    Swap(&data->block.playPolicy);
    Swap(&data->block.frameCount);
}

void Endian::Swap(nn::g3d::viewer::detail::FrameCtrlPacket* data)
{
    Swap(&data->header);
    Swap(&data->block.frame);
    Swap(&data->block.frameStep);
    Swap(&data->block.playPolicy);
    Swap(&data->block.frameCount);
    Swap(&data->block.startFrame);
}


void Endian::Swap(nw::g3d::edit::detail::MultiFileDataBlock* data)
{
    u16 count = data->numFile;
    Swap(&data->numFile);
    Swap(&data->loopCount);

    FileInfoData* fileInfoData = data->ofsFileInfo.to_ptr<FileInfoData>();
    for (u16 i = 0; i < count; ++i)
    {
        Swap(&fileInfoData->fileInfo[i].ofsFileName);
        Swap(&fileInfoData->fileInfo[i].fileSize);
        Swap(&fileInfoData->fileInfo[i].fileDataKind);
        Swap(&fileInfoData->fileInfo[i].fileAlignment);
    }
    Swap(&data->ofsFileInfo);
}

void Endian::Swap(nw::g3d::edit::detail::ModelOptimizedShaderPacket* data)
{
    Swap(&data->header);
    Swap(&data->block.modelObjKey);

    MaterialShaderIndexData* indexData = data->block.ofsMaterialShaderIndices.to_ptr<MaterialShaderIndexData>();
    u32 count = indexData->numMaterial;
    Swap(&indexData->numMaterial);
    for (u32 i = 0; i < count; ++i)
    {
        Swap(&indexData->index[i]);
    }
    Swap(&data->block.ofsMaterialShaderIndices);

    Swap(&data->block.multiFile);
}

void Endian::Swap(nw::g3d::edit::detail::AnimEditInfoPacket* data)
{
    Swap(&data->header);
    Swap(&data->block.animationKey);
    Swap(&data->block.fValue);
    Swap(&data->block.modelKey);
}

void Endian::Swap(nw::g3d::edit::detail::RenderInfoUpdateBlock* data)
{
    Swap(&data->modelKey);
    Swap(&data->materialIndex);

    u32 dataSize = data->renderInfoDataSize;
    Swap(&data->renderInfoDataSize);

    nw::g3d::res::ResDicPatriciaData* dicData = reinterpret_cast<nw::g3d::res::ResDicPatriciaData*>(&data->renderInfoData[0]);
    s32 numData = dicData->numData;
    {
        nw::g3d::res::ResDicPatriciaData::Node* node = &dicData->node[0];
        Swap(&node->idxLeft);
        Swap(&node->idxRight);
        Swap(&node->refBit);

        {
            s32 offset = node->ofsData;
            Swap(&offset);
            node->ofsData = offset;
        }
    }

    for ( s32 i = 0; i < numData; ++i)
    {
        nw::g3d::res::ResDicPatriciaData::Node* node = &dicData->node[i+1];
        {
            nw::g3d::res::ResName* name = const_cast<nw::g3d::res::ResName*>(node->ofsName.GetResName());
            Swap(&name->ref().len);
            s32 offset = node->ofsName.offset;
            Swap(&offset);
            node->ofsName.offset = offset;
        }
        Swap(&node->idxLeft);
        Swap(&node->idxRight);
        Swap(&node->refBit);

        {
            nw::g3d::res::ResRenderInfoData* renderInfoData = node->ofsData.to_ptr<nw::g3d::res::ResRenderInfoData>();
            {
                //nw::g3d::res::ResName* name = const_cast<nw::g3d::res::ResName*>(renderInfoData->ofsName.GetResName());
                //Swap(&name->ref().len); //ここは、上記の辞書で同じ場所を指しているので len のスワップは行わない。
                s32 offset = renderInfoData->ofsName.offset;
                Swap(&offset);
                renderInfoData->ofsName.offset = offset;
            }

            u16 arrayLength = renderInfoData->arrayLength;
            Swap(&renderInfoData->arrayLength);
            if (renderInfoData->type == nw::g3d::res::ResRenderInfo::STRING)
            {
                for (u16 arrayCount = 0; arrayCount < arrayLength; ++arrayCount)
                {
                    nw::g3d::res::BinString* ofsString = &renderInfoData->ofsString[arrayCount];
                    nw::g3d::res::ResName* name = const_cast<nw::g3d::res::ResName*>(ofsString->GetResName());
                    Swap(&name->ref().len);
                    s32 offset = ofsString->offset;
                    Swap(&offset);
                    ofsString->offset = offset;
                }
            }
            else
            {
                for (u16 arrayCount = 0; arrayCount < arrayLength; ++arrayCount)
                {
                    Swap(&renderInfoData->iValue[arrayCount]);
                }
            }

            s32 offsetData = node->ofsData;
            Swap(&offsetData);
            node->ofsData = offsetData;
        }
    }
    Swap(&dicData->numData);
    Swap(&dicData->size);
}

void Endian::Swap(nw::g3d::edit::detail::RenderInfoRecvBlock* data)
{
    using namespace nw::g3d::edit::detail;
    Swap(&data->modelKey);

    u16 materialIndexCount = data->numMaterialIndex;
    u16 renderInfoCount = data->numRenderInfo;

    Swap(&data->numMaterialIndex);
    Swap(&data->numRenderInfo);

    {
        s32* materialIndexArray = data->ofsMaterialIndexArray.to_ptr<s32>();
        for (u16 i = 0; i < materialIndexCount; ++i)
        {
            Swap(&materialIndexArray[i]);
        }
        s32 offset = data->ofsMaterialIndexArray;
        Swap(&offset);
        data->ofsMaterialIndexArray = offset;
    }

    {
        SetupRenderInfoData* renderInfoArray = data->ofsRenderInfoArray.to_ptr<SetupRenderInfoData>();
        for (u16 i = 0; i < renderInfoCount; ++i)
        {
            SetupRenderInfoData& renderInfo = renderInfoArray[i];
            {
                nw::g3d::res::ResName* name = const_cast<nw::g3d::res::ResName*>(renderInfo.ofsName.GetResName());
                Swap(&name->ref().len);
                s32 offset = renderInfo.ofsName.offset;
                Swap(&offset);
                renderInfo.ofsName.offset = offset;
            }

            {
                SetupRenderInfoChoiceData* choice = renderInfo.ofsChoice.to_ptr<SetupRenderInfoChoiceData>();
                u32 choiceCount = choice->numChoice;
                if (renderInfo.type == ResRenderInfo::STRING)
                {
                    for (u32 j = 0; j < choiceCount; ++j)
                    {
                        nw::g3d::res::ResName* name = const_cast<nw::g3d::res::ResName*>(choice->ofsString[j].GetResName());
                        Swap(&name->ref().len);
                        s32 offset = choice->ofsString[j].offset;
                        Swap(&offset);
                        choice->ofsString[j].offset = offset;
                    }
                }
                else
                {
                    for (u32 j = 0; j < choiceCount; ++j)
                    {
                        Swap(&choice->iValue[j]);
                    }
                }

                Swap(&choice->numChoice);
            }

            {
                SetupRenderInfoDefaultData* defaultValue = renderInfo.ofsDefault.to_ptr<SetupRenderInfoDefaultData>();
                u32 defaultCount = defaultValue->numDefault;
                if (renderInfo.type == nw::g3d::res::ResRenderInfo::STRING)
                {
                    for (u32 j = 0; j < defaultCount; ++j)
                    {
                        nw::g3d::res::ResName* name = const_cast<nw::g3d::res::ResName*>(defaultValue->ofsString[j].GetResName());
                        Swap(&name->ref().len);
                        s32 offset = defaultValue->ofsString[j].offset;
                        Swap(&offset);
                        defaultValue->ofsString[j].offset = offset;
                    }
                }
                else
                {
                    for (u32 j = 0; j < defaultCount; ++j)
                    {
                        Swap(&defaultValue->iValue[j]);
                    }
                }
                Swap(&defaultValue->numDefault);
            }
            Swap(&renderInfo.type);

            {
                s32 offset = renderInfo.ofsChoice;
                Swap(&offset);
                renderInfo.ofsChoice = offset;
            }

            {
                s32 offset = renderInfo.ofsDefault;
                Swap(&offset);
                renderInfo.ofsDefault = offset;
            }

            {
                s32 offset = data->ofsRenderInfoArray;
                Swap(&offset);
                data->ofsRenderInfoArray = offset;
            }
        }
    }
}

//--------------------------------------------------------------------------------
// 受信用
void Endian::Swap(nw::g3d::edit::detail::AttachBlock* data)
{
    Swap(&data->attachedKey);
    Swap(&data->attachKind);
    Swap(&data->flag);
}

void Endian::Swap(nw::g3d::edit::detail::FileLoadedBlock* data)
{
    Swap(&data->fileDataKind);
    Swap(&data->toolKey);
    Swap(&data->resFileKey);
    Swap(&data->resModelKey);
    Swap(&data->modelObjKey);
    Swap(&data->newResFileKey);
}

void Endian::Swap(nw::g3d::edit::detail::RuntimeErrorNotificationInfo* data)
{
    Swap(&data->runtimeErrorCode);
}

void Endian::Swap(nn::g3d::viewer::detail::MessageNotificationInfo* data)
{
    Swap(&data->messageType);
    Swap(&data->messageDestination);
    Swap(&data->messageDataSize);
}

void Endian::Swap(nw::g3d::edit::detail::ModelLayoutEditBlock* data)
{
    Swap(&data->modelKey);

    for (int i = 0; i < 3; ++i)
    {
        Swap(&data->scale.a[i]);
        Swap(&data->rotate.a[i]);
        Swap(&data->translate.a[i]);
    }
}

void Endian::Swap(nw::g3d::edit::detail::FrameCtrlBlock* data)
{
    Swap(&data->frame);
    Swap(&data->frameStep);
    Swap(&data->playPolicy);
    Swap(&data->frameCount);
}

void Endian::Swap(nn::g3d::viewer::detail::FrameCtrlBlock* data)
{
    Swap(&data->frame);
    Swap(&data->frameStep);
    Swap(&data->playPolicy);
    Swap(&data->frameCount);
    Swap(&data->startFrame);
}

void Endian::Swap(nw::g3d::edit::detail::RenderInfoSendInfo* info)
{
    Swap(&info->labelInfoNum);
    Swap(&info->modelKey);
    Swap(&info->materialIndex);

    u32 labelSize = info->labelInfoNum;
    size_t headerOffset = sizeof(nw::g3d::edit::detail::PacketHeader);
    size_t labelInfoSize = sizeof(RenderInfoLabelInfo);
    RenderInfoLabelInfo* firstLabelInfo = g3d::ut::AddOffset<RenderInfoLabelInfo>(info, sizeof(RenderInfoSendInfo));
    {
        for (u32 i = 0; i < labelSize; ++i)
        {
            RenderInfoLabelInfo* labelInfo = g3d::ut::AddOffset<RenderInfoLabelInfo>(firstLabelInfo, i * labelInfoSize);

            Swap(&labelInfo->labelOffset);
            Swap(&labelInfo->itemOffset);
            Swap(&labelInfo->itemNum);

            Swap(&labelInfo->valueNum);
            Swap(&labelInfo->valueOffset);

            Swap(&labelInfo->renderInfoType);

            u32 itemNum = labelInfo->itemNum;
            u32 itemOffset = labelInfo->itemOffset;

            u32 valueNum = labelInfo->valueNum;
            u32 valueOffset = labelInfo->valueOffset;

            ResRenderInfo::Type type = static_cast<ResRenderInfo::Type>(labelInfo->renderInfoType);

            // Defaultなど値周りの変換処理
            {
                u32* valueOffsetArray = g3d::ut::AddOffset<u32>(info, valueOffset - headerOffset);
                for (u32 j = 0; j < valueNum; ++j)
                {
                    Swap(&valueOffsetArray[j]);
                }
            }

            // 文字列型以外はあとの処理はしない
            if (type != ResRenderInfo::STRING)
            {
                continue;
            }

            {
                RenderInfoChoiceInfo* itemOffsetArray = g3d::ut::AddOffset<RenderInfoChoiceInfo>(info, itemOffset - headerOffset);
                for (u32 j = 0; j < itemNum; ++j)
                {
                    RenderInfoChoiceInfo* item = &itemOffsetArray[j];
                    Swap(&item->choiceOffset);
                    Swap(&item->aliasOffset);
                    Swap(&item->aliasSize);
                }
            }
        }
    }
}

void Endian::Swap(nw::g3d::edit::detail::ShaderProgramSendInfo* info)
{
    Swap(&info->optionInfoNum);
    Swap(&info->shaderArchiveKey);
    Swap(&info->shadingModelIndex);
    Swap(&info->shaderProgramIndex);

    u32 optionInfoCount = info->optionInfoNum;
    size_t optionInfoSize = sizeof(ShaderProgramOptionInfo);
    ShaderProgramOptionInfo* firstOptionInfo = g3d::ut::AddOffset<ShaderProgramOptionInfo>(info, sizeof(ShaderProgramSendInfo));
    {
        for (u32 i = 0; i < optionInfoCount; ++i)
        {
            ShaderProgramOptionInfo* optionInfo = g3d::ut::AddOffset<ShaderProgramOptionInfo>(firstOptionInfo, i * optionInfoSize);
            Swap(&optionInfo->optionOffset);
            Swap(&optionInfo->choiceOffset);
        }
    }
}

void Endian::Swap(nw::g3d::edit::detail::PickupSendInfo* info)
{
    Swap(&info->materialPickupNum);

    size_t materialPickupInfoSize = sizeof(MaterialPickupInfo);
    MaterialPickupInfo* firstInfo = g3d::ut::AddOffset<MaterialPickupInfo>(info, sizeof(PickupSendInfo));
    {
        for (u32 i = 0; i < info->materialPickupNum; ++i)
        {
            MaterialPickupInfo* info = g3d::ut::AddOffset<MaterialPickupInfo>(firstInfo, i * materialPickupInfoSize);
            Swap(&info->modelObjKey);
            Swap(&info->materialIndex);
        }
    }
}

}}} // namespace nw::g3d::tool


size_t ResAbstractionUtility::GetResAnimCurveSize()
{
    switch (HIO::GetInstance()->TargetDeviceType)
    {
    case HIOBase::TargetType::Cafe:
    case HIOBase::TargetType::OldPc:
        return sizeof(nw::g3d::ResAnimCurve);
    case HIOBase::TargetType::Htc:
        return sizeof(nn::g3d::ResAnimCurveData);
    default:
        unexpected();
    }
}

void NintendoWare::G3d::Edit::ReadResShaderFileAlignment(size_t* pOutAlignment, System::String^ fileName)
{
    switch (HIO::GetInstance()->TargetDeviceType)
    {
    case HIOBase::TargetType::Cafe:
    case HIOBase::TargetType::OldPc:
        // nw::g3d::edit ではアライメントはランタイム側でファイルから取得する
        *pOutAlignment = 1;
        break;
    case HIOBase::TargetType::Htc:
        {
            System::String^ windowsFileName = fileName;
            if (fileName->Substring(0, 1)->Equals("/"))
            {
                // Cafe のパスを Windows パスに変換
                System::String^ driveLetter = fileName->Substring(1, 1);
                windowsFileName = driveLetter + ":" + fileName->Substring(2);
            }

            std::string path = marshal_as<std::string>(windowsFileName);
            size_t fileSize = ExamineFileSize(windowsFileName);
            ScopedAllocator allocator(fileSize);
            void* pBytes = static_cast<void*>(allocator.GetAllocatedBuffer());
            FILE *fp;
            errno_t error = fopen_s(&fp, path.c_str(), "rb" );
            G3D_HTC_THROW_EXCEPTION_UNLESS(error == 0, System::String::Format("Could not open {0}", windowsFileName));

            size_t readSize = fread(allocator.GetAllocatedBuffer(), sizeof( unsigned char ), fileSize, fp);
            G3D_HTC_THROW_EXCEPTION_UNLESS(readSize == fileSize, "failed to read file");

            GetShaderFileAlignment(pOutAlignment, pBytes);

            fclose( fp );
        }
        break;
    default:
        unexpected();
    }
}
