﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define _WINSOCK_DEPRECATED_NO_WARNINGS
#include "Socket.h"

using namespace NintendoWare::G3d::Edit;

namespace {
    NintendoWare::G3d::Edit::HtcErrorDetailCode ConvertSocketError(int errorCode)
    {
        switch (errorCode)
        {
        case WSAEACCES: return NintendoWare::G3d::Edit::HtcErrorDetailCode::SocketErrorAccess;
        case WSAEAFNOSUPPORT: return NintendoWare::G3d::Edit::HtcErrorDetailCode::SocketErrorAdressFamilyNoSupport;
        case WSAEINVAL: return NintendoWare::G3d::Edit::HtcErrorDetailCode::SocketErrorInvalid;
        case WSAEMFILE: return NintendoWare::G3d::Edit::HtcErrorDetailCode::SocketErrorMaxFileTable;
        case WSAENOBUFS: return NintendoWare::G3d::Edit::HtcErrorDetailCode::SocketErrorNoBuffurs;
        case WSAEPROTONOSUPPORT: return NintendoWare::G3d::Edit::HtcErrorDetailCode::SocketErrorProtocolNoSupport;
        case WSAEPROTOTYPE: return NintendoWare::G3d::Edit::HtcErrorDetailCode::SocketErrorProtocolType;
        case WSAETIMEDOUT: return NintendoWare::G3d::Edit::HtcErrorDetailCode::SocketErrorTimeout;
        case WSAECONNREFUSED: return NintendoWare::G3d::Edit::HtcErrorDetailCode::SocketErrorConnectionRefused;
        case WSAENETDOWN: return NintendoWare::G3d::Edit::HtcErrorDetailCode::SocketErrorNetOrHostDown;
        case WSAEHOSTDOWN: return NintendoWare::G3d::Edit::HtcErrorDetailCode::SocketErrorNetOrHostDown;
        case WSAENETUNREACH: return NintendoWare::G3d::Edit::HtcErrorDetailCode::SocketErrorNetOrHostUnreach;
        case WSAEHOSTUNREACH: return NintendoWare::G3d::Edit::HtcErrorDetailCode::SocketErrorNetOrHostUnreach;
        case WSAEWOULDBLOCK: return NintendoWare::G3d::Edit::HtcErrorDetailCode::SocketErrorWouldBlock;
        case WSAENOTSOCK: return NintendoWare::G3d::Edit::HtcErrorDetailCode::SocketErrorNotSock;
        default: return NintendoWare::G3d::Edit::HtcErrorDetailCode::SocketErrorUnknown;
        }
    }
}

namespace nw { namespace g3d { namespace tool {

Socket::Socket()
    : m_Socket( INVALID_SOCKET )
    , m_pReadBuffer(NULL)
    , m_ReadBufferSize(0)
    , m_ReadSize(0)
    , m_ReadingSize(0)
    , m_TotalReadingSize(0)
    , m_IsReadEnd(true)
    , m_IsDebugLogEnabled(false)
{

}

Socket::~Socket()
{
    Close();
}

NintendoWare::G3d::Edit::HtcResult^ Socket::Open()
{
    NintendoWare::G3d::Edit::HtcResult^ result = gcnew NintendoWare::G3d::Edit::HtcResult();
    if ( m_Socket != INVALID_SOCKET )
    {
        result->ResultCode = NintendoWare::G3d::Edit::HtcResultCode::SocketAlreadyOpened;
        return result;
    }

    m_Socket = socket( AF_UNSPEC, SOCK_STREAM, IPPROTO_TCP );
    if ( m_Socket == INVALID_SOCKET )
    {
        result->ResultCode = NintendoWare::G3d::Edit::HtcResultCode::CreateSocketFailed;
        result->ErrorDetailRawValue = WSAGetLastError();
        result->ErrorDetailCode = ConvertSocketError(result->ErrorDetailRawValue);
        return result;
    }

    result->ResultCode = NintendoWare::G3d::Edit::HtcResultCode::Success;
    return result;
}

void Socket::Close()
{
    if ( m_Socket == INVALID_SOCKET )
    {
        return;
    }

    closesocket( m_Socket );
    m_Socket = INVALID_SOCKET;
}

bool Socket::SetBlockMode()
{
    bool funcResult = m_Socket != SOCKET_ERROR;
    if (funcResult)
    {
        // ブロッキングモードに設定
        DWORD  nonBlock = 0;
        ioctlsocket(m_Socket, FIONBIO, &nonBlock);
    }

    return funcResult;
}


NintendoWare::G3d::Edit::HtcResult^ Socket::Connect( u16 port, const char* addr )
{
    NintendoWare::G3d::Edit::HtcResult^ result = gcnew NintendoWare::G3d::Edit::HtcResult();
    if ( m_Socket == INVALID_SOCKET )
    {
        result->ResultCode = NintendoWare::G3d::Edit::HtcResultCode::InvalidSocket;
        return result;
    }

    sockaddr_in saddr = { 0 };
    saddr.sin_family = AF_INET;
    saddr.sin_addr.S_un.S_addr = inet_addr( addr );
    saddr.sin_port = htons( port );

    int socketConnectResult = connect( m_Socket, reinterpret_cast<sockaddr*>( &saddr ), sizeof( saddr ) );
    if (socketConnectResult == SOCKET_ERROR)
    {
        result->ResultCode = NintendoWare::G3d::Edit::HtcResultCode::SocketConnectFailed;
        result->ErrorDetailRawValue = WSAGetLastError();
        result->ErrorDetailCode = ConvertSocketError(result->ErrorDetailRawValue);
        return result;
    }

    // ノンブロッキングモードに設定
    DWORD  nonBlock = 1;
    ioctlsocket(m_Socket, FIONBIO, &nonBlock);

    result->ResultCode = NintendoWare::G3d::Edit::HtcResultCode::Success;
    return result;
}

NintendoWare::G3d::Edit::HtcResult^ Socket::Recv(int* receiveBytes, char* buf, int len )
{
    NintendoWare::G3d::Edit::HtcResult^ result = gcnew NintendoWare::G3d::Edit::HtcResult();
    if ( m_Socket == INVALID_SOCKET )
    {
        result->ResultCode = NintendoWare::G3d::Edit::HtcResultCode::InvalidSocket;
        return result;
    }

    *receiveBytes = recv( m_Socket, buf, len, 0 );
    if (*receiveBytes == SOCKET_ERROR)
    {
        result->ResultCode = NintendoWare::G3d::Edit::HtcResultCode::SocketRecvFailed;
        result->ErrorDetailRawValue = WSAGetLastError();
        result->ErrorDetailCode = ConvertSocketError(result->ErrorDetailRawValue);
        return result;
    }
    else if (*receiveBytes == 0)
    {
        result->ResultCode = NintendoWare::G3d::Edit::HtcResultCode::SocketRecvFailed;
        result->ErrorDetailRawValue = 0;
        result->ErrorDetailCode = HtcErrorDetailCode::SocketErrorGracefullyClosed;
        return result;
    }

    result->ResultCode = NintendoWare::G3d::Edit::HtcResultCode::Success;
    return result;
}

NintendoWare::G3d::Edit::HtcResult^ Socket::Send(int* sentBytes, char* buf, int len )
{
    NintendoWare::G3d::Edit::HtcResult^ result = gcnew NintendoWare::G3d::Edit::HtcResult();
    if ( m_Socket == INVALID_SOCKET )
    {
        result->ResultCode = NintendoWare::G3d::Edit::HtcResultCode::InvalidSocket;
        return result;
    }

    *sentBytes = send( m_Socket, buf, len, 0 );
    if (*sentBytes == SOCKET_ERROR)
    {
        result->ResultCode = NintendoWare::G3d::Edit::HtcResultCode::SocketSendFailed;
        result->ErrorDetailRawValue = WSAGetLastError();
        result->ErrorDetailCode = ConvertSocketError(result->ErrorDetailRawValue);
        return result;
    }

    result->ResultCode = NintendoWare::G3d::Edit::HtcResultCode::Success;
    return result;
}

bool Socket::ReadSync( void* buf, int size )
{
    bool result = false;
#if 0
    int bytesReceived = 0;
    char* pData = static_cast<char*>(buf);

    while ( bytesReceived < size )
    {
        int socketResult = Recv( pData + bytesReceived, size - bytesReceived );

        if ( socketResult < 0 )
        {
            break;
        }

        bytesReceived += socketResult;
    }

    if ( bytesReceived == size )
    {
        result = true;
    }
#endif
    return result;
}

bool Socket::WriteSync( const void* buf, int size )
{
    bool result = true;

    int bytesSended = 0;
    const char* pChar = reinterpret_cast<const char*>(buf);
    char* pData = const_cast<char*>( pChar );

    while ( bytesSended < size )
    {
        int sendBytes;
        NintendoWare::G3d::Edit::HtcResult^ sendResult = Send(&sendBytes, pData + bytesSended, size - bytesSended );
        if (!sendResult->IsSuccess)
        {
            // MEMO: WSAEWOULDBLOCK が続いて無限ループに陥る可能性がある？
            if (sendResult->ErrorDetailCode != HtcErrorDetailCode::SocketErrorWouldBlock)
            {
                closesocket(m_Socket);
                m_Socket = INVALID_SOCKET;
                return false;
            }

            sendBytes = 0;
        }
        bytesSended += sendBytes;
    }

    if ( bytesSended != size )
    {
        result = false;
    }

    return result;
}

bool Socket::ReadASync( void* buf, int size )
{
    if (buf == NULL || size <= 0)
    {
        return false;
    }
    if (IsReading())
    {
        return false;
    }

    if (!m_IsReadEnd)
    {
        return false;
    }

    if (m_Socket == INVALID_SOCKET)
    {
        return false;
    }

    m_IsReadEnd = false;
    m_pReadBuffer = static_cast<char*>(buf);
    m_ReadBufferSize = size;

    m_ReadSize = size;
    m_TotalReadingSize = m_ReadingSize = 0;

    int recvBytes;
    HtcResult^ recvResult = Recv(&recvBytes, m_pReadBuffer, m_ReadBufferSize);
    if (!recvResult->IsSuccess)
    {
        // ソケットが使えなくなっている場合は、ソケットを閉じる
        if (recvResult->ErrorDetailCode != HtcErrorDetailCode::SocketErrorWouldBlock)
        {
            ResetReadFlag();
            closesocket(m_Socket);
            m_Socket = INVALID_SOCKET;
            return false;
        }
        else
        {
            return true;
        }
    }
    m_ReadingSize += static_cast<u32>(recvBytes);
    m_TotalReadingSize = m_ReadingSize;

    if (m_IsDebugLogEnabled)
    {
        System::Diagnostics::Debug::Write("(m_ReadBufferSize = ");
        System::Diagnostics::Debug::Write(m_ReadBufferSize);
        System::Diagnostics::Debug::Write(") > (");
        System::Diagnostics::Debug::Write("m_TotalReadingSize = ");
        System::Diagnostics::Debug::Write(m_TotalReadingSize);
        System::Diagnostics::Debug::WriteLine(")");
    }

    return true;
}

bool
Socket::IsReading() const
{
    return m_ReadBufferSize > m_TotalReadingSize;
}

void
Socket::ResetReadFlag()
{
    m_TotalReadingSize = m_ReadSize = m_ReadingSize = 0;
    m_IsReadEnd = true;
    m_ReadBufferSize = 0;
    m_pReadBuffer = NULL;
}

void
Socket::Poll()
{
    if ( IsConnected() )
    {
        ReadASyncInternal();
    }
}

void
Socket::ReadASyncInternal()
{
    if ( m_Socket == INVALID_SOCKET )
    {
        return;
    }

    // 読み込みが終了している場合は処理を抜ける
    if (m_IsReadEnd)
    {
        return;
    }

    int recvBytes;
    HtcResult^ recvResult = Recv(&recvBytes, m_pReadBuffer + m_ReadingSize, m_ReadBufferSize - m_ReadingSize);
    bool isServerSideGracefullyClosed = recvBytes == 0;
    if (!recvResult->IsSuccess || isServerSideGracefullyClosed)
    {
        // ソケットが使えなくなっている場合は、ソケットを閉じる
        if (recvResult->ErrorDetailCode != HtcErrorDetailCode::SocketErrorWouldBlock)
        {
            ResetReadFlag();
            closesocket(m_Socket);
            m_Socket = INVALID_SOCKET;
        }

        return;
    }

    m_ReadingSize += static_cast<u32>(recvBytes);
    m_TotalReadingSize += m_ReadingSize;

    if (m_IsDebugLogEnabled)
    {
        System::Diagnostics::Debug::Write("(m_ReadBufferSize = ");
        System::Diagnostics::Debug::Write(m_ReadBufferSize);
        System::Diagnostics::Debug::Write(") > (");
        System::Diagnostics::Debug::Write("m_TotalReadingSize = ");
        System::Diagnostics::Debug::Write(m_TotalReadingSize);
        System::Diagnostics::Debug::WriteLine(")");
    }

    if (m_ReadBufferSize == m_TotalReadingSize)
    {
        m_IsReadEnd = true;
    }
}

bool
Socket::IsConnected() const
{
    return m_Socket != INVALID_SOCKET;
}

}}} // namespace nw::g3d::tool
