﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/g3d//ut/g3d_Inlines.h>
#include <nw/g3d/edit/g3d_EditDefs.h>
#include <vector>

namespace nw {
namespace g3d {
namespace tool {

//! @brief ブロックを表す構造体です。
struct Chunk
{
    size_t size;
    ptrdiff_t offset;
    size_t alignment;
};

//! @brief 文字列をまとめてバイナリ化するためのクラスです。
class StringPool
{
public:
    StringPool()
        : m_Strings()
        , m_Paths()
        , m_ResName()
        , m_ResNamePath()
        , m_Chunk()
        , m_pAddress(nullptr)
    {
        m_Chunk.size = 0;
        m_Chunk.offset = -1;
        m_Chunk.alignment = 0;
    }

    void Add(const std::string& str) { m_Strings.push_back(&str); }
    void AddPath(const std::string& str) { m_Paths.push_back(&str); }
    void Finalize();
    void Write(void* pBuffer);

    size_t GetBlockSize() const { return m_Chunk.size; }
    ptrdiff_t GetBlockOffset() const { return m_Chunk.offset; }
    void SetBlockOffset(ptrdiff_t offset) { m_Chunk.offset = offset; }
    const char* GetPtr(const std::string& str) const;
    const char* GetPathPtr(const std::string& str) const;

    //! @brief ブロックのメイン構造体のポインタを取得します。
    void* GetPtr(void* pBuffer) const
    {
        if (m_Chunk.size > 0)
        {
            return nw::g3d::ut::AddOffset(pBuffer, m_Chunk.offset);
        }
        return nullptr;
    }

private:
    //! @brief m_Strings と m_Paths をソートします。
    void Sort();

    //! @brief StringPool に必要なブロックサイズを計算します。
    void CalcSize();

    std::vector<const std::string*> m_Strings;
    std::vector<const std::string*> m_Paths;
    std::vector<const ResNameData*> m_ResName; // バイナリ上の実ポインタ
    std::vector<const ResNameData*> m_ResNamePath;
    Chunk m_Chunk;
    void* m_pAddress;
};

class EditBinaryBlock
{
public:
    enum Category
    {
        BIN_MAIN,
        BIN_INDEX_STREAM_PADDING,
        BIN_INDEX_STREAM,
        BIN_VERTEX_STREAM_PADDING,
        BIN_VERTEX_STREAM,
        BIN_TEXTURE_PADDING,
        BIN_TEXTURE,
        BIN_EXTERNAL_FILE_PADDING,
        BIN_EXTERNAL_FILE,
        BIN_FILE_PADDING,
        NUM_CATEGORY
    };

    class Context;

    virtual void Build(Context& ctx) {} // 1. 階層構造の構築、文字列の収集、辞書への文字列登録。
    virtual void CalcSize() {} // 2. ブロックのサイズ計算。
    virtual void CalcOffset(Context& ctx); // 3. ブロックのオフセット計算。
    virtual void Convert(const Context& ctx) {} // 4. ブロック内データの変換。
    virtual void Adjust() {}; // 5. ブロックデータ間での調整。
    // バインドとエンディアンスワップはランタイムのメソッドを利用する。

    //! @brief ブロックのメイン構造体のポインタを取得します。
    virtual void* GetPtr(void* pBuffer) const
    {
        if (m_Block[BIN_MAIN].size > 0)
        {
            // 自身の MAIN 領域から割り当てる。
            return nw::g3d::ut::AddOffset(pBuffer, m_Block[BIN_MAIN].offset);
        }
        return nullptr;
    }

    template <typename T>
    T* GetPtr(void* pBuffer) const
    {
        return static_cast<T*>(GetPtr(pBuffer));
    }

    size_t GetBlockSize(Category category) const { return m_Block[category].size; }
    void SetBlockOffset(Category category, ptrdiff_t offset) { m_Block[category].offset = offset; }

protected:
    EditBinaryBlock()
    {
        for (int index = 0; index < NUM_CATEGORY; ++index)
        {
            m_Block[index].size = 0;
            m_Block[index].offset = -1;
            m_Block[index].alignment = 0;
        }
    }

    static size_t CalcChunk(Chunk* pChunk, int count)
    {
        int idx = 0;
        pChunk[idx].offset = 0;
        for (; idx < count - 1; ++idx)
        {
            pChunk[idx + 1].offset = pChunk[idx].offset + pChunk[idx].size;
        }
        return pChunk[idx].offset + pChunk[idx].size;
    }

    void SetBlockSize(Category category, size_t size, size_t alignment = 0)
    {
        m_Block[category].size = size;
        m_Block[category].alignment = alignment;
    }

    ptrdiff_t GetBlockOffset(Category category) const { return m_Block[category].offset; }

    // 指定カテゴリ内の指定オフセット位置のポインタを取得する。
    void* GetPtr(const Context& ctx, Category category, ptrdiff_t offset) const;

    template <typename T>
    T* GetPtr(const Context& ctx, Category category, ptrdiff_t offset) const
    {
        return static_cast<T*>(GetPtr(ctx, category, offset));
    }

private:
    Chunk m_Block[NUM_CATEGORY];
};

class EditBinaryBlock::Context
{
public:
    Context() : blocks(), pool(), pBuf(nullptr), bufSize(0)
    {
        for (int index = 0; index < NUM_CATEGORY; ++index)
        {
            chunk[index].size = 0;
            chunk[index].offset = -1;
            chunk[index].alignment = 0;
        }
    }

    std::vector<EditBinaryBlock*> blocks;
    StringPool pool;
    std::vector<Chunk*> textureChunks;
    Chunk chunk[NUM_CATEGORY];
    void* pBuf;
    size_t bufSize;
};

}}} // namespace nw::g3d::tool
