﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "EditBinAnimCurve.h"
#include "HIO.h"

using namespace NintendoWare::G3d::Edit;
using namespace Nintendo::InGameEditing::Communication;

namespace
{

template<typename T>
inline T Round(float src)
{
    if (src > 0.0f)
    {
        return static_cast<T>(std::floor(src + 0.5f));
    }
    else
    {
        return static_cast<T>(std::ceil(src - 0.5f));
    }
}

inline static float Pi() { return 3.141592653589793f; } // 円周率

inline float RadToDeg(float rad)
{
    return rad * (180.0f / Pi());
}

inline float DegToRad(float deg)
{
    return deg * (Pi() / 180.0f);
}

}

namespace nw { namespace g3d { namespace tool {

void EditBinAnimCurve::AddCurveValue(float frame, float value, float inSlope, float outSlope)
{
    CurveValue curveValue;
    curveValue.frame = frame;
    curveValue.value = value;
    curveValue.inSlope = inSlope;
    curveValue.outSlope = outSlope;
    m_CurveValueArray.push_back(curveValue);
}

void EditBinAnimCurve::Build()
{
    size_t size = m_CurveValueArray.size();
    for (size_t index = 0; index < size - 1; ++index)
    {
        const CurveValue& curveValue = m_CurveValueArray[index];
        const CurveValue& nextCurveValue = m_CurveValueArray[index + 1];

        FVSPair pair;
        if (m_InterpolationType == LINEAR)
        {
            pair.linear.frame0 = curveValue.frame;
            pair.linear.value0 = curveValue.value;
            pair.linear.frame1 = nextCurveValue.frame;
            pair.linear.value1 = nextCurveValue.value;
        }
        else if (m_InterpolationType == STEP)
        {
            pair.step.frame0 = curveValue.frame;
            pair.step.value0 = curveValue.value;
        }
        else // HERMITE
        {
            pair.hermite.frame0 = curveValue.frame;
            pair.hermite.value0 = curveValue.value;
            pair.hermite.in_slope0 = curveValue.inSlope;
            pair.hermite.out_slope0 = curveValue.outSlope;
            pair.hermite.frame1 = nextCurveValue.frame;
            pair.hermite.value1 = nextCurveValue.value;
            pair.hermite.in_slope1 = nextCurveValue.inSlope;
        }
        m_FVSPairArray.push_back(pair);
    }

    // 最終フレームの登録
    {
        const CurveValue& endFrameCurveValue = m_CurveValueArray[size - 1];
        FVSPair pair;
        if (m_InterpolationType == LINEAR)
        {
            pair.linear.frame0 = endFrameCurveValue.frame;
            pair.linear.value0 = endFrameCurveValue.value;
            pair.linear.frame1 = endFrameCurveValue.frame;
            pair.linear.value1 = endFrameCurveValue.value;
        }
        else if (m_InterpolationType == STEP)
        {
            pair.step.frame0 = endFrameCurveValue.frame;
            pair.step.value0 = endFrameCurveValue.value;
        }
        else // HERMITE
        {
            pair.hermite.frame0 = endFrameCurveValue.frame;
            pair.hermite.value0 = endFrameCurveValue.value;
            pair.hermite.in_slope0 = endFrameCurveValue.inSlope;
            pair.hermite.out_slope0 = endFrameCurveValue.outSlope;
            pair.hermite.frame1 = endFrameCurveValue.frame;
            pair.hermite.value1 = endFrameCurveValue.value;
            pair.hermite.in_slope1 = endFrameCurveValue.inSlope;
        }
        m_FVSPairArray.push_back(pair);
    }
}

void EditBinAnimCurve::CalcSize()
{
    // TODO: ベイク対応
    static const int typeSize[] = { 4, 4, 2, 1 };
    static const int compSize[] = { 4, 2, 1 };
    const int frame_type = m_FrameType;
    const int key_type = m_KeyType;
    const int curve_type = 0;
    auto size =  m_FVSPairArray.size();
    m_Chunk[FRAME].size = Align(typeSize[frame_type] * size);

    if (m_Type == BOOL)
    {
        m_Chunk[KEY].size = Align(size, ALIGNMENT_FLAG) >> 3;
    }
    else if(m_InterpolationType == STEP && m_Type == FLOAT)
    {
        // リニアに変換される
        m_Chunk[KEY].size = Align(typeSize[key_type] * compSize[LINEAR] * size);
    }
    else
    {
        // まずは、float hermite のみ
        m_Chunk[KEY].size = Align(typeSize[key_type] * compSize[m_InterpolationType] * size);
    }
    SetBlockSize(BIN_MAIN, CalcChunk(m_Chunk, NUM_CHUNK));
}

void EditBinAnimCurve::Convert(const Context& ctx)
{
    u16 flag = 0;
    flag |= m_FrameTypeFlag;
    flag |= m_KeyTypeFlag;
    flag |= (m_PreWrapTypeFlag << ResAnimCurve::WRAP_PRE_SHIFT);
    flag |= (m_PostWrapTypeFlag << ResAnimCurve::WRAP_POST_SHIFT);

    switch (HIO::GetInstance()->TargetDeviceType)
    {
    case HIOBase::TargetType::Cafe:
    case HIOBase::TargetType::OldPc:
        {
            ResAnimCurveData& curve = *GetPtr<ResAnimCurveData>(ctx.pBuf);

            curve.flag = flag;

            curve.numKey = static_cast<u16>(m_FVSPairArray.size());
            curve.targetOffset = static_cast<u32>(m_TargetOffset);

            curve.fDelta = m_FVSPairArray.back().f[1] - m_FVSPairArray.front().f[1];
            if (m_IsDegreeValue)
            {
                curve.fDelta = DegToRad(curve.fDelta);
            }

            QuantizeFrame(ctx, curve);

            switch(m_InterpolationType)
            {
            case HERMITE:
                QuantizeHermite(ctx, curve);
                break;
            case LINEAR:
                QuantizeLinear(ctx, curve);
                break;
            case STEP:
                switch(m_Type)
                {
                case FLOAT:
                    QuantizeStepFloat(ctx, curve);
                    break;
                case INT:
                    QuantizeStepInt(ctx, curve);
                    break;
                case BOOL:
                    QuantizeStepBool(ctx, curve);
                    break;
                }
            }

            const void* pKey = GetPtr(ctx, BIN_MAIN, m_Chunk[KEY].offset);
            curve.ofsKeyArray.set_ptr(pKey);
        }
        break;
    case HIOBase::TargetType::Htc:
        {
            nn::g3d::ResAnimCurveData& curve = *GetPtr<nn::g3d::ResAnimCurveData>(ctx.pBuf);

            curve.flag = flag;

            curve.keyCount = static_cast<u16>(m_FVSPairArray.size());
            curve.targetOffset = static_cast<u32>(m_TargetOffset);

            curve.fDelta = m_FVSPairArray.back().f[1] - m_FVSPairArray.front().f[1];
            if (m_IsDegreeValue)
            {
                curve.fDelta = DegToRad(curve.fDelta);
            }

            QuantizeFrame(ctx, curve);

            switch(m_InterpolationType)
            {
            case HERMITE:
                QuantizeHermite(ctx, curve);
                break;
            case LINEAR:
                QuantizeLinear(ctx, curve);
                break;
            case STEP:
                switch(m_Type)
                {
                case FLOAT:
                    QuantizeStepFloat(ctx, curve);
                    break;
                case INT:
                    QuantizeStepInt(ctx, curve);
                    break;
                case BOOL:
                    QuantizeStepBool(ctx, curve);
                    break;
                }
            }

            void* pKey = GetPtr(ctx, BIN_MAIN, m_Chunk[KEY].offset);
            curve.pKeyArray.SetOffset(&curve.pKeyArray, pKey);
        }
        break;
    default:
        unexpected();
    }
}

void EditBinAnimCurve::QuantizeFrame(const Context& ctx, ResAnimCurveData& curve)
{
    curve.ofsFrameArray.set_ptr(GetPtr(ctx, BIN_MAIN, m_Chunk[FRAME].offset));

    if (m_FrameType == FRAME32)
    {
        int dstLastIndex = 0;
        float* pDst = curve.ofsFrameArray.to_ptr<float>();
        for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter, ++pDst, ++dstLastIndex)
        {
            *pDst = iter->f[0];
        }

        pDst = curve.ofsFrameArray.to_ptr<float>();
        curve.startFrame = pDst[0];
        curve.endFrame = pDst[dstLastIndex-1];
    }
    else if (m_FrameType == FRAME16)
    {
        size_t frameSize = 0;
        int dstLastIndex = 0;
        s16* pDst = curve.ofsFrameArray.to_ptr<s16>();
        for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter, ++pDst, ++dstLastIndex)
        {
            float frame = iter->f[0];
            frame = nw::g3d::math::Math::Clamp(frame, -1024.5f, 1024.4f);

            // F32 浮動小数点数を S10.5 固定小数点数に変換
            *pDst = Round<s16>(iter->f[0] * 32.0f);
            frameSize += sizeof(s16);
        }

        size_t padding = m_Chunk[FRAME].size - frameSize;
        memset(pDst, 0, padding);

        pDst = curve.ofsFrameArray.to_ptr<s16>();
        curve.startFrame = static_cast<float>(pDst[0]) * (1.0f / 32.0f);
        curve.endFrame = static_cast<float>(pDst[dstLastIndex-1]) * (1.0f / 32.0f);
    }
    else// FRAME8
    {
        size_t frameSize = 0;
        int dstLastIndex = 0;
        u8* pDst = curve.ofsFrameArray.to_ptr<u8>();
        for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter, ++pDst, ++dstLastIndex)
        {
            float frame = iter->f[0];
            frame = nw::g3d::math::Math::Clamp(frame, -0.5f, 255.4f);

            *pDst = Round<u8>(frame);
            frameSize += sizeof(u8);
        }

        size_t padding = m_Chunk[FRAME].size - frameSize;
        memset(pDst, 0, padding);

        pDst = curve.ofsFrameArray.to_ptr<u8>();
        curve.startFrame = pDst[0];
        curve.endFrame = pDst[dstLastIndex-1];
    }
}

void EditBinAnimCurve::QuantizeHermite(const Context& ctx, ResAnimCurveData& curve)
{
    curve.flag |= ResAnimCurve::CURVE_CUBIC;
    curve.ofsKeyArray.set_ptr(GetPtr(ctx, BIN_MAIN, m_Chunk[KEY].offset));

    float coef[4];
    curve.fScale = m_Scale;
    curve.fOffset = m_Offset;

    struct Coef32 { float coef[4]; };
    struct Coef16 { s16 coef[4]; };
    struct Coef8 { s8 coef[4]; };

    void * pDst = curve.ofsKeyArray.to_ptr();
    for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter)
    {
        float f = iter->hermite.frame1 - iter->hermite.frame0;
        float v0 = iter->hermite.value0;
        float v1 = iter->hermite.value1;
        float so = iter->hermite.out_slope0;
        float si = iter->hermite.in_slope1;
        if (m_IsDegreeValue)
        {
            v0 = DegToRad(v0);
            v1 = DegToRad(v1);
            so = DegToRad(so);
            si = DegToRad(si);
        }
        float v = v1 - v0;
        coef[3] = -2 * v + (so + si) * f;
        coef[2] = 3 * v - (2 * so + si) * f;
        coef[1] = so * f;
        coef[0] = v0;

        if (m_KeyType == KEY32)
        {
            Coef32* coefDst = static_cast<Coef32*>(pDst);
            coefDst->coef[3] = coef[3];
            coefDst->coef[2] = coef[2];
            coefDst->coef[1] = coef[1];
            coefDst->coef[0] = coef[0];
            pDst = AddOffset(pDst, sizeof(Coef32));
        }
        else if (m_KeyType == KEY16)
        {
            Coef16* coefDst = static_cast<Coef16*>(pDst);
            coefDst->coef[3] = Round<s16>(coef[3] / curve.fScale);
            coefDst->coef[2] = Round<s16>(coef[2] / curve.fScale);
            coefDst->coef[1] = Round<s16>(coef[1] / curve.fScale);
            coefDst->coef[0] = Round<s16>((coef[0] - curve.fOffset) / curve.fScale);
            pDst = AddOffset(pDst, sizeof(Coef16));
        }
        else// KEY8
        {
            Coef8* coefDst = static_cast<Coef8*>(pDst);
            coefDst->coef[3] = Round<s8>(coef[3] / curve.fScale);
            coefDst->coef[2] = Round<s8>(coef[2] / curve.fScale);
            coefDst->coef[1] = Round<s8>(coef[1] / curve.fScale);
            coefDst->coef[0] = Round<s8>((coef[0] - curve.fOffset) / curve.fScale);
            pDst = AddOffset(pDst, sizeof(Coef8));
        }
    }
}

void EditBinAnimCurve::QuantizeLinear(const Context& ctx, ResAnimCurveData& curve)
{
    curve.flag |= ResAnimCurve::CURVE_LINEAR;
    curve.ofsKeyArray.set_ptr(GetPtr(ctx, BIN_MAIN, m_Chunk[KEY].offset));

    float coef[2];
    curve.fScale = m_Scale;
    curve.fOffset = m_Offset;

    struct Coef32 { float coef[2]; };
    struct Coef16 { s16 coef[2]; };
    struct Coef8 { s8 coef[2]; };

    void * pDst = curve.ofsKeyArray.to_ptr();
    for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter)
    {
        float v0 = iter->linear.value0;
        float v1 = iter->linear.value1;
        if (m_IsDegreeValue)
        {
            v0 = DegToRad(v0);
            v1 = DegToRad(v1);
        }
        float v = v1 - v0;
        coef[1] = v;
        coef[0] = v0;


        if (m_KeyType == KEY32)
        {
            Coef32* coefDst = static_cast<Coef32*>(pDst);
            coefDst->coef[1] = coef[1];
            coefDst->coef[0] = coef[0];
            pDst = AddOffset(pDst, sizeof(Coef32));
        }
        else if (m_KeyType == KEY16)
        {
            Coef16* coefDst = static_cast<Coef16*>(pDst);
            coefDst->coef[1] = Round<s16>(coef[1] / curve.fScale);
            coefDst->coef[0] = Round<s16>((coef[0] - curve.fOffset) / curve.fScale);
            pDst = AddOffset(pDst, sizeof(Coef16));
        }
        else // KEY8
        {
            Coef8* coefDst = static_cast<Coef8*>(pDst);
            coefDst->coef[1] = Round<s8>(coef[1] / curve.fScale);
            coefDst->coef[0] = Round<s8>((coef[0] - curve.fOffset) / curve.fScale);
            pDst = AddOffset(pDst, sizeof(Coef8));
        }
    }
}
void EditBinAnimCurve::QuantizeStepFloat(const Context& ctx, ResAnimCurveData& curve)
{
    // リニア扱い。
    curve.flag |= ResAnimCurve::CURVE_LINEAR;
    curve.ofsKeyArray.set_ptr(GetPtr(ctx, BIN_MAIN, m_Chunk[KEY].offset));

    float coef[2];
    curve.fScale = m_Scale;
    curve.fOffset = m_Offset;

    struct Coef32 { float coef[2]; };
    struct Coef16 { s16 coef[2]; };
    struct Coef8 { s8 coef[2]; };

    void * pDst = curve.ofsKeyArray.to_ptr();
    for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter)
    {
        coef[1] = 0.0f;
        coef[0] = iter->step.value0;
        if (m_IsDegreeValue)
        {
            coef[0] = DegToRad(coef[0]);
        }

        if (m_KeyType == KEY32)
        {
            Coef32* coefDst = static_cast<Coef32*>(pDst);
            coefDst->coef[1] = coef[1];
            coefDst->coef[0] = coef[0];
            pDst = AddOffset(pDst, sizeof(Coef32));
        }
        else if (m_KeyType == KEY16)
        {
            Coef16* coefDst = static_cast<Coef16*>(pDst);
            coefDst->coef[1] = Round<s16>(coef[1] / curve.fScale);
            coefDst->coef[0] = Round<s16>((coef[0] - curve.fOffset) / curve.fScale);
            pDst = AddOffset(pDst, sizeof(Coef16));
        }
        else // KEY8
        {
            Coef8* coefDst = static_cast<Coef8*>(pDst);
            coefDst->coef[1] = Round<s8>(coef[1] / curve.fScale);
            coefDst->coef[0] = Round<s8>((coef[0] - curve.fOffset) / curve.fScale);
            pDst = AddOffset(pDst, sizeof(Coef8));
        }
    }
}
void EditBinAnimCurve::QuantizeStepInt(const Context& ctx, ResAnimCurveData& curve)
{
    curve.flag |= ResAnimCurve::CURVE_STEP_INT;
    curve.ofsKeyArray.set_ptr(GetPtr(ctx, BIN_MAIN, m_Chunk[KEY].offset));
    curve.iScale = 0;
    curve.iOffset = static_cast<int>(m_Offset);

    if (m_KeyType == KEY32)
    {
        s32* pDst = curve.ofsKeyArray.to_ptr<s32>();
        for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter, ++pDst)
        {
            *pDst = Round<s32>(iter->step.value0);
        }
    }
    else if (m_KeyType == KEY16)
    {
        s16* pDst = curve.ofsKeyArray.to_ptr<s16>();
        for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter, ++pDst)
        {
            *pDst = Round<s16>(iter->step.value0 - curve.iOffset);
        }
    }
    else // KEY8
    {
        s8* pDst = curve.ofsKeyArray.to_ptr<s8>();
        for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter, ++pDst)
        {
            *pDst = Round<s8>(iter->step.value0 - curve.iOffset);
        }
    }
}
void EditBinAnimCurve::QuantizeStepBool(const Context& ctx, ResAnimCurveData& curve)
{
    curve.flag |= ResAnimCurve::CURVE_STEP_BOOL;
    curve.ofsKeyArray.set_ptr(GetPtr(ctx, BIN_MAIN, m_Chunk[KEY].offset));

    u32* pDst = curve.ofsKeyArray.to_ptr<u32>();
    size_t byteSize = Align(m_FVSPairArray.size(), ALIGNMENT_FLAG) >> 3;
    memset(pDst, 0, byteSize);
    int index = 0;
    for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter, ++index)
    {
        if (iter->step.value0 != 0.0f)
        {
            pDst[index >> 5] |= 0x1 << (index & 0x1F);
        }
    }
    curve.iScale = 1;
    curve.iOffset = 0;
}


void EditBinAnimCurve::QuantizeFrame(const Context& ctx, nn::g3d::ResAnimCurveData& curve)
{
    curve.pFrameArray.SetOffset(&curve.pFrameArray, GetPtr(ctx, BIN_MAIN, m_Chunk[FRAME].offset));

    if (m_FrameType == FRAME32)
    {
        int dstLastIndex = 0;
        float* pDst = static_cast<float*>(curve.pFrameArray.ToPtr(&curve.pFrameArray));
        for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter, ++pDst, ++dstLastIndex)
        {
            *pDst = iter->f[0];
        }

        pDst = static_cast<float*>(curve.pFrameArray.ToPtr(&curve.pFrameArray));
        curve.startFrame = pDst[0];
        curve.endFrame = pDst[dstLastIndex-1];
    }
    else if (m_FrameType == FRAME16)
    {
        size_t frameSize = 0;
        int dstLastIndex = 0;
        s16* pDst = static_cast<s16*>(curve.pFrameArray.ToPtr(&curve.pFrameArray));
        for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter, ++pDst, ++dstLastIndex)
        {
            float frame = iter->f[0];
            frame = nw::g3d::math::Math::Clamp(frame, -1024.5f, 1024.4f);

            // F32 浮動小数点数を S10.5 固定小数点数に変換
            *pDst = Round<s16>(iter->f[0] * 32.0f);
            frameSize += sizeof(s16);
        }

        size_t padding = m_Chunk[FRAME].size - frameSize;
        memset(pDst, 0, padding);

        pDst = static_cast<s16*>(curve.pFrameArray.ToPtr(&curve.pFrameArray));
        curve.startFrame = static_cast<float>(pDst[0]) * (1.0f / 32.0f);
        curve.endFrame = static_cast<float>(pDst[dstLastIndex-1]) * (1.0f / 32.0f);
    }
    else// FRAME8
    {
        size_t frameSize = 0;
        int dstLastIndex = 0;
        u8* pDst = static_cast<u8*>(curve.pFrameArray.ToPtr(&curve.pFrameArray));
        for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter, ++pDst, ++dstLastIndex)
        {
            float frame = iter->f[0];
            frame = nw::g3d::math::Math::Clamp(frame, -0.5f, 255.4f);

            *pDst = Round<u8>(frame);
            frameSize += sizeof(u8);
        }

        size_t padding = m_Chunk[FRAME].size - frameSize;
        memset(pDst, 0, padding);

        pDst = static_cast<u8*>(curve.pFrameArray.ToPtr(&curve.pFrameArray));
        curve.startFrame = pDst[0];
        curve.endFrame = pDst[dstLastIndex-1];
    }
}


void EditBinAnimCurve::QuantizeHermite(const Context& ctx, nn::g3d::ResAnimCurveData& curve)
{
    curve.flag |= ResAnimCurve::CURVE_CUBIC;
    curve.pKeyArray.Set(GetPtr(ctx, BIN_MAIN, m_Chunk[KEY].offset));

    float coef[4];
    curve.fScale = m_Scale;
    curve.fOffset = m_Offset;

    struct Coef32 { float coef[4]; };
    struct Coef16 { s16 coef[4]; };
    struct Coef8 { s8 coef[4]; };

    void * pDst = curve.pKeyArray.ToPtr(0);
    for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter)
    {
        float f = iter->hermite.frame1 - iter->hermite.frame0;
        float v0 = iter->hermite.value0;
        float v1 = iter->hermite.value1;
        float so = iter->hermite.out_slope0;
        float si = iter->hermite.in_slope1;
        if (m_IsDegreeValue)
        {
            v0 = DegToRad(v0);
            v1 = DegToRad(v1);
            so = DegToRad(so);
            si = DegToRad(si);
        }
        float v = v1 - v0;
        coef[3] = -2 * v + (so + si) * f;
        coef[2] = 3 * v - (2 * so + si) * f;
        coef[1] = so * f;
        coef[0] = v0;

        if (m_KeyType == KEY32)
        {
            Coef32* coefDst = static_cast<Coef32*>(pDst);
            coefDst->coef[3] = coef[3];
            coefDst->coef[2] = coef[2];
            coefDst->coef[1] = coef[1];
            coefDst->coef[0] = coef[0];
            pDst = AddOffset(pDst, sizeof(Coef32));
        }
        else if (m_KeyType == KEY16)
        {
            Coef16* coefDst = static_cast<Coef16*>(pDst);
            coefDst->coef[3] = Round<s16>(coef[3] / curve.fScale);
            coefDst->coef[2] = Round<s16>(coef[2] / curve.fScale);
            coefDst->coef[1] = Round<s16>(coef[1] / curve.fScale);
            coefDst->coef[0] = Round<s16>((coef[0] - curve.fOffset) / curve.fScale);
            pDst = AddOffset(pDst, sizeof(Coef16));
        }
        else// KEY8
        {
            Coef8* coefDst = static_cast<Coef8*>(pDst);
            coefDst->coef[3] = Round<s8>(coef[3] / curve.fScale);
            coefDst->coef[2] = Round<s8>(coef[2] / curve.fScale);
            coefDst->coef[1] = Round<s8>(coef[1] / curve.fScale);
            coefDst->coef[0] = Round<s8>((coef[0] - curve.fOffset) / curve.fScale);
            pDst = AddOffset(pDst, sizeof(Coef8));
        }
    }
}

void EditBinAnimCurve::QuantizeLinear(const Context& ctx, nn::g3d::ResAnimCurveData& curve)
{
    curve.flag |= ResAnimCurve::CURVE_LINEAR;
    curve.pKeyArray.Set(GetPtr(ctx, BIN_MAIN, m_Chunk[KEY].offset));

    float coef[2];
    curve.fScale = m_Scale;
    curve.fOffset = m_Offset;

    struct Coef32 { float coef[2]; };
    struct Coef16 { s16 coef[2]; };
    struct Coef8 { s8 coef[2]; };

    void * pDst = curve.pKeyArray.ToPtr(0);
    for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter)
    {
        float v0 = iter->linear.value0;
        float v1 = iter->linear.value1;
        if (m_IsDegreeValue)
        {
            v0 = DegToRad(v0);
            v1 = DegToRad(v1);
        }
        float v = v1 - v0;
        coef[1] = v;
        coef[0] = v0;


        if (m_KeyType == KEY32)
        {
            Coef32* coefDst = static_cast<Coef32*>(pDst);
            coefDst->coef[1] = coef[1];
            coefDst->coef[0] = coef[0];
            pDst = AddOffset(pDst, sizeof(Coef32));
        }
        else if (m_KeyType == KEY16)
        {
            Coef16* coefDst = static_cast<Coef16*>(pDst);
            coefDst->coef[1] = Round<s16>(coef[1] / curve.fScale);
            coefDst->coef[0] = Round<s16>((coef[0] - curve.fOffset) / curve.fScale);
            pDst = AddOffset(pDst, sizeof(Coef16));
        }
        else // KEY8
        {
            Coef8* coefDst = static_cast<Coef8*>(pDst);
            coefDst->coef[1] = Round<s8>(coef[1] / curve.fScale);
            coefDst->coef[0] = Round<s8>((coef[0] - curve.fOffset) / curve.fScale);
            pDst = AddOffset(pDst, sizeof(Coef8));
        }
    }
}
void EditBinAnimCurve::QuantizeStepFloat(const Context& ctx, nn::g3d::ResAnimCurveData& curve)
{
    // リニア扱い。
    curve.flag |= ResAnimCurve::CURVE_LINEAR;
    curve.pKeyArray.Set(GetPtr(ctx, BIN_MAIN, m_Chunk[KEY].offset));

    float coef[2];
    curve.fScale = m_Scale;
    curve.fOffset = m_Offset;

    struct Coef32 { float coef[2]; };
    struct Coef16 { s16 coef[2]; };
    struct Coef8 { s8 coef[2]; };

    void * pDst = curve.pKeyArray.ToPtr(0);
    for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter)
    {
        coef[1] = 0.0f;
        coef[0] = iter->step.value0;
        if (m_IsDegreeValue)
        {
            coef[0] = DegToRad(coef[0]);
        }

        if (m_KeyType == KEY32)
        {
            Coef32* coefDst = static_cast<Coef32*>(pDst);
            coefDst->coef[1] = coef[1];
            coefDst->coef[0] = coef[0];
            pDst = AddOffset(pDst, sizeof(Coef32));
        }
        else if (m_KeyType == KEY16)
        {
            Coef16* coefDst = static_cast<Coef16*>(pDst);
            coefDst->coef[1] = Round<s16>(coef[1] / curve.fScale);
            coefDst->coef[0] = Round<s16>((coef[0] - curve.fOffset) / curve.fScale);
            pDst = AddOffset(pDst, sizeof(Coef16));
        }
        else // KEY8
        {
            Coef8* coefDst = static_cast<Coef8*>(pDst);
            coefDst->coef[1] = Round<s8>(coef[1] / curve.fScale);
            coefDst->coef[0] = Round<s8>((coef[0] - curve.fOffset) / curve.fScale);
            pDst = AddOffset(pDst, sizeof(Coef8));
        }
    }
}
void EditBinAnimCurve::QuantizeStepInt(const Context& ctx, nn::g3d::ResAnimCurveData& curve)
{
    curve.flag |= ResAnimCurve::CURVE_STEP_INT;
    curve.pKeyArray.Set(GetPtr(ctx, BIN_MAIN, m_Chunk[KEY].offset));
    curve.iScale = 0;
    curve.iOffset = static_cast<int>(m_Offset);

    if (m_KeyType == KEY32)
    {
        s32* pDst = static_cast<s32*>(curve.pKeyArray.ToPtr(0));
        for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter, ++pDst)
        {
            *pDst = Round<s32>(iter->step.value0);
        }
    }
    else if (m_KeyType == KEY16)
    {
        s16* pDst = static_cast<s16*>(curve.pKeyArray.ToPtr(0));
        for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter, ++pDst)
        {
            *pDst = Round<s16>(iter->step.value0 - curve.iOffset);
        }
    }
    else // KEY8
    {
        s8* pDst = static_cast<s8*>(curve.pKeyArray.ToPtr(0));
        for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter, ++pDst)
        {
            *pDst = Round<s8>(iter->step.value0 - curve.iOffset);
        }
    }
}
void EditBinAnimCurve::QuantizeStepBool(const Context& ctx, nn::g3d::ResAnimCurveData& curve)
{
    curve.flag |= ResAnimCurve::CURVE_STEP_BOOL;
    curve.pKeyArray.Set(GetPtr(ctx, BIN_MAIN, m_Chunk[KEY].offset));

    u32* pDst = static_cast<u32*>(curve.pKeyArray.ToPtr(0));
    size_t byteSize = Align(m_FVSPairArray.size(), ALIGNMENT_FLAG) >> 3;
    memset(pDst, 0, byteSize);
    int index = 0;
    for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter, ++index)
    {
        if (iter->step.value0 != 0.0f)
        {
            pDst[index >> 5] |= 0x1 << (index & 0x1F);
        }
    }
    curve.iScale = 1;
    curve.iOffset = 0;
}

}}} // namespace nw::g3d::tool
