﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// g3dhiolib.h

#pragma once

#include "G3dHioLibDefs.h"
#include "Client.h"
#include "CommandWriter.h"
#include "RecvInfo.h"
#include <nn/g3d/viewer/detail/g3d_ViewerPacketDefine.h>

namespace NintendoWare {
namespace G3d {
namespace Edit {

private ref class CommDevice : ICommDevice
{
protected:
    static CommDevice()
    {

    }

    CommDevice(u_short portNumber, RecvInfo^ recvInfo, bool isSwap)
    : m_Client( new nw::g3d::tool::Client(isSwap) )
    , m_PortNumber( portNumber )
    , m_PingPortNumber(0)
    , m_IsConnected( false )
    , m_RecvInfo(recvInfo)
    , m_IsCodePageChanged(false)
    {
        m_RenderInfoPack = gcnew RenderInfoPack();
    }

    static bool EqualsHIO(System::String^ str)
    {
        return str->Equals("HIO");
    }
public:
    ~CommDevice()
    {
        this->!CommDevice();
    }

    !CommDevice()
    {
        if (m_Client != NULL)
        {
            delete m_Client;
            m_Client = NULL;
        }
    }

    static Nintendo::InGameEditing::Communication::ConnectionManager^      m_ConnectionManager;

    virtual event System::ComponentModel::PropertyChangedEventHandler^ PropertyChanged;

    void OnPropertyChanged(System::String^ propertyName)
    {
        PropertyChanged(this, gcnew System::ComponentModel::PropertyChangedEventArgs(propertyName));
    }

    void SetRecvInfo(RecvInfo^ recvInfo)
    {
        m_RecvInfo = recvInfo;
    }

    virtual property bool IsConnected
    {
        bool get()
        {
            return m_IsConnected;
        }

        void set(bool value)
        {
            if (m_IsConnected == value)
            {
                return;
            }

            m_IsConnected = value;

            System::String^ funcName = __FUNCTION__;
            funcName = funcName->Replace("::set", "");
            funcName = funcName->Substring(funcName->LastIndexOf(":")+1);
            this->OnPropertyChanged(funcName);
        }
    }

    virtual property bool IsPingConnected
    {
        bool get()
        {
            return m_IsPingConnected;
        }

        void set(bool value)
        {
            if (m_IsPingConnected == value)
            {
                return;
            }

            m_IsPingConnected = value;

            System::String^ funcName = __FUNCTION__;
            funcName = funcName->Replace("::set", "");
            funcName = funcName->Substring(funcName->LastIndexOf(":")+1);
            this->OnPropertyChanged(funcName);
        }
    }

    virtual property u_short Port
    {
        u_short get()
        {
            return m_PortNumber;
        }

        void set( u_short value )
        {
            m_PortNumber = value;
        }
    }

    virtual property bool IsBeginFreezeReceived
    {
        bool get()
        {
            return m_Client->IsBeginFreezeReceived();
        }
    }

    virtual property bool IsEndFreezeReceived
    {
        bool get()
        {
            return m_Client->IsEndFreezeReceived();
        }
    }

    virtual property bool IsAttachReceived
    {
        bool get()
        {
            return m_Client->IsAttachReceived();
        }
    }

    virtual property bool IsDetachReceived
    {
        bool get()
        {
            return m_Client->IsDetachReceived();
        }
    }

    virtual property bool IsAttachShaderArchiveBinary
    {
        bool get()
        {
            return m_Client->IsAttachShaderArchiveBinary();
        }
    }

    virtual property u32 AttachKind
    {
        u32 get()
        {
            return m_Client->GetAttachKind();
        }
    }

    virtual property u32 ToolKey
    {
        u32 get()
        {
            return m_Client->GetToolKey();
        }
    }

    virtual property u32 ResFileKey
    {
        u32 get()
        {
            return m_Client->GetResFileKey();
        }
    }

    virtual property u32 ResModelKey
    {
        u32 get()
        {
            return m_Client->GetResModelKey();
        }
    }

    virtual property u32 ModelObjKey
    {
        u32 get()
        {
            return m_Client->GetModelObjKey();
        }
    }

    virtual property u32 NewResFileKey
    {
        u32 get()
        {
            return m_Client->GetNewResFileKey();
        }
    }

    virtual property u32 ShaderArchiveKey
    {
        u32 get()
        {
            return m_Client->GetShaderArchiveKey();
        }
    }

    virtual property bool IsFileLoaded
    {
        bool get()
        {
            return m_Client->IsFileLoaded();
        }
    }

    virtual property bool IsModelFileLoaded
    {
        bool get()
        {
            return m_Client->IsModelFileLoaded();
        }
    }

    virtual property bool IsFileReloaded
    {
        bool get()
        {
            return m_Client->IsFileReloaded();
        }
    }

    virtual property bool IsModelFileReloaded
    {
        bool get()
        {
            return m_Client->IsModelFileReloaded();
        }
    }

    virtual property System::String^ FileName
    {
        System::String^ get()
        {
            System::String^ fileName = gcnew System::String(m_Client->GetFileName());
            return fileName;
        }
    }

    virtual property System::String^ AttachPathName
    {
        System::String^ get()
        {
            System::String^ name = gcnew System::String(m_Client->GetAttachPathName());
            return name;
        }
    }

    virtual property bool IsRenderInfoReceived
    {
        bool get()
        {
            return m_Client->IsRenderInfoReceived();
        }
    }

    virtual property RenderInfoPack^ RecvRenderInfoPack
    {
        RenderInfoPack^ get()
        {
            return m_RenderInfoPack;
        }
    }

    virtual property bool IsModelLayoutReceived
    {
        bool get()
        {
            return m_Client->IsModelLayoutReceived();
        }
    }

    virtual property bool IsModifiedShaderProgramReceived
    {
        bool get()
        {
            return m_Client->IsModifiedShaderProgramReceived();
        }
    }

    virtual property bool IsPickupReceived
    {
        bool get()
        {
            return m_Client->IsPickupReceived();
        }
    }

    virtual property bool IsPlayFrameCtrlReceived
    {
        bool get()
        {
            return m_Client->IsPlayFrameCtrlReceived();
        }
    }

    virtual property bool IsStopFrameCtrlReceived
    {
        bool get()
        {
            return m_Client->IsStopFrameCtrlReceived();
        }
    }

    virtual property bool IsSendFrameReceived
    {
        bool get()
        {
            return m_Client->IsSendFrameReceived();
        }
    }

    virtual property bool IsSendFrameStepReceived
    {
        bool get()
        {
            return m_Client->IsSendFrameStepReceived();
        }
    }

    virtual property bool IsSendModelNextAnimReceived
    {
        bool get()
        {
            return m_Client->IsSendModelNextAnimReceived();
        }
    }

    virtual property bool IsSendModelPrevAnimReceived
    {
        bool get()
        {
            return m_Client->IsSendModelPrevAnimReceived();
        }
    }

    virtual property float Frame
    {
        float get()
        {
            return m_Client->GetFrame();
        }
    }

    virtual property float FrameStep
    {
        float get()
        {
            return m_Client->GetFrameStep();
        }
    }

    virtual property bool IsAbnormalPacketReceived
    {
        bool get()
        {
            return m_Client->IsAbnormalPacketReceived();
        }
    }

    virtual property bool IsIncorrectVersionReceived
    {
        bool get()
        {
            return m_Client->IsIncorrectVersionReceived();
        }
    }

    virtual property bool IsPingReceived
    {
        bool get()
        {
            return m_Client->IsPingReceived();
        }
    }

    virtual property bool HasRuntimeErrorReceived
    {
        bool get()
        {
            return m_Client->HasReceivedRuntimeError();
        }
    }

    virtual property bool IsShowMessageRequested
    {
        bool get()
        {
            return m_Client->IsShowMessageRequested();
        }
    }

    virtual property bool IsCodePageChanged
    {
        bool get()
        {
            return m_IsCodePageChanged;
        }
    }

    virtual property System::String^ Message
    {
        System::String^ get()
        {
            System::String^ message = gcnew System::String(
                m_Client->GetMessage(),
                0,
                static_cast<int>(strlen(m_Client->GetMessage())),
                System::Text::Encoding::GetEncoding(m_Client->GetMessageCodePage()));

            return message;
        }
    }

    virtual property NintendoWare::G3d::Edit::MessageType MessageType
    {
        NintendoWare::G3d::Edit::MessageType get()
        {
            return m_Client->GetMessageType();
        }
    }

    virtual property NintendoWare::G3d::Edit::MessageDestination MessageDestination
    {
        NintendoWare::G3d::Edit::MessageDestination get()
        {
            return m_Client->GetMessageDestination();
        }
    }

    virtual property int ShaderModelIndex
    {
        int get()
        {
            return m_Client->GetModifiedShaderProgram()->shadingModelIndex;
        }
    }

    virtual property int ShaderProgramIndex
    {
        int get()
        {
            return m_Client->GetModifiedShaderProgram()->shaderProgramIndex;
        }
    }

    virtual property UINT16 CodePage
    {
        UINT16 get()
        {
            return m_Client->GetCodePage();
        }
    }

    virtual RuntimeError GetRuntimeError()
    {
        return m_Client->GetRuntimeError();
    }

    virtual RuntimeState GetRuntimeState()
    {
        RuntimeState state = RuntimeState::Normal;

        u32 pingState = m_Client->GetPingState();
        if (pingState & nw::g3d::edit::detail::PING_STATE_LOCK)
        {
            state = state | RuntimeState::Lock;
        }
        if (pingState & nw::g3d::edit::detail::PING_STATE_KEEPING)
        {
            state = state | RuntimeState::Keeping;
        }
        if (pingState & nw::g3d::edit::detail::PING_STATE_UNLOCK)
        {
            state = state | RuntimeState::Unlock;
        }
        if (pingState & nn::g3d::viewer::detail::PingState_Closed)
        {
            state = state | RuntimeState::Closed;
        }

        return state;
    }

    virtual PlatformAddressType GetRuntimeAddressType()
    {
        u32 pingFlag = m_Client->GetPingFlag();
        switch (nn::g3d::viewer::detail::GetPlatformAddressType(pingFlag))
        {
        case nn::g3d::viewer::detail::PlatformAddressType_Address32:
            return NintendoWare::G3d::Edit::PlatformAddressType::Adress32;
        case nn::g3d::viewer::detail::PlatformAddressType_Address64:
            return NintendoWare::G3d::Edit::PlatformAddressType::Adress64;
        default:
            return NintendoWare::G3d::Edit::PlatformAddressType::Unknown;
        }
    }

    virtual bool SendData(array<System::Byte>^ buffer)
    {
        return CommandWriter::SendData(m_Client->GetSocket(), buffer);
    }

    virtual void SetupRenderInfoPack()
    {
        m_RenderInfoPack->Items->Clear();

        const nw::g3d::tool::Client::RenderInfo* pRenderInfo = m_Client->GetRenderInfo();
        m_RenderInfoPack->MaterialIndex = pRenderInfo->materialIndex;
        m_RenderInfoPack->ModelObjKey = pRenderInfo->modelKey;

        for (size_t i = 0, end = pRenderInfo->labels.size(); i < end; ++i)
        {
            const nw::g3d::tool::Client::RenderInfoLabel& label = pRenderInfo->labels[i];
            System::String^ labelName = gcnew System::String(label.labelName.c_str());
            switch(label.type)
            {
            case nw::g3d::tool::Client::RENDER_INFO_STRING:
                {
                    size_t itemEnd = label.items.size();
                    System::String^ defaultValue = nullptr;
                    if (itemEnd > 0)
                    {
                        defaultValue = gcnew System::String(label.items[0].choice.c_str());
                    }
                    else
                    {
                        defaultValue = gcnew System::String(System::String::Empty);
                    }

                    StringRenderInfo^ renderInfo = gcnew StringRenderInfo(labelName);
                    for (size_t j = 0; j < itemEnd; ++j)
                    {
                        System::String^ choice = gcnew System::String(label.items[j].choice.c_str());
                        System::Collections::Generic::List<UINT8>^ alias = gcnew System::Collections::Generic::List<UINT8>();
                        if (label.items[j].alias.size() > 0)
                        {
                            array<UINT8>^ aliasBuffer = gcnew array<UINT8>(static_cast<int>(label.items[j].alias.size()));
                            pin_ptr<u8> bufferPtr = &aliasBuffer[0];
                            std::memcpy(bufferPtr, &label.items[j].alias[0], label.items[j].alias.size());
                            bufferPtr = nullptr;
                            alias->AddRange(aliasBuffer);
                        }
                        StringRenderInfo::Item^ item = gcnew StringRenderInfo::Item(choice, alias->ToArray());
                        renderInfo->AddItem(item);
                    }

                    // スロットの登録
                    {
                        for (size_t i = 0, end = label.values.size(); i < end; ++i)
                        {
                            System::String^ stringValue = gcnew System::String(label.values[i].sDefault.c_str());
                            StringRenderInfo::Value^ value = gcnew StringRenderInfo::Value(stringValue);
                            renderInfo->AddValue(value);
                        }
                    }
                    m_RenderInfoPack->Items->Add(renderInfo);
                }
                break;
            case nw::g3d::tool::Client::RENDER_INFO_INT:
                {
                    IntRenderInfo^ renderInfo = gcnew IntRenderInfo(labelName, label.iMinValue, label.iMaxValue);

                    // スロットの登録
                    {
                        for (size_t i = 0, end = label.values.size(); i < end; ++i)
                        {
                            IntRenderInfo::Value^ value = gcnew IntRenderInfo::Value(label.values[i].iDefault);
                            renderInfo->AddValue(value);
                        }
                    }
                    m_RenderInfoPack->Items->Add(renderInfo);
                }
                break;
            case nw::g3d::tool::Client::RENDER_INFO_FLOAT:
                {
                    FloatRenderInfo^ renderInfo = gcnew FloatRenderInfo(labelName, label.fMinValue, label.fMaxValue);

                    // スロットの登録
                    {
                        for (size_t i = 0, end = label.values.size(); i < end; ++i)
                        {
                            FloatRenderInfo::Value^ value = gcnew FloatRenderInfo::Value(label.values[i].fDefault);
                            renderInfo->AddValue(value);
                        }
                    }
                    m_RenderInfoPack->Items->Add(renderInfo);
                }
                break;
            }
        }
    }

    virtual ModifiedShaderProgramRecvData^ CreateModifiedShaderProgramData()
    {
        const nw::g3d::tool::Client::ModifiedShaderProgram* shaderProgram = m_Client->GetModifiedShaderProgram();
        ModifiedShaderProgramRecvData^ command = gcnew ModifiedShaderProgramRecvData(
            shaderProgram->shaderArchiveKey,
            shaderProgram->shadingModelIndex,
            shaderProgram->shaderProgramIndex);

        size_t infoCount = shaderProgram->optionInfos.size();
        for (size_t i = 0; i < infoCount; ++i)
        {
            const nw::g3d::tool::Client::ModifiedShaderProgramOptionInfo& info =
                shaderProgram->optionInfos[i];

            System::String^ option = gcnew System::String(info.option.c_str());
            System::String^ choice = gcnew System::String(info.choice.c_str());

            command->AddShaderCompileInfo(option, choice);
        }
        return command;
    }

    virtual PickupRecvData^ CreatePickupData()
    {
        const nw::g3d::tool::Client::Pickup* pickup = m_Client->GetPickup();
        PickupRecvData^ command = gcnew PickupRecvData();

        size_t infoCount = pickup->materialPickups.size();
        for (size_t i = 0; i < infoCount; ++i)
        {
            const nw::g3d::tool::Client::MaterialPickup& materialPickupInfo =
                pickup->materialPickups[i];
            command->AddMaterialSelectionInfo(materialPickupInfo.modelObjKey, materialPickupInfo.materialIndex);
        }
        return command;
    }

    virtual HtcResult^ OpenConnection()
    {
        return m_Client->Open();
    }

    virtual HtcResult^ Connect()
    {
        HtcResult^ result = gcnew HtcResult();
        result->ResultCode = HtcResultCode::NotImplemented;
        return result;
    }

    virtual void CloseConnection()
    {
        IsConnected = false;
        m_Client->Close();
    }

    virtual void Poll()
    {
        if (!m_IsConnected)
        {
            return;
        }
        m_Client->Poll();
    }

    virtual HtcResult^ OpenPingConnection()
    {
        return m_Client->OpenPing();
    }

    virtual HtcResult^ ConnectPing()
    {
        HtcResult^ result = gcnew HtcResult();
        result->ResultCode = HtcResultCode::NotImplemented;
        return result;
    }

    virtual void ClosePingConnection()
    {
        IsPingConnected = false;
        m_Client->ClosePing();
    }

    virtual void PollPing()
    {
        if (!m_IsPingConnected)
        {
            return;
        }
        m_Client->PollPing();
        IsPingConnected = m_Client->IsPingConnected();
        if (IsPingConnected)
        {
            if (m_RecvInfo->CodePage != m_Client->GetCodePage())
            {
                // コードページ変更コールバックを PollPing で呼ぶと
                // 3DEditor でデッドロックの危険性があるので Poll で呼ぶ(SIGLO-24373)
                m_IsCodePageChanged = true;
                m_RecvInfo->CodePage = m_Client->GetCodePage();
            }
        }
        if (m_Client->IsIncorrectVersionReceived())
        {
            m_RecvInfo->SetRuntimeVersion(m_Client->GetLastIncorrectVersion());
        }
    }


    virtual void ResetBeginFreezeFlag()
    {
        m_Client->ResetBeginFreezeFlag();
    }

    virtual void ResetEndFreezeFlag()
    {
        m_Client->ResetEndFreezeFlag();
    }

    virtual void ResetAttachFlag()
    {
        m_Client->ResetAttachFlag();
    }

    virtual void ResetDetachFlag()
    {
        m_Client->ResetDetachFlag();
    }

    virtual void ResetFileLoadFlag()
    {
        m_Client->ResetFileLoadFlag();
    }

    virtual void ResetRenderInfoFlag()
    {
        m_Client->ResetRenderInfoFlag();
    }

    virtual void ResetModelLayoutFlag()
    {
        m_Client->ResetModelLayoutFlag();
    }

    virtual void ResetModifiedShaderProgramFlag()
    {
        m_Client->ResetModifiedShaderProgramFlag();
    }

    virtual void ResetPickupFlag()
    {
        m_Client->ResetPickupFlag();
    }

    virtual void ResetPlayFrameCtrlFlag()
    {
        m_Client->ResetPlayFrameCtrlFlag();
    }

    virtual void ResetStopFrameCtrlFlag()
    {
        m_Client->ResetStopFrameCtrlFlag();
    }

    virtual void ResetSendFrameFlag()
    {
        m_Client->ResetSendFrameFlag();
    }

    virtual void ResetSendFrameStepFlag()
    {
        m_Client->ResetSendFrameStepFlag();
    }

    virtual void ResetSendModelNextAnimFlag()
    {
        m_Client->ResetSendModelNextAnimFlag();
    }

    virtual void ResetSendModelPrevAnimFlag()
    {
        m_Client->ResetSendModelPrevAnimFlag();
    }

    virtual void ResetAbnormalPacketFlag()
    {
        m_Client->ResetAbnormalPacketFlag();
    }

    virtual void ResetIncorrectVersionFlag()
    {
        m_Client->ResetIncorrectVersionFlag();
    }

    virtual void ResetRuntimeErrorFlag()
    {
        m_Client->ResetRuntimeErrorFlag();
    }

    virtual void ResetShowMessageRequestedFlag()
    {
        m_Client->ResetShowMessageRequestedFlag();
    }

    virtual Math::Vector3^ GetModelLayoutScale()
    {
        return gcnew Math::Vector3(
            m_Client->GetModelLayout()->scale.x,
            m_Client->GetModelLayout()->scale.y,
            m_Client->GetModelLayout()->scale.z);
    }

    virtual Math::Vector3^ GetModelLayoutTranslate()
    {
        return gcnew Math::Vector3(
            m_Client->GetModelLayout()->translate.x,
            m_Client->GetModelLayout()->translate.y,
            m_Client->GetModelLayout()->translate.z);
    }

    virtual Math::Vector3^ GetModelLayoutRotate()
    {
        return gcnew Math::Vector3(
            m_Client->GetModelLayout()->rotate.x,
            m_Client->GetModelLayout()->rotate.y,
            m_Client->GetModelLayout()->rotate.z);
    }

    virtual void ResetCodePageChangedFlag()
    {
        m_IsCodePageChanged = false;
    }

protected:
    nw::g3d::tool::Client* m_Client;
    u_short                 m_PortNumber;
    u_short                 m_PingPortNumber;
    bool                    m_IsConnected;
    bool                    m_IsPingConnected;
    RenderInfoPack^         m_RenderInfoPack;
    RecvInfo^               m_RecvInfo;
    bool m_IsCodePageChanged;

    static System::Net::IPEndPoint^ ResolvePortMapping(System::String^ peerType, System::String^ htcsPeerName, System::String^ portName)
    {
        if (m_ConnectionManager == nullptr)
        {
            m_ConnectionManager = gcnew Nintendo::InGameEditing::Communication::ConnectionManager();
        }

        try
        {
            m_ConnectionManager->Start(gcnew System::String("HTCS"), -1);
            // retry の最大が 10 ぐらいだとつながりにくいらしい
            for (int retry = 0; retry < 30; retry++)
            {
                // TODO: IsAvailable のプロトコル名に HTCS を指定するべきか？
                if (NintendoWare::G3d::Edit::CommDevice::m_ConnectionManager->IsAvailable(System::String::Empty))
                {
                    auto targetInfos = NintendoWare::G3d::Edit::CommDevice::m_ConnectionManager->TargetInfos;
                    for (int i= 0; i< targetInfos->Count; i++)
                    {
                        Nintendo::InGameEditing::Communication::TargetInfo^ info = targetInfos[i];
                        if (info->PeerType->Equals(peerType)
                            // TODO: PeerNameを3DEditorから設定できるようにするときに条件を整理する
                            && ( System::String::IsNullOrEmpty(htcsPeerName) || info->HtcsPeerName->Equals(htcsPeerName)) )
                        {
                            for (int j=0;j < info->PortInfos->Count; j++)
                            {
                                Nintendo::InGameEditing::Communication::PortInfo^ portInfo = info->PortInfos[j];
                                if (portInfo->PortName->Equals(portName))
                                {
                                    return portInfo->EndPoint;
                                }
                            }
                        }
                    }
                }

                // 短すぎるとダメな場合があるらしい
                System::Threading::Thread::Sleep(100);
            }
        }
        finally
        {
            m_ConnectionManager->Stop(gcnew System::String("HTCS"));
        }

        return nullptr;
    }
};

}}} // namespace NintendoWare.G3d.Edit
