﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "stdafx.h"

#include "msclr/marshal.h"
#include "msclr/marshal_windows.h"
#include "msclr/marshal_cppstd.h"
#include "msclr/marshal_atl.h"
#include "HIO.h"

#include "AnimCommandMaker.h"
#include "CommandUtility.h"
#include "Utility.h"
#include <nw/g3d/res/g3d_ResUtility.h>

using namespace System;
using namespace msclr::interop;
using namespace nw::g3d::edit::detail;
using namespace NintendoWare::G3d::Edit;

namespace {

u32 GetBindAnimInfoSize()
{
    return sizeof(BindAnimInfo);
}

u32 GetAnimCurveInfoSize()
{
    return sizeof(AnimCurveInfo);
}

u32 GetAnimCurveValue()
{
    return sizeof(AnimCurveValue);
}

u32 GetFrameCtrlBlockSize()
{
    return sizeof(nn::g3d::viewer::detail::FrameCtrlBlock);
}

u32 GetAnimEditInfoBlcokSize()
{
    return sizeof(AnimEditInfoBlock);
}

}

namespace NintendoWare { namespace G3d { namespace Edit {

bool AnimCommandMaker::CreateAnimCurve(nw::g3d::tool::EditBinAnimCurve* animCurve, UINT32 targetOffset, ICurve^ curve)
{
    if (animCurve == NULL || curve == nullptr)
    {
        return false;
    }

    animCurve->SetTargetOffset(targetOffset);
    animCurve->SetDegreeValue(curve->IsDegreeValue);
    animCurve->SetScale(curve->Scale);
    animCurve->SetOffset(curve->Offset);

    switch(curve->InterpolationType)
    {
    case InterpolationType::Hermite:
        animCurve->SetInterpolationType(nw::g3d::tool::EditBinAnimCurve::HERMITE);
        break;
    case InterpolationType::Linear:
        animCurve->SetInterpolationType(nw::g3d::tool::EditBinAnimCurve::LINEAR);
        break;
    case InterpolationType::Step:
        animCurve->SetInterpolationType(nw::g3d::tool::EditBinAnimCurve::STEP);
        break;
    }

    switch(curve->KeyType)
    {
    case KeyType::Key32:
        animCurve->SetKeyType(nw::g3d::tool::EditBinAnimCurve::KEY32);
        break;
    case KeyType::Key16:
        animCurve->SetKeyType(nw::g3d::tool::EditBinAnimCurve::KEY16);
        break;
    case KeyType::Key8:
        animCurve->SetKeyType(nw::g3d::tool::EditBinAnimCurve::KEY8);
        break;
    }

    switch(curve->FrameType)
    {
    case FrameType::Frame32:
        animCurve->SetFrameType(nw::g3d::tool::EditBinAnimCurve::FRAME32);
        break;
    case FrameType::Frame16:
        animCurve->SetFrameType(nw::g3d::tool::EditBinAnimCurve::FRAME16);
        break;
    case FrameType::Frame8:
        animCurve->SetFrameType(nw::g3d::tool::EditBinAnimCurve::FRAME8);
        break;
    }

    switch(curve->PreWrap)
    {
    case WrapType::Clamp:
        animCurve->SetPreWrapType(nw::g3d::tool::EditBinAnimCurve::Clamp);
        break;
    case WrapType::Repeat:
        animCurve->SetPreWrapType(nw::g3d::tool::EditBinAnimCurve::Repeat);
        break;
    case WrapType::Mirror:
        animCurve->SetPreWrapType(nw::g3d::tool::EditBinAnimCurve::Mirror);
        break;
    case WrapType::RelativeRepeat:
        animCurve->SetPreWrapType(nw::g3d::tool::EditBinAnimCurve::RelativeRepeat);
        break;
    }

    switch(curve->PostWrap)
    {
    case WrapType::Clamp:
        animCurve->SetPostWrapType(nw::g3d::tool::EditBinAnimCurve::Clamp);
        break;
    case WrapType::Repeat:
        animCurve->SetPostWrapType(nw::g3d::tool::EditBinAnimCurve::Repeat);
        break;
    case WrapType::Mirror:
        animCurve->SetPostWrapType(nw::g3d::tool::EditBinAnimCurve::Mirror);
        break;
    case WrapType::RelativeRepeat:
        animCurve->SetPostWrapType(nw::g3d::tool::EditBinAnimCurve::RelativeRepeat);
        break;
    }

    switch(curve->ValueType)
    {
    case ValueType::Float:
        animCurve->SetType(nw::g3d::tool::EditBinAnimCurve::FLOAT);
        break;
    case ValueType::Int:
        animCurve->SetType(nw::g3d::tool::EditBinAnimCurve::INT);
        animCurve->SetInterpolationType(nw::g3d::tool::EditBinAnimCurve::STEP);
        break;
    case ValueType::Bool:
        animCurve->SetType(nw::g3d::tool::EditBinAnimCurve::BOOL);
        animCurve->SetInterpolationType(nw::g3d::tool::EditBinAnimCurve::STEP);
        break;
    }

    array<ICurveValue^>^ curveValues = curve->ValueArray;
    for each(ICurveValue^ curveValue in curveValues)
    {
        animCurve->AddCurveValue(curveValue->Frame, curveValue->Value, curveValue->InSlope, curveValue->OutSlope);
    }

    animCurve->Build();
    animCurve->CalcSize();
    animCurve->SetBlockOffset(nw::g3d::tool::EditBinaryBlock::BIN_MAIN, 0);

    return true;
}

int AnimCommandMaker::CalcBindAnimationsPacketSize(array<UINT32>^ animationKeys)
{
    if (animationKeys == nullptr)
    {
        return -1;
    }

    size_t animationKeyCount = static_cast<u32>(animationKeys->Length);
    size_t animationKeySize = animationKeyCount * sizeof(u32);

    size_t totalSize =
        nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + GetBindAnimInfoSize() + animationKeySize, NW_G3D_EDIT_ALIGNMENT);
    return static_cast<int>(totalSize);
}

int AnimCommandMaker::CalcEditAnimCurvePacketSize(nw::g3d::tool::EditBinAnimCurve* animCurve)
{
    if (animCurve == NULL)
    {
        return -1;
    }

    size_t resAnimCurveDataSize = ResAbstractionUtility::GetResAnimCurveSize();
    size_t curveDataSize = animCurve->GetBlockSize(nw::g3d::tool::EditBinaryBlock::BIN_MAIN);
    size_t curveOffset = CommandUtility::GetPacketHeaderSize() + GetAnimCurveInfoSize() + GetAnimCurveValue();

    size_t totalSize =
        nw::g3d::tool::Align(curveOffset + resAnimCurveDataSize + curveDataSize, NW_G3D_EDIT_ALIGNMENT);
    return static_cast<int>(totalSize);
}

int AnimCommandMaker::CalcEditFrameCtrlPacketSize()
{
    return static_cast<int>(nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + GetFrameCtrlBlockSize(), NW_G3D_EDIT_ALIGNMENT));
}

int AnimCommandMaker::CalcAnimEditInfoPacketSize()
{
    return static_cast<int>(nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + GetAnimEditInfoBlcokSize(), NW_G3D_EDIT_ALIGNMENT));
}

bool AnimCommandMaker::MakeBindAnimationsPacket(
    array<Byte>^ packetBuffer,
    CommandFlag command,
    UINT32 modelKey,
    array<UINT32>^ animationKeys,
    bool isAttachModel,
    TargetEndianKind endianKind)
{
    if (packetBuffer == nullptr)
    {
        return false;
    }

    if (animationKeys == nullptr)
    {
        return false;
    }

    size_t totalSize = CalcBindAnimationsPacketSize(animationKeys);

    if (packetBuffer->Length != totalSize)
    {
        return false;
    }

    pin_ptr<u8> bufferPtr = &packetBuffer[0];

    BindAnimPacket* packet = reinterpret_cast<BindAnimPacket*>(bufferPtr);
    packet->header = CommandUtility::GetPacketHeader();
    packet->header.command = command;
    packet->header.dataSize = static_cast<u32>(totalSize - CommandUtility::GetPacketHeaderSize());

    packet->block.info.modelKey = modelKey;
    packet->block.info.animationKeySize = animationKeys->Length;
    packet->block.info.attachModelFlag = static_cast<u32>(isAttachModel);

    for (u32 i = 0, end = packet->block.info.animationKeySize; i < end; ++i)
    {
        packet->block.animationKeys[i] = animationKeys[i];
    }

    if (endianKind == TargetEndianKind::BigEndian)
    {
        u32 keySize = packet->block.info.animationKeySize;
        nw::g3d::tool::Endian::Swap(packet);
        nw::g3d::tool::Endian::SwapBindAnimationKeys(&packet->block, keySize);
    }

    // pin_ptr 解除
    bufferPtr = nullptr;
    return true;
}

void DebugLogCurveKeysAndFrames(void* resAnimCurveData)
{

    // デバッグ用
    if (NintendoWare::G3d::Edit::HIO::GetInstance()->TargetDeviceType == HIOBase::TargetType::Htc)
    {
        nn::g3d::ResAnimCurveData* curve = static_cast<nn::g3d::ResAnimCurveData*>(resAnimCurveData);
        {
            const float* pKeyFloat = static_cast<float*>(curve->pKeyArray.ToPtr(&curve->pKeyArray));
            //const float *pKeyFloat = static_cast<float*>(pKey);
            const float* end = pKeyFloat + curve->keyCount;

            System::Diagnostics::Debug::Write("key offset: ");
            System::Diagnostics::Debug::WriteLine(curve->pKeyArray.GetOffset().ToString());

            for (int i = 0; pKeyFloat != end; ++pKeyFloat, ++i)
            {
                float value = *pKeyFloat;
                System::Diagnostics::Debug::Write("key[");
                System::Diagnostics::Debug::Write(i.ToString());
                System::Diagnostics::Debug::Write("]: ");
                System::Diagnostics::Debug::WriteLine(value.ToString());
            }
        }

        {
            const u8* pData = static_cast<u8*>(curve->pFrameArray.ToPtr(&curve->pFrameArray));
            //const float *pKeyFloat = static_cast<float*>(pKey);
            const u8* end = pData + curve->keyCount;
            for (int i = 0; pData != end; ++pData, ++i)
            {
                u8 value = *pData;
                System::Diagnostics::Debug::Write("frame[");
                System::Diagnostics::Debug::Write(i.ToString());
                System::Diagnostics::Debug::Write("]: ");
                System::Diagnostics::Debug::WriteLine(value.ToString());
            }
        }
    }
    else
    {
        nw::g3d::res::ResAnimCurveData* curve = static_cast<nw::g3d::res::ResAnimCurveData*>(resAnimCurveData);
        {
            const float* pKeyFloat = static_cast<float*>(curve->ofsKeyArray.to_ptr());
            //const float *pKeyFloat = static_cast<float*>(pKey);
            const float* end = pKeyFloat + curve->numKey;
            for (int i = 0; pKeyFloat != end; ++pKeyFloat, ++i)
            {
                float value = *pKeyFloat;
                System::Diagnostics::Debug::Write("key[");
                System::Diagnostics::Debug::Write(i.ToString());
                System::Diagnostics::Debug::Write("]: ");
                System::Diagnostics::Debug::WriteLine(value.ToString());
            }
        }

        {
            const u8* pData = static_cast<u8*>(curve->ofsFrameArray.to_ptr());
            //const float *pKeyFloat = static_cast<float*>(pKey);
            const u8* end = pData + curve->numKey;
            for (int i = 0; pData != end; ++pData, ++i)
            {
                u8 value = *pData;
                System::Diagnostics::Debug::Write("frame[");
                System::Diagnostics::Debug::Write(i.ToString());
                System::Diagnostics::Debug::Write("]: ");
                System::Diagnostics::Debug::WriteLine(value.ToString());
            }
        }
    }
}

bool AnimCommandMaker::MakeEditAnimCurvePacket(
    array<Byte>^ packetBuffer,
    nw::g3d::tool::EditBinAnimCurve* animCurve,
    CommandFlag command,
    EditTargetKind editTargetKind,
    UINT32 animationKey,
    UINT32 index,
    INT32 curveIndex,
    TargetEndianKind endianKind)
{
    if (packetBuffer == nullptr)
    {
        return false;
    }

    if (animCurve == NULL)
    {
        return false;
    }

    size_t resAnimCurveDataSize = ResAbstractionUtility::GetResAnimCurveSize();
    nw::g3d::tool::EditBinaryBlock::Context ctx;
    ctx.chunk[nw::g3d::tool::EditBinaryBlock::BIN_MAIN].offset = resAnimCurveDataSize;
    size_t curveDataSize = animCurve->GetBlockSize(nw::g3d::tool::EditBinaryBlock::BIN_MAIN);
    size_t curveOffset = CommandUtility::GetPacketHeaderSize() + GetAnimCurveInfoSize() + GetAnimCurveValue();

    size_t totalSize =
        nw::g3d::tool::Align(curveOffset + resAnimCurveDataSize + curveDataSize, NW_G3D_EDIT_ALIGNMENT);

    if (packetBuffer->Length != totalSize)
    {
        return false;
    }

    pin_ptr<u8> bufferPtr = &packetBuffer[0];

    void* resAnimCurveData = &bufferPtr[curveOffset];

    ctx.pBuf = resAnimCurveData;
    animCurve->Convert(ctx);

    AnimCurvePacket* packet = reinterpret_cast<AnimCurvePacket*>(bufferPtr);
    packet->header = CommandUtility::GetPacketHeader();
    packet->header.command = command;
    packet->header.dataSize = static_cast<s32>(totalSize - CommandUtility::GetPacketHeaderSize());

    packet->block.info.animationKey = animationKey;
    packet->block.info.animationKind = editTargetKind;
    packet->block.info.padding = 0;

    packet->block.value.index = static_cast<s32>(index);
    packet->block.value.curveIndex = static_cast<s32>(curveIndex);
    packet->block.value.curveSize = static_cast<u32>(resAnimCurveDataSize + curveDataSize);

    if (endianKind == TargetEndianKind::BigEndian)
    {
        nw::g3d::tool::Endian::Swap(packet);
        nw::g3d::tool::Endian::SwapResAnimCurveData(resAnimCurveData);
    }

    // TODO: nn::g3dが安定するまでのデバッグ用なので、安定したら消す
    // DebugLogCurveKeysAndFrames(resAnimCurveData);

    // pin_ptr 解除
    bufferPtr = nullptr;
    return true;
}

bool AnimCommandMaker::MakeEditFrameCtrlPacket(
    array<Byte>^ packetBuffer,
    nn::g3d::viewer::detail::CommandFlag command,
    float frame,
    float frameStep,
    nn::g3d::viewer::detail::EditPlayPolicyKind kind,
    float startFrame,
    float frameCount,
    TargetEndianKind endianKind)
{
    if (packetBuffer == nullptr)
    {
        return false;
    }

    if (command != nn::g3d::viewer::detail::ANIMATION_FRAME_STEP_COMMAND_FLAG &&
        command != nn::g3d::viewer::detail::ANIMATION_PLAY_FRAME_CTRL_COMMAND_FLAG &&
        command != nn::g3d::viewer::detail::ANIMATION_STOP_FRAME_CTRL_COMMAND_FLAG &&
        command != nn::g3d::viewer::detail::ANIMATION_PLAY_POLICY_COMMAND_FLAG &&
        command != nn::g3d::viewer::detail::ANIMATION_FRAME_COUNT_COMMAND_FLAG &&
        command != nn::g3d::viewer::detail::ANIMATION_START_FRAME_COMMAND_FLAG)
    {
        return false;
    }

    size_t totalSize =
        nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + GetFrameCtrlBlockSize(), NW_G3D_EDIT_ALIGNMENT);

    if (packetBuffer->Length != totalSize)
    {
        return false;
    }

    pin_ptr<u8> bufferPtr = &packetBuffer[0];

    nn::g3d::viewer::detail::FrameCtrlPacket* packet = reinterpret_cast<nn::g3d::viewer::detail::FrameCtrlPacket*>(bufferPtr);
    packet->header = CommandUtility::GetNnG3dViewerPacketHeader();
    packet->header.command = command;
    packet->header.dataSize = static_cast<s32>(totalSize - CommandUtility::GetPacketHeaderSize());

    packet->block.frame = frame;
    packet->block.frameStep = frameStep;
    packet->block.playPolicy = kind;
    packet->block.frameCount = frameCount;
    packet->block.startFrame = startFrame;

    if (endianKind == TargetEndianKind::BigEndian)
    {
        nw::g3d::tool::Endian::Swap(packet);
    }
    return true;
}

bool AnimCommandMaker::MakeAnimEditInfoPacket(
        array<Byte>^ packetBuffer,
        UINT32 modelKey,
        UINT32 animationKey,
        nw::g3d::edit::detail::CommandFlag command,
        float value,
        TargetEndianKind endianKind)
{
    if (packetBuffer == nullptr)
    {
        return false;
    }

    if (command != MODEL_ANIMATION_PLAY_COMMAND_FLAG &&
        command != MODEL_ANIMATION_STOP_COMMAND_FLAG &&
        command != MODEL_ANIMATION_EDIT_RETARGET_HOST_MODEL_COMMAND_FLAG &&
        command != MODEL_ANIMATION_EDIT_MIRRORING_ENABLED_COMMAND_FLAG)
    {
        return false;
    }
    size_t totalSize = CalcAnimEditInfoPacketSize();
    if (packetBuffer->Length != totalSize)
    {
        return false;
    }

    pin_ptr<u8> bufferPtr = &packetBuffer[0];

    AnimEditInfoPacket* packet = reinterpret_cast<AnimEditInfoPacket*>(bufferPtr);
    packet->header = CommandUtility::GetPacketHeader();
    packet->header.command = command;
    packet->header.dataSize = static_cast<s32>(totalSize - CommandUtility::GetPacketHeaderSize());

    packet->block.animationKey = animationKey;
    packet->block.fValue = value;
    packet->block.modelKey = modelKey;
    packet->block.padding[1] = 0;

    if (endianKind == TargetEndianKind::BigEndian)
    {
        nw::g3d::tool::Endian::Swap(packet);
    }

    return true;
}

}}} // namespace NintendoWare.G3d.Edit
