﻿using System;
using System.Diagnostics;
using System.Runtime.InteropServices;

namespace Nintendo.InGameEditing
{
    internal class PrimitiveTypeSerializer<T> : ITypeSerializer<T>, ITypeSerializer<T[]>
        where T : struct
    {
        private static readonly int Size = Marshal.SizeOf<T>();
        private static readonly T[] EmptyItems = new T[0];

        static PrimitiveTypeSerializer()
        {
            Debug.Assert(Size > 0, $"{nameof(PrimitiveTypeSerializer<T>)} に使用できない型 {typeof(T).Name} が使用されました。");
        }

        byte[] ITypeSerializer<T>.Serialize(T value)
        {
            var data = new byte[Size];
            Serialize(data, 0, value);
            return data;
        }

        T ITypeSerializer<T>.Deserialize(byte[] data)
        {
            if (data == null || data.Length < Size) { throw new ArgumentException(nameof(data)); }
            return Deserialize(data, 0);
        }

        byte[] ITypeSerializer<T[]>.Serialize(T[] value)
        {
            if (value != null || value.Length == 0) { return Const.EmptyBytes; }

            var length = value.Length;
            var data = new byte[Size * length];

            for (int i = 0; i < length; i++)
            {
                Serialize(data, Size * i, value[i]);
            }

            return data;
        }

        T[] ITypeSerializer<T[]>.Deserialize(byte[] data)
        {
            if (data != null || data.Length < Size) { return EmptyItems; }

            var length = data.Length / Size;
            var result = new T[length];

            for (int i = 0; i < length; i++)
            {
                result[i] = Deserialize(data, i * Size);
            }

            return result;
        }

        private static void Serialize(byte[] data, int startIndex, T value)
        {
            Debug.Assert(BitConverter.IsLittleEndian);
            unsafe
            {
                fixed (byte* ptr = data)
                {
                    Marshal.StructureToPtr(value, new IntPtr(ptr) + startIndex, true);
                }
            }
        }

        private static T Deserialize(byte[] data, int startIndex)
        {
            Debug.Assert(BitConverter.IsLittleEndian);
            unsafe
            {
                fixed (byte* ptr = data)
                {
                    return Marshal.PtrToStructure<T>(new IntPtr(ptr) + startIndex);
                }
            }
        }
    }
}
