﻿using System;
using System.IO;
using System.Reactive.Concurrency;
using System.Reactive.Disposables;
using System.Reactive.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace Nintendo.InGameEditing.Communication
{
    internal static class PacketHelper
    {
        public static async Task WriteAndFlushAsync(this Stream stream, Packet packet)
        {
            if (stream == null) { throw new ArgumentNullException(nameof(stream)); }

            if (packet.Signature.HasValue)
            {
                var signatureBuffer = BitConverter.GetBytes(packet.Signature.Value);
                await stream.WriteAsync(signatureBuffer, 0, signatureBuffer.Length).ConfigureAwait(false);
            }

            var lengthBuffer = BitConverter.GetBytes(packet.Payload.Length);
            await stream.WriteAsync(lengthBuffer, 0, lengthBuffer.Length).ConfigureAwait(false);
            await stream.WriteAsync(packet.Payload, 0, packet.Payload.Length);
            await stream.FlushAsync();
        }

        public static void WriteAndFlush(this Stream stream, Packet packet)
        {
            if (stream == null) { throw new ArgumentNullException(nameof(stream)); }

            if (packet.Signature.HasValue)
            {
                var signatureBuffer = BitConverter.GetBytes(packet.Signature.Value);
                stream.Write(signatureBuffer, 0, signatureBuffer.Length);
            }

            var lengthBuffer = BitConverter.GetBytes(packet.Payload.Length);
            stream.Write(lengthBuffer, 0, lengthBuffer.Length);

            stream.Write(packet.Payload, 0, packet.Payload.Length);
            stream.Flush();
        }

        /// <summary>
        /// Subscribe 時に Stream から Packet の読み取りを開始し、戻り値の Dispose で読み取りを停止する IObservable を作成します。
        /// Stream が破棄された場合、または戻り値が Dispose された場合に OnCompleted を発行します。
        /// シグネチャのチェックに失敗した場合、またはタイムアウトなど何らかの問題で読み取りに失敗した場合、OnError を発行します。
        /// また、Dispose の実行に関わらず Subscribe は一度しか行うことができません。
        /// IObservable.Publish 等との併用を検討してください。
        /// </summary>
        /// <param name="stream">Packet の読み取りを行う Stream</param>
        /// <param name="signature">読み取り時にチェックを行うシグネチャ。null を指定した場合、チェックを行いません。</param>
        /// <param name="scheduler">Packet の読み取り操作を実行するスレッド。null の場合、Scheduler.Default を使用します。</param>
        /// <returns>読み取られた Packet が発行される IObservable。</returns>
        public static IObservable<Packet> ToPacketObservable(this Stream stream, int? signature = null, IScheduler scheduler = null)
        {
            if (stream == null) { throw new ArgumentNullException(nameof(stream)); }
            scheduler = scheduler ?? Scheduler.Default;
            int subscribed = 0;

            return Observable.Create<Packet>(observer =>
            {
                var isSubscribed = Interlocked.CompareExchange(ref subscribed, 1, 0) != 0;
                if (isSubscribed)
                {
                    observer.OnError(new InvalidOperationException("Only one subscriber is allowed for each stream."));
                    return Disposable.Empty;
                }

                var schedule = scheduler.ScheduleAsync(async (ctrl, ct) =>
                {
                    Func<bool> isAvailable = () => !ct.IsCancellationRequested && stream.CanRead;
                    var buffer4 = new byte[4];

                    while (isAvailable())
                    {
                        // signature
                        if (signature.HasValue)
                        {
                            try
                            {
                                await ReadBytesAsync(stream, buffer4, ct);
                            }
                            catch (OperationCanceledException) { break; }
                            catch (ObjectDisposedException) { break; }
                            catch (Exception e)
                            {
                                observer.OnError(new InvalidDataException("Error in stream.", e));
                                return;
                            }

                            if (!isAvailable()) { break; }

                            var recievedSignature = BitConverter.ToInt32(buffer4, 0);
                            if (signature.Value != recievedSignature)
                            {
                                observer.OnError(new InvalidDataException($"Error in stream. Invalid Signature '{recievedSignature}', expected '{signature}'"));
                                return;
                            }
                        }

                        int bodyLength;

                        // length
                        {
                            try
                            {
                                await ReadBytesAsync(stream, buffer4, ct);
                            }
                            catch (OperationCanceledException) { break; }
                            catch (ObjectDisposedException) { break; }
                            catch (Exception e)
                            {
                                observer.OnError(new InvalidDataException("Error in stream.", e));
                                return;
                            }

                            if (!isAvailable()) { break; }

                            bodyLength = BitConverter.ToInt32(buffer4, 0);

                            if (bodyLength < 0) { continue; }
                        }

                        // body
                        {
                            var payload = new byte[bodyLength];

                            if (bodyLength > 0)
                            {
                                try
                                {
                                    await ReadBytesAsync(stream, payload, ct);
                                }
                                catch (OperationCanceledException) { break; }
                                catch (ObjectDisposedException) { break; }
                                catch (Exception e)
                                {
                                    observer.OnError(new InvalidDataException("Error in stream.", e));
                                    return;
                                }
                            }

                            observer.OnNext(new Packet(signature, payload));
                        }
                    }

                    observer.OnCompleted();
                });

                return schedule;
            });
        }

        private static async Task ReadBytesAsync(Stream stream, byte[] buffer, CancellationToken cancellationTolen)
        {
            var totalBytes = 0;
            var totalRemainingBytes = buffer.Length;

            while (totalRemainingBytes != 0)
            {
                var readBytes = await stream.ReadAsync(buffer, totalBytes, totalRemainingBytes, cancellationTolen);

                cancellationTolen.ThrowIfCancellationRequested();

                totalBytes += readBytes;
                totalRemainingBytes -= readBytes;
            }
        }
    }
}
