﻿using System;
using System.Collections.Generic;
using System.Threading;
using Nintendo.InGameEditing.Utilities;

namespace Nintendo.InGameEditing.Communication
{
    /// <summary>
    /// Host - TargetManager 通信のための ConnectionManager のラッパー
    /// </summary>
    internal sealed class HtcsConnectionManager
    {
        private static readonly IReadOnlyList<TargetInfo> EmptyTargetInfos = new TargetInfo[0];
        private static readonly Lazy<HtcsConnectionManager> instance = new Lazy<HtcsConnectionManager>(() => new HtcsConnectionManager());

        public static HtcsConnectionManager Instance => instance.Value;

        private readonly WeakDelegateManager weakDelegateManager = new WeakDelegateManager();
        private ConnectionManager manager;

        public event EventHandler<ConnectionInfoUpdatedEventArgs> ConnectionChanged
        {
            add { weakDelegateManager.Add(value); }
            remove { weakDelegateManager.Remove(value); }
        }

        private HtcsConnectionManager() { }

        /// <summary>
        /// NintendoTargetManager に接続しているかどうかを取得します。
        /// </summary>
        public bool IsConnected => manager?.IsAvailable() ?? false;

         /// <summary>
         /// ターゲット情報を取得します。
         /// </summary>
        public IReadOnlyList<TargetInfo> TargetInfos => manager?.TargetInfos ?? EmptyTargetInfos;

        /// <summary>
        /// NintendoTargetManager への接続を開始します。
        /// </summary>
        public void Start()
        {
            if (manager != null) { return; }

            var newManager = new ConnectionManager();
            newManager.ConnectionInfoUpdated += Manager_ConnectionInfoUpdated;
            newManager.Start();

            var oldManager = Interlocked.CompareExchange(ref manager, newManager, null);
            if (oldManager != null)
            {
                newManager.Dispose();
            }
        }

        /// <summary>
        /// NintendoTargetManager への接続を開始した状態かどうかを取得します。
        /// </summary>
        public bool IsStarted => manager != null;

        /// <summary>
        /// NintendoTargetManager から切断します。
        /// </summary>
        public void Stop()
        {
            var oldManager = Interlocked.Exchange(ref manager, null);
            if (oldManager == null) { return; }

            // manager.Dispose が ConnectionInfoUpdated を呼ぶ可能性がある = Dispose が例外を投げる可能性がある = 危ないので先に抜く。
            oldManager.ConnectionInfoUpdated -= Manager_ConnectionInfoUpdated;

            try { oldManager.Dispose(); }
            catch { /* ignored */ }
        }

        private void Manager_ConnectionInfoUpdated(object sender, ConnectionInfoUpdatedEventArgs e) => RaiseConnectionChanged(e);

        private void RaiseConnectionChanged(ConnectionInfoUpdatedEventArgs e) => weakDelegateManager.Invoke(this, e);

    }
}
