﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Xml.Linq;

namespace Nintendo.InGameEditing.Communication.Htcs
{
    /// <summary>
    /// ターゲットリストです。
    /// </summary>
    internal class TargetList : IEnumerable<Target>, IEquatable<TargetList>
    {
        /// <summary>
        /// XML上での要素名です。
        /// </summary>
        internal const string XElementName = "TargetList";

        /// <summary>
        /// ターゲット情報のリストです。
        /// </summary>
        private readonly List<Target> items = new List<Target>();

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TargetList()
        {
            // 何もしない
        }

        /// <summary>
        /// コピー用コンストラクタ
        /// </summary>
        internal TargetList(IEnumerable<Target> targetEnum)
        {
            items.Clear();
            items.AddRange(targetEnum);
        }

        /// <summary>
        /// パース用コンストラクタ
        /// </summary>
        internal TargetList(XElement xElement)
        {
            var elements = xElement?.Elements(Target.XElementName);
            if (elements == null)
            {
                return;
            }

            items.AddRange(elements.Select(x => new Target(x)));
        }

        /// <summary>
        /// インスタンスが同じ内容を保持しているかを比較します。
        /// </summary>
        /// <param name="other">もう1つのインスタンス</param>
        /// <returns>一致していればtrue,そうでなければfalse.</returns>
        public bool Equals(TargetList other)
        {
            if (this.items.Count != other.items.Count)
            {
                return false;
            }

            return !this.items.Where((t, i) => !t.Equals(other.items[i])).Any();
        }

        #region IEnumerable<Target> メンバー

        /// <summary>
        /// Enumeratorを取得します。
        /// </summary>
        public IEnumerator<Target> GetEnumerator()
        {
            return items.GetEnumerator();
        }

        #endregion

        #region IEnumerable メンバー

        /// <summary>
        /// Enumeratorを取得します。
        /// </summary>
        System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
        {
            return items.GetEnumerator();
        }

        #endregion

        /// <summary>
        /// ターゲット情報を追加します。
        /// </summary>
        internal void Add(Target item)
        {
            items.Add(item);
        }

        /// <summary>
        /// ターゲット情報を削除します。
        /// </summary>
        internal void Remove(Target target)
        {
            items.RemoveAt(items.FindIndex(s => s == target));
        }

        /// <summary>
        /// XMLに変換します。
        /// </summary>
        internal XElement ToXElement()
        {
            return new XElement(XElementName,
                items.Select(s => s.ToXElement()));
        }
    }
}
