﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Threading;
using System.Xml.Linq;
using Nintendo.Tm;

namespace Nintendo.InGameEditing.Communication.Htcs
{
    /// <summary>
    /// Htcs標準のコネクタです。
    /// </summary>
    internal class HtcConnector : TargetConnector
    {
        /// <summary>
        /// HtcDaemonとの通信に使用するポート番号のデフォルト値です。
        /// </summary>
        private const int DefaultHostDaemonPort = 8003;

        /// <summary>
        /// HtcDaemonからの処理結果通知をプールするキューです。
        /// </summary>
        private readonly Queue<HtcDaemonAckEvent> AckQueue = new Queue<HtcDaemonAckEvent>();

        /// <summary>
        /// HtcDaemonからのメッセージを読み取るリーダーです。
        /// </summary>
        private StreamReader reader = null;

        private static int GetHtcsControlPort()
        {
            try
            {
                return TargetManager.GetHtcsControlPort();
            }
            catch (Exception e)
            {
                Logger.WriteLine(e.Message);
                return DefaultHostDaemonPort;
            }
        }

        /// <summary>
        /// デストラクタ
        /// </summary>
        ~HtcConnector()
        {
            this.Dispose(false);
        }

        /// <summary>
        /// ディスポーザー
        /// </summary>
        public override void Dispose()
        {
            if (this.reader != null)
            {
                this.reader.Close();
                this.reader = null;
            }

            base.Dispose();
        }

        protected override int Port => GetHtcsControlPort();

        /// <summary>
        /// ホスト側をサーバにするためにポートマッピング情報をデーモンに登録します。
        /// </summary>
        public uint RegisterPort(PortMapItem portMapItem)
        {
            using (HtcDaemonAckEvent waitAckEvent = new HtcDaemonAckEvent())
            {
                lock (AckQueue)
                {
                    using (var sw = new StreamWriter(this.Client.GetStream(), new UTF8Encoding(false), 1024, true))
                    {
                        var command = new RegisterPortMapCommand(portMapItem, "Auto");
                        string xml = command.ToXElement().ToString(SaveOptions.DisableFormatting);
                        sw.WriteLine(xml);
                    }
                    AckQueue.Enqueue(waitAckEvent);
                }
                waitAckEvent.Wait();
                return waitAckEvent.GetResult().GetValue();
            }
        }

        /// <summary>
        /// ホスト側をサーバにするためにポートマッピング情報をデーモンに登録します。
        /// </summary>
        public uint RegisterPort(HtcsPeerName htcsPeerName, string htcsPortName, System.Net.IPEndPoint endPoint)
        {
            HtcsPortDescriptor htcsPortDesc = new HtcsPortDescriptor(htcsPeerName, htcsPortName);
            PortMapItem portMapping = new PortMapItem(htcsPortDesc, endPoint);
            return RegisterPort(portMapping);
        }

        /// <summary>
        /// デーモンに登録したポートマッピング情報を解除します。
        /// </summary>
        public uint UnregisterPort(HtcsPortDescriptor htcsPortDescriptor)
        {
            using (HtcDaemonAckEvent waitAckEvent = new HtcDaemonAckEvent())
            {
                lock (AckQueue)
                {
                    using (var sw = new StreamWriter(this.Client.GetStream(), new UTF8Encoding(false), 1024, true))
                    {
                        var command = new UnregisterPortMapCommand(htcsPortDescriptor, "Auto");
                        string xml = command.ToXElement().ToString(SaveOptions.DisableFormatting);
                        sw.WriteLine(xml);
                    }
                    AckQueue.Enqueue(waitAckEvent);
                }
                waitAckEvent.Wait();
                return waitAckEvent.GetResult().GetValue();
            }
        }

        /// <summary>
        /// デーモンに登録したポートマッピング情報を解除します。
        /// </summary>
        public uint UnregisterPort(HtcsPeerName htcsPeerName, string htcsPortName)
        {
            HtcsPortDescriptor htcsPortDesc = new HtcsPortDescriptor(htcsPeerName, htcsPortName);
            return UnregisterPort(htcsPortDesc);
        }

        /// <summary>
        /// HtcDaemonから通知された接続情報を処理します。
        /// </summary>
        /// <returns></returns>
        protected override bool ReceiveInfo()
        {
            if (this.reader == null && this.Client != null)
            {
                this.reader = new StreamReader(this.Client.GetStream(), new UTF8Encoding(false), false, 1024, true);
            }

            bool result = this.reader != null;

            try
            {
                var line = result ? this.reader.ReadLine() : null;
                if (line == null)
                {
                    result = false;
                }
                else
                {
                    var element = XElement.Parse(line);
                    if (element.Name == HtcsInfo.XElementName)
                    {
                        result = ReceivePortMap(element);
                    }
                    else if (element.Name == CommandResult.XElementName)
                    {
                        result = ReceiveAck(element);
                    }
                }
            }
            catch (EndOfStreamException)
            {
                Logger.WriteLine("Disconnected HtcDaemon.");
                result = false;
            }
            catch (IOException)
            {
                Logger.WriteLine("Disconnected HtcDaemon.");
                result = false;
            }
            catch (ObjectDisposedException)
            {
                Logger.WriteLine("Disconnected HtcDaemon.");
                result = false;
            }

            if (!result && this.reader != null)
            {
                // 接続終了時のreaderのクリーンアップ
                this.reader.Close();
                this.reader = null;
            }

            return result;
        }

        /// <summary>
        /// ポートマッピング情報を更新し、イベントを発生させます。
        /// </summary>
        private bool ReceivePortMap(XElement element)
        {
            var htcsInfo = new HtcsInfo(element);
            this.TargetList = htcsInfo.TargetList;
            this.PortMap = htcsInfo.PortMap;

            this.RaiseConnectionInfoUpdated();

            return true;
        }

        /// <summary>
        /// HtcDaemonから通知をキューに詰みます。
        /// </summary>
        private bool ReceiveAck(XElement element)
        {
            HtcDaemonAckEvent signal;
            lock (AckQueue)
            {
                signal = AckQueue.Dequeue();
            }

            signal.SetResult(new CommandResult(element));
            signal.Signal();

            return true;
        }
    }

    internal class HtcDaemonAckEvent : IDisposable
    {
        private bool disposed;
        private readonly AutoResetEvent signal = new AutoResetEvent(false);
        private CommandResult result;

        ~HtcDaemonAckEvent()
        {
            this.Dispose(false);
        }

        public void Dispose()
        {
            this.Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (!this.disposed)
            {
                if (disposing)
                {
                    // ここでマネージドリソースをDisposeする
                    signal.Dispose();
                }

                // ここでアンマネージドリソースを解放する
                this.disposed = true;
            }
        }

        public void Wait()
        {
            signal.WaitOne();
        }

        public void Signal()
        {
            signal.Set();
        }

        public void SetResult(CommandResult res)
        {
            result = res;
        }

        public CommandResult GetResult()
        {
            return result;
        }
    }
}
