﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Net;
using System.Text;
using Nintendo.InGameEditing.Communication.Htcs;

namespace Nintendo.InGameEditing.Communication.Hio
{
    /// <summary>
    /// HIOホストブリッジへのコネクタです。
    /// </summary>
    public class HioConnector : TargetConnector
    {
        /// <summary>
        /// ホストブリッジとの通信に使用するポート番号です。
        /// </summary>
        private const int HioBridgePort = 6003;

        /// <summary>
        /// メッセージ連結用のストリームです。
        /// </summary>
        private MemoryStream memoryStream = null;

        /// <summary>
        /// デストラクタ
        /// </summary>
        ~HioConnector()
        {
            this.Dispose(false);
        }

        /// <summary>
        /// ディスポーザー
        /// </summary>
        public override void Dispose()
        {
            if (this.memoryStream != null)
            {
                this.memoryStream.Close();
                this.memoryStream = null;
            }

            base.Dispose();
        }

        protected override int Port => HioBridgePort;

        /// <summary>
        /// ホストブリッジからの受信内容を処理します。
        /// </summary>
        protected override bool ReceiveInfo()
        {
            bool result = this.Client != null;
            if (result)
            {
                if (this.memoryStream == null)
                {
                    this.memoryStream = new MemoryStream();
                }
                else
                {
                    this.memoryStream.SetLength(0);
                }

                try
                {
                    byte[] buffer = new byte[256];

                    // 読み込み処理.
                    do
                    {
                        int numBuytesRead = this.Client.GetStream().Read(buffer, 0, buffer.Length);
                        if (numBuytesRead > 0)
                        {
                            this.memoryStream.Write(buffer, 0, numBuytesRead);
                        }
                    }
                    while (this.Client.Available > 0);

                    // 文字列にする.
                    string info = Encoding.ASCII.GetString(this.memoryStream.ToArray());

                    // 各チャンネル情報ごとに分ける.
                    this.MakeHtcsPamameters(SplitChannelInfos(info));
                }
                catch (EndOfStreamException)
                {
                    Logger.WriteLine("Disconnected HIO Bridge.");
                    result = false;
                }
                catch (IOException)
                {
                    Logger.WriteLine("Disconnected HIO Bridge.");
                    result = false;
                }
                catch (ObjectDisposedException)
                {
                    Logger.WriteLine("Disconnected HIO Bridge.");
                    result = false;
                }
            }

            if (!result && this.memoryStream != null)
            {
                // 接続終了時のmemoryStreamのクリーンアップ
                this.memoryStream.Close();
                this.memoryStream = null;
            }

            return result;
        }

        /// <summary>
        /// パースしたデータからHtcs互換のパラメータインスタンスを生成し、マネージャに通知します。
        /// </summary>
        private void MakeHtcsPamameters(IList<string> channels)
        {
            this.ClearConnections();
            this.AddTargetInfo("HIO", "Cafe");

            // 各チャンネル情報を調べる.
            foreach (string channel in channels)
            {
                var tokens = channel.Split(new[] { ':' }, 2);
                if (tokens.Length != 2)
                {
                    continue;
                }

                string channelName = tokens[0];
                int portNo;

                // パースしてみる.
                if (int.TryParse(tokens[1], out portNo) == false)
                {
                    // 処理をスキップ.
                    continue;
                }

                // HTCS互換のデータ形式に変換してイベントをトリガーする
                if (channelName[0] == '+' || channelName[0] == '-')
                {
                    this.AddPortInfo(new HtcsPortDescriptor(
                        new HtcsPeerName("Cafe"), channelName.Substring(1)),
                        new IPEndPoint(IPAddress.Loopback, portNo));
                }
            }

            this.RaiseConnectionInfoUpdated();
        }

        /// <summary>
        /// チャンネル情報を解析します。
        /// </summary>
        private static IList<string> SplitChannelInfos(string msg)
        {
            List<string> list = new List<string>();

            for (int idx = 0; idx < msg.Length;)
            {
                if (msg[idx] == '+' || msg[idx] == '-')
                {
                    int j = idx + 1;

                    for (; j < msg.Length; ++j)
                    {
                        if (msg[j] == '+' || msg[j] == '-')
                        {
                            break;
                        }
                    }

                    list.Add(msg.Substring(idx, j - idx));
                    idx = j;
                }
                else
                {
                    idx++;
                }
            }

            return list;
        }
    }
}
