﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using Nintendo.Foundation.ComponentModel;
using Nintendo.Foundation.Contracts;
using System;
using System.IO;
using System.Runtime.InteropServices;
using System.Text;

namespace Nintendo.Atk.Binary
{
    /// <summary>
    /// bfwav データの読み込み機能を提供します。
    /// </summary>
    public sealed class BfwavReader : DisposableObject
    {
        private Stream stream;
        private bool leaveOpenStream;
        private IntPtr bfwavHandle;
        private GCHandle bfwavDataGCHandle;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="stream">bfwav データを読み込むストリームを指定します。</param>
        public BfwavReader(Stream stream)
            : this(stream, false)
        {
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="stream">bfwav データを読み込むストリームを指定します。</param>
        /// <param name="leaveOpen">BfwavReader 破棄時にストリームを開いたままにする場合は true を指定します。</param>
        public BfwavReader(Stream stream, bool leaveOpen)
        {
            Ensure.Argument.NotNull(stream);

            this.OpenStream(stream);

            this.stream = stream;
            this.leaveOpenStream = leaveOpen;
        }

        /// <summary>
        /// 波形バイナリ情報を読み込みます。
        /// </summary>
        public WaveBinaryInfo ReadInfo()
        {
            Ensure.Operation.True(this.bfwavHandle != IntPtr.Zero);

            var info = new AtkNativeWaveBinaryInfo();
            var getResult = AtkNative.GetBfwavWaveBinaryInfo(this.bfwavHandle, ref info);

            if (getResult.IsFailed())
            {
                throw new InvalidOperationException(getResult.ToString());
            }

            return info.ToWaveBinaryInfo();
        }

        /// <summary>
        /// 指定チャンネルをデコードし、PCM サンプル列を返します。
        /// </summary>
        /// <param name="channel">デコードするチャンネルを指定します。</param>
        /// <returns>デコード後の PCM サンプル配列を返します。</returns>
        public short[] DecodeAll(int channel)
        {
            Ensure.Operation.True(this.bfwavHandle != IntPtr.Zero);

            var bufferSize = AtkNative.GetRequiredBufferSizeForDecodeBfwav(this.bfwavHandle);

            var decodedSamples = new short[bufferSize / sizeof(short)];

            GCHandle decodedSamplesGCHandle = GCHandle.Alloc(decodedSamples, GCHandleType.Pinned);

            try
            {
                var decodeResult = AtkNative.DecodeBfwav(
                    this.bfwavHandle,
                    channel,
                    decodedSamplesGCHandle.AddrOfPinnedObject(),
                    bufferSize);

                if (decodeResult.IsFailed())
                {
                    throw new InvalidOperationException(decodeResult.ToString());
                }
            }
            finally
            {
                decodedSamplesGCHandle.Free();
            }

            return decodedSamples;
        }

        /// <summary>
        /// マネージドインスタンスを破棄します。
        /// </summary>
        protected sealed override void DisposeManagedInstance()
        {
            if (this.stream != null && !this.leaveOpenStream)
            {
                this.stream.Close();
            }
        }

        /// <summary>
        /// アンマネージドインスタンスを破棄します。
        /// </summary>
        protected sealed override void DisposeUnmanagedInstance()
        {
            AtkNative.CloseBfwav(this.bfwavHandle);

            if (this.bfwavDataGCHandle.IsAllocated)
            {
                this.bfwavDataGCHandle.Free();
            }
        }

        private void OpenStream(Stream stream)
        {
            Assertion.Argument.NotNull(stream);

            using (var reader = new BinaryReader(stream, Encoding.ASCII, true))
            {
                var fileData = reader.ReadBytes((int)stream.Length);

                IntPtr handle = IntPtr.Zero;
                GCHandle fileDataGCHandle = GCHandle.Alloc(fileData, GCHandleType.Pinned);

                try
                {
                    var openResult = AtkNative.OpenBfwav(fileDataGCHandle.AddrOfPinnedObject(), out handle);

                    if (openResult.IsFailed())
                    {
                        throw new InvalidOperationException(openResult.ToString());
                    }

                    this.bfwavHandle = handle;
                    this.bfwavDataGCHandle = fileDataGCHandle;
                }
                catch
                {
                    if (handle != IntPtr.Zero)
                    {
                        AtkNative.CloseBfwav(handle);
                    }

                    if (fileDataGCHandle.IsAllocated)
                    {
                        fileDataGCHandle.Free();
                    }

                    throw;
                }
            }
        }
    }
}
