﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/mem/mem_StandardAllocator.h>
#include <nn/atk.h>
#include <nn/atk/viewer/atk_SoundEdit.h>
#include "WrappedSoundHandle.h"


//  HTCS 通信を有効にするかどうかのマクロ
#if defined( NN_SDK_BUILD_DEBUG ) || defined( NN_SDK_BUILD_DEVELOP )
#define     ENABLE_HTCS_CONNECTION
#endif


//! @brief  SoundArchive に関するインスタンスを保持します
class SoundArchiveContext
{
public:
    //! @brief  同時に扱える SoundHandle の数
    static const int SoundHandleCountMax = 4;

public:
    //! @brief  初期化します
    void Initialize(void* pMemory, size_t memorySize) NN_NOEXCEPT;
    //! @brief  終了処理をします
    void Finalize() NN_NOEXCEPT;
    //! @brief  更新処理をします
    void Update() NN_NOEXCEPT;

    //! @brief  サウンドアーカイブ内のサウンドの数を取得します
    uint32_t GetSoundCount() const NN_NOEXCEPT;
    //! @brief  itemId のサウンドのラベルを取得します
    //          ラベルの取得に失敗したときは nullptr を返します
    const char* GetSoundLabel(nn::atk::SoundArchive::ItemId itemId) const NN_NOEXCEPT;

    //! @biref  handleIndex 番目の SoundHandle を取得します
    WrappedSoundHandle& GetSoundHandle(int handleIndex) NN_NOEXCEPT;
    //! @biref  handleIndex 番目の SoundHandle を取得します
    const WrappedSoundHandle& GetSoundHandle(int handleIndex) const NN_NOEXCEPT;

    //! @brief  SoundArchivePlayer を取得します
    //          SoundArchive が開かれていないときは nullptr を返します
    nn::atk::SoundArchivePlayer* GetSoundArchivePlayer() NN_NOEXCEPT;

    //! @biref  接続されているかを取得します
    bool IsConnected() const NN_NOEXCEPT;

    //! @brief  ファイルを開きます
    void Open(const char* filePath) NN_NOEXCEPT;
    //! @brief  itemId のサウンドを読み込みます
    void LoadSound(nn::atk::SoundArchive::ItemId itemId) NN_NOEXCEPT;

private:
    //  LoadSoundCallBack の userParam に設定する構造体です
    struct LoadSoundCallBackParameter
    {
        SoundArchiveContext* pSoundArchiveContext;
        WrappedSoundHandle* pWrappedSoundHandle;
    };

private:
    nn::atk::SoundHeap          m_SoundHeap;
    nn::atk::FsSoundArchive     m_SoundArchive;
    nn::atk::SoundDataManager   m_SoundDataManager;
    nn::atk::SoundArchivePlayer m_SoundArchivePlayer;

    nn::audio::MemoryPoolType  m_MemoryPoolForStreamBuffer;
    nn::mem::StandardAllocator m_Allocator;
    void* m_pMemoryForSoundHeap;
    void* m_pMemoryForInfoBlock;
    void* m_pMemoryForLabelBlock;
    void* m_pMemoryForSoudDataManager;
    void* m_pMemoryForSoundArchivePlayer;
    void* m_pMemoryForStreamBuffer;
    void* m_pMemoryForStreamCacheBuffer;

    bool m_IsFileOpened;

    WrappedSoundHandle m_WrappedSoundHandle[SoundHandleCountMax];

#if defined( ENABLE_HTCS_CONNECTION )
    NN_STATIC_ASSERT( nn::atk::viewer::SoundObjectController::SoundControllerCount == SoundHandleCountMax );

    LoadSoundCallBackParameter m_LoadSoundCallBackParameter[SoundHandleCountMax];
    nn::atk::viewer::SoundEdit m_SoundEdit;
    bool                       m_IsConnected;
    void*                      m_pMemoryForSoundEdit;
#else
    nn::atk::SoundHandle       m_SoundHandle[SoundHandleCountMax];
#endif

private:
    //! @brief  ファイルを閉じます
    void Close() NN_NOEXCEPT;

#if defined( ENABLE_HTCS_CONNECTION )
    //! @brief  soundId のサウンドを読み込みます。この関数は SoundController のコールバックとして機能します
    static void LoadSoundCallBack(nn::atk::viewer::SoundController& target, nn::atk::SoundArchive::ItemId soundId, void* userParam) NN_NOEXCEPT;
#endif
};
