﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>
#include "gfxutil/GfxContext.h"
#include "gfxutil/Label.h"

namespace
{
    //  ラベルのデフォルトの位置
    const float LabelDefaultPositionX = 0.0f;
    const float LabelDefaultPositionY = 0.0f;
    //  ラベルのデフォルトのスケール
    const float LabelDefaultScaleX    = 1.0f;
    const float LabelDefaultScaleY    = 1.0f;
    //  ラベルのデフォルトの枠の大きさ
    const float LabelDefaultEdgeSize  = 0.0f;
    //  ラベルのデフォルトのテキスト
    const char* LabelDefaultText = "Label";
    //  ラベルのデフォルトのテキスト配置
    const gfxutil::Label::DrawAlign LabelDefaultDrawAlign = gfxutil::Label::DrawAlign_Right;
    //  ラベルのデフォルトの文字の色
    const nn::util::Unorm8x4 LabelDefaultColor    = GetUnorm8x4( 192, 192, 192, 255 );
    //  ラベルのデフォルトの背景の色
    const nn::util::Uint8x4 LabelDefaultBackColor = GetUint8x4(   0,   0,   0,   0 );
    //  ラベルのデフォルトの枠の色
    const nn::util::Uint8x4 LabelDefaultEdgeColor = GetUint8x4(   0,   0,   0,   0 );
}

namespace gfxutil
{

NN_DEFINE_STATIC_CONSTANT( const int Label::TextLengthMax );

//  コンストラクタです
Label::Label() NN_NOEXCEPT
    : m_DrawSize( GetFloat2( 0, 0 ) )  //  CalculateDrawSize で計算されるため、ここでは仮の値を代入
    , m_IsWidthFixed( false )
{
    SetText( LabelDefaultText );
    SetPosition( LabelDefaultPositionX, LabelDefaultPositionY );
    SetScale( LabelDefaultScaleX, LabelDefaultScaleY );
    SetEdgeSize( LabelDefaultEdgeSize );
    SetDrawAlign( LabelDefaultDrawAlign );
    SetColor( LabelDefaultColor );
    SetBackColor( LabelDefaultBackColor );
    SetEdgeColor( LabelDefaultEdgeColor );
}
//  描画します
void Label::Draw(GfxContext& gfxContext) NN_NOEXCEPT
{
#if  defined( ATKPLAYER_BUILD_CONFIG_ENABLE_GFX )
    gfxutil::FontRenderer& fontRenderer = gfxContext.GetFontRenderer();

    //  サイズの計算
    CalculateDrawSize( fontRenderer );

    //  テキストの位置を計算
    nn::util::Float2 position = m_Position;
    switch( m_DrawAlign )
    {
    case DrawAlign_Right:
        {
        }
        break;

    case DrawAlign_Center:
        {
            position.x -= 0.5f * m_DrawSize.x;
        }
        break;

    case DrawAlign_Left:
        {
            position.x -= m_DrawSize.x;
        }
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }

    //  枠の描画
    if( m_EdgeSize >= 0.0f )
    {
        gfxContext.DrawQuad(
            nn::util::Vector3f( position.x - m_EdgeSize, position.y - m_EdgeSize, 0.0f ),
            GetFloat2( m_DrawSize.x + m_EdgeSize, m_EdgeSize ),
            m_EdgeColor
        );
        gfxContext.DrawQuad(
            nn::util::Vector3f( position.x - m_EdgeSize, position.y, 0.0f ),
            GetFloat2( m_EdgeSize, m_DrawSize.y + m_EdgeSize ),
            m_EdgeColor
        );
        gfxContext.DrawQuad(
            nn::util::Vector3f( position.x + m_DrawSize.x, position.y - m_EdgeSize, 0.0f ),
            GetFloat2( m_EdgeSize, m_DrawSize.y + m_EdgeSize ),
            m_EdgeColor
        );
        gfxContext.DrawQuad(
            nn::util::Vector3f( position.x, position.y + m_DrawSize.y, 0.0f ),
            GetFloat2( m_DrawSize.x + m_EdgeSize, m_EdgeSize ),
            m_EdgeColor
        );
    }

    //  背景の描画
    gfxContext.DrawQuad(
        nn::util::Vector3f( position.x, position.y, 0.0f ),
        m_DrawSize,
        m_BackColor
    );

    //  テキストの描画
    fontRenderer.SetPosition( position.x, position.y );
    fontRenderer.SetColor( m_Color );
    fontRenderer.SetScale( m_TextScale.x, m_TextScale.y );
    fontRenderer.Print( m_Text );
#else
    NN_UNUSED( gfxContext );
#endif  //  ATKPLAYER_BUILD_CONFIG_ENABLE_GFX
}

//  テキストを取得します
const char* Label::GetText() const NN_NOEXCEPT
{
    return m_Text;
}
//  位置を取得します
const nn::util::Float2& Label::GetPosition() const NN_NOEXCEPT
{
    return m_Position;
}
//  描画されるサイズを計算します
const nn::util::Float2 Label::CalculateDrawSize(FontRenderer& fontRenderer) NN_NOEXCEPT
{
#if  defined( ATKPLAYER_BUILD_CONFIG_ENABLE_GFX )
    if( m_IsSizeChanged )
    {
        m_IsSizeChanged = false;

        fontRenderer.SetScale( m_TextScale.x, m_TextScale.y );

        if( !m_IsWidthFixed )
        {
            m_DrawSize.x = fontRenderer.CalculateWidth( m_Text );
        }

        m_DrawSize.y = fontRenderer.CalculateHeight( m_Text );
    }
#else
    NN_UNUSED( fontRenderer );
#endif
    return m_DrawSize;
}

//  テキストを設定します
void Label::SetText(const char* text, ...) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( text );

    std::va_list args;
    va_start( args, text );
    nn::util::VSNPrintf( m_Text, TextLengthMax, text, args );
    va_end( args );

    m_IsSizeChanged = true;
}
//  テキストの描画位置を設定します
void Label::SetDrawAlign(DrawAlign align) NN_NOEXCEPT
{
    m_DrawAlign = align;
}
//  テキストのスケールを設定します
void Label::SetScale(float scaleX, float scaleY) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_GREATER_EQUAL( scaleX, 0.0f );
    NN_ABORT_UNLESS_GREATER_EQUAL( scaleY, 0.0f );

    m_TextScale.x = scaleX;
    m_TextScale.y = scaleY;

    m_IsSizeChanged = true;
}
//  描画される幅を設定します
void Label::SetDrawWidth(float width) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_GREATER_EQUAL( width, 0 );

    m_DrawSize.x = width;
    m_IsWidthFixed = true;
}

//  位置を設定します
void Label::SetPosition(float x, float y) NN_NOEXCEPT
{
    m_Position.x = x;
    m_Position.y = y;
}
//  枠の大きさを設定します
void Label::SetEdgeSize(float size) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_GREATER_EQUAL( size, 0.0f );
    m_EdgeSize = size;
}

//  テキストの色を設定します
void Label::SetColor(const nn::util::Unorm8x4& color) NN_NOEXCEPT
{
    m_Color = color;
}
//  背景の色を設定します
void Label::SetBackColor(const nn::util::Uint8x4& color) NN_NOEXCEPT
{
    m_BackColor = color;
}
//  枠の色を設定します
void Label::SetEdgeColor(const nn::util::Uint8x4& color) NN_NOEXCEPT
{
    m_EdgeColor = color;
}

}
